/*
 * Decompiled with CFR 0.152.
 */
package io.trino.plugin.iceberg.procedure;

import com.google.common.base.Enums;
import com.google.common.base.Verify;
import com.google.common.collect.ImmutableList;
import com.google.common.collect.ImmutableMap;
import com.google.common.collect.Streams;
import com.google.inject.Inject;
import com.google.inject.Provider;
import io.airlift.log.Logger;
import io.airlift.slice.Slices;
import io.trino.filesystem.FileEntry;
import io.trino.filesystem.FileIterator;
import io.trino.filesystem.Location;
import io.trino.filesystem.TrinoFileSystem;
import io.trino.filesystem.TrinoFileSystemFactory;
import io.trino.filesystem.TrinoInputFile;
import io.trino.plugin.hive.HiveMetadata;
import io.trino.plugin.hive.HiveStorageFormat;
import io.trino.plugin.hive.metastore.Column;
import io.trino.plugin.hive.metastore.HiveMetastore;
import io.trino.plugin.hive.metastore.HiveMetastoreFactory;
import io.trino.plugin.hive.metastore.MetastoreUtil;
import io.trino.plugin.hive.metastore.Partition;
import io.trino.plugin.hive.metastore.PrincipalPrivileges;
import io.trino.plugin.hive.metastore.RawHiveMetastoreFactory;
import io.trino.plugin.hive.metastore.Storage;
import io.trino.plugin.hive.metastore.StorageFormat;
import io.trino.plugin.hive.metastore.Table;
import io.trino.plugin.hive.util.HiveUtil;
import io.trino.plugin.iceberg.IcebergConfig;
import io.trino.plugin.iceberg.IcebergErrorCode;
import io.trino.plugin.iceberg.IcebergFileFormat;
import io.trino.plugin.iceberg.IcebergSecurityConfig;
import io.trino.plugin.iceberg.PartitionData;
import io.trino.plugin.iceberg.PartitionFields;
import io.trino.plugin.iceberg.TypeConverter;
import io.trino.plugin.iceberg.catalog.TrinoCatalog;
import io.trino.plugin.iceberg.catalog.TrinoCatalogFactory;
import io.trino.plugin.iceberg.fileio.ForwardingInputFile;
import io.trino.spi.ErrorCodeSupplier;
import io.trino.spi.StandardErrorCode;
import io.trino.spi.TrinoException;
import io.trino.spi.classloader.ThreadContextClassLoader;
import io.trino.spi.connector.ConnectorSession;
import io.trino.spi.connector.SchemaTableName;
import io.trino.spi.connector.TableNotFoundException;
import io.trino.spi.predicate.TupleDomain;
import io.trino.spi.procedure.Procedure;
import io.trino.spi.type.ArrayType;
import io.trino.spi.type.MapType;
import io.trino.spi.type.RowType;
import io.trino.spi.type.SmallintType;
import io.trino.spi.type.TinyintType;
import io.trino.spi.type.TypeManager;
import io.trino.spi.type.VarcharType;
import java.io.IOException;
import java.lang.invoke.MethodHandle;
import java.lang.invoke.MethodHandles;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Locale;
import java.util.Map;
import java.util.Objects;
import java.util.Optional;
import java.util.concurrent.atomic.AtomicInteger;
import java.util.stream.Stream;
import org.apache.iceberg.AppendFiles;
import org.apache.iceberg.DataFile;
import org.apache.iceberg.DataFiles;
import org.apache.iceberg.Metrics;
import org.apache.iceberg.MetricsConfig;
import org.apache.iceberg.PartitionSpec;
import org.apache.iceberg.Schema;
import org.apache.iceberg.SortOrder;
import org.apache.iceberg.StructLike;
import org.apache.iceberg.Transaction;
import org.apache.iceberg.avro.Avro;
import org.apache.iceberg.io.InputFile;
import org.apache.iceberg.mapping.MappingUtil;
import org.apache.iceberg.mapping.NameMapping;
import org.apache.iceberg.mapping.NameMappingParser;
import org.apache.iceberg.orc.OrcMetrics;
import org.apache.iceberg.parquet.TrinoParquetUtil;
import org.apache.iceberg.types.Type;
import org.apache.iceberg.types.TypeUtil;
import org.apache.iceberg.types.Types;

public class MigrateProcedure
implements Provider<Procedure> {
    private static final Logger log = Logger.get(MigrateProcedure.class);
    public static final String PROVIDER_PROPERTY_KEY = "provider";
    public static final String PROVIDER_PROPERTY_VALUE = "iceberg";
    private static final MetricsConfig METRICS_CONFIG = MetricsConfig.getDefault();
    private final TrinoCatalogFactory catalogFactory;
    private final HiveMetastoreFactory metastoreFactory;
    private final TrinoFileSystemFactory fileSystemFactory;
    private final TypeManager typeManager;
    private final int formatVersion;
    private final boolean isUsingSystemSecurity;
    private static final MethodHandle MIGRATE;

    @Inject
    public MigrateProcedure(TrinoCatalogFactory catalogFactory, @RawHiveMetastoreFactory HiveMetastoreFactory metastoreFactory, TrinoFileSystemFactory fileSystemFactory, TypeManager typeManager, IcebergConfig icebergConfig, IcebergSecurityConfig securityConfig) {
        this.catalogFactory = Objects.requireNonNull(catalogFactory, "catalogFactory is null");
        this.metastoreFactory = Objects.requireNonNull(metastoreFactory, "metastoreFactory is null");
        this.fileSystemFactory = Objects.requireNonNull(fileSystemFactory, "fileSystemFactory is null");
        this.typeManager = Objects.requireNonNull(typeManager, "typeManager is null");
        this.formatVersion = icebergConfig.getFormatVersion();
        this.isUsingSystemSecurity = securityConfig.getSecuritySystem() == IcebergSecurityConfig.IcebergSecurity.SYSTEM;
    }

    public Procedure get() {
        return new Procedure("system", "migrate", (List)ImmutableList.of((Object)new Procedure.Argument("SCHEMA_NAME", (io.trino.spi.type.Type)VarcharType.VARCHAR), (Object)new Procedure.Argument("TABLE_NAME", (io.trino.spi.type.Type)VarcharType.VARCHAR), (Object)new Procedure.Argument("RECURSIVE_DIRECTORY", (io.trino.spi.type.Type)VarcharType.VARCHAR, false, (Object)Slices.utf8Slice((String)"fail"))), MIGRATE.bindTo(this));
    }

    public void migrate(ConnectorSession session, String schemaName, String tableName, String recursiveDirectory) {
        try (ThreadContextClassLoader ignored = new ThreadContextClassLoader(this.getClass().getClassLoader());){
            this.doMigrate(session, schemaName, tableName, recursiveDirectory);
        }
    }

    public void doMigrate(ConnectorSession session, String schemaName, String tableName, String recursiveDirectory) {
        SchemaTableName sourceTableName = new SchemaTableName(schemaName, tableName);
        TrinoCatalog catalog = this.catalogFactory.create(session.getIdentity());
        HiveMetastore metastore = this.metastoreFactory.createMetastore(Optional.of(session.getIdentity()));
        RecursiveDirectory recursive = (RecursiveDirectory)((Object)Enums.getIfPresent(RecursiveDirectory.class, (String)recursiveDirectory.toUpperCase(Locale.ENGLISH)).toJavaUtil().orElseThrow(() -> new TrinoException((ErrorCodeSupplier)StandardErrorCode.INVALID_PROCEDURE_ARGUMENT, "Invalid recursive_directory: " + recursiveDirectory)));
        Table hiveTable = (Table)metastore.getTable(schemaName, tableName).orElseThrow(() -> new TableNotFoundException(sourceTableName));
        String transactionalProperty = (String)hiveTable.getParameters().get("transactional");
        if (Boolean.parseBoolean(transactionalProperty)) {
            throw new TrinoException((ErrorCodeSupplier)StandardErrorCode.NOT_SUPPORTED, "Migrating transactional tables is unsupported");
        }
        if (!"MANAGED_TABLE".equalsIgnoreCase(hiveTable.getTableType()) && !"EXTERNAL_TABLE".equalsIgnoreCase(hiveTable.getTableType())) {
            throw new TrinoException((ErrorCodeSupplier)StandardErrorCode.NOT_SUPPORTED, "The procedure doesn't support migrating %s table type".formatted(hiveTable.getTableType()));
        }
        if (HiveUtil.isDeltaLakeTable((Table)hiveTable)) {
            throw new TrinoException((ErrorCodeSupplier)StandardErrorCode.NOT_SUPPORTED, "The procedure doesn't support migrating Delta Lake tables");
        }
        if (HiveUtil.isHudiTable((Table)hiveTable)) {
            throw new TrinoException((ErrorCodeSupplier)StandardErrorCode.NOT_SUPPORTED, "The procedure doesn't support migrating Hudi tables");
        }
        if (HiveUtil.isIcebergTable((Table)hiveTable)) {
            throw new TrinoException((ErrorCodeSupplier)StandardErrorCode.NOT_SUPPORTED, "The table is already an Iceberg table");
        }
        Schema schema = this.toIcebergSchema(Streams.concat((Stream[])new Stream[]{hiveTable.getDataColumns().stream(), hiveTable.getPartitionColumns().stream()}).toList());
        NameMapping nameMapping = MappingUtil.create((Schema)schema);
        HiveStorageFormat storageFormat = HiveMetadata.extractHiveStorageFormat((StorageFormat)hiveTable.getStorage().getStorageFormat());
        String location = hiveTable.getStorage().getLocation();
        Map<String, String> properties = this.icebergTableProperties(location, hiveTable.getParameters(), nameMapping, MigrateProcedure.toIcebergFileFormat(storageFormat));
        PartitionSpec partitionSpec = PartitionFields.parsePartitionFields(schema, MigrateProcedure.getPartitionColumnNames(hiveTable));
        try {
            ImmutableList.Builder dataFilesBuilder = ImmutableList.builder();
            if (hiveTable.getPartitionColumns().isEmpty()) {
                log.debug("Building data files from %s", new Object[]{location});
                dataFilesBuilder.addAll(this.buildDataFiles(session, recursive, storageFormat, location, partitionSpec, new PartitionData(new Object[0]), nameMapping));
            } else {
                Map<String, Optional<Partition>> partitions = this.listAllPartitions(metastore, hiveTable);
                int fileCount = 1;
                for (Map.Entry<String, Optional<Partition>> partition : partitions.entrySet()) {
                    Storage storage = partition.getValue().orElseThrow().getStorage();
                    log.debug("Building data files from '%s' for partition %d of %d", new Object[]{storage.getLocation(), fileCount++, partitions.size()});
                    HiveStorageFormat partitionStorageFormat = HiveMetadata.extractHiveStorageFormat((StorageFormat)storage.getStorageFormat());
                    org.apache.iceberg.PartitionData partitionData = DataFiles.data((PartitionSpec)partitionSpec, (String)partition.getKey());
                    dataFilesBuilder.addAll(this.buildDataFiles(session, recursive, partitionStorageFormat, storage.getLocation(), partitionSpec, (StructLike)partitionData, nameMapping));
                }
            }
            log.debug("Start new transaction");
            Transaction transaction = catalog.newCreateTableTransaction(session, sourceTableName, schema, PartitionFields.parsePartitionFields(schema, MigrateProcedure.toPartitionFields(hiveTable)), SortOrder.unsorted(), location, properties);
            ImmutableList dataFiles = dataFilesBuilder.build();
            log.debug("Append data %d data files", new Object[]{dataFiles.size()});
            org.apache.iceberg.Table table = transaction.table();
            AppendFiles append = table.newAppend();
            dataFiles.forEach(arg_0 -> ((AppendFiles)append).appendFile(arg_0));
            append.commit();
            log.debug("Set preparatory table properties in a metastore for migrations");
            PrincipalPrivileges principalPrivileges = this.isUsingSystemSecurity ? PrincipalPrivileges.NO_PRIVILEGES : MetastoreUtil.buildInitialPrivilegeSet((String)session.getUser());
            Table newTable = Table.builder((Table)hiveTable).setParameter("metadata_location", location).setParameter("table_type", PROVIDER_PROPERTY_VALUE.toUpperCase(Locale.ENGLISH)).build();
            metastore.replaceTable(schemaName, tableName, newTable, principalPrivileges);
            transaction.commitTransaction();
            log.debug("Successfully migrated %s table to Iceberg format", new Object[]{sourceTableName});
        }
        catch (Exception e) {
            throw new TrinoException((ErrorCodeSupplier)IcebergErrorCode.ICEBERG_COMMIT_ERROR, "Failed to migrate table", (Throwable)e);
        }
    }

    private Map<String, String> icebergTableProperties(String location, Map<String, String> hiveTableProperties, NameMapping nameMapping, IcebergFileFormat fileFormat) {
        HashMap<String, String> icebergTableProperties = new HashMap<String, String>();
        icebergTableProperties.putAll(hiveTableProperties);
        icebergTableProperties.remove("path");
        icebergTableProperties.remove("transient_lastDdlTime");
        icebergTableProperties.remove("serialization.format");
        icebergTableProperties.put("migrated", "true");
        icebergTableProperties.putIfAbsent("location", location);
        icebergTableProperties.put(PROVIDER_PROPERTY_KEY, PROVIDER_PROPERTY_VALUE);
        icebergTableProperties.put("metadata_location", location);
        icebergTableProperties.put("schema.name-mapping.default", NameMappingParser.toJson((NameMapping)nameMapping));
        icebergTableProperties.put("write.format.default", fileFormat.name());
        icebergTableProperties.put("format-version", String.valueOf(this.formatVersion));
        return ImmutableMap.copyOf(icebergTableProperties);
    }

    private Schema toIcebergSchema(List<Column> columns) {
        AtomicInteger nextFieldId = new AtomicInteger(1);
        ArrayList<Types.NestedField> icebergColumns = new ArrayList<Types.NestedField>();
        for (Column column : columns) {
            int index = icebergColumns.size();
            Type type = MigrateProcedure.toIcebergType(this.typeManager.getType(column.getType().getTypeSignature()), nextFieldId);
            Types.NestedField field = Types.NestedField.of((int)index, (boolean)false, (String)column.getName(), (Type)type, (String)column.getComment().orElse(null));
            icebergColumns.add(field);
        }
        Types.StructType icebergSchema = Types.StructType.of(icebergColumns);
        icebergSchema = TypeUtil.assignFreshIds((Type)icebergSchema, nextFieldId::getAndIncrement);
        return new Schema(icebergSchema.asStructType().fields());
    }

    private static Type toIcebergType(io.trino.spi.type.Type type, AtomicInteger nextFieldId) {
        if (type instanceof ArrayType || type instanceof MapType || type instanceof RowType) {
            throw new TrinoException((ErrorCodeSupplier)StandardErrorCode.NOT_SUPPORTED, "Migrating %s type is not supported".formatted(type));
        }
        if (type.equals(TinyintType.TINYINT) || type.equals(SmallintType.SMALLINT)) {
            return Types.IntegerType.get();
        }
        return TypeConverter.toIcebergTypeForNewColumn(type, nextFieldId);
    }

    public Map<String, Optional<Partition>> listAllPartitions(HiveMetastore metastore, Table table) {
        List partitionNames = (List)table.getPartitionColumns().stream().map(Column::getName).collect(ImmutableList.toImmutableList());
        Optional partitions = metastore.getPartitionNamesByFilter(table.getDatabaseName(), table.getTableName(), partitionNames, TupleDomain.all());
        if (partitions.isEmpty()) {
            return ImmutableMap.of();
        }
        return metastore.getPartitionsByNames(table, (List)partitions.get());
    }

    private List<DataFile> buildDataFiles(ConnectorSession session, RecursiveDirectory recursive, HiveStorageFormat format, String location, PartitionSpec partitionSpec, StructLike partition, NameMapping nameMapping) throws IOException {
        TrinoFileSystem fileSystem = this.fileSystemFactory.create(session);
        FileIterator files = fileSystem.listFiles(Location.of((String)location));
        ImmutableList.Builder dataFilesBuilder = ImmutableList.builder();
        while (files.hasNext()) {
            FileEntry file = files.next();
            String fileLocation = file.location().toString();
            String relativePath = fileLocation.substring(location.length());
            if (relativePath.contains("/_") || relativePath.contains("/.") || recursive == RecursiveDirectory.FALSE && MigrateProcedure.isRecursive(location, fileLocation)) continue;
            if (recursive == RecursiveDirectory.FAIL && MigrateProcedure.isRecursive(location, fileLocation)) {
                throw new TrinoException((ErrorCodeSupplier)StandardErrorCode.NOT_SUPPORTED, "Recursive directory must not exist when recursive_directory argument is 'fail': " + file.location());
            }
            Metrics metrics = MigrateProcedure.loadMetrics(fileSystem.newInputFile(file.location()), format, nameMapping);
            DataFile dataFile = MigrateProcedure.buildDataFile(file, partition, partitionSpec, format.name(), metrics);
            dataFilesBuilder.add((Object)dataFile);
        }
        ImmutableList dataFiles = dataFilesBuilder.build();
        log.debug("Found %d files in '%s'", new Object[]{dataFiles.size(), location});
        return dataFiles;
    }

    private static boolean isRecursive(String baseLocation, String location) {
        Verify.verify((boolean)location.startsWith(baseLocation), (String)"%s should start with %s", (Object)location, (Object)baseLocation);
        String suffix = location.substring(baseLocation.length() + 1).replaceFirst("^/+", "");
        return suffix.contains("/");
    }

    private static IcebergFileFormat toIcebergFileFormat(HiveStorageFormat storageFormat) {
        return switch (storageFormat) {
            case HiveStorageFormat.ORC -> IcebergFileFormat.ORC;
            case HiveStorageFormat.PARQUET -> IcebergFileFormat.PARQUET;
            case HiveStorageFormat.AVRO -> IcebergFileFormat.AVRO;
            default -> throw new TrinoException((ErrorCodeSupplier)StandardErrorCode.NOT_SUPPORTED, "Unsupported storage format: " + storageFormat);
        };
    }

    private static Metrics loadMetrics(TrinoInputFile file, HiveStorageFormat storageFormat, NameMapping nameMapping) {
        ForwardingInputFile inputFile = new ForwardingInputFile(file);
        return switch (storageFormat) {
            case HiveStorageFormat.ORC -> OrcMetrics.fromInputFile((InputFile)inputFile, (MetricsConfig)METRICS_CONFIG, (NameMapping)nameMapping);
            case HiveStorageFormat.PARQUET -> TrinoParquetUtil.fileMetrics(inputFile, METRICS_CONFIG, nameMapping);
            case HiveStorageFormat.AVRO -> new Metrics(Long.valueOf(Avro.rowCount((InputFile)inputFile)), null, null, null, null);
            default -> throw new TrinoException((ErrorCodeSupplier)StandardErrorCode.NOT_SUPPORTED, "Unsupported storage format: " + storageFormat);
        };
    }

    private static List<String> toPartitionFields(Table table) {
        ImmutableList.Builder fields = ImmutableList.builder();
        fields.addAll(MigrateProcedure.getPartitionColumnNames(table));
        return fields.build();
    }

    private static List<String> getPartitionColumnNames(Table table) {
        return (List)table.getPartitionColumns().stream().map(Column::getName).collect(ImmutableList.toImmutableList());
    }

    private static DataFile buildDataFile(FileEntry file, StructLike partition, PartitionSpec spec, String format, Metrics metrics) {
        return DataFiles.builder((PartitionSpec)spec).withPath(file.location().toString()).withFormat(format).withFileSizeInBytes(file.length()).withMetrics(metrics).withPartition(partition).build();
    }

    static {
        try {
            MIGRATE = MethodHandles.lookup().unreflect(MigrateProcedure.class.getMethod("migrate", ConnectorSession.class, String.class, String.class, String.class));
        }
        catch (ReflectiveOperationException e) {
            throw new AssertionError((Object)e);
        }
    }

    private static enum RecursiveDirectory {
        TRUE,
        FALSE,
        FAIL;

    }
}

