/*
 * Decompiled with CFR 0.152.
 */
package io.trino.plugin.iceberg.util;

import com.google.common.base.Verify;
import com.google.common.collect.ImmutableList;
import com.google.common.collect.ImmutableMap;
import com.google.common.collect.ImmutableSet;
import io.airlift.slice.Slice;
import io.trino.filesystem.TrinoInputFile;
import io.trino.orc.OrcColumn;
import io.trino.orc.OrcDataSource;
import io.trino.orc.OrcReader;
import io.trino.orc.OrcReaderOptions;
import io.trino.orc.metadata.ColumnMetadata;
import io.trino.orc.metadata.Footer;
import io.trino.orc.metadata.OrcColumnId;
import io.trino.orc.metadata.OrcType;
import io.trino.orc.metadata.statistics.BooleanStatistics;
import io.trino.orc.metadata.statistics.ColumnStatistics;
import io.trino.orc.metadata.statistics.DateStatistics;
import io.trino.orc.metadata.statistics.DecimalStatistics;
import io.trino.orc.metadata.statistics.DoubleStatistics;
import io.trino.orc.metadata.statistics.IntegerStatistics;
import io.trino.orc.metadata.statistics.StringStatistics;
import io.trino.orc.metadata.statistics.TimestampStatistics;
import io.trino.plugin.hive.FileFormatDataSourceStats;
import io.trino.plugin.iceberg.TrinoOrcDataSource;
import io.trino.plugin.iceberg.util.OrcIcebergIds;
import java.io.IOException;
import java.io.UncheckedIOException;
import java.math.BigDecimal;
import java.nio.ByteBuffer;
import java.util.ArrayList;
import java.util.List;
import java.util.Map;
import java.util.Optional;
import java.util.Set;
import java.util.function.Function;
import org.apache.iceberg.Metrics;
import org.apache.iceberg.MetricsConfig;
import org.apache.iceberg.MetricsModes;
import org.apache.iceberg.MetricsUtil;
import org.apache.iceberg.Schema;
import org.apache.iceberg.expressions.Literal;
import org.apache.iceberg.mapping.MappingUtil;
import org.apache.iceberg.mapping.NameMapping;
import org.apache.iceberg.types.Conversions;
import org.apache.iceberg.types.Type;
import org.apache.iceberg.types.Types;
import org.apache.iceberg.util.BinaryUtil;
import org.apache.iceberg.util.UnicodeUtil;

public final class OrcMetrics {
    private OrcMetrics() {
    }

    /*
     * Enabled aggressive block sorting
     * Enabled unnecessary exception pruning
     * Enabled aggressive exception aggregation
     */
    public static Metrics fileMetrics(TrinoInputFile file, MetricsConfig metricsConfig, Schema schema) {
        OrcReaderOptions options = new OrcReaderOptions();
        try (TrinoOrcDataSource dataSource = new TrinoOrcDataSource(file, options, new FileFormatDataSourceStats());){
            Optional reader = OrcReader.createOrcReader((OrcDataSource)dataSource, (OrcReaderOptions)options);
            if (reader.isEmpty()) {
                Metrics metrics = new Metrics(Long.valueOf(0L), null, null, null, null);
                return metrics;
            }
            Footer footer = ((OrcReader)reader.get()).getFooter();
            Optional<NameMapping> nameMapping = Optional.of(MappingUtil.create((Schema)schema));
            Map mappedColumns = (Map)OrcIcebergIds.fileColumnsByIcebergId((OrcReader)reader.get(), nameMapping).values().stream().collect(ImmutableMap.toImmutableMap(OrcColumn::getColumnId, Function.identity()));
            ArrayList<OrcType> mappedTypes = new ArrayList<OrcType>();
            ColumnMetadata types = footer.getTypes();
            for (int i = 0; i < types.size(); ++i) {
                OrcColumnId id = new OrcColumnId(i);
                mappedTypes.add(Optional.ofNullable((OrcColumn)mappedColumns.get(id)).map(OrcMetrics::toBasicOrcType).orElseGet(() -> (OrcType)types.get(id)));
            }
            Metrics metrics = OrcMetrics.computeMetrics(metricsConfig, schema, (ColumnMetadata<OrcType>)new ColumnMetadata(mappedTypes), footer.getNumberOfRows(), footer.getFileStats());
            return metrics;
        }
        catch (IOException e) {
            throw new UncheckedIOException("Failed to read file footer: " + String.valueOf(file.location()), e);
        }
    }

    private static OrcType toBasicOrcType(OrcColumn column) {
        return new OrcType(column.getColumnType(), (List)column.getNestedColumns().stream().map(OrcColumn::getColumnId).collect(ImmutableList.toImmutableList()), null, Optional.empty(), Optional.empty(), Optional.empty(), column.getAttributes());
    }

    public static Metrics computeMetrics(MetricsConfig metricsConfig, Schema icebergSchema, ColumnMetadata<OrcType> orcColumns, long fileRowCount, Optional<ColumnMetadata<ColumnStatistics>> columnStatistics) {
        if (columnStatistics.isEmpty()) {
            return new Metrics(Long.valueOf(fileRowCount), null, null, null, null, null, null);
        }
        Set<OrcColumnId> excludedColumns = OrcMetrics.getExcludedColumns(orcColumns);
        ImmutableMap.Builder valueCountsBuilder = ImmutableMap.builder();
        ImmutableMap.Builder nullCountsBuilder = ImmutableMap.builder();
        ImmutableMap.Builder nanCountsBuilder = ImmutableMap.builder();
        ImmutableMap.Builder lowerBoundsBuilder = ImmutableMap.builder();
        ImmutableMap.Builder upperBoundsBuilder = ImmutableMap.builder();
        for (int i = 1; i < orcColumns.size(); ++i) {
            OrcColumnId orcColumnId = new OrcColumnId(i);
            if (excludedColumns.contains(orcColumnId)) continue;
            OrcType orcColumn = (OrcType)orcColumns.get(orcColumnId);
            ColumnStatistics orcColumnStats = (ColumnStatistics)columnStatistics.get().get(orcColumnId);
            int icebergId = OrcMetrics.getIcebergId(orcColumn);
            Types.NestedField icebergField = icebergSchema.findField(icebergId);
            MetricsModes.MetricsMode metricsMode = MetricsUtil.metricsMode((Schema)icebergSchema, (MetricsConfig)metricsConfig, (int)icebergId);
            if (metricsMode.equals((Object)MetricsModes.None.get())) continue;
            Verify.verify((icebergField != null ? 1 : 0) != 0, (String)"Cannot find Iceberg column with ID %s in schema %s", (int)icebergId, (Object)icebergSchema);
            valueCountsBuilder.put((Object)icebergId, (Object)fileRowCount);
            if (orcColumnStats.hasNumberOfValues()) {
                nullCountsBuilder.put((Object)icebergId, (Object)(fileRowCount - orcColumnStats.getNumberOfValues()));
            }
            if (orcColumnStats.getNumberOfNanValues() > 0L) {
                nanCountsBuilder.put((Object)icebergId, (Object)orcColumnStats.getNumberOfNanValues());
            }
            if (metricsMode.equals((Object)MetricsModes.Counts.get())) continue;
            OrcMetrics.toIcebergMinMax(orcColumnStats, icebergField.type(), metricsMode).ifPresent(minMax -> {
                lowerBoundsBuilder.put((Object)icebergId, (Object)minMax.getMin());
                upperBoundsBuilder.put((Object)icebergId, (Object)minMax.getMax());
            });
        }
        ImmutableMap valueCounts = valueCountsBuilder.buildOrThrow();
        ImmutableMap nullCounts = nullCountsBuilder.buildOrThrow();
        ImmutableMap nanCounts = nanCountsBuilder.buildOrThrow();
        ImmutableMap lowerBounds = lowerBoundsBuilder.buildOrThrow();
        ImmutableMap upperBounds = upperBoundsBuilder.buildOrThrow();
        return new Metrics(Long.valueOf(fileRowCount), null, (Map)(valueCounts.isEmpty() ? null : valueCounts), (Map)(nullCounts.isEmpty() ? null : nullCounts), (Map)(nanCounts.isEmpty() ? null : nanCounts), (Map)(lowerBounds.isEmpty() ? null : lowerBounds), (Map)(upperBounds.isEmpty() ? null : upperBounds));
    }

    private static Set<OrcColumnId> getExcludedColumns(ColumnMetadata<OrcType> orcColumns) {
        ImmutableSet.Builder excludedColumns = ImmutableSet.builder();
        OrcMetrics.populateExcludedColumns(orcColumns, OrcColumnId.ROOT_COLUMN, false, (ImmutableSet.Builder<OrcColumnId>)excludedColumns);
        return excludedColumns.build();
    }

    private static void populateExcludedColumns(ColumnMetadata<OrcType> orcColumns, OrcColumnId orcColumnId, boolean exclude, ImmutableSet.Builder<OrcColumnId> excludedColumns) {
        if (exclude) {
            excludedColumns.add((Object)orcColumnId);
        }
        OrcType orcColumn = (OrcType)orcColumns.get(orcColumnId);
        switch (orcColumn.getOrcTypeKind()) {
            case LIST: 
            case MAP: {
                for (OrcColumnId child : orcColumn.getFieldTypeIndexes()) {
                    OrcMetrics.populateExcludedColumns(orcColumns, child, true, excludedColumns);
                }
                return;
            }
            case STRUCT: {
                for (OrcColumnId child : orcColumn.getFieldTypeIndexes()) {
                    OrcMetrics.populateExcludedColumns(orcColumns, child, exclude, excludedColumns);
                }
                return;
            }
        }
    }

    private static int getIcebergId(OrcType orcColumn) {
        String icebergId = (String)orcColumn.getAttributes().get("iceberg.id");
        Verify.verify((icebergId != null ? 1 : 0) != 0, (String)"ORC column %s doesn't have an associated Iceberg ID", (Object)orcColumn);
        return Integer.parseInt(icebergId);
    }

    private static Optional<IcebergMinMax> toIcebergMinMax(ColumnStatistics orcColumnStats, Type icebergType, MetricsModes.MetricsMode metricsModes) {
        BooleanStatistics booleanStatistics = orcColumnStats.getBooleanStatistics();
        if (booleanStatistics != null) {
            boolean hasTrueValues = booleanStatistics.getTrueValueCount() != 0L;
            boolean hasFalseValues = orcColumnStats.getNumberOfValues() != booleanStatistics.getTrueValueCount();
            return Optional.of(new IcebergMinMax(icebergType, !hasFalseValues, hasTrueValues, metricsModes));
        }
        IntegerStatistics integerStatistics = orcColumnStats.getIntegerStatistics();
        if (integerStatistics != null) {
            Number min = integerStatistics.getMin();
            Number max = integerStatistics.getMax();
            if (min == null || max == null) {
                return Optional.empty();
            }
            if (icebergType.typeId() == Type.TypeID.INTEGER) {
                min = Math.toIntExact(min);
                max = Math.toIntExact(max);
            }
            return Optional.of(new IcebergMinMax(icebergType, min, max, metricsModes));
        }
        DoubleStatistics doubleStatistics = orcColumnStats.getDoubleStatistics();
        if (doubleStatistics != null) {
            Number min = doubleStatistics.getMin();
            Number max = doubleStatistics.getMax();
            if (min == null || max == null) {
                return Optional.empty();
            }
            if (icebergType.typeId() == Type.TypeID.FLOAT) {
                min = Float.valueOf(min.floatValue());
                max = Float.valueOf(max.floatValue());
            }
            return Optional.of(new IcebergMinMax(icebergType, min, max, metricsModes));
        }
        StringStatistics stringStatistics = orcColumnStats.getStringStatistics();
        if (stringStatistics != null) {
            Slice min = stringStatistics.getMin();
            Slice max = stringStatistics.getMax();
            if (min == null || max == null) {
                return Optional.empty();
            }
            return Optional.of(new IcebergMinMax(icebergType, min.toStringUtf8(), max.toStringUtf8(), metricsModes));
        }
        DateStatistics dateStatistics = orcColumnStats.getDateStatistics();
        if (dateStatistics != null) {
            Integer min = dateStatistics.getMin();
            Integer max = dateStatistics.getMax();
            if (min == null || max == null) {
                return Optional.empty();
            }
            return Optional.of(new IcebergMinMax(icebergType, min, max, metricsModes));
        }
        DecimalStatistics decimalStatistics = orcColumnStats.getDecimalStatistics();
        if (decimalStatistics != null) {
            BigDecimal min = decimalStatistics.getMin();
            BigDecimal max = decimalStatistics.getMax();
            if (min == null || max == null) {
                return Optional.empty();
            }
            min = min.setScale(((Types.DecimalType)icebergType).scale());
            max = max.setScale(((Types.DecimalType)icebergType).scale());
            return Optional.of(new IcebergMinMax(icebergType, min, max, metricsModes));
        }
        TimestampStatistics timestampStatistics = orcColumnStats.getTimestampStatistics();
        if (timestampStatistics != null) {
            Long min = timestampStatistics.getMin();
            Long max = timestampStatistics.getMax();
            if (min == null || max == null) {
                return Optional.empty();
            }
            return Optional.of(new IcebergMinMax(icebergType, min * 1000L, max * 1000L + 999L, metricsModes));
        }
        return Optional.empty();
    }

    private static class IcebergMinMax {
        private final ByteBuffer min;
        private final ByteBuffer max;

        private IcebergMinMax(Type type, Object min, Object max, MetricsModes.MetricsMode metricsMode) {
            if (metricsMode instanceof MetricsModes.Full) {
                this.min = Conversions.toByteBuffer((Type)type, (Object)min);
                this.max = Conversions.toByteBuffer((Type)type, (Object)max);
            } else if (metricsMode instanceof MetricsModes.Truncate) {
                MetricsModes.Truncate truncateMode = (MetricsModes.Truncate)metricsMode;
                int truncateLength = truncateMode.length();
                switch (type.typeId()) {
                    case STRING: {
                        this.min = UnicodeUtil.truncateStringMin((Literal)Literal.of((CharSequence)((CharSequence)min)), (int)truncateLength).toByteBuffer();
                        this.max = UnicodeUtil.truncateStringMax((Literal)Literal.of((CharSequence)((CharSequence)max)), (int)truncateLength).toByteBuffer();
                        break;
                    }
                    case FIXED: 
                    case BINARY: {
                        this.min = BinaryUtil.truncateBinaryMin((Literal)Literal.of((ByteBuffer)((ByteBuffer)min)), (int)truncateLength).toByteBuffer();
                        this.max = BinaryUtil.truncateBinaryMax((Literal)Literal.of((ByteBuffer)((ByteBuffer)max)), (int)truncateLength).toByteBuffer();
                        break;
                    }
                    default: {
                        this.min = Conversions.toByteBuffer((Type)type, (Object)min);
                        this.max = Conversions.toByteBuffer((Type)type, (Object)max);
                        break;
                    }
                }
            } else {
                throw new UnsupportedOperationException("Unsupported metrics mode for Iceberg Max/Min Bound: " + String.valueOf(metricsMode));
            }
        }

        public ByteBuffer getMin() {
            return this.min;
        }

        public ByteBuffer getMax() {
            return this.max;
        }
    }
}

