/*
 * Decompiled with CFR 0.152.
 */
package io.trino.plugin.hive.util;

import com.fasterxml.jackson.core.type.TypeReference;
import com.fasterxml.jackson.databind.ObjectMapper;
import com.google.common.annotations.VisibleForTesting;
import com.google.common.base.Preconditions;
import com.google.common.collect.ArrayListMultimap;
import com.google.common.collect.ComparisonChain;
import com.google.common.collect.ImmutableList;
import com.google.common.collect.ListMultimap;
import com.google.common.collect.Multimaps;
import io.trino.filesystem.FileEntry;
import io.trino.filesystem.FileIterator;
import io.trino.filesystem.Location;
import io.trino.filesystem.TrinoFileSystem;
import io.trino.filesystem.TrinoInputFile;
import io.trino.filesystem.TrinoInputStream;
import io.trino.filesystem.TrinoOutputFile;
import io.trino.plugin.hive.HiveErrorCode;
import io.trino.plugin.hive.util.ValidWriteIdList;
import io.trino.spi.ErrorCodeSupplier;
import io.trino.spi.TrinoException;
import java.io.IOException;
import java.io.InputStream;
import java.io.OutputStream;
import java.nio.charset.StandardCharsets;
import java.util.ArrayList;
import java.util.Collection;
import java.util.Comparator;
import java.util.List;
import java.util.Locale;
import java.util.Map;
import java.util.Objects;
import java.util.Optional;

public final class AcidTables {
    private AcidTables() {
    }

    public static boolean isInsertOnlyTable(Map<String, String> parameters) {
        return "insert_only".equalsIgnoreCase(parameters.get("transactional_properties"));
    }

    public static boolean isTransactionalTable(Map<String, String> parameters) {
        return "true".equalsIgnoreCase(parameters.get("transactional")) || "true".equalsIgnoreCase(parameters.get("transactional".toUpperCase(Locale.ENGLISH)));
    }

    public static boolean isFullAcidTable(Map<String, String> parameters) {
        return AcidTables.isTransactionalTable(parameters) && !AcidTables.isInsertOnlyTable(parameters);
    }

    public static Location bucketFileName(Location subdir, int bucket) {
        return subdir.appendPath("bucket_%05d".formatted(bucket));
    }

    public static String deltaSubdir(long writeId, int statementId) {
        return "delta_%07d_%07d_%04d".formatted(writeId, writeId, statementId);
    }

    public static String deleteDeltaSubdir(long writeId, int statementId) {
        return "delete_" + AcidTables.deltaSubdir(writeId, statementId);
    }

    public static void writeAcidVersionFile(TrinoFileSystem fileSystem, Location deltaOrBaseDir) throws IOException {
        TrinoOutputFile file = fileSystem.newOutputFile(AcidTables.versionFilePath(deltaOrBaseDir));
        try (OutputStream out = file.createOrOverwrite();){
            out.write(50);
        }
    }

    public static int readAcidVersionFile(TrinoFileSystem fileSystem, Location deltaOrBaseDir) throws IOException {
        TrinoInputFile file = fileSystem.newInputFile(AcidTables.versionFilePath(deltaOrBaseDir));
        if (!file.exists()) {
            return 0;
        }
        try (TrinoInputStream in = file.newStream();){
            byte[] bytes = in.readNBytes(1);
            if (bytes.length == 1) {
                int n = Integer.parseInt(new String(bytes, StandardCharsets.UTF_8));
                return n;
            }
            int n = 0;
            return n;
        }
    }

    private static Location versionFilePath(Location deltaOrBaseDir) {
        return deltaOrBaseDir.appendPath("_orc_acid_version");
    }

    public static AcidState getAcidState(TrinoFileSystem fileSystem, Location directory, ValidWriteIdList writeIdList) throws IOException {
        ArrayListMultimap groupedFiles = ArrayListMultimap.create();
        ArrayList<FileEntry> originalFiles = new ArrayList<FileEntry>();
        for (FileEntry file : AcidTables.listFiles(fileSystem, directory)) {
            String name;
            String suffix = AcidTables.listingSuffix(directory.toString(), file.location().toString());
            int slash = suffix.indexOf(47);
            String string = name = slash == -1 ? "" : suffix.substring(0, slash);
            if (name.startsWith("base_") || name.startsWith("delta_") || name.startsWith("delete_delta_")) {
                if (suffix.indexOf(47, slash + 1) != -1) {
                    throw new TrinoException((ErrorCodeSupplier)HiveErrorCode.HIVE_INVALID_BUCKET_FILES, "Found file in sub-directory of ACID directory: " + String.valueOf(file.location()));
                }
                groupedFiles.put((Object)name, (Object)file);
                continue;
            }
            if (file.length() <= 0L) continue;
            originalFiles.add(file);
        }
        ArrayList<ParsedDelta> workingDeltas = new ArrayList<ParsedDelta>();
        String oldestBase = null;
        long oldestBaseWriteId = Long.MAX_VALUE;
        String bestBasePath = null;
        long bestBaseWriteId = 0L;
        Object bestBaseFiles = ImmutableList.of();
        for (Map.Entry entry2 : Multimaps.asMap((ListMultimap)groupedFiles).entrySet()) {
            String name = (String)entry2.getKey();
            String baseDir = String.valueOf(directory) + "/" + name;
            List files = (List)entry2.getValue();
            if (name.startsWith("base_")) {
                ParsedBase base = AcidTables.parseBase(name);
                long writeId = base.writeId();
                if (oldestBaseWriteId > writeId) {
                    oldestBase = baseDir;
                    oldestBaseWriteId = writeId;
                }
                if (bestBasePath != null && bestBaseWriteId >= writeId || !AcidTables.isValidBase(base, writeIdList, fileSystem, baseDir)) continue;
                bestBasePath = baseDir;
                bestBaseWriteId = writeId;
                bestBaseFiles = files;
                continue;
            }
            String deltaPrefix = name.startsWith("delta_") ? "delta_" : "delete_delta_";
            ParsedDelta delta = AcidTables.parseDelta(baseDir, deltaPrefix, files);
            if (!writeIdList.isWriteIdRangeValid(delta.min(), delta.max())) continue;
            workingDeltas.add(delta);
        }
        if (oldestBase != null && bestBasePath == null) {
            long[] exceptions = writeIdList.getInvalidWriteIds();
            String minOpenWriteId = exceptions != null && exceptions.length > 0 ? String.valueOf(exceptions[0]) : "x";
            throw new IOException("Not enough history available for (%s,%s). Oldest available base: %s".formatted(writeIdList.getHighWatermark(), minOpenWriteId, oldestBase));
        }
        if (bestBasePath != null) {
            originalFiles.clear();
        }
        originalFiles.sort(Comparator.comparing(entry -> entry.location().toString()));
        workingDeltas.sort(null);
        ArrayList<ParsedDelta> deltas = new ArrayList<ParsedDelta>();
        long current = bestBaseWriteId;
        int lastStatementId = -1;
        ParsedDelta prev = null;
        for (ParsedDelta next : workingDeltas) {
            if (next.max() > current) {
                if (!writeIdList.isWriteIdRangeValid(current + 1L, next.max())) continue;
                deltas.add(next);
                current = next.max();
                lastStatementId = next.statementId();
                prev = next;
                continue;
            }
            if (next.max() == current && lastStatementId >= 0) {
                deltas.add(next);
                prev = next;
                continue;
            }
            if (prev == null || next.max() != prev.max() || next.min() != prev.min() || next.statementId() != prev.statementId()) continue;
            deltas.add(next);
            prev = next;
        }
        Optional<Location> baseDirectory = Optional.ofNullable(bestBasePath).map(Location::of);
        return new AcidState(baseDirectory, (List<FileEntry>)bestBaseFiles, deltas, originalFiles);
    }

    private static boolean isValidBase(ParsedBase base, ValidWriteIdList writeIdList, TrinoFileSystem fileSystem, String baseDir) throws IOException {
        if (base.writeId() == Long.MIN_VALUE) {
            return true;
        }
        if (base.visibilityId() > 0L || AcidTables.isCompacted(fileSystem, baseDir)) {
            return writeIdList.isValidBase(base.writeId());
        }
        return writeIdList.isWriteIdValid(base.writeId());
    }

    private static boolean isCompacted(TrinoFileSystem fileSystem, String baseDir) throws IOException {
        Map metadata;
        Location location = Location.of((String)baseDir).appendPath("_metadata_acid");
        TrinoInputFile file = fileSystem.newInputFile(location);
        if (!file.exists()) {
            return false;
        }
        try (TrinoInputStream in = file.newStream();){
            metadata = (Map)new ObjectMapper().readValue((InputStream)in, (TypeReference)new TypeReference<Map<String, String>>(){});
        }
        catch (IOException e) {
            throw new IOException("Failed to read %s: %s".formatted(file.location(), e.getMessage()), e);
        }
        String version = (String)metadata.get("thisFileVersion");
        if (!"0".equals(version)) {
            throw new IOException("Unexpected ACID metadata version: " + version);
        }
        String format = (String)metadata.get("dataFormat");
        if (!"compacted".equals(format)) {
            throw new IOException("Unexpected value for ACID dataFormat: " + format);
        }
        return true;
    }

    @VisibleForTesting
    static ParsedDelta parseDelta(String path, String deltaPrefix, List<FileEntry> files) {
        String fileName = path.substring(path.lastIndexOf(47) + 1);
        Preconditions.checkArgument((boolean)fileName.startsWith(deltaPrefix), (String)"File does not start with '%s': %s", (Object)deltaPrefix, (Object)path);
        int visibility = fileName.indexOf("_v");
        if (visibility != -1) {
            fileName = fileName.substring(0, visibility);
        }
        boolean deleteDelta = deltaPrefix.equals("delete_delta_");
        String rest = fileName.substring(deltaPrefix.length());
        int split = rest.indexOf(95);
        int split2 = rest.indexOf(95, split + 1);
        long min = Long.parseLong(rest.substring(0, split));
        if (split2 == -1) {
            long max = Long.parseLong(rest.substring(split + 1));
            return new ParsedDelta(min, max, path, -1, deleteDelta, files);
        }
        long max = Long.parseLong(rest.substring(split + 1, split2));
        int statementId = Integer.parseInt(rest.substring(split2 + 1));
        return new ParsedDelta(min, max, path, statementId, deleteDelta, files);
    }

    @VisibleForTesting
    static ParsedBase parseBase(String name) {
        Preconditions.checkArgument((boolean)name.startsWith("base_"), (String)"File does not start with 'base_': %s", (Object)name);
        name = name.substring("base_".length());
        int index = name.indexOf("_v");
        if (index == -1) {
            return new ParsedBase(Long.parseLong(name), 0L);
        }
        return new ParsedBase(Long.parseLong(name.substring(0, index)), Long.parseLong(name.substring(index + 2)));
    }

    private static List<FileEntry> listFiles(TrinoFileSystem fileSystem, Location directory) throws IOException {
        ArrayList<FileEntry> files = new ArrayList<FileEntry>();
        FileIterator iterator = fileSystem.listFiles(directory);
        while (iterator.hasNext()) {
            FileEntry file = iterator.next();
            String path = file.location().path();
            if (path.contains("/_") || path.contains("/.")) continue;
            files.add(file);
        }
        return files;
    }

    private static String listingSuffix(String directory, String file) {
        Preconditions.checkArgument((boolean)file.startsWith(directory), (String)"file '%s' does not start with directory '%s'", (Object)file, (Object)directory);
        Preconditions.checkArgument((file.length() - directory.length() >= 2 ? 1 : 0) != 0, (Object)"file name is too short");
        Preconditions.checkArgument((file.charAt(directory.length()) == '/' ? 1 : 0) != 0, (Object)"no slash after directory prefix");
        Preconditions.checkArgument((file.charAt(directory.length() + 1) != '/' ? 1 : 0) != 0, (Object)"extra slash after directory prefix");
        return file.substring(directory.length() + 1);
    }

    public record ParsedBase(long writeId, long visibilityId) {
    }

    public record ParsedDelta(long min, long max, String path, int statementId, boolean deleteDelta, List<FileEntry> files) implements Comparable<ParsedDelta>
    {
        public ParsedDelta {
            Objects.requireNonNull(path, "path is null");
            files = ImmutableList.copyOf((Collection)Objects.requireNonNull(files, "files is null"));
        }

        @Override
        public int compareTo(ParsedDelta other) {
            return ComparisonChain.start().compare(this.min, other.min).compare(other.max, this.max).compare(this.statementId, other.statementId).compare((Comparable)((Object)this.path), (Comparable)((Object)other.path)).result();
        }
    }

    public record AcidState(Optional<Location> baseDirectory, List<FileEntry> baseFiles, List<ParsedDelta> deltas, List<FileEntry> originalFiles) {
        public AcidState {
            Objects.requireNonNull(baseDirectory, "baseDirectory is null");
            baseFiles = ImmutableList.copyOf((Collection)Objects.requireNonNull(baseFiles, "baseFiles is null"));
            deltas = ImmutableList.copyOf((Collection)Objects.requireNonNull(deltas, "deltas is null"));
            originalFiles = ImmutableList.copyOf((Collection)Objects.requireNonNull(originalFiles, "originalFiles is null"));
        }
    }
}

