/*
 * Decompiled with CFR 0.152.
 */
package io.trino.plugin.iceberg;

import com.google.common.base.Preconditions;
import com.google.common.base.Verify;
import com.google.common.collect.ImmutableList;
import com.google.common.collect.Iterables;
import com.google.common.collect.Streams;
import io.airlift.concurrent.MoreFutures;
import io.trino.Session;
import io.trino.filesystem.FileIterator;
import io.trino.filesystem.Location;
import io.trino.filesystem.TrinoFileSystem;
import io.trino.plugin.iceberg.IcebergTestUtils;
import io.trino.plugin.iceberg.fileio.ForwardingFileIo;
import io.trino.sql.query.QueryAssertions;
import io.trino.testing.BaseConnectorSmokeTest;
import io.trino.testing.QueryRunner;
import io.trino.testing.TestingAccessControlManager;
import io.trino.testing.TestingConnectorBehavior;
import io.trino.testing.TestingConnectorSession;
import io.trino.testing.TestingNames;
import io.trino.testing.sql.TestTable;
import java.io.IOException;
import java.time.ZoneOffset;
import java.time.ZonedDateTime;
import java.time.format.DateTimeFormatter;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Optional;
import java.util.Set;
import java.util.concurrent.CyclicBarrier;
import java.util.concurrent.ExecutorService;
import java.util.concurrent.Executors;
import java.util.concurrent.Future;
import java.util.concurrent.TimeUnit;
import java.util.regex.Matcher;
import java.util.regex.Pattern;
import java.util.stream.Collectors;
import java.util.stream.IntStream;
import java.util.stream.Stream;
import org.apache.iceberg.FileFormat;
import org.apache.iceberg.ManifestFile;
import org.apache.iceberg.TableMetadata;
import org.apache.iceberg.TableMetadataParser;
import org.apache.iceberg.io.FileIO;
import org.assertj.core.api.AbstractBooleanAssert;
import org.assertj.core.api.AssertProvider;
import org.assertj.core.api.Assertions;
import org.assertj.core.api.ListAssert;
import org.junit.jupiter.api.BeforeAll;
import org.junit.jupiter.api.RepeatedTest;
import org.junit.jupiter.api.Test;
import org.junit.jupiter.api.TestInstance;
import org.junit.jupiter.api.Timeout;
import org.junit.jupiter.api.parallel.Execution;
import org.junit.jupiter.api.parallel.ExecutionMode;

@TestInstance(value=TestInstance.Lifecycle.PER_CLASS)
public abstract class BaseIcebergConnectorSmokeTest
extends BaseConnectorSmokeTest {
    protected final FileFormat format;
    protected TrinoFileSystem fileSystem;

    public BaseIcebergConnectorSmokeTest(FileFormat format) {
        this.format = Objects.requireNonNull(format, "format is null");
    }

    @BeforeAll
    public void initFileSystem() {
        this.fileSystem = IcebergTestUtils.getFileSystemFactory((QueryRunner)this.getDistributedQueryRunner()).create(TestingConnectorSession.SESSION);
    }

    protected boolean hasBehavior(TestingConnectorBehavior connectorBehavior) {
        return switch (connectorBehavior) {
            case TestingConnectorBehavior.SUPPORTS_TOPN_PUSHDOWN -> false;
            default -> super.hasBehavior(connectorBehavior);
        };
    }

    @Test
    public void testShowCreateTable() {
        String schemaName = (String)this.getSession().getSchema().orElseThrow();
        Assertions.assertThat((String)((String)this.computeScalar("SHOW CREATE TABLE region"))).matches((CharSequence)("CREATE TABLE iceberg." + schemaName + ".region \\(\n   regionkey bigint,\n   name varchar,\n   comment varchar\n\\)\nWITH \\(\n   format = '" + this.format.name() + "',\n   format_version = 2,\n" + String.format("   location = '.*/" + schemaName + "/region.*',\n   max_commit_retry = 4\n", new Object[0]) + "\\)"));
    }

    @Test
    public void testHiddenPathColumn() {
        try (TestTable table = this.newTrinoTable("hidden_file_path", "(a int, b VARCHAR)", (List)ImmutableList.of((Object)"(1, 'a')"));){
            String filePath = (String)this.computeScalar(String.format("SELECT file_path FROM \"%s$files\"", table.getName()));
            this.assertQuery("SELECT DISTINCT \"$path\" FROM " + table.getName(), "VALUES '" + filePath + "'");
            this.assertQuery(String.format("SELECT a FROM %s WHERE \"$path\" = '%s'", table.getName(), filePath), "VALUES 1");
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    @RepeatedTest(value=4)
    @Timeout(value=120L)
    @Execution(value=ExecutionMode.SAME_THREAD)
    public void testDeleteRowsConcurrently() throws Exception {
        int threads = 4;
        CyclicBarrier barrier = new CyclicBarrier(threads);
        ExecutorService executor = Executors.newFixedThreadPool(threads);
        ImmutableList rows = ImmutableList.of((Object)"(1, 0, 0, 0)", (Object)"(0, 1, 0, 0)", (Object)"(0, 0, 1, 0)", (Object)"(0, 0, 0, 1)");
        String[] expectedErrors = new String[]{"Failed to commit the transaction during write:", "Failed to replace table due to concurrent updates:", "Failed to commit during write:"};
        try (TestTable table = this.newTrinoTable("test_concurrent_delete", "(col0 INTEGER, col1 INTEGER, col2 INTEGER, col3 INTEGER)");){
            String tableName = table.getName();
            this.assertUpdate("INSERT INTO " + tableName + " VALUES " + String.join((CharSequence)", ", (Iterable<? extends CharSequence>)rows), 4L);
            List futures = (List)IntStream.range(0, threads).mapToObj(threadNumber -> executor.submit(() -> {
                barrier.await(10L, TimeUnit.SECONDS);
                String columnName = "col" + threadNumber;
                try {
                    this.getQueryRunner().execute(String.format("DELETE FROM %s WHERE %s = 1", tableName, columnName));
                    return true;
                }
                catch (Exception e) {
                    Assertions.assertThat((String)e.getMessage()).containsAnyOf((CharSequence[])expectedErrors);
                    return false;
                }
            })).collect(ImmutableList.toImmutableList());
            Stream expectedRows = Streams.mapWithIndex(futures.stream(), (arg_0, arg_1) -> BaseIcebergConnectorSmokeTest.lambda$testDeleteRowsConcurrently$2((List)rows, arg_0, arg_1));
            List expectedValues = (List)expectedRows.filter(Optional::isPresent).map(Optional::get).collect(ImmutableList.toImmutableList());
            ((ListAssert)Assertions.assertThat((List)expectedValues).as("Expected at least one delete operation to pass", new Object[0])).hasSizeLessThan(rows.size());
            ((QueryAssertions.QueryAssert)Assertions.assertThat((AssertProvider)this.query("SELECT * FROM " + tableName))).matches("VALUES " + String.join((CharSequence)", ", expectedValues));
        }
        finally {
            executor.shutdownNow();
            Assertions.assertThat((boolean)executor.awaitTermination(10L, TimeUnit.SECONDS)).isTrue();
        }
    }

    @Test
    public void testCreateOrReplaceTable() {
        try (TestTable table = this.newTrinoTable("test_create_or_replace", " AS SELECT BIGINT '42' a, DOUBLE '-38.5' b");){
            ((QueryAssertions.QueryAssert)Assertions.assertThat((AssertProvider)this.query("SELECT a, b FROM " + table.getName()))).matches("VALUES (BIGINT '42', -385e-1)");
            long v1SnapshotId = this.getMostRecentSnapshotId(table.getName());
            this.assertUpdate("CREATE OR REPLACE TABLE " + table.getName() + " AS SELECT BIGINT '-42' a, DOUBLE '38.5' b", 1L);
            ((QueryAssertions.QueryAssert)Assertions.assertThat((AssertProvider)this.query("SELECT a, b FROM " + table.getName()))).matches("VALUES (BIGINT '-42', 385e-1)");
            ((QueryAssertions.QueryAssert)Assertions.assertThat((AssertProvider)this.query("SELECT COUNT(snapshot_id) FROM \"" + table.getName() + "$history\""))).matches("VALUES BIGINT '2'");
            ((QueryAssertions.QueryAssert)Assertions.assertThat((AssertProvider)this.query("SELECT a, b  FROM " + table.getName() + " FOR VERSION AS OF " + v1SnapshotId))).matches("VALUES (BIGINT '42', -385e-1)");
        }
    }

    @Test
    public void testCreateOrReplaceTableChangeColumnNamesAndTypes() {
        String tableName = "test_create_or_replace_" + TestingNames.randomNameSuffix();
        this.assertUpdate("CREATE TABLE " + tableName + " AS SELECT BIGINT '42' a, DOUBLE '-38.5' b", 1L);
        ((QueryAssertions.QueryAssert)Assertions.assertThat((AssertProvider)this.query("SELECT CAST(a AS bigint), b FROM " + tableName))).matches("VALUES (BIGINT '42', -385e-1)");
        long v1SnapshotId = this.getMostRecentSnapshotId(tableName);
        this.assertUpdate("CREATE OR REPLACE TABLE " + tableName + " AS SELECT VARCHAR 'test' c, VARCHAR 'test2' d", 1L);
        ((QueryAssertions.QueryAssert)Assertions.assertThat((AssertProvider)this.query("SELECT c, d FROM " + tableName))).matches("VALUES (VARCHAR 'test', VARCHAR 'test2')");
        ((QueryAssertions.QueryAssert)Assertions.assertThat((AssertProvider)this.query("SELECT a, b  FROM " + tableName + " FOR VERSION AS OF " + v1SnapshotId))).matches("VALUES (BIGINT '42', -385e-1)");
        this.assertUpdate("DROP TABLE " + tableName);
    }

    @Test
    public void testRegisterTableWithTableLocation() {
        String tableName = "test_register_table_with_table_location_" + TestingNames.randomNameSuffix();
        this.assertUpdate(String.format("CREATE TABLE %s (a int, b varchar, c boolean)", tableName));
        this.assertUpdate(String.format("INSERT INTO %s values(1, 'INDIA', true)", tableName), 1L);
        this.assertUpdate(String.format("INSERT INTO %s values(2, 'USA', false)", tableName), 1L);
        String tableLocation = this.getTableLocation(tableName);
        this.dropTableFromMetastore(tableName);
        this.assertUpdate("CALL system.register_table (CURRENT_SCHEMA, '" + tableName + "', '" + tableLocation + "')");
        ((QueryAssertions.QueryAssert)Assertions.assertThat((AssertProvider)this.query(String.format("SELECT * FROM %s", tableName)))).matches("VALUES ROW(INT '1', VARCHAR 'INDIA', BOOLEAN 'true'), ROW(INT '2', VARCHAR 'USA', BOOLEAN 'false')");
        this.assertUpdate(String.format("DROP TABLE %s", tableName));
    }

    @Test
    public void testRegisterTableWithComments() {
        String tableName = "test_register_table_with_comments_" + TestingNames.randomNameSuffix();
        this.assertUpdate(String.format("CREATE TABLE %s (a int, b varchar, c boolean)", tableName));
        this.assertUpdate(String.format("INSERT INTO %s values(1, 'INDIA', true)", tableName), 1L);
        this.assertUpdate(String.format("COMMENT ON TABLE %s is 'my-table-comment'", tableName));
        this.assertUpdate(String.format("COMMENT ON COLUMN %s.a is 'a-comment'", tableName));
        this.assertUpdate(String.format("COMMENT ON COLUMN %s.b is 'b-comment'", tableName));
        this.assertUpdate(String.format("COMMENT ON COLUMN %s.c is 'c-comment'", tableName));
        String tableLocation = this.getTableLocation(tableName);
        this.dropTableFromMetastore(tableName);
        this.assertUpdate("CALL system.register_table (CURRENT_SCHEMA, '" + tableName + "', '" + tableLocation + "')");
        Assertions.assertThat((String)this.getTableComment(tableName)).isEqualTo("my-table-comment");
        Assertions.assertThat((String)this.getColumnComment(tableName, "a")).isEqualTo("a-comment");
        Assertions.assertThat((String)this.getColumnComment(tableName, "b")).isEqualTo("b-comment");
        Assertions.assertThat((String)this.getColumnComment(tableName, "c")).isEqualTo("c-comment");
        this.assertUpdate(String.format("DROP TABLE %s", tableName));
    }

    @Test
    public void testRegisterTableWithShowCreateTable() {
        String tableName = "test_register_table_with_show_create_table_" + TestingNames.randomNameSuffix();
        this.assertUpdate(String.format("CREATE TABLE %s (a int, b varchar, c boolean)", tableName));
        this.assertUpdate(String.format("INSERT INTO %s values(1, 'INDIA', true)", tableName), 1L);
        String tableLocation = this.getTableLocation(tableName);
        String showCreateTableOld = (String)this.computeActual("SHOW CREATE TABLE " + tableName).getOnlyValue();
        this.dropTableFromMetastore(tableName);
        this.assertUpdate("CALL system.register_table (CURRENT_SCHEMA, '" + tableName + "', '" + tableLocation + "')");
        String showCreateTableNew = (String)this.computeActual("SHOW CREATE TABLE " + tableName).getOnlyValue();
        Assertions.assertThat((String)showCreateTableOld).isEqualTo(showCreateTableNew);
        this.assertUpdate(String.format("DROP TABLE %s", tableName));
    }

    @Test
    public void testRegisterTableWithReInsert() {
        String tableName = "test_register_table_with_re_insert_" + TestingNames.randomNameSuffix();
        this.assertUpdate(String.format("CREATE TABLE %s (a int, b varchar, c boolean)", tableName));
        this.assertUpdate(String.format("INSERT INTO %s values(1, 'INDIA', true)", tableName), 1L);
        this.assertUpdate(String.format("INSERT INTO %s values(2, 'USA', false)", tableName), 1L);
        String tableLocation = this.getTableLocation(tableName);
        this.dropTableFromMetastore(tableName);
        this.assertUpdate("CALL system.register_table (CURRENT_SCHEMA, '" + tableName + "', '" + tableLocation + "')");
        this.assertUpdate(String.format("INSERT INTO %s values(3, 'POLAND', true)", tableName), 1L);
        ((QueryAssertions.QueryAssert)Assertions.assertThat((AssertProvider)this.query(String.format("SELECT * FROM %s", tableName)))).matches("VALUES ROW(INT '1', VARCHAR 'INDIA', BOOLEAN 'true'), ROW(INT '2', VARCHAR 'USA', BOOLEAN 'false'), ROW(INT '3', VARCHAR 'POLAND', BOOLEAN 'true')");
        this.assertUpdate(String.format("DROP TABLE %s", tableName));
    }

    @Test
    public void testRegisterTableWithDroppedTable() {
        String tableName = "test_register_table_with_dropped_table_" + TestingNames.randomNameSuffix();
        this.assertUpdate(String.format("CREATE TABLE %s (a int, b varchar, c boolean)", tableName));
        this.assertUpdate(String.format("INSERT INTO %s values(1, 'INDIA', true)", tableName), 1L);
        String tableLocation = this.getTableLocation(tableName);
        String tableNameNew = tableName + "_new";
        this.assertUpdate(String.format("DROP TABLE %s", tableName));
        this.assertQueryFails(String.format("CALL system.register_table (CURRENT_SCHEMA, '%s', '%s')", tableNameNew, tableLocation), ".*No versioned metadata file exists at location.*");
    }

    @Test
    public void testRegisterTableWithDifferentTableName() {
        String tableName = "test_register_table_with_different_table_name_" + TestingNames.randomNameSuffix();
        this.assertUpdate(String.format("CREATE TABLE %s (a int, b varchar, c boolean)", tableName));
        this.assertUpdate(String.format("INSERT INTO %s values(1, 'INDIA', true)", tableName), 1L);
        this.assertUpdate(String.format("INSERT INTO %s values(2, 'USA', false)", tableName), 1L);
        String tableLocation = this.getTableLocation(tableName);
        String tableNameNew = tableName + "_new";
        this.dropTableFromMetastore(tableName);
        this.assertUpdate(String.format("CALL system.register_table (CURRENT_SCHEMA, '%s', '%s')", tableNameNew, tableLocation));
        this.assertUpdate(String.format("INSERT INTO %s values(3, 'POLAND', true)", tableNameNew), 1L);
        ((QueryAssertions.QueryAssert)Assertions.assertThat((AssertProvider)this.query(String.format("SELECT * FROM %s", tableNameNew)))).matches("VALUES ROW(INT '1', VARCHAR 'INDIA', BOOLEAN 'true'), ROW(INT '2', VARCHAR 'USA', BOOLEAN 'false'), ROW(INT '3', VARCHAR 'POLAND', BOOLEAN 'true')");
        this.assertUpdate(String.format("DROP TABLE %s", tableNameNew));
    }

    @Test
    public void testRegisterTableWithMetadataFile() {
        String tableName = "test_register_table_with_metadata_file_" + TestingNames.randomNameSuffix();
        this.assertUpdate(String.format("CREATE TABLE %s (a int, b varchar, c boolean)", tableName));
        this.assertUpdate(String.format("INSERT INTO %s values(1, 'INDIA', true)", tableName), 1L);
        this.assertUpdate(String.format("INSERT INTO %s values(2, 'USA', false)", tableName), 1L);
        String tableLocation = this.getTableLocation(tableName);
        String metadataLocation = this.getMetadataLocation(tableName);
        String metadataFileName = metadataLocation.substring(metadataLocation.lastIndexOf("/") + 1);
        this.dropTableFromMetastore(tableName);
        this.assertUpdate("CALL iceberg.system.register_table (CURRENT_SCHEMA, '" + tableName + "', '" + tableLocation + "', '" + metadataFileName + "')");
        this.assertUpdate(String.format("INSERT INTO %s values(3, 'POLAND', true)", tableName), 1L);
        ((QueryAssertions.QueryAssert)Assertions.assertThat((AssertProvider)this.query(String.format("SELECT * FROM %s", tableName)))).matches("VALUES ROW(INT '1', VARCHAR 'INDIA', BOOLEAN 'true'), ROW(INT '2', VARCHAR 'USA', BOOLEAN 'false'), ROW(INT '3', VARCHAR 'POLAND', BOOLEAN 'true')");
        this.assertUpdate(String.format("DROP TABLE %s", tableName));
    }

    @Test
    public void testCreateTableWithTrailingSpaceInLocation() {
        String tableName = "test_create_table_with_trailing_space_" + TestingNames.randomNameSuffix();
        String tableLocationWithTrailingSpace = this.schemaPath() + tableName + " ";
        this.assertQuerySucceeds(String.format("CREATE TABLE %s WITH (location = '%s') AS SELECT 1 AS a, 'INDIA' AS b, true AS c", tableName, tableLocationWithTrailingSpace));
        this.assertQuery("SELECT * FROM " + tableName, "VALUES (1, 'INDIA', true)");
        Assertions.assertThat((String)this.getTableLocation(tableName)).isEqualTo(tableLocationWithTrailingSpace);
        this.assertUpdate("DROP TABLE " + tableName);
    }

    @Test
    public void testRegisterTableWithTrailingSpaceInLocation() {
        String tableName = "test_create_table_with_trailing_space_" + TestingNames.randomNameSuffix();
        String tableLocationWithTrailingSpace = this.schemaPath() + tableName + " ";
        this.assertQuerySucceeds(String.format("CREATE TABLE %s WITH (location = '%s') AS SELECT 1 AS a, 'INDIA' AS b, true AS c", tableName, tableLocationWithTrailingSpace));
        String registeredTableName = "test_register_table_with_trailing_space_" + TestingNames.randomNameSuffix();
        this.assertUpdate(String.format("CALL system.register_table(CURRENT_SCHEMA, '%s', '%s')", registeredTableName, tableLocationWithTrailingSpace));
        this.assertQuery("SELECT * FROM " + registeredTableName, "VALUES (1, 'INDIA', true)");
        Assertions.assertThat((String)this.getTableLocation(registeredTableName)).isEqualTo(tableLocationWithTrailingSpace);
        this.assertUpdate("DROP TABLE " + registeredTableName);
        this.dropTableFromMetastore(tableName);
    }

    @Test
    public void testUnregisterTable() {
        String tableName = "test_unregister_table_" + TestingNames.randomNameSuffix();
        this.assertUpdate("CREATE TABLE " + tableName + " AS SELECT 1 a", 1L);
        String tableLocation = this.getTableLocation(tableName);
        this.assertUpdate("CALL system.unregister_table(CURRENT_SCHEMA, '" + tableName + "')");
        this.assertQueryFails("SELECT * FROM " + tableName, ".* Table .* does not exist");
        this.assertUpdate("CALL iceberg.system.register_table(CURRENT_SCHEMA, '" + tableName + "', '" + tableLocation + "')");
        this.assertQuery("SELECT * FROM " + tableName, "VALUES 1");
        this.assertUpdate("DROP TABLE " + tableName);
    }

    @Test
    public void testUnregisterBrokenTable() {
        String tableName = "test_unregister_broken_table_" + TestingNames.randomNameSuffix();
        this.assertUpdate("CREATE TABLE " + tableName + " AS SELECT 1 a", 1L);
        String tableLocation = this.getTableLocation(tableName);
        this.deleteDirectory(tableLocation);
        this.assertUpdate("CALL system.unregister_table(CURRENT_SCHEMA, '" + tableName + "')");
        this.assertQueryFails("SELECT * FROM " + tableName, ".* Table .* does not exist");
    }

    protected abstract void deleteDirectory(String var1);

    @Test
    public void testUnregisterTableNotExistingSchema() {
        String schemaName = "test_unregister_table_not_existing_schema_" + TestingNames.randomNameSuffix();
        this.assertQueryFails("CALL system.unregister_table('" + schemaName + "', 'non_existent_table')", "Schema " + schemaName + " not found");
    }

    @Test
    public void testUnregisterTableNotExistingTable() {
        String tableName = "test_unregister_table_not_existing_table_" + TestingNames.randomNameSuffix();
        this.assertQueryFails("CALL system.unregister_table(CURRENT_SCHEMA, '" + tableName + "')", "Table .* not found");
    }

    @Test
    public void testRepeatUnregisterTable() {
        String tableName = "test_repeat_unregister_table_not_" + TestingNames.randomNameSuffix();
        this.assertQueryFails("CALL system.unregister_table(CURRENT_SCHEMA, '" + tableName + "')", "Table .* not found");
        this.assertUpdate("CREATE TABLE " + tableName + " AS SELECT 1 a", 1L);
        String tableLocation = this.getTableLocation(tableName);
        this.assertUpdate("CALL system.unregister_table(CURRENT_SCHEMA, '" + tableName + "')");
        this.assertQueryFails("CALL system.unregister_table(CURRENT_SCHEMA, '" + tableName + "')", "Table .* not found");
        this.assertUpdate("CALL iceberg.system.register_table(CURRENT_SCHEMA, '" + tableName + "', '" + tableLocation + "')");
        this.assertQuery("SELECT * FROM " + tableName, "VALUES 1");
        this.assertUpdate("DROP TABLE " + tableName);
    }

    @Test
    public void testUnregisterTableAccessControl() {
        String tableName = "test_unregister_table_access_control_" + TestingNames.randomNameSuffix();
        this.assertUpdate("CREATE TABLE " + tableName + " AS SELECT 1 a", 1L);
        this.assertAccessDenied("CALL system.unregister_table(CURRENT_SCHEMA, '" + tableName + "')", "Cannot drop table .*", new TestingAccessControlManager.TestingPrivilege[]{TestingAccessControlManager.privilege((String)tableName, (TestingAccessControlManager.TestingPrivilegeType)TestingAccessControlManager.TestingPrivilegeType.DROP_TABLE)});
        this.assertQuery("SELECT * FROM " + tableName, "VALUES 1");
        this.assertUpdate("DROP TABLE " + tableName);
    }

    @Test
    public void testCreateTableWithNonExistingSchemaVerifyLocation() {
        String schemaName = "non_existing_schema_" + TestingNames.randomNameSuffix();
        String tableName = "test_create_table_in_non_existent_schema_" + TestingNames.randomNameSuffix();
        String tableLocation = this.schemaPath() + "/" + tableName;
        this.assertQueryFails("CREATE TABLE " + schemaName + "." + tableName + " (a int, b int) WITH (location = '" + tableLocation + "')", "Schema (.*) not found");
        ((AbstractBooleanAssert)Assertions.assertThat((boolean)this.locationExists(tableLocation)).as("location should not exist", new Object[0])).isFalse();
        this.assertQueryFails("CREATE TABLE " + schemaName + "." + tableName + " (a, b) WITH (location = '" + tableLocation + "') AS VALUES (1, 2), (3, 4)", "Schema (.*) not found");
        ((AbstractBooleanAssert)Assertions.assertThat((boolean)this.locationExists(tableLocation)).as("location should not exist", new Object[0])).isFalse();
    }

    @Test
    public void testSortedNationTable() {
        Session withSmallRowGroups = IcebergTestUtils.withSmallRowGroups(this.getSession());
        try (TestTable table = this.newTrinoTable("test_sorted_nation_table", "WITH (sorted_by = ARRAY['comment'], format = '" + this.format.name() + "') AS SELECT * FROM nation WITH NO DATA");){
            this.assertUpdate(withSmallRowGroups, "INSERT INTO " + table.getName() + " SELECT * FROM nation", 25L);
            for (Object filePath : this.computeActual("SELECT file_path from \"" + table.getName() + "$files\"").getOnlyColumnAsSet()) {
                Assertions.assertThat((boolean)this.isFileSorted(Location.of((String)((String)filePath)), "comment")).isTrue();
            }
            this.assertQuery("SELECT * FROM " + table.getName(), "SELECT * FROM nation");
        }
    }

    @Test
    public void testFileSortingWithLargerTable() {
        Session withSmallRowGroups = Session.builder((Session)this.getSession()).setCatalogSessionProperty("iceberg", "orc_writer_max_stripe_rows", "200").setCatalogSessionProperty("iceberg", "parquet_writer_block_size", "20kB").setCatalogSessionProperty("iceberg", "parquet_writer_batch_size", "200").build();
        try (TestTable table = this.newTrinoTable("test_sorted_lineitem_table", "WITH (sorted_by = ARRAY['comment'], format = '" + this.format.name() + "') AS TABLE tpch.tiny.lineitem WITH NO DATA");){
            this.assertUpdate(withSmallRowGroups, "INSERT INTO " + table.getName() + " TABLE tpch.tiny.lineitem", "VALUES 60175");
            for (Object filePath : this.computeActual("SELECT file_path from \"" + table.getName() + "$files\"").getOnlyColumnAsSet()) {
                Assertions.assertThat((boolean)this.isFileSorted(Location.of((String)((String)filePath)), "comment")).isTrue();
            }
            this.assertQuery("SELECT * FROM " + table.getName(), "SELECT * FROM lineitem");
        }
    }

    @Test
    public void testDropTableWithMissingMetadataFile() throws Exception {
        String tableName = "test_drop_table_with_missing_metadata_file_" + TestingNames.randomNameSuffix();
        this.assertUpdate("CREATE TABLE " + tableName + " AS SELECT 1 x, 'INDIA' y", 1L);
        Location metadataLocation = Location.of((String)this.getMetadataLocation(tableName));
        Location tableLocation = Location.of((String)this.getTableLocation(tableName));
        this.fileSystem.deleteFile(metadataLocation);
        ((AbstractBooleanAssert)Assertions.assertThat((boolean)this.fileSystem.newInputFile(metadataLocation).exists()).describedAs("Current metadata file should not exist", new Object[0])).isFalse();
        this.assertUpdate("DROP TABLE " + tableName);
        Assertions.assertThat((boolean)this.getQueryRunner().tableExists(this.getSession(), tableName)).isFalse();
        ((AbstractBooleanAssert)Assertions.assertThat((boolean)this.fileSystem.listFiles(tableLocation).hasNext()).describedAs("Table location should not exist", new Object[0])).isFalse();
    }

    @Test
    public void testDropTableWithMissingSnapshotFile() throws Exception {
        String tableName = "test_drop_table_with_missing_snapshot_file_" + TestingNames.randomNameSuffix();
        this.assertUpdate("CREATE TABLE " + tableName + " AS SELECT 1 x, 'INDIA' y", 1L);
        String metadataLocation = this.getMetadataLocation(tableName);
        TableMetadata tableMetadata = TableMetadataParser.read((FileIO)new ForwardingFileIo(this.fileSystem), (String)metadataLocation);
        Location tableLocation = Location.of((String)tableMetadata.location());
        Location currentSnapshotFile = Location.of((String)tableMetadata.currentSnapshot().manifestListLocation());
        this.fileSystem.deleteFile(currentSnapshotFile);
        ((AbstractBooleanAssert)Assertions.assertThat((boolean)this.fileSystem.newInputFile(currentSnapshotFile).exists()).describedAs("Current snapshot file should not exist", new Object[0])).isFalse();
        this.assertUpdate("DROP TABLE " + tableName);
        Assertions.assertThat((boolean)this.getQueryRunner().tableExists(this.getSession(), tableName)).isFalse();
        ((AbstractBooleanAssert)Assertions.assertThat((boolean)this.fileSystem.listFiles(tableLocation).hasNext()).describedAs("Table location should not exist", new Object[0])).isFalse();
    }

    @Test
    public void testDropTableWithMissingManifestListFile() throws Exception {
        String tableName = "test_drop_table_with_missing_manifest_list_file_" + TestingNames.randomNameSuffix();
        this.assertUpdate("CREATE TABLE " + tableName + " AS SELECT 1 x, 'INDIA' y", 1L);
        String metadataLocation = this.getMetadataLocation(tableName);
        ForwardingFileIo fileIo = new ForwardingFileIo(this.fileSystem);
        TableMetadata tableMetadata = TableMetadataParser.read((FileIO)fileIo, (String)metadataLocation);
        Location tableLocation = Location.of((String)tableMetadata.location());
        Location manifestListFile = Location.of((String)((ManifestFile)tableMetadata.currentSnapshot().allManifests((FileIO)fileIo).get(0)).path());
        this.fileSystem.deleteFile(manifestListFile);
        ((AbstractBooleanAssert)Assertions.assertThat((boolean)this.fileSystem.newInputFile(manifestListFile).exists()).describedAs("Manifest list file should not exist", new Object[0])).isFalse();
        this.assertUpdate("DROP TABLE " + tableName);
        Assertions.assertThat((boolean)this.getQueryRunner().tableExists(this.getSession(), tableName)).isFalse();
        ((AbstractBooleanAssert)Assertions.assertThat((boolean)this.fileSystem.listFiles(tableLocation).hasNext()).describedAs("Table location should not exist", new Object[0])).isFalse();
    }

    @Test
    public void testDropTableWithMissingDataFile() throws Exception {
        String tableName = "test_drop_table_with_missing_data_file_" + TestingNames.randomNameSuffix();
        this.assertUpdate("CREATE TABLE " + tableName + " AS SELECT 1 x, 'INDIA' y", 1L);
        this.assertUpdate("INSERT INTO " + tableName + " VALUES (2, 'POLAND')", 1L);
        Location tableLocation = Location.of((String)this.getTableLocation(tableName));
        Location tableDataPath = tableLocation.appendPath("data");
        FileIterator fileIterator = this.fileSystem.listFiles(tableDataPath);
        Assertions.assertThat((boolean)fileIterator.hasNext()).isTrue();
        Location dataFile = fileIterator.next().location();
        this.fileSystem.deleteFile(dataFile);
        ((AbstractBooleanAssert)Assertions.assertThat((boolean)this.fileSystem.newInputFile(dataFile).exists()).describedAs("Data file should not exist", new Object[0])).isFalse();
        this.assertUpdate("DROP TABLE " + tableName);
        Assertions.assertThat((boolean)this.getQueryRunner().tableExists(this.getSession(), tableName)).isFalse();
        ((AbstractBooleanAssert)Assertions.assertThat((boolean)this.fileSystem.listFiles(tableLocation).hasNext()).describedAs("Table location should not exist", new Object[0])).isFalse();
    }

    @Test
    public void testDropTableWithNonExistentTableLocation() throws Exception {
        String tableName = "test_drop_table_with_non_existent_table_location_" + TestingNames.randomNameSuffix();
        this.assertUpdate("CREATE TABLE " + tableName + " AS SELECT 1 x, 'INDIA' y", 1L);
        this.assertUpdate("INSERT INTO " + tableName + " VALUES (2, 'POLAND')", 1L);
        Location tableLocation = Location.of((String)this.getTableLocation(tableName));
        this.fileSystem.deleteDirectory(tableLocation);
        ((AbstractBooleanAssert)Assertions.assertThat((boolean)this.fileSystem.listFiles(tableLocation).hasNext()).describedAs("Table location should not exist", new Object[0])).isFalse();
        this.assertUpdate("DROP TABLE " + tableName);
        Assertions.assertThat((boolean)this.getQueryRunner().tableExists(this.getSession(), tableName)).isFalse();
    }

    @Test
    public void testMetadataTables() {
        try (TestTable table = this.newTrinoTable("test_metadata_tables", "(id int, part varchar) WITH (partitioning = ARRAY['part'])");){
            this.assertUpdate("INSERT INTO " + table.getName() + " VALUES (1, 'p1')", 1L);
            this.assertUpdate("INSERT INTO " + table.getName() + " VALUES (2, 'p1')", 1L);
            this.assertUpdate("INSERT INTO " + table.getName() + " VALUES (3, 'p2')", 1L);
            List snapshotIds = (List)this.computeActual("SELECT snapshot_id FROM \"" + table.getName() + "$snapshots\" ORDER BY committed_at DESC").getOnlyColumn().map(Long.class::cast).collect(ImmutableList.toImmutableList());
            List historySnapshotIds = (List)this.computeActual("SELECT snapshot_id FROM \"" + table.getName() + "$history\" ORDER BY made_current_at DESC").getOnlyColumn().map(Long.class::cast).collect(ImmutableList.toImmutableList());
            long filesCount = (Long)this.computeScalar("SELECT count(*) FROM \"" + table.getName() + "$files\"");
            long partitionsCount = (Long)this.computeScalar("SELECT count(*) FROM \"" + table.getName() + "$partitions\"");
            Assertions.assertThat((List)snapshotIds).hasSize(4);
            Assertions.assertThat((List)snapshotIds).hasSameElementsAs((Iterable)historySnapshotIds);
            Assertions.assertThat((long)filesCount).isEqualTo(3L);
            Assertions.assertThat((long)partitionsCount).isEqualTo(2L);
        }
    }

    @Test
    public void testPartitionFilterRequired() {
        String tableName = "test_partition_" + TestingNames.randomNameSuffix();
        Session session = Session.builder((Session)this.getSession()).setCatalogSessionProperty("iceberg", "query_partition_filter_required", "true").build();
        this.assertUpdate(session, "CREATE TABLE " + tableName + " (id integer, a varchar, b varchar, ds varchar) WITH (partitioning = ARRAY['ds'])");
        this.assertUpdate(session, "INSERT INTO " + tableName + " (id, a, ds) VALUES (1, 'a', 'a')", 1L);
        String query = "SELECT id FROM " + tableName + " WHERE a = 'a'";
        String failureMessage = "Filter required for .*" + tableName + " on at least one of the partition columns: ds";
        this.assertQueryFails(session, query, failureMessage);
        this.assertQueryFails(session, "EXPLAIN " + query, failureMessage);
        this.assertUpdate(session, "DROP TABLE " + tableName);
    }

    protected abstract boolean isFileSorted(Location var1, String var2);

    @Test
    public void testTableChangesFunction() {
        DateTimeFormatter instantMillisFormatter = DateTimeFormatter.ofPattern("uuuu-MM-dd'T'HH:mm:ss.SSSVV").withZone(ZoneOffset.UTC);
        try (TestTable table = this.newTrinoTable("test_table_changes_function_", "AS SELECT nationkey, name FROM tpch.tiny.nation WITH NO DATA");){
            long initialSnapshot = this.getMostRecentSnapshotId(table.getName());
            this.assertUpdate("INSERT INTO " + table.getName() + " SELECT nationkey, name FROM nation", 25L);
            long snapshotAfterInsert = this.getMostRecentSnapshotId(table.getName());
            String snapshotAfterInsertTime = this.getSnapshotTime(table.getName(), snapshotAfterInsert).format(instantMillisFormatter);
            this.assertQuery("SELECT nationkey, name, _change_type, _change_version_id, to_iso8601(_change_timestamp), _change_ordinal " + "FROM TABLE(system.table_changes(CURRENT_SCHEMA, '%s', %s, %s))".formatted(table.getName(), initialSnapshot, snapshotAfterInsert), "SELECT nationkey, name, 'insert', %s, '%s', 0 FROM nation".formatted(snapshotAfterInsert, snapshotAfterInsertTime));
            this.assertQuery("SELECT nationkey, name, _change_type, _change_version_id, to_iso8601(_change_timestamp), _change_ordinal " + "FROM TABLE(system.table_changes(schema_name => CURRENT_SCHEMA, table_name => '%s', start_snapshot_id => %s, end_snapshot_id => %s))".formatted(table.getName(), initialSnapshot, snapshotAfterInsert), "SELECT nationkey, name, 'insert', %s, '%s', 0 FROM nation".formatted(snapshotAfterInsert, snapshotAfterInsertTime));
            this.assertUpdate("DELETE FROM " + table.getName(), 25L);
            long snapshotAfterDelete = this.getMostRecentSnapshotId(table.getName());
            String snapshotAfterDeleteTime = this.getSnapshotTime(table.getName(), snapshotAfterDelete).format(instantMillisFormatter);
            this.assertQuery("SELECT nationkey, name, _change_type, _change_version_id, to_iso8601(_change_timestamp), _change_ordinal " + "FROM TABLE(system.table_changes(CURRENT_SCHEMA, '%s', %s, %s))".formatted(table.getName(), snapshotAfterInsert, snapshotAfterDelete), "SELECT nationkey, name, 'delete', %s, '%s', 0 FROM nation".formatted(snapshotAfterDelete, snapshotAfterDeleteTime));
            this.assertQuery("SELECT nationkey, name, _change_type, _change_version_id, to_iso8601(_change_timestamp), _change_ordinal " + "FROM TABLE(system.table_changes(CURRENT_SCHEMA, '%s', %s, %s))".formatted(table.getName(), initialSnapshot, snapshotAfterDelete), "SELECT nationkey, name, 'insert', %s, '%s', 0 FROM nation UNION SELECT nationkey, name, 'delete', %s, '%s', 1 FROM nation".formatted(snapshotAfterInsert, snapshotAfterInsertTime, snapshotAfterDelete, snapshotAfterDeleteTime));
        }
    }

    @Test
    public void testRowLevelDeletesWithTableChangesFunction() {
        try (TestTable table = this.newTrinoTable("test_row_level_deletes_with_table_changes_function_", "AS SELECT nationkey, regionkey, name FROM tpch.tiny.nation WITH NO DATA");){
            this.assertUpdate("INSERT INTO " + table.getName() + " SELECT nationkey, regionkey, name FROM nation", 25L);
            long snapshotAfterInsert = this.getMostRecentSnapshotId(table.getName());
            this.assertUpdate("DELETE FROM " + table.getName() + " WHERE regionkey = 2", 5L);
            long snapshotAfterDelete = this.getMostRecentSnapshotId(table.getName());
            this.assertQueryFails("SELECT * FROM TABLE(system.table_changes(CURRENT_SCHEMA, '%s', %s, %s))".formatted(table.getName(), snapshotAfterInsert, snapshotAfterDelete), "Table uses features which are not yet supported by the table_changes function");
        }
    }

    @Test
    public void testCreateOrReplaceWithTableChangesFunction() {
        DateTimeFormatter instantMillisFormatter = DateTimeFormatter.ofPattern("uuuu-MM-dd'T'HH:mm:ss.SSSVV").withZone(ZoneOffset.UTC);
        try (TestTable table = this.newTrinoTable("test_table_changes_function_", "AS SELECT nationkey, name FROM tpch.tiny.nation WITH NO DATA");){
            long initialSnapshot = this.getMostRecentSnapshotId(table.getName());
            this.assertUpdate("INSERT INTO " + table.getName() + " SELECT nationkey, name FROM nation", 25L);
            long snapshotAfterInsert = this.getMostRecentSnapshotId(table.getName());
            String snapshotAfterInsertTime = this.getSnapshotTime(table.getName(), snapshotAfterInsert).format(instantMillisFormatter);
            this.assertQuery("SELECT nationkey, name, _change_type, _change_version_id, to_iso8601(_change_timestamp), _change_ordinal " + "FROM TABLE(system.table_changes(CURRENT_SCHEMA, '%s', %s, %s))".formatted(table.getName(), initialSnapshot, snapshotAfterInsert), "SELECT nationkey, name, 'insert', %s, '%s', 0 FROM nation".formatted(snapshotAfterInsert, snapshotAfterInsertTime));
            this.assertUpdate("CREATE OR REPLACE TABLE " + table.getName() + " AS SELECT nationkey, name FROM nation LIMIT 0", 0L);
            long snapshotAfterCreateOrReplace = this.getMostRecentSnapshotId(table.getName());
            this.assertQueryFails("SELECT * FROM TABLE(system.table_changes(CURRENT_SCHEMA, '%s', %s, %s))".formatted(table.getName(), initialSnapshot, snapshotAfterCreateOrReplace), "Starting snapshot \\(exclusive\\) %s is not a parent ancestor of end snapshot %s".formatted(initialSnapshot, snapshotAfterCreateOrReplace));
            this.assertUpdate("INSERT INTO " + table.getName() + " SELECT nationkey, name FROM nation", 25L);
            long snapshotAfterInsertIntoCreateOrReplace = this.getMostRecentSnapshotId(table.getName());
            String snapshotAfterInsertTimeIntoCreateOrReplace = this.getSnapshotTime(table.getName(), snapshotAfterInsertIntoCreateOrReplace).format(instantMillisFormatter);
            this.assertQuery("SELECT nationkey, name, _change_type, _change_version_id, to_iso8601(_change_timestamp), _change_ordinal " + "FROM TABLE(system.table_changes(CURRENT_SCHEMA, '%s', %s, %s))".formatted(table.getName(), snapshotAfterCreateOrReplace, snapshotAfterInsertIntoCreateOrReplace), "SELECT nationkey, name, 'insert', %s, '%s', 0 FROM nation".formatted(snapshotAfterInsertIntoCreateOrReplace, snapshotAfterInsertTimeIntoCreateOrReplace));
        }
    }

    @Test
    public void testMetadataDeleteAfterCommitEnabled() throws IOException {
        if (!this.hasBehavior(TestingConnectorBehavior.SUPPORTS_CREATE_TABLE)) {
            return;
        }
        int metadataPreviousVersionCount = 5;
        String tableName = "test_metadata_delete_after_commit_enabled" + TestingNames.randomNameSuffix();
        this.assertUpdate("CREATE TABLE " + tableName + "(_bigint BIGINT, _varchar VARCHAR)");
        this.assertUpdate("ALTER TABLE " + tableName + " SET PROPERTIES extra_properties = MAP(ARRAY['write.metadata.delete-after-commit.enabled'], ARRAY['true'])");
        this.assertUpdate("ALTER TABLE " + tableName + " SET PROPERTIES extra_properties = MAP(ARRAY['write.metadata.previous-versions-max'], ARRAY['" + metadataPreviousVersionCount + "'])");
        String tableLocation = this.getTableLocation(tableName);
        Map<String, Long> historyMetadataFiles = IcebergTestUtils.getMetadataFileAndUpdatedMillis(this.fileSystem, tableLocation);
        for (int i = 0; i < 10; ++i) {
            this.assertUpdate("INSERT INTO " + tableName + " VALUES (1, 'a')", 1L);
            Map<String, Long> metadataFiles = IcebergTestUtils.getMetadataFileAndUpdatedMillis(this.fileSystem, tableLocation);
            historyMetadataFiles.putAll(metadataFiles);
            Assertions.assertThat((int)metadataFiles.size()).isLessThanOrEqualTo(1 + metadataPreviousVersionCount);
            Set expectMetadataFiles = historyMetadataFiles.entrySet().stream().sorted(Map.Entry.comparingByValue().reversed()).limit(metadataPreviousVersionCount + 1).map(Map.Entry::getKey).collect(Collectors.toSet());
            Assertions.assertThat(metadataFiles.keySet()).containsAll(expectMetadataFiles);
        }
        this.assertUpdate("DROP TABLE " + tableName);
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    @Test
    public void testIcebergTablesSystemTable() throws Exception {
        String firstSchema = "first_schema_" + TestingNames.randomNameSuffix();
        String secondSchema = "second_schema_" + TestingNames.randomNameSuffix();
        this.createSchema(firstSchema);
        this.createSchema(secondSchema);
        try (AutoCloseable autoCloseable = this.createTable(firstSchema, "first_schema_table1", "(id int)");
             AutoCloseable autoCloseable2 = this.createTable(firstSchema, "first_schema_table2", "(id int)");
             AutoCloseable autoCloseable3 = this.createTable(secondSchema, "second_schema_table", "(id int)");
             AutoCloseable autoCloseable4 = this.createSparkIcebergTable(firstSchema);){
            ((QueryAssertions.QueryAssert)Assertions.assertThat((AssertProvider)this.query("SELECT * FROM iceberg.system.iceberg_tables WHERE table_schema = '%s'".formatted(firstSchema)))).matches("SELECT table_schema, table_name FROM iceberg.information_schema.tables WHERE table_schema='%s'".formatted(firstSchema));
            ((QueryAssertions.QueryAssert)Assertions.assertThat((AssertProvider)this.query("SELECT * FROM iceberg.system.iceberg_tables WHERE table_schema in ('%s', '%s')".formatted(firstSchema, secondSchema)))).matches("SELECT table_schema, table_name FROM iceberg.information_schema.tables WHERE table_schema IN ('%s', '%s')".formatted(firstSchema, secondSchema));
        }
        finally {
            this.dropSchema(firstSchema);
            this.dropSchema(secondSchema);
        }
    }

    protected void dropSchema(String schema) throws Exception {
        this.assertQuerySucceeds("DROP SCHEMA " + schema);
    }

    protected AutoCloseable createTable(String schema, String tableName, String tableDefinition) throws Exception {
        Session schemaSession = Session.builder((Session)this.getQueryRunner().getDefaultSession()).setSchema(schema).build();
        return new TestTable(sql -> this.getQueryRunner().execute(schemaSession, sql), tableName, tableDefinition);
    }

    protected void createSchema(String schemaName) throws Exception {
        String defaultSchemaName = (String)this.getSession().getSchema().orElseThrow();
        String schemaLocation = this.schemaPath().replaceAll(defaultSchemaName, schemaName);
        this.assertQuerySucceeds("CREATE SCHEMA " + schemaName + " WITH (location = '%s')".formatted(schemaLocation));
    }

    protected AutoCloseable createSparkIcebergTable(String schema) {
        return () -> {};
    }

    private long getMostRecentSnapshotId(String tableName) {
        return (Long)Iterables.getOnlyElement((Iterable)this.getQueryRunner().execute(String.format("SELECT snapshot_id FROM \"%s$snapshots\" ORDER BY committed_at DESC LIMIT 1", tableName)).getOnlyColumnAsSet());
    }

    private ZonedDateTime getSnapshotTime(String tableName, long snapshotId) {
        return (ZonedDateTime)Iterables.getOnlyElement((Iterable)this.getQueryRunner().execute(String.format("SELECT committed_at FROM \"%s$snapshots\" WHERE snapshot_id = %s", tableName, snapshotId)).getOnlyColumnAsSet());
    }

    protected String getTableLocation(String tableName) {
        Pattern locationPattern = Pattern.compile(".*location = '(.*?)'.*", 32);
        Matcher m = locationPattern.matcher((String)this.computeActual("SHOW CREATE TABLE " + tableName).getOnlyValue());
        if (m.find()) {
            String location = m.group(1);
            Verify.verify((!m.find() ? 1 : 0) != 0, (String)"Unexpected second match", (Object[])new Object[0]);
            return location;
        }
        throw new IllegalStateException("Location not found in SHOW CREATE TABLE result");
    }

    protected abstract void dropTableFromMetastore(String var1);

    protected abstract String getMetadataLocation(String var1);

    protected abstract String schemaPath();

    protected abstract boolean locationExists(String var1);

    private static /* synthetic */ Optional lambda$testDeleteRowsConcurrently$2(List rows, Future future, long index) {
        Optional value = MoreFutures.tryGetFutureValue((Future)future, (int)20, (TimeUnit)TimeUnit.SECONDS);
        Preconditions.checkState((boolean)value.isPresent(), (String)"Task %s did not complete in time", (long)index);
        boolean deleteSuccessful = (Boolean)value.get();
        return deleteSuccessful ? Optional.empty() : Optional.of((String)rows.get((int)index));
    }
}

