/*
 * Decompiled with CFR 0.152.
 */
package io.trino.plugin.iceberg;

import com.google.common.base.Verify;
import com.google.common.primitives.Primitives;
import io.airlift.slice.Slice;
import io.airlift.slice.Slices;
import io.trino.plugin.iceberg.PartitionTransforms;
import io.trino.plugin.iceberg.TypeConverter;
import io.trino.spi.block.Block;
import io.trino.spi.block.BlockBuilder;
import io.trino.spi.type.BigintType;
import io.trino.spi.type.DateType;
import io.trino.spi.type.DecimalType;
import io.trino.spi.type.Decimals;
import io.trino.spi.type.IntegerType;
import io.trino.spi.type.LongTimestampWithTimeZone;
import io.trino.spi.type.TestingTypeManager;
import io.trino.spi.type.TimeType;
import io.trino.spi.type.TimeZoneKey;
import io.trino.spi.type.TimestampType;
import io.trino.spi.type.TimestampWithTimeZoneType;
import io.trino.spi.type.TypeManager;
import io.trino.spi.type.TypeUtils;
import io.trino.spi.type.UuidType;
import io.trino.spi.type.VarbinaryType;
import io.trino.spi.type.VarcharType;
import io.trino.type.UuidOperators;
import java.math.BigDecimal;
import java.nio.ByteBuffer;
import java.nio.charset.StandardCharsets;
import java.time.LocalDate;
import java.time.LocalDateTime;
import java.time.LocalTime;
import java.time.ZoneOffset;
import java.util.UUID;
import java.util.function.Function;
import org.apache.iceberg.transforms.Transform;
import org.apache.iceberg.transforms.Transforms;
import org.apache.iceberg.types.Type;
import org.apache.iceberg.types.Types;
import org.assertj.core.api.AbstractIntegerAssert;
import org.assertj.core.api.Assertions;
import org.junit.jupiter.api.Test;
import org.junit.jupiter.params.ParameterizedTest;
import org.junit.jupiter.params.provider.MethodSource;

public class TestIcebergBucketing {
    private static final TypeManager TYPE_MANAGER = new TestingTypeManager();

    @Test
    public void testBucketNumberCompare() {
        this.assertBucketAndHashEquals("int", null, null);
        this.assertBucketAndHashEquals("int", 0, 1669671676);
        this.assertBucketAndHashEquals("int", 300000, 1798339266);
        this.assertBucketAndHashEquals("int", Integer.MIN_VALUE, 74448856);
        this.assertBucketAndHashEquals("int", Integer.MAX_VALUE, 1819228606);
        this.assertBucketAndHashEquals("long", null, null);
        this.assertBucketAndHashEquals("long", 0L, 1669671676);
        this.assertBucketAndHashEquals("long", 300000000000L, 371234369);
        this.assertBucketAndHashEquals("long", Long.MIN_VALUE, 1366273829);
        this.assertBucketAndHashEquals("long", Long.MAX_VALUE, 40977599);
        this.assertBucketAndHashEquals("decimal(1, 1)", null, null);
        this.assertBucketAndHashEquals("decimal(1, 0)", "0", 1364076727);
        this.assertBucketAndHashEquals("decimal(1, 0)", "1", 1683673515);
        this.assertBucketAndHashEquals("decimal(1, 0)", "9", 1771774483);
        this.assertBucketAndHashEquals("decimal(1, 0)", "-9", 156162024);
        this.assertBucketAndHashEquals("decimal(3, 1)", "0.1", 1683673515);
        this.assertBucketAndHashEquals("decimal(3, 1)", "1.0", 307159211);
        this.assertBucketAndHashEquals("decimal(3, 1)", "12.3", 1308316337);
        this.assertBucketAndHashEquals("decimal(3, 1)", "-12.3", 1847027525);
        this.assertBucketAndHashEquals("decimal(18, 10)", "0", 1364076727);
        this.assertBucketAndHashEquals("decimal(38, 10)", null, null);
        this.assertBucketAndHashEquals("decimal(38, 10)", "999999.9999999999", 1053577599);
        this.assertBucketAndHashEquals("decimal(38, 10)", "-999999.9999999999", 1054888790);
        this.assertBucketAndHashEquals("decimal(38, 0)", "99999999999999999999999999999999999999", 1067515814);
        this.assertBucketAndHashEquals("decimal(38, 0)", "-99999999999999999999999999999999999999", 193266010);
        this.assertBucketAndHashEquals("decimal(38, 10)", "9999999999999999999999999999.9999999999", 1067515814);
        this.assertBucketAndHashEquals("decimal(38, 10)", "-9999999999999999999999999999.9999999999", 193266010);
        this.assertBucketAndHashEquals("decimal(38, 10)", "123456789012345.0", 93815101);
        this.assertBucketAndHashEquals("decimal(38, 10)", "-123456789012345.0", 522439017);
        this.assertBucketAndHashEquals("string", null, null);
        this.assertBucketAndHashEquals("string", "", 0);
        this.assertBucketAndHashEquals("string", "test string", 671244848);
        this.assertBucketAndHashEquals("string", "Trino rocks", 2131833594);
        this.assertBucketAndHashEquals("string", "\u5f3a\u5927\u7684Trino\u5f15\u64ce", 822296301);
        this.assertBucketAndHashEquals("string", "\ud83d\udcb0", 661122892);
        this.assertBucketAndHashEquals("string", "\ud843\udffc\ud843\udffd\ud843\udffe\ud843\udfff", 2094039023);
        this.assertBucketAndHashEquals("binary", null, null);
        this.assertBucketAndHashEquals("binary", ByteBuffer.wrap(new byte[0]), 0);
        this.assertBucketAndHashEquals("binary", ByteBuffer.wrap("hello trino".getBytes(StandardCharsets.UTF_8)), 493441885);
        this.assertBucketAndHashEquals("binary", ByteBuffer.wrap("\ud843\udffc\ud843\udffd\ud843\udffe\ud843\udfff".getBytes(StandardCharsets.UTF_16)), 1291558121);
        this.assertBucketAndHashEquals("uuid", null, null);
        this.assertBucketAndHashEquals("uuid", UUID.fromString("00000000-0000-0000-0000-000000000000"), 20237816);
        this.assertBucketAndHashEquals("uuid", UUID.fromString("1-2-3-4-5"), 1802237169);
        this.assertBucketAndHashEquals("uuid", UUID.fromString("406caec7-68b9-4778-81b2-a12ece70c8b1"), 1231261529);
        this.assertBucketAndHashEquals("fixed[3]", null, null);
        this.assertBucketAndHashEquals("fixed[3]", ByteBuffer.wrap(new byte[]{0, 0, 0}), 99660839);
        this.assertBucketAndHashEquals("fixed[3]", ByteBuffer.wrap(new byte[]{1, 2, 3}), 13750788);
        this.assertBucketAndHashEquals("fixed[3]", ByteBuffer.wrap(new byte[]{127, -128, 1}), 107475887);
        this.assertBucketAndHashEquals("fixed[3]", ByteBuffer.wrap(new byte[]{-1, -1, -1}), 1058185254);
        this.assertBucketAndHashEquals("fixed[3]", ByteBuffer.wrap(new byte[]{127, 127, 127}), 533318325);
        this.assertBucketAndHashEquals("fixed[3]", ByteBuffer.wrap(new byte[]{-128, -128, -128}), 1945840528);
        this.assertBucketAndHashEquals("date", null, null);
        this.assertBucketAndHashEquals("date", 0, 1669671676);
        this.assertBucketAndHashEquals("date", 1, 1392991556);
        this.assertBucketAndHashEquals("date", Math.toIntExact(LocalDate.of(2005, 9, 10).toEpochDay()), 1958311396);
        this.assertBucketAndHashEquals("date", Math.toIntExact(LocalDate.of(1965, 1, 2).toEpochDay()), 1149697962);
        this.assertBucketAndHashEquals("time", null, null);
        this.assertBucketAndHashEquals("time", 0L, 1669671676);
        this.assertBucketAndHashEquals("time", 1L, 1392991556);
        this.assertBucketAndHashEquals("time", LocalTime.of(17, 13, 15, 123000000).toNanoOfDay() / 1000L, 539121226);
        this.assertBucketAndHashEquals("time", 86399999999L, 1641029256);
        this.assertBucketAndHashEquals("timestamp", null, null);
        this.assertBucketAndHashEquals("timestamp", 0L, 1669671676);
        this.assertBucketAndHashEquals("timestamp", 1L, 1392991556);
        this.assertBucketAndHashEquals("timestamp", -1L, 1651860712);
        this.assertBucketAndHashEquals("timestamp", -13L, 1222449245);
        this.assertBucketAndHashEquals("timestamp", LocalDateTime.of(2005, 9, 10, 13, 30, 15).toEpochSecond(ZoneOffset.UTC) * 1000000L + 123456L, 1162062113);
        this.assertBucketAndHashEquals("timestamp", LocalDateTime.of(1965, 1, 2, 13, 30, 15).toEpochSecond(ZoneOffset.UTC) * 1000000L + 123456L, 236109233);
        this.assertBucketAndHashEquals("timestamptz", null, null);
        this.assertBucketAndHashEquals("timestamptz", 0L, 1669671676);
        this.assertBucketAndHashEquals("timestamptz", 1L, 1392991556);
        this.assertBucketAndHashEquals("timestamptz", -1L, 1651860712);
        this.assertBucketAndHashEquals("timestamptz", -13L, 1222449245);
        this.assertBucketAndHashEquals("timestamptz", LocalDateTime.of(2005, 9, 10, 13, 30, 15).toEpochSecond(ZoneOffset.UTC) * 1000000L + 123456L, 1162062113);
        this.assertBucketAndHashEquals("timestamptz", LocalDateTime.of(1965, 1, 2, 13, 30, 15).toEpochSecond(ZoneOffset.UTC) * 1000000L + 123456L, 236109233);
    }

    @Test
    public void testBucketingSpecValues() {
        this.assertBucketAndHashEquals("int", 34, 2017239379);
        this.assertBucketAndHashEquals("long", 34L, 2017239379);
        this.assertBucketAndHashEquals("decimal(4, 2)", "14.20", 1646729059);
        this.assertBucketAndHashEquals("decimal(10, 2)", "14.20", 1646729059);
        this.assertBucketAndHashEquals("decimal(22, 2)", "14.20", 1646729059);
        this.assertBucketAndHashEquals("date", Math.toIntExact(LocalDate.of(2017, 11, 16).toEpochDay()), 1494153226);
        this.assertBucketAndHashEquals("time", LocalTime.of(22, 31, 8).toNanoOfDay() / 1000L, 1484720659);
        this.assertBucketAndHashEquals("timestamp", LocalDateTime.of(2017, 11, 16, 22, 31, 8).toEpochSecond(ZoneOffset.UTC) * 1000000L, 99539207);
        this.assertBucketAndHashEquals("timestamptz", LocalDateTime.of(2017, 11, 16, 14, 31, 8).toEpochSecond(ZoneOffset.ofHours(-8)) * 1000000L, 99539207);
        this.assertBucketAndHashEquals("string", "iceberg", 1210000089);
        this.assertBucketAndHashEquals("uuid", UUID.fromString("f79c3e09-677c-4bbd-a479-3f349cb785e7"), 1488055340);
        this.assertBucketAndHashEquals("fixed[4]", ByteBuffer.wrap(new byte[]{0, 1, 2, 3}), 1958800441);
        this.assertBucketAndHashEquals("binary", ByteBuffer.wrap(new byte[]{0, 1, 2, 3}), 1958800441);
    }

    @ParameterizedTest
    @MethodSource(value={"unsupportedBucketingTypes"})
    public void testUnsupportedTypes(Type type) {
        Assertions.assertThatThrownBy(() -> this.computeIcebergBucket(type, null, 1)).hasMessage("Cannot bucket by type: %s", new Object[]{type});
        Assertions.assertThatThrownBy(() -> this.computeTrinoBucket(type, null, 1)).hasMessage("Unsupported type for 'bucket': %s", new Object[]{TypeConverter.toTrinoType((Type)type, (TypeManager)TYPE_MANAGER)});
    }

    public static Object[][] unsupportedBucketingTypes() {
        return new Object[][]{{Types.BooleanType.get()}, {Types.FloatType.get()}, {Types.DoubleType.get()}};
    }

    private void assertBucketAndHashEquals(String icebergTypeName, Object icebergValue, Integer expectedHash) {
        Type.PrimitiveType icebergType = Types.fromPrimitiveString((String)icebergTypeName);
        if (icebergValue != null && icebergType.typeId() == Type.TypeID.DECIMAL) {
            icebergValue = new BigDecimal((String)icebergValue).setScale(((Types.DecimalType)icebergType).scale());
        }
        this.assertBucketEquals((Type)icebergType, icebergValue);
        this.assertHashEquals((Type)icebergType, icebergValue, expectedHash);
    }

    private void assertBucketEquals(Type icebergType, Object icebergValue) {
        this.assertBucketNumberEquals(icebergType, icebergValue, Integer.MAX_VALUE);
        this.assertBucketNumberEquals(icebergType, icebergValue, 2);
        this.assertBucketNumberEquals(icebergType, icebergValue, 7);
        this.assertBucketNumberEquals(icebergType, icebergValue, 31);
        this.assertBucketNumberEquals(icebergType, icebergValue, 32);
        this.assertBucketNumberEquals(icebergType, icebergValue, 100);
        this.assertBucketNumberEquals(icebergType, icebergValue, 10000);
        this.assertBucketNumberEquals(icebergType, icebergValue, 524287);
        this.assertBucketNumberEquals(icebergType, icebergValue, 0x40000000);
    }

    private void assertBucketNumberEquals(Type icebergType, Object icebergValue, int bucketCount) {
        Integer icebergBucket = this.computeIcebergBucket(icebergType, icebergValue, bucketCount);
        Integer trinoBucket = this.computeTrinoBucket(icebergType, icebergValue, bucketCount);
        ((AbstractIntegerAssert)Assertions.assertThat((Integer)trinoBucket).describedAs(String.format("icebergType=%s, bucketCount=%s, icebergBucket=%d, trinoBucket=%d;", icebergType, bucketCount, icebergBucket, trinoBucket), new Object[0])).isEqualTo((Object)icebergBucket);
    }

    private void assertHashEquals(Type icebergType, Object icebergValue, Integer expectedHash) {
        Integer icebergBucketHash = this.computeIcebergBucket(icebergType, icebergValue, Integer.MAX_VALUE);
        Integer trinoBucketHash = this.computeTrinoBucket(icebergType, icebergValue, Integer.MAX_VALUE);
        ((AbstractIntegerAssert)Assertions.assertThat((Integer)icebergBucketHash).describedAs(String.format("expected Iceberg %s(%s) bucket with %sd buckets to be %d, got %d", icebergType, icebergValue, Integer.MAX_VALUE, expectedHash, icebergBucketHash), new Object[0])).isEqualTo((Object)expectedHash);
        ((AbstractIntegerAssert)Assertions.assertThat((Integer)trinoBucketHash).describedAs(String.format("expected Trino %s(%s) bucket with %sd buckets to be %d, got %d", icebergType, icebergValue, Integer.MAX_VALUE, expectedHash, trinoBucketHash), new Object[0])).isEqualTo((Object)expectedHash);
    }

    private Integer computeIcebergBucket(Type type, Object icebergValue, int bucketCount) {
        Transform bucketTransform = Transforms.bucket((int)bucketCount);
        return (Integer)bucketTransform.bind(type).apply(icebergValue);
    }

    private Integer computeTrinoBucket(Type icebergType, Object icebergValue, int bucketCount) {
        io.trino.spi.type.Type trinoType = TypeConverter.toTrinoType((Type)icebergType, (TypeManager)TYPE_MANAGER);
        PartitionTransforms.ColumnTransform transform = PartitionTransforms.bucket((io.trino.spi.type.Type)trinoType, (int)bucketCount);
        Function blockTransform = transform.blockTransform();
        BlockBuilder blockBuilder = trinoType.createBlockBuilder(null, 1);
        Object trinoValue = TestIcebergBucketing.toTrinoValue(icebergType, icebergValue);
        Verify.verify((trinoValue == null || Primitives.wrap((Class)trinoType.getJavaType()).isInstance(trinoValue) ? 1 : 0) != 0, (String)"Unexpected value for %s: %s", (Object)trinoType, trinoValue != null ? trinoValue.getClass() : null);
        TypeUtils.writeNativeValue((io.trino.spi.type.Type)trinoType, (BlockBuilder)blockBuilder, (Object)trinoValue);
        Block block = blockBuilder.build();
        Block bucketBlock = (Block)blockTransform.apply(block);
        Verify.verify((bucketBlock.getPositionCount() == 1 ? 1 : 0) != 0);
        Integer trinoBucketWithBlock = bucketBlock.isNull(0) ? null : Integer.valueOf(IntegerType.INTEGER.getInt(bucketBlock, 0));
        Long trinoBucketWithValue = (Long)transform.valueTransform().apply(block, 0);
        Integer trinoBucketWithValueAsInteger = trinoBucketWithValue == null ? null : Integer.valueOf(Math.toIntExact(trinoBucketWithValue));
        Assertions.assertThat((Integer)trinoBucketWithValueAsInteger).isEqualTo((Object)trinoBucketWithBlock);
        return trinoBucketWithBlock;
    }

    private static Object toTrinoValue(Type icebergType, Object icebergValue) {
        io.trino.spi.type.Type trinoType = TypeConverter.toTrinoType((Type)icebergType, (TypeManager)TYPE_MANAGER);
        if (icebergValue == null) {
            return null;
        }
        if (trinoType == IntegerType.INTEGER) {
            return (long)((Integer)icebergValue).intValue();
        }
        if (trinoType == BigintType.BIGINT) {
            return (long)((Long)icebergValue);
        }
        if (trinoType instanceof DecimalType) {
            DecimalType trinoDecimalType = (DecimalType)trinoType;
            if (trinoDecimalType.isShort()) {
                return Decimals.encodeShortScaledValue((BigDecimal)((BigDecimal)icebergValue), (int)trinoDecimalType.getScale());
            }
            return Decimals.encodeScaledValue((BigDecimal)((BigDecimal)icebergValue), (int)trinoDecimalType.getScale());
        }
        if (trinoType == VarcharType.VARCHAR) {
            return Slices.utf8Slice((String)((String)icebergValue));
        }
        if (trinoType == VarbinaryType.VARBINARY) {
            return Slices.wrappedBuffer((byte[])((ByteBuffer)icebergValue).array());
        }
        if (trinoType == UuidType.UUID) {
            UUID uuidValue = (UUID)icebergValue;
            return UuidOperators.castFromVarcharToUuid((Slice)Slices.utf8Slice((String)uuidValue.toString()));
        }
        if (trinoType == DateType.DATE) {
            return (long)((Integer)icebergValue).intValue();
        }
        if (trinoType == TimeType.TIME_MICROS) {
            return (Long)icebergValue * 1000000L;
        }
        if (trinoType == TimestampType.TIMESTAMP_MICROS) {
            return (long)((Long)icebergValue);
        }
        if (trinoType == TimestampWithTimeZoneType.TIMESTAMP_TZ_MICROS) {
            long epochMicros = (Long)icebergValue;
            return LongTimestampWithTimeZone.fromEpochMillisAndFraction((long)Math.floorDiv(epochMicros, 1000), (int)(Math.floorMod(epochMicros, 1000) * 1000000), (short)TimeZoneKey.UTC_KEY.getKey());
        }
        throw new UnsupportedOperationException("Unsupported type: " + String.valueOf(trinoType));
    }
}

