/*
 * Decompiled with CFR 0.152.
 */
package io.trino.plugin.iceberg;

import com.google.common.collect.ImmutableList;
import com.google.common.collect.Lists;
import com.google.common.collect.MoreCollectors;
import com.google.common.math.IntMath;
import io.trino.Session;
import io.trino.plugin.iceberg.IcebergQueryRunner;
import io.trino.sql.query.QueryAssertions;
import io.trino.testing.AbstractTestQueryFramework;
import io.trino.testing.DataProviders;
import io.trino.testing.MaterializedRow;
import io.trino.testing.QueryRunner;
import io.trino.testing.TestingAccessControlManager;
import io.trino.testing.TestingNames;
import io.trino.tpch.TpchTable;
import java.math.RoundingMode;
import java.util.List;
import java.util.stream.Collectors;
import org.assertj.core.api.AssertProvider;
import org.assertj.core.api.Assertions;
import org.junit.jupiter.api.Test;
import org.junit.jupiter.params.ParameterizedTest;
import org.junit.jupiter.params.provider.MethodSource;
import org.junit.jupiter.params.provider.ValueSource;

public class TestIcebergStatistics
extends AbstractTestQueryFramework {
    protected QueryRunner createQueryRunner() throws Exception {
        return IcebergQueryRunner.builder().setInitialTables(TpchTable.NATION).build();
    }

    @ParameterizedTest
    @ValueSource(booleans={true, false})
    public void testAnalyze(boolean collectOnStatsOnWrites) {
        Session writeSession = TestIcebergStatistics.withStatsOnWrite(this.getSession(), collectOnStatsOnWrites);
        String tableName = "test_analyze_" + collectOnStatsOnWrites;
        this.assertUpdate(writeSession, "CREATE TABLE " + tableName + " AS SELECT * FROM tpch.sf1.nation", 25L);
        String goodStatsInitial = "VALUES\n  ('nationkey', null, 25, 0, null, '0', '24'),\n  ('regionkey', null, 5, 0, null, '0', '4'),\n  ('comment', 2162.0, 25, 0, null, null, null),\n  ('name', 583.0, 25, 0, null, null, null),\n  (null, null, null, null, 25, null, null)";
        if (collectOnStatsOnWrites) {
            this.assertQuery("SHOW STATS FOR " + tableName, goodStatsInitial);
        } else {
            this.assertQuery("SHOW STATS FOR " + tableName, "VALUES\n  ('nationkey', null, null, 0, null, '0', '24'),\n  ('regionkey', null, null, 0, null, '0', '4'),\n  ('comment', 2162.0, null, 0, null, null, null),\n  ('name', 583.0, null, 0, null, null, null),\n  (null, null, null, null, 25, null, null)");
        }
        this.assertUpdate("ANALYZE " + tableName);
        this.assertQuery("SHOW STATS FOR " + tableName, goodStatsInitial);
        this.assertUpdate("ANALYZE " + tableName);
        this.assertQuery("SHOW STATS FOR " + tableName, goodStatsInitial);
        this.assertUpdate(writeSession, "INSERT INTO " + tableName + " SELECT * FROM tpch.sf1.nation", 25L);
        String goodStatsAfterFirstInsert = "VALUES\n  ('nationkey', null, 25, 0, null, '0', '24'),\n  ('regionkey', null, 5, 0, null, '0', '4'),\n  ('comment', 4325.0, 25, 0, null, null, null),\n  ('name', 1166.0, 25, 0, null, null, null),\n  (null, null, null, null, 50, null, null)";
        this.assertUpdate("ANALYZE " + tableName);
        this.assertQuery("SHOW STATS FOR " + tableName, goodStatsAfterFirstInsert);
        this.assertUpdate(writeSession, "INSERT INTO " + tableName + " SELECT nationkey + 25, reverse(name), regionkey + 5, reverse(comment) FROM tpch.sf1.nation", 25L);
        String goodStatsAfterSecondInsert = "VALUES\n  ('nationkey', null, 50, 0, null, '0', '49'),\n  ('regionkey', null, 10, 0, null, '0', '9'),\n  ('comment', 6463.0, 50, 0, null, null, null),\n  ('name', 1768.0, 50, 0, null, null, null),\n  (null, null, null, null, 75, null, null)\n";
        if (collectOnStatsOnWrites) {
            this.assertQuery("SHOW STATS FOR " + tableName, goodStatsAfterSecondInsert);
        } else {
            this.assertQuery("SHOW STATS FOR " + tableName, "VALUES\n  ('nationkey', null, 25, 0, null, '0', '49'),\n  ('regionkey', null, 5, 0, null, '0', '9'),\n  ('comment', 6463.0, 25, 0, null, null, null),\n  ('name', 1768.0, 25, 0, null, null, null),\n  (null, null, null, null, 75, null, null)\n");
        }
        this.assertUpdate("ANALYZE " + tableName);
        this.assertQuery("SHOW STATS FOR " + tableName, goodStatsAfterSecondInsert);
        this.assertUpdate("DROP TABLE " + tableName);
    }

    @Test
    public void testAnalyzeWithSchemaEvolution() {
        String tableName = "test_analyze_with_schema_evolution";
        this.assertUpdate("CREATE TABLE " + tableName + " AS SELECT * FROM tpch.sf1.nation", 25L);
        this.assertUpdate("ANALYZE " + tableName);
        this.assertUpdate("ALTER TABLE " + tableName + " ADD COLUMN info varchar");
        this.assertUpdate("UPDATE " + tableName + " SET info = format('%s %s', name, comment)", 25L);
        this.assertUpdate("ALTER TABLE " + tableName + " DROP COLUMN comment");
        double nameDataSize = (Double)((MaterializedRow)this.computeActual("SHOW STATS FOR " + tableName).getMaterializedRows().stream().filter(row -> "name".equals(row.getField(0))).collect(MoreCollectors.onlyElement())).getField(1);
        Assertions.assertThat((double)nameDataSize).isBetween(Double.valueOf(1000.0), Double.valueOf(3000.0));
        this.assertQuery("SHOW STATS FOR " + tableName, "VALUES\n  ('nationkey', null, 25, 0, null, '0', '24'),\n  ('regionkey', null, 5, 0, null, '0', '4'),\n  ('name', %s, 25, 0, null, null, null),\n  ('info', null, null, null, null, null, null),\n  (null, null, null, null, 50, null, null)\n".formatted(nameDataSize));
        this.assertUpdate("ANALYZE " + tableName);
        double infoDataSize = (Double)((MaterializedRow)this.computeActual("SHOW STATS FOR " + tableName).getMaterializedRows().stream().filter(row -> "info".equals(row.getField(0))).collect(MoreCollectors.onlyElement())).getField(1);
        Assertions.assertThat((double)infoDataSize).isBetween(Double.valueOf(2000.0), Double.valueOf(5000.0));
        this.assertQuery("SHOW STATS FOR " + tableName, "VALUES\n  ('nationkey', null, 25, 0, null, '0', '24'),\n  ('regionkey', null, 5, 0, null, '0', '4'),\n  ('name', %s, 25, 0, null, null, null),\n  ('info', %s, 25, 0.1, null, null, null),\n  (null, null, null, null, 50, null, null)\n".formatted(nameDataSize, infoDataSize));
        this.assertUpdate("DROP TABLE " + tableName);
    }

    @ParameterizedTest
    @ValueSource(booleans={true, false})
    public void testAnalyzePartitioned(boolean collectOnStatsOnWrites) {
        Session writeSession = TestIcebergStatistics.withStatsOnWrite(this.getSession(), collectOnStatsOnWrites);
        String tableName = "test_analyze_partitioned_" + collectOnStatsOnWrites;
        this.assertUpdate(writeSession, "CREATE TABLE " + tableName + " WITH (partitioning = ARRAY['regionkey']) AS SELECT * FROM tpch.sf1.nation", 25L);
        String goodStatsInitial = "VALUES\n  ('nationkey', null, 25, 0, null, '0', '24'),\n  ('regionkey', null, 5, 0, null, '0', '4'),\n  ('comment', 3507.0, 25, 0, null, null, null),\n  ('name', 1182.0, 25, 0, null, null, null),\n  (null, null, null, null, 25, null, null)\n";
        if (collectOnStatsOnWrites) {
            this.assertQuery("SHOW STATS FOR " + tableName, goodStatsInitial);
        } else {
            this.assertQuery("SHOW STATS FOR " + tableName, "VALUES\n  ('nationkey', null, null, 0, null, '0', '24'),\n  ('regionkey', null, null, 0, null, '0', '4'),\n  ('comment', 3507.0, null, 0, null, null, null),\n  ('name', 1182.0, null, 0, null, null, null),\n  (null, null, null, null, 25, null, null)\n");
        }
        this.assertUpdate("ANALYZE " + tableName);
        this.assertQuery("SHOW STATS FOR " + tableName, goodStatsInitial);
        this.assertUpdate(writeSession, "INSERT INTO " + tableName + " SELECT * FROM tpch.sf1.nation", 25L);
        this.assertUpdate("ANALYZE " + tableName);
        this.assertQuery("SHOW STATS FOR " + tableName, "VALUES\n  ('nationkey', null, 25, 0, null, '0', '24'),\n  ('regionkey', null, 5, 0, null, '0', '4'),\n  ('comment', 7014.0, 25, 0, null, null, null),\n  ('name', 2365.0, 25, 0, null, null, null),\n  (null, null, null, null, 50, null, null)\n");
        this.assertUpdate(writeSession, "INSERT INTO " + tableName + " SELECT nationkey + 25, reverse(name), regionkey + 5, reverse(comment) FROM tpch.sf1.nation", 25L);
        String goodStatsAfterSecondInsert = "VALUES\n  ('nationkey', null, 50, 0, null, '0', '49'),\n  ('regionkey', null, 10, 0, null, '0', '9'),\n  ('comment', 10493.999999999998, 50, 0, null, null, null),\n  ('name', 3564.0000000000005, 50, 0, null, null, null),\n  (null, null, null, null, 75, null, null)\n";
        if (collectOnStatsOnWrites) {
            this.assertQuery("SHOW STATS FOR " + tableName, goodStatsAfterSecondInsert);
        } else {
            this.assertQuery("SHOW STATS FOR " + tableName, "VALUES\n  ('nationkey', null, 25, 0, null, '0', '49'),\n  ('regionkey', null, 5, 0, null, '0', '9'),\n  ('comment', 10493.999999999998, 25, 0, null, null, null),\n  ('name', 3564.0000000000005, 25, 0, null, null, null),\n  (null, null, null, null, 75, null, null)\n");
        }
        this.assertUpdate("ANALYZE " + tableName);
        this.assertQuery("SHOW STATS FOR " + tableName, goodStatsAfterSecondInsert);
        this.assertUpdate("DROP TABLE " + tableName);
    }

    @Test
    public void testAnalyzeEmpty() {
        String tableName = "test_analyze_empty";
        Session noStatsOnWrite = TestIcebergStatistics.withStatsOnWrite(this.getSession(), false);
        this.assertUpdate(noStatsOnWrite, "CREATE TABLE " + tableName + " AS SELECT * FROM tpch.sf1.nation WITH NO DATA", 0L);
        this.assertQuery("SHOW STATS FOR " + tableName, "VALUES\n  ('nationkey', 0, 0, 1, null, null, null),\n  ('regionkey', 0, 0, 1, null, null, null),\n  ('comment', 0, 0, 1, null, null, null),\n  ('name', 0, 0, 1, null, null, null),\n  (null, null, null, null, 0, null, null)\n");
        this.assertUpdate("ANALYZE " + tableName);
        this.assertQuery("SHOW STATS FOR " + tableName, "VALUES\n  ('nationkey', 0, 0, 1, null, null, null),\n  ('regionkey', 0, 0, 1, null, null, null),\n  ('comment', 0, 0, 1, null, null, null),\n  ('name', 0, 0, 1, null, null, null),\n  (null, null, null, null, 0, null, null)\n");
        this.assertUpdate(noStatsOnWrite, "INSERT INTO " + tableName + " SELECT * FROM tpch.sf1.nation", 25L);
        this.assertUpdate("ANALYZE " + tableName);
        this.assertQuery("SHOW STATS FOR " + tableName, "VALUES\n  ('nationkey', null, 25, 0, null, '0', '24'),\n  ('regionkey', null, 5, 0, null, '0', '4'),\n  ('comment', 2162.0, 25, 0, null, null, null),\n  ('name', 583.0, 25, 0, null, null, null),\n  (null, null, null, null, 25, null, null)\n");
        this.assertUpdate("DROP TABLE " + tableName);
    }

    @ParameterizedTest
    @MethodSource(value={"testCollectStatisticsOnWriteDataProvider"})
    public void testCollectStatisticsOnWrite(boolean collectOnStatsOnCreateTable, boolean partitioned) {
        String tableName = "test_collect_stats_insert_" + collectOnStatsOnCreateTable + partitioned;
        this.assertUpdate(TestIcebergStatistics.withStatsOnWrite(this.getSession(), collectOnStatsOnCreateTable), "CREATE TABLE " + tableName + " " + (partitioned ? "WITH (partitioning=ARRAY['regionkey']) " : "") + "AS SELECT * FROM tpch.sf1.nation WHERE nationkey < 12 AND regionkey < 3", 7L);
        this.assertQuery("SHOW STATS FOR " + tableName, collectOnStatsOnCreateTable ? "VALUES\n  ('nationkey', null, 7, 0, null, '0', '9'),\n  ('regionkey', null, 3, 0, null, '0', '2'),\n  ('comment', %s, 7, 0, null, null, null),\n  ('name', %s, 7, 0, null, null, null),\n  (null, null, null, null, 7, null, null)\n".formatted(partitioned ? "1301.0" : "936.0", partitioned ? "469.0" : "270.0") : "VALUES\n  ('nationkey', null, null, 0, null, '0', '9'),\n  ('regionkey', null, null, 0, null, '0', '2'),\n  ('comment', %s, null, 0, null, null, null),\n  ('name', %s, null, 0, null, null, null),\n  (null, null, null, null, 7, null, null)\n".formatted(partitioned ? "1301.0" : "936.0", partitioned ? "469.0" : "270.0"));
        this.assertUpdate(TestIcebergStatistics.withStatsOnWrite(this.getSession(), true), "INSERT INTO " + tableName + " SELECT * FROM tpch.sf1.nation WHERE nationkey >= 12 OR regionkey >= 3", 18L);
        this.assertQuery("SHOW STATS FOR " + tableName, collectOnStatsOnCreateTable ? "VALUES\n  ('nationkey', null, 25, 0, null, '0', '24'),\n  ('regionkey', null, 5, 0, null, '0', '4'),\n  ('comment', %s, 25, 0, null, null, null),\n  ('name', %s, 25, 0, null, null, null),\n  (null, null, null, null, 25, null, null)\n".formatted(partitioned ? "4058.0" : "2627.0", partitioned ? "1447.0" : "726.0") : "VALUES\n  ('nationkey', null, null, 0, null, '0', '24'),\n  ('regionkey', null, null, 0, null, '0', '4'),\n  ('comment', %s, null, 0, null, null, null),\n  ('name', %s, null, 0, null, null, null),\n  (null, null, null, null, 25, null, null)\n".formatted(partitioned ? "4058.0" : "2627.0", partitioned ? "1447.0" : "726.0"));
        this.assertUpdate("DROP TABLE " + tableName);
    }

    @ParameterizedTest
    @MethodSource(value={"testCollectStatisticsOnWriteDataProvider"})
    public void testCollectStatisticsOnWriteToEmptyTable(boolean collectOnStatsOnCreateTable, boolean partitioned) {
        String tableName = "test_collect_stats_insert_into_empty_" + collectOnStatsOnCreateTable + partitioned;
        this.assertUpdate(TestIcebergStatistics.withStatsOnWrite(this.getSession(), collectOnStatsOnCreateTable), "CREATE TABLE " + tableName + " " + (partitioned ? "WITH (partitioning=ARRAY['regionkey']) " : "") + "AS TABLE tpch.sf1.nation WITH NO DATA", 0L);
        this.assertQuery("SHOW STATS FOR " + tableName, "VALUES\n  ('nationkey', 0, 0, 1, null, null, null),\n  ('regionkey', 0, 0, 1, null, null, null),\n  ('comment', 0, 0, 1, null, null, null),\n  ('name', 0, 0, 1, null, null, null),\n  (null, null, null, null, 0, null, null)\n");
        this.assertUpdate(TestIcebergStatistics.withStatsOnWrite(this.getSession(), true), "INSERT INTO " + tableName + " TABLE tpch.sf1.nation", 25L);
        this.assertQuery("SHOW STATS FOR " + tableName, "VALUES\n  ('nationkey', null, 25, 0, null, '0', '24'),\n  ('regionkey', null, 5, 0, null, '0', '4'),\n  ('comment', %f, 25, 0, null, null, null),\n  ('name', %f, 25, 0, null, null, null),\n  (null, null, null, null, 25, null, null)\n".formatted(partitioned ? 3507.0 : 2162.0, partitioned ? 1182.0 : 583.0));
        this.assertUpdate("DROP TABLE " + tableName);
    }

    public Object[][] testCollectStatisticsOnWriteDataProvider() {
        return DataProviders.cartesianProduct((Object[][][])new Object[][][]{DataProviders.trueFalse(), DataProviders.trueFalse()});
    }

    @ParameterizedTest
    @ValueSource(booleans={true, false})
    public void testAnalyzeAfterStatsDrift(boolean withOptimize) {
        String tableName = "test_analyze_stats_drift_" + withOptimize;
        Session session = TestIcebergStatistics.withStatsOnWrite(this.getSession(), true);
        this.assertUpdate(session, "CREATE TABLE " + tableName + " AS SELECT nationkey, regionkey FROM tpch.sf1.nation", 25L);
        this.assertQuery("SHOW STATS FOR " + tableName, "VALUES\n  ('nationkey', null, 25, 0, null, '0', '24'),\n  ('regionkey', null, 5, 0, null, '0', '4'),\n  (null, null, null, null, 25, null, null)");
        List idsToRemove = (List)this.computeActual("SELECT nationkey FROM tpch.sf1.nation WHERE regionkey IN (2, 4)").getOnlyColumn().map(value -> Long.toString((Long)value)).collect(ImmutableList.toImmutableList());
        for (List ids : Lists.partition((List)idsToRemove, (int)IntMath.divide((int)idsToRemove.size(), (int)2, (RoundingMode)RoundingMode.UP))) {
            String idsLiteral = ids.stream().collect(Collectors.joining(", ", "(", ")"));
            this.assertUpdate("DELETE FROM " + tableName + " WHERE nationkey IN " + idsLiteral, ids.size());
        }
        this.assertQuery("SHOW STATS FOR " + tableName, "VALUES\n  ('nationkey', null, 25, 0, null, '0', '24'),\n  ('regionkey', null, 5, 0, null, '0', '4'),\n  (null, null, null, null, 25, null, null)\n");
        if (withOptimize) {
            this.assertUpdate("ALTER TABLE " + tableName + " EXECUTE optimize");
            this.assertQuery("SHOW STATS FOR " + tableName, "VALUES\n  ('nationkey', null, 15, 0, null, '0', '24'),\n  ('regionkey', null, 4, 0, null, '0', '3'),\n  (null, null, null, null, 15, null, null)\n");
        }
        this.assertUpdate("ANALYZE " + tableName + " WITH(columns=ARRAY['nationkey'])");
        this.assertQuery("SHOW STATS FOR " + tableName, withOptimize ? "VALUES\n  ('nationkey', null, 15, 0, null, '0', '24'),\n  ('regionkey', null, 4, 0, null, '0', '3'), -- not updated yet\n  (null, null, null, null, 15, null, null)\n" : "VALUES\n  ('nationkey', null, 15, 0, null, '0', '24'),\n  ('regionkey', null, 5, 0, null, '0', '4'), -- not updated yet\n  (null, null, null, null, 25, null, null)\n");
        this.assertUpdate("ANALYZE " + tableName);
        this.assertQuery("SHOW STATS FOR " + tableName, withOptimize ? "VALUES\n  ('nationkey', null, 15, 0, null, '0', '24'),\n  ('regionkey', null, 3, 0, null, '0', '3'),\n  (null, null, null, null, 15, null, null)\n" : "VALUES\n  ('nationkey', null, 15, 0, null, '0', '24'),\n  ('regionkey', null, 3, 0, null, '0', '4'),\n  (null, null, null, null, 25, null, null)\n");
        this.assertUpdate("DROP TABLE " + tableName);
    }

    @Test
    public void testAnalyzeSomeColumns() {
        String tableName = "test_analyze_some_columns";
        Session noStatsOnWrite = TestIcebergStatistics.withStatsOnWrite(this.getSession(), false);
        this.assertUpdate(noStatsOnWrite, "CREATE TABLE " + tableName + " AS SELECT * FROM tpch.sf1.nation", 25L);
        this.assertQueryFails("ANALYZE " + tableName + " WITH (columns = NULL)", "\\Qline 1:41: Invalid null value for catalog 'iceberg' analyze property 'columns' from [null]");
        this.assertQueryFails("ANALYZE " + tableName + " WITH (columns = ARRAY[])", "\\QCannot specify empty list of columns for analysis");
        this.assertQueryFails("ANALYZE " + tableName + " WITH (columns = ARRAY['nationkey', 'blah'])", "\\QInvalid columns specified for analysis: [blah]");
        this.assertQueryFails("ANALYZE " + tableName + " WITH (columns = ARRAY['NationKey'])", "\\QInvalid columns specified for analysis: [NationKey]");
        this.assertQueryFails("ANALYZE " + tableName + " WITH (columns = ARRAY['nationkey', NULL])", "\\Qline 1:41: Unable to set catalog 'iceberg' analyze property 'columns' to [ARRAY['nationkey',null]]: Invalid null value in analyze columns property");
        this.assertUpdate("ANALYZE " + tableName + " WITH (columns = ARRAY['nationkey', 'regionkey'])");
        this.assertQuery("SHOW STATS FOR " + tableName, "VALUES\n  ('nationkey', null, 25, 0, null, '0', '24'),\n  ('regionkey', null, 5, 0, null, '0', '4'),\n  ('comment', 2162.0, null, 0, null, null, null),\n  ('name', 583.0, null, 0, null, null, null),\n  (null, null, null, null, 25, null, null)\n");
        this.assertUpdate(noStatsOnWrite, "INSERT INTO " + tableName + " SELECT nationkey + 25, concat(name, '1'), regionkey + 5, concat(comment, '21') FROM tpch.sf1.nation", 25L);
        this.assertUpdate("ANALYZE " + tableName + " WITH (columns = ARRAY['nationkey', 'regionkey'])");
        this.assertQuery("SHOW STATS FOR " + tableName, "VALUES\n  ('nationkey', null, 50, 0, null, '0', '49'),\n  ('regionkey', null, 10, 0, null, '0', '9'),\n  ('comment', 4441.0, null, 0, null, null, null),\n  ('name', 1193.0, null, 0, null, null, null),\n  (null, null, null, null, 50, null, null)\n");
        this.assertUpdate("ALTER TABLE " + tableName + " EXECUTE DROP_EXTENDED_STATS");
        this.assertUpdate("ANALYZE " + tableName);
        this.assertQuery("SHOW STATS FOR " + tableName, "VALUES\n  ('nationkey', null, 50, 0, null, '0', '49'),\n  ('regionkey', null, 10, 0, null, '0', '9'),\n  ('comment', 4441.0, 50, 0, null, null, null),\n  ('name', 1193.0, 50, 0, null, null, null),\n  (null, null, null, null, 50, null, null)\n");
        this.assertUpdate(noStatsOnWrite, "INSERT INTO " + tableName + " SELECT nationkey + 50, concat(name, '2'), regionkey + 10, concat(comment, '22') FROM tpch.sf1.nation", 25L);
        this.assertQuery("SHOW STATS FOR " + tableName, "VALUES\n  ('nationkey', null, 50, 0, null, '0', '74'),\n  ('regionkey', null, 10, 0, null, '0', '14'),\n  ('comment', 6701.0, 50, 0, null, null, null),\n  ('name', 1803.0, 50, 0, null, null, null),\n  (null, null, null, null, 75, null, null)\n");
        this.assertUpdate("ANALYZE " + tableName + " WITH (columns = ARRAY['nationkey', 'regionkey'])");
        this.assertQuery("SHOW STATS FOR " + tableName, "VALUES\n  ('nationkey', null, 75, 0, null, '0', '74'),\n  ('regionkey', null, 15, 0, null, '0', '14'),\n  ('comment', 6701.0, 50, 0, null, null, null), -- result of previous analyze\n  ('name', 1803.0, 50, 0, null, null, null), -- result of previous analyze\n  (null, null, null, null, 75, null, null)\n");
        this.assertUpdate("ANALYZE " + tableName);
        this.assertQuery("SHOW STATS FOR " + tableName, "VALUES\n  ('nationkey', null, 75, 0, null, '0', '74'),\n  ('regionkey', null, 15, 0, null, '0', '14'),\n  ('comment', 6701.0, 75, 0, null, null, null),\n  ('name', 1803.0, 75, 0, null, null, null),\n  (null, null, null, null, 75, null, null)\n");
        this.assertUpdate("DROP TABLE " + tableName);
    }

    @Test
    public void testAnalyzeSnapshot() {
        String tableName = "test_analyze_snapshot_" + TestingNames.randomNameSuffix();
        this.assertUpdate("CREATE TABLE " + tableName + " (a) AS VALUES 11", 1L);
        long snapshotId = this.getCurrentSnapshotId(tableName);
        this.assertUpdate("INSERT INTO " + tableName + " VALUES 22", 1L);
        ((QueryAssertions.QueryAssert)Assertions.assertThat((AssertProvider)this.query("ANALYZE \"%s@%d\"".formatted(tableName, snapshotId)))).failure().hasMessage(String.format("line 1:1: Table 'iceberg.tpch.\"%s@%s\"' does not exist", tableName, snapshotId));
        ((QueryAssertions.QueryAssert)Assertions.assertThat((AssertProvider)this.query("SELECT * FROM " + tableName))).matches("VALUES 11, 22");
        this.assertUpdate("DROP TABLE " + tableName);
    }

    @Test
    public void testAnalyzeSystemTable() {
        ((QueryAssertions.QueryAssert)Assertions.assertThat((AssertProvider)this.query("ANALYZE \"nation$files\""))).nonTrinoExceptionFailure().hasMessage("Cannot record write for catalog not part of transaction");
        ((QueryAssertions.QueryAssert)Assertions.assertThat((AssertProvider)this.query("ANALYZE \"nation$snapshots\""))).nonTrinoExceptionFailure().hasMessage("Cannot record write for catalog not part of transaction");
    }

    @Test
    public void testDropExtendedStats() {
        String tableName = "test_drop_extended_stats";
        this.assertUpdate("CREATE TABLE " + tableName + " AS SELECT * FROM tpch.sf1.nation", 25L);
        String baseStats = "VALUES\n  ('nationkey', null, null, 0, null, '0', '24'),\n  ('regionkey', null, null, 0, null, '0', '4'),\n  ('comment', 2162.0, null, 0, null, null, null),\n  ('name',  583.0, null, 0, null, null, null),\n  (null,  null, null, null, 25, null, null)\n";
        String extendedStats = "VALUES\n  ('nationkey', null, 25, 0, null, '0', '24'),\n  ('regionkey', null, 5, 0, null, '0', '4'),\n  ('comment', 2162.0, 25, 0, null, null, null),\n  ('name',  583.0, 25, 0, null, null, null),\n  (null,  null, null, null, 25, null, null)\n";
        this.assertQuery("SHOW STATS FOR " + tableName, extendedStats);
        this.assertUpdate("ALTER TABLE " + tableName + " EXECUTE DROP_EXTENDED_STATS");
        this.assertQuery("SHOW STATS FOR " + tableName, baseStats);
        this.assertUpdate("ANALYZE " + tableName);
        this.assertQuery("SHOW STATS FOR " + tableName, extendedStats);
        this.assertUpdate("DROP TABLE " + tableName);
    }

    @Test
    public void testDropMissingStats() {
        String tableName = "test_drop_missing_stats";
        this.assertUpdate("CREATE TABLE " + tableName + " AS SELECT * FROM tpch.sf1.nation", 25L);
        this.assertUpdate("ALTER TABLE " + tableName + " EXECUTE DROP_EXTENDED_STATS");
        this.assertQuery("SHOW STATS FOR " + tableName, "VALUES\n  ('nationkey', null, null, 0, null, '0', '24'),\n  ('regionkey', null, null, 0, null, '0', '4'),\n  ('comment', 2162.0, null, 0, null, null, null),\n  ('name',  583.0, null, 0, null, null, null),\n  (null,  null, null, null, 25, null, null)\n");
        this.assertUpdate("DROP TABLE " + tableName);
    }

    @Test
    public void testDropStatsAccessControl() {
        String catalog = (String)this.getSession().getCatalog().orElseThrow();
        String schema = (String)this.getSession().getSchema().orElseThrow();
        String tableName = "test_deny_drop_stats";
        this.assertUpdate("CREATE TABLE " + tableName + " AS SELECT * FROM tpch.sf1.nation", 25L);
        this.assertAccessDenied("ALTER TABLE " + tableName + " EXECUTE DROP_EXTENDED_STATS", "Cannot execute table procedure DROP_EXTENDED_STATS on iceberg.tpch.test_deny_drop_stats", new TestingAccessControlManager.TestingPrivilege[]{TestingAccessControlManager.privilege((String)String.format("%s.%s.%s.DROP_EXTENDED_STATS", catalog, schema, tableName), (TestingAccessControlManager.TestingPrivilegeType)TestingAccessControlManager.TestingPrivilegeType.EXECUTE_TABLE_PROCEDURE)});
        this.assertUpdate("DROP TABLE " + tableName);
    }

    @Test
    public void testDropStatsSnapshot() {
        String tableName = "test_drop_stats_snapshot_" + TestingNames.randomNameSuffix();
        this.assertUpdate("CREATE TABLE " + tableName + " (a) AS VALUES 11", 1L);
        long snapshotId = this.getCurrentSnapshotId(tableName);
        this.assertUpdate("INSERT INTO " + tableName + " VALUES 22", 1L);
        ((QueryAssertions.QueryAssert)Assertions.assertThat((AssertProvider)this.query("ALTER TABLE \"%s@%d\" EXECUTE DROP_EXTENDED_STATS".formatted(tableName, snapshotId)))).failure().hasMessage(String.format("line 1:7: Table 'iceberg.tpch.\"%s@%s\"' does not exist", tableName, snapshotId));
        ((QueryAssertions.QueryAssert)Assertions.assertThat((AssertProvider)this.query("SELECT * FROM " + tableName))).matches("VALUES 11, 22");
        this.assertUpdate("DROP TABLE " + tableName);
    }

    @Test
    public void testDropStatsSystemTable() {
        ((QueryAssertions.QueryAssert)Assertions.assertThat((AssertProvider)this.query("ALTER TABLE \"nation$files\" EXECUTE DROP_EXTENDED_STATS"))).failure().hasMessage("This connector does not support table procedures");
        ((QueryAssertions.QueryAssert)Assertions.assertThat((AssertProvider)this.query("ALTER TABLE \"nation$snapshots\" EXECUTE DROP_EXTENDED_STATS"))).failure().hasMessage("This connector does not support table procedures");
    }

    @Test
    public void testAnalyzeAndRollbackToSnapshot() {
        String schema = (String)this.getSession().getSchema().orElseThrow();
        String tableName = "test_analyze_and_rollback";
        this.assertUpdate("CREATE TABLE " + tableName + " AS SELECT * FROM tpch.sf1.nation", 25L);
        long createSnapshot = this.getCurrentSnapshotId(tableName);
        this.assertUpdate("ANALYZE " + tableName);
        long analyzeSnapshot = this.getCurrentSnapshotId(tableName);
        Assertions.assertThat((long)analyzeSnapshot).isEqualTo(createSnapshot);
        this.assertUpdate("INSERT INTO " + tableName + " SELECT * FROM tpch.sf1.nation WHERE nationkey = 1", 1L);
        Assertions.assertThat((long)this.getCurrentSnapshotId(tableName)).isNotEqualTo(createSnapshot);
        this.assertQuery("SHOW STATS FOR " + tableName, "VALUES\n  ('nationkey', null, 25, 0, null, '0', '24'),\n  ('regionkey', null, 5, 0, null, '0', '4'),\n  ('comment', 2448.0, 25, 0, null, null, null),\n  ('name',  704.0, 25, 0, null, null, null),\n  (null,  null, null, null, 26, null, null)\n");
        this.assertUpdate(String.format("ALTER TABLE %s.%s EXECUTE rollback_to_snapshot(%s)", schema, tableName, createSnapshot));
        this.assertQuery("SHOW STATS FOR " + tableName, "VALUES\n  ('nationkey', null, 25, 0, null, '0', '24'),\n  ('regionkey', null, 5, 0, null, '0', '4'),\n  ('comment', 2162.0, 25, 0, null, null, null),\n  ('name',  583.0, 25, 0, null, null, null),\n  (null,  null, null, null, 25, null, null)\n");
        this.assertUpdate("DROP TABLE " + tableName);
    }

    @Test
    public void testAnalyzeAndDeleteOrphanFiles() {
        String tableName = "test_analyze_and_delete_orphan_files";
        this.assertUpdate("CREATE TABLE " + tableName + " AS SELECT * FROM tpch.sf1.nation", 25L);
        this.assertUpdate("ANALYZE " + tableName);
        this.assertQuerySucceeds(Session.builder((Session)this.getSession()).setCatalogSessionProperty("iceberg", "remove_orphan_files_min_retention", "0s").build(), "ALTER TABLE " + tableName + " EXECUTE REMOVE_ORPHAN_FILES (retention_threshold => '0s')");
        this.assertQuery("SHOW STATS FOR " + tableName, "VALUES\n  ('nationkey', null, 25, 0, null, '0', '24'),\n  ('regionkey', null, 5, 0, null, '0', '4'),\n  ('comment', 2162.0, 25, 0, null, null, null),\n  ('name',  583.0, 25, 0, null, null, null),\n  (null,  null, null, null, 25, null, null)\n");
        this.assertUpdate("DROP TABLE " + tableName);
    }

    @Test
    public void testEmptyNoScalarColumns() {
        String tableName = "empty_table_without_scalar_columns";
        this.assertUpdate("CREATE TABLE " + tableName + " (a row(x integer), b row(y varchar))");
        this.assertQuery("SHOW STATS FOR " + tableName, "VALUES\n  ('a', 0, 0, 1, null, null, null),\n  ('b', 0, 0, 1, null, null, null),\n  (null,  null, null, null, 0, null, null)\n");
        this.assertQueryFails("ANALYZE " + tableName + " WITH (columns = ARRAY[])", "Cannot specify empty list of columns for analysis");
        this.assertQueryFails("ANALYZE " + tableName + " WITH (columns = ARRAY['a'])", "Invalid columns specified for analysis: \\[a]");
        this.assertQueryFails("ANALYZE " + tableName + " WITH (columns = ARRAY['a.x'])", "Invalid columns specified for analysis: \\[a.x]");
        this.assertQueryFails("ANALYZE " + tableName + " WITH (columns = ARRAY['b'])", "Invalid columns specified for analysis: \\[b]");
        this.assertUpdate("ANALYZE " + tableName);
        this.assertQuery("SHOW STATS FOR " + tableName, "VALUES\n  ('a', 0, 0, 1, null, null, null),\n  ('b', 0, 0, 1, null, null, null),\n  (null,  null, null, null, 0, null, null)\n");
        this.assertUpdate(TestIcebergStatistics.withStatsOnWrite(this.getSession(), true), "INSERT INTO " + tableName + " VALUES (ROW(52), ROW('hot')), (ROW(53), ROW('dog'))", 2L);
        this.assertQuery("SHOW STATS FOR " + tableName, "VALUES\n  ('a', null, null, null, null, null, null),\n  ('b', null, null, null, null, null, null),\n  (null,  null, null, null, 2, null, null)\n");
        this.assertUpdate("DROP TABLE " + tableName);
    }

    @Test
    public void testNoScalarColumns() {
        String tableName = "table_without_scalar_columns";
        this.assertUpdate("CREATE TABLE " + tableName + " (a row(x integer), b row(y varchar))");
        this.assertUpdate(TestIcebergStatistics.withStatsOnWrite(this.getSession(), false), "INSERT INTO " + tableName + " VALUES (ROW(42), ROW('ala')), (ROW(43), ROW('has a cat'))", 2L);
        this.assertQuery("SHOW STATS FOR " + tableName, "VALUES\n  ('a', null, null, null, null, null, null),\n  ('b', null, null, null, null, null, null),\n  (null,  null, null, null, 2, null, null)\n");
        this.assertQueryFails("ANALYZE " + tableName + " WITH (columns = ARRAY[])", "Cannot specify empty list of columns for analysis");
        this.assertQueryFails("ANALYZE " + tableName + " WITH (columns = ARRAY['a'])", "Invalid columns specified for analysis: \\[a]");
        this.assertQueryFails("ANALYZE " + tableName + " WITH (columns = ARRAY['a.x'])", "Invalid columns specified for analysis: \\[a.x]");
        this.assertQueryFails("ANALYZE " + tableName + " WITH (columns = ARRAY['b'])", "Invalid columns specified for analysis: \\[b]");
        this.assertUpdate("ANALYZE " + tableName);
        this.assertQuery("SHOW STATS FOR " + tableName, "VALUES\n  ('a', null, null, null, null, null, null),\n  ('b', null, null, null, null, null, null),\n  (null,  null, null, null, 2, null, null)\n");
        this.assertUpdate(TestIcebergStatistics.withStatsOnWrite(this.getSession(), true), "INSERT INTO " + tableName + " VALUES (ROW(52), ROW('hot')), (ROW(53), ROW('dog'))", 2L);
        this.assertQuery("SHOW STATS FOR " + tableName, "VALUES\n  ('a', null, null, null, null, null, null),\n  ('b', null, null, null, null, null, null),\n  (null,  null, null, null, 4, null, null)\n");
        this.assertUpdate("DROP TABLE " + tableName);
    }

    @Test
    public void testShowStatsAsOf() {
        Session writeSession = TestIcebergStatistics.withStatsOnWrite(this.getSession(), false);
        this.assertUpdate(writeSession, "CREATE TABLE show_stats_as_of(key integer)");
        this.assertUpdate(writeSession, "INSERT INTO show_stats_as_of VALUES 3", 1L);
        long beforeAnalyzedSnapshot = this.getCurrentSnapshotId("show_stats_as_of");
        this.assertUpdate(writeSession, "INSERT INTO show_stats_as_of VALUES 4", 1L);
        this.assertUpdate("ANALYZE show_stats_as_of");
        long analyzedSnapshot = this.getCurrentSnapshotId("show_stats_as_of");
        this.assertUpdate(writeSession, "INSERT INTO show_stats_as_of VALUES 5", 1L);
        long laterSnapshot = this.getCurrentSnapshotId("show_stats_as_of");
        this.assertQuery("SHOW STATS FOR (SELECT * FROM show_stats_as_of FOR VERSION AS OF " + beforeAnalyzedSnapshot + ")", "VALUES\n  ('key', null, null, 0, null, '3', '3'), -- NDV not present, as ANALYZE was run on a later snapshot\n  (null,  null, null, null, 1, null, null)\n");
        this.assertQuery("SHOW STATS FOR (SELECT * FROM show_stats_as_of FOR VERSION AS OF " + analyzedSnapshot + ")", "VALUES\n  ('key', null, 2, 0, null, '3', '4'), -- NDV present, this is the snapshot ANALYZE was run for\n  (null,  null, null, null, 2, null, null)\n");
        this.assertQuery("SHOW STATS FOR (SELECT * FROM show_stats_as_of FOR VERSION AS OF " + laterSnapshot + ")", "VALUES\n  ('key', null, 2, 0, null, '3', '5'), -- NDV present, stats \"inherited\" from previous snapshot\n  (null,  null, null, null, 3, null, null)\n");
        this.assertUpdate("DROP TABLE show_stats_as_of");
    }

    @Test
    public void testShowStatsAfterExpiration() {
        String catalog = (String)this.getSession().getCatalog().orElseThrow();
        Session writeSession = TestIcebergStatistics.withStatsOnWrite(this.getSession(), false);
        this.assertUpdate(writeSession, "CREATE TABLE show_stats_after_expiration(key integer)");
        this.assertUpdate(writeSession, "INSERT INTO show_stats_after_expiration VALUES 1", 1L);
        this.assertUpdate(writeSession, "INSERT INTO show_stats_after_expiration VALUES 2", 1L);
        this.assertUpdate(writeSession, "INSERT INTO show_stats_after_expiration VALUES 3", 1L);
        long beforeAnalyzedSnapshot = this.getCurrentSnapshotId("show_stats_after_expiration");
        this.assertUpdate(Session.builder((Session)this.getSession()).setCatalogSessionProperty(catalog, "expire_snapshots_min_retention", "0s").build(), "ALTER TABLE show_stats_after_expiration EXECUTE expire_snapshots(retention_threshold => '0d')");
        ((QueryAssertions.QueryAssert)Assertions.assertThat((AssertProvider)this.query("SELECT count(*) FROM \"show_stats_after_expiration$snapshots\""))).matches("VALUES BIGINT '1'");
        this.assertUpdate(writeSession, "INSERT INTO show_stats_after_expiration VALUES 4", 1L);
        this.assertUpdate("ANALYZE show_stats_after_expiration");
        long analyzedSnapshot = this.getCurrentSnapshotId("show_stats_after_expiration");
        this.assertUpdate(writeSession, "INSERT INTO show_stats_after_expiration VALUES 5", 1L);
        long laterSnapshot = this.getCurrentSnapshotId("show_stats_after_expiration");
        this.assertQuery("SHOW STATS FOR (SELECT * FROM show_stats_after_expiration FOR VERSION AS OF " + beforeAnalyzedSnapshot + ")", "VALUES\n  ('key', null, null, 0, null, '1', '3'), -- NDV not present, as ANALYZE was run on a later snapshot\n  (null,  null, null, null, 3, null, null)\n");
        this.assertQuery("SHOW STATS FOR (SELECT * FROM show_stats_after_expiration FOR VERSION AS OF " + analyzedSnapshot + ")", "VALUES\n  ('key', null, 4, 0, null, '1', '4'), -- NDV present, this is the snapshot ANALYZE was run for\n  (null,  null, null, null, 4, null, null)\n");
        this.assertQuery("SHOW STATS FOR (SELECT * FROM show_stats_after_expiration FOR VERSION AS OF " + laterSnapshot + ")", "VALUES\n  ('key', null, 4, 0, null, '1', '5'), -- NDV present, stats \"inherited\" from previous snapshot\n  (null,  null, null, null, 5, null, null)\n");
        this.assertQuery("SHOW STATS FOR show_stats_after_expiration", "VALUES\n  ('key', null, 4, 0, null, '1', '5'), -- NDV present, stats \"inherited\" from previous snapshot\n  (null,  null, null, null, 5, null, null)\n");
        this.assertUpdate("ANALYZE show_stats_after_expiration");
        this.assertQuery("SHOW STATS FOR show_stats_after_expiration", "VALUES\n  ('key', null, 5, 0, null, '1', '5'), -- NDV present, stats \"inherited\" from previous snapshot\n  (null,  null, null, null, 5, null, null)\n");
        this.assertUpdate("DROP TABLE show_stats_after_expiration");
    }

    @Test
    public void testShowStatsAfterOptimize() {
        String tableName = "show_stats_after_optimize_" + TestingNames.randomNameSuffix();
        String catalog = (String)this.getSession().getCatalog().orElseThrow();
        Session writeSession = TestIcebergStatistics.withStatsOnWrite(this.getSession(), false);
        Session minimalSnapshotRetentionSession = Session.builder((Session)this.getSession()).setCatalogSessionProperty(catalog, "expire_snapshots_min_retention", "0s").build();
        String expireSnapshotQuery = "ALTER TABLE " + tableName + " EXECUTE expire_snapshots(retention_threshold => '0d')";
        this.assertUpdate(writeSession, "CREATE TABLE " + tableName + "(key integer)");
        this.assertUpdate(writeSession, "INSERT INTO " + tableName + " VALUES 1", 1L);
        this.assertUpdate(writeSession, "INSERT INTO " + tableName + " VALUES 2", 1L);
        this.assertUpdate(writeSession, "INSERT INTO " + tableName + " VALUES 3", 1L);
        this.assertUpdate("ANALYZE " + tableName);
        this.assertUpdate(writeSession, "INSERT INTO " + tableName + " VALUES 4", 1L);
        this.assertQuery("SHOW STATS FOR " + tableName, "VALUES\n  ('key', null, 3, 0, null, '1', '4'), -- NDV present, stats \"inherited\" from previous snapshot\n  (null,  null, null, null, 4, null, null)\n");
        this.assertUpdate(minimalSnapshotRetentionSession, expireSnapshotQuery);
        this.assertQuery("SHOW STATS FOR " + tableName, "VALUES\n  ('key', null, null, 0, null, '1', '4'), -- NDV not present as expire_snapshot removed stats for previous snapshots\n  (null,  null, null, null, 4, null, null)\n");
        this.assertUpdate("ANALYZE " + tableName);
        this.assertQuery("SHOW STATS FOR " + tableName, "VALUES\n  ('key', null, 4, 0, null, '1', '4'), -- NDV present\n  (null,  null, null, null, 4, null, null)\n");
        this.assertUpdate("ALTER TABLE " + tableName + " EXECUTE optimize");
        this.assertUpdate(minimalSnapshotRetentionSession, expireSnapshotQuery);
        this.assertQuery("SHOW STATS FOR " + tableName, "VALUES\n  ('key', null, 4, 0, null, '1', '4'), -- NDV present\n  (null,  null, null, null, 4, null, null)\n");
        this.assertUpdate("DROP TABLE " + tableName);
    }

    @Test
    public void testStatsAfterDeletingAllRows() {
        String tableName = "test_stats_after_deleting_all_rows_";
        this.assertUpdate("CREATE TABLE " + tableName + " AS SELECT * FROM tpch.sf1.nation", 25L);
        ((QueryAssertions.QueryAssert)Assertions.assertThat((AssertProvider)this.query("SHOW STATS FOR " + tableName))).result().projected(new String[]{"column_name", "distinct_values_count", "row_count"}).skippingTypesCheck().containsAll("VALUES ('nationkey', DOUBLE '25', null), ('name', DOUBLE '25', null), ('regionkey', DOUBLE '5', null), ('comment', DOUBLE '25', null), (null, null, DOUBLE '25')");
        this.assertUpdate("DELETE FROM " + tableName + " WHERE nationkey < 50", 25L);
        ((QueryAssertions.QueryAssert)Assertions.assertThat((AssertProvider)this.query("SHOW STATS FOR " + tableName))).result().projected(new String[]{"column_name", "distinct_values_count", "row_count"}).skippingTypesCheck().containsAll("VALUES ('nationkey', DOUBLE '25', null), ('name', DOUBLE '25', null), ('regionkey', DOUBLE '5', null), ('comment', DOUBLE '25', null), (null, null, DOUBLE '25')");
    }

    @Test
    public void testNaN() {
        String tableName = "test_nan";
        this.assertUpdate("CREATE TABLE " + tableName + " AS SELECT 1 AS c1, double 'NaN' AS c2", 1L);
        this.assertQuery("SHOW STATS FOR " + tableName, "VALUES\n  ('c1', null, 1.0, 0.0, null, 1, 1),\n  ('c2', null, 1.0, 0.0, null, null, null),\n  (null, null, null, null, 1.0, null, null)\n");
    }

    private long getCurrentSnapshotId(String tableName) {
        return (Long)this.computeActual(String.format("SELECT snapshot_id FROM \"%s$snapshots\" ORDER BY committed_at DESC FETCH FIRST 1 ROW WITH TIES", tableName)).getOnlyValue();
    }

    private static Session withStatsOnWrite(Session session, boolean enabled) {
        String catalog = (String)session.getCatalog().orElseThrow();
        return Session.builder((Session)session).setCatalogSessionProperty(catalog, "collect_extended_statistics_on_write", Boolean.toString(enabled)).build();
    }
}

