/*
 * Decompiled with CFR 0.152.
 */
package io.trino.plugin.iceberg.procedure;

import com.google.common.base.Joiner;
import com.google.common.base.MoreObjects;
import com.google.common.base.Verify;
import com.google.common.collect.ImmutableList;
import com.google.common.collect.ImmutableSet;
import io.airlift.log.Logger;
import io.trino.filesystem.FileEntry;
import io.trino.filesystem.FileIterator;
import io.trino.filesystem.Location;
import io.trino.filesystem.TrinoFileSystem;
import io.trino.filesystem.TrinoInputFile;
import io.trino.metastore.HiveMetastore;
import io.trino.metastore.HiveMetastoreFactory;
import io.trino.metastore.Partition;
import io.trino.metastore.Storage;
import io.trino.metastore.StorageFormat;
import io.trino.parquet.ParquetDataSource;
import io.trino.parquet.ParquetReaderOptions;
import io.trino.parquet.metadata.ParquetMetadata;
import io.trino.parquet.reader.MetadataReader;
import io.trino.plugin.base.metrics.FileFormatDataSourceStats;
import io.trino.plugin.base.util.Procedures;
import io.trino.plugin.hive.HiveMetadata;
import io.trino.plugin.hive.HiveStorageFormat;
import io.trino.plugin.hive.parquet.TrinoParquetDataSource;
import io.trino.plugin.iceberg.IcebergErrorCode;
import io.trino.plugin.iceberg.IcebergSessionProperties;
import io.trino.plugin.iceberg.catalog.TrinoCatalog;
import io.trino.plugin.iceberg.fileio.ForwardingInputFile;
import io.trino.plugin.iceberg.util.OrcMetrics;
import io.trino.plugin.iceberg.util.ParquetUtil;
import io.trino.spi.ErrorCodeSupplier;
import io.trino.spi.StandardErrorCode;
import io.trino.spi.TrinoException;
import io.trino.spi.connector.ConnectorSession;
import io.trino.spi.connector.SchemaTableName;
import java.io.IOException;
import java.io.UncheckedIOException;
import java.util.List;
import java.util.Map;
import java.util.Optional;
import java.util.Set;
import java.util.stream.Stream;
import org.apache.iceberg.AppendFiles;
import org.apache.iceberg.BaseTable;
import org.apache.iceberg.DataFile;
import org.apache.iceberg.DataFiles;
import org.apache.iceberg.FileScanTask;
import org.apache.iceberg.Metrics;
import org.apache.iceberg.MetricsConfig;
import org.apache.iceberg.PartitionData;
import org.apache.iceberg.PartitionSpec;
import org.apache.iceberg.Schema;
import org.apache.iceberg.StructLike;
import org.apache.iceberg.Table;
import org.apache.iceberg.Transaction;
import org.apache.iceberg.avro.Avro;
import org.apache.iceberg.io.CloseableIterable;
import org.apache.iceberg.io.InputFile;
import org.apache.iceberg.mapping.MappingUtil;
import org.apache.iceberg.mapping.NameMapping;
import org.apache.iceberg.mapping.NameMappingParser;
import org.apache.iceberg.types.Types;

public final class MigrationUtils {
    private static final Logger log = Logger.get(MigrationUtils.class);
    private static final Joiner.MapJoiner PARTITION_JOINER = Joiner.on((String)"/").withKeyValueSeparator("=");
    private static final MetricsConfig METRICS_CONFIG = MetricsConfig.getDefault();

    private MigrationUtils() {
    }

    public static List<DataFile> buildDataFiles(TrinoFileSystem fileSystem, RecursiveDirectory recursive, HiveStorageFormat format, String location, PartitionSpec partitionSpec, Optional<StructLike> partition, Schema schema) throws IOException {
        FileIterator files = fileSystem.listFiles(Location.of((String)location));
        ImmutableList.Builder dataFilesBuilder = ImmutableList.builder();
        while (files.hasNext()) {
            FileEntry file = files.next();
            String fileLocation = file.location().toString();
            String relativePath = fileLocation.substring(location.length());
            if (relativePath.contains("/_") || relativePath.contains("/.") || recursive == RecursiveDirectory.FALSE && MigrationUtils.isRecursive(location, fileLocation)) continue;
            if (recursive == RecursiveDirectory.FAIL && MigrationUtils.isRecursive(location, fileLocation)) {
                throw new TrinoException((ErrorCodeSupplier)StandardErrorCode.NOT_SUPPORTED, "Recursive directory must not exist when recursive_directory argument is 'fail': " + String.valueOf(file.location()));
            }
            Metrics metrics = MigrationUtils.loadMetrics(fileSystem.newInputFile(file.location(), file.length()), format, schema);
            DataFile dataFile = MigrationUtils.buildDataFile(fileLocation, file.length(), partition, partitionSpec, format.name(), metrics);
            dataFilesBuilder.add((Object)dataFile);
        }
        ImmutableList dataFiles = dataFilesBuilder.build();
        log.debug("Found %d files in '%s'", new Object[]{dataFiles.size(), location});
        return dataFiles;
    }

    private static boolean isRecursive(String baseLocation, String location) {
        Verify.verify((boolean)location.startsWith(baseLocation), (String)"%s should start with %s", (Object)location, (Object)baseLocation);
        String suffix = location.substring(baseLocation.length() + 1).replaceFirst("^/+", "");
        return suffix.contains("/");
    }

    public static Metrics loadMetrics(TrinoInputFile file, HiveStorageFormat storageFormat, Schema schema) {
        return switch (storageFormat) {
            case HiveStorageFormat.ORC -> OrcMetrics.fileMetrics(file, METRICS_CONFIG, schema);
            case HiveStorageFormat.PARQUET -> MigrationUtils.parquetMetrics(file, METRICS_CONFIG, MappingUtil.create((Schema)schema));
            case HiveStorageFormat.AVRO -> new Metrics(Long.valueOf(Avro.rowCount((InputFile)new ForwardingInputFile(file))), null, null, null, null);
            default -> throw new TrinoException((ErrorCodeSupplier)StandardErrorCode.NOT_SUPPORTED, "Unsupported storage format: " + String.valueOf(storageFormat));
        };
    }

    private static Metrics parquetMetrics(TrinoInputFile file, MetricsConfig metricsConfig, NameMapping nameMapping) {
        Metrics metrics;
        TrinoParquetDataSource dataSource = new TrinoParquetDataSource(file, ParquetReaderOptions.defaultOptions(), new FileFormatDataSourceStats());
        try {
            ParquetMetadata metadata = MetadataReader.readFooter((ParquetDataSource)dataSource, Optional.empty());
            metrics = ParquetUtil.footerMetrics(metadata, Stream.empty(), metricsConfig, nameMapping);
        }
        catch (Throwable throwable) {
            try {
                try {
                    dataSource.close();
                }
                catch (Throwable throwable2) {
                    throwable.addSuppressed(throwable2);
                }
                throw throwable;
            }
            catch (IOException e) {
                throw new UncheckedIOException("Failed to read file footer: " + String.valueOf(file.location()), e);
            }
        }
        dataSource.close();
        return metrics;
    }

    public static void addFiles(ConnectorSession session, TrinoFileSystem fileSystem, TrinoCatalog catalog, SchemaTableName targetName, String location, HiveStorageFormat format, RecursiveDirectory recursiveDirectory) {
        BaseTable table = catalog.loadTable(session, targetName);
        PartitionSpec partitionSpec = table.spec();
        Procedures.checkProcedureArgument((boolean)partitionSpec.isUnpartitioned(), (String)"The procedure does not support partitioned tables", (Object[])new Object[0]);
        try {
            List<DataFile> dataFiles = MigrationUtils.buildDataFilesFromLocation(fileSystem, recursiveDirectory, format, location, partitionSpec, Optional.empty(), table.schema());
            MigrationUtils.addFiles(session, (Table)table, dataFiles);
        }
        catch (Exception e) {
            throw new TrinoException((ErrorCodeSupplier)IcebergErrorCode.ICEBERG_COMMIT_ERROR, "Failed to add files: " + String.valueOf(MoreObjects.firstNonNull((Object)e.getMessage(), (Object)e)), (Throwable)e);
        }
    }

    private static List<DataFile> buildDataFilesFromLocation(TrinoFileSystem fileSystem, RecursiveDirectory recursive, HiveStorageFormat format, String location, PartitionSpec partitionSpec, Optional<StructLike> partition, Schema schema) throws IOException {
        if (fileSystem.directoryExists(Location.of((String)location)).orElse(false).booleanValue()) {
            return MigrationUtils.buildDataFiles(fileSystem, recursive, format, location, partitionSpec, partition, schema);
        }
        TrinoInputFile file = fileSystem.newInputFile(Location.of((String)location));
        if (file.exists()) {
            Metrics metrics = MigrationUtils.loadMetrics(file, format, schema);
            return ImmutableList.of((Object)MigrationUtils.buildDataFile(file.location().toString(), file.length(), partition, partitionSpec, format.name(), metrics));
        }
        throw new TrinoException((ErrorCodeSupplier)StandardErrorCode.NOT_FOUND, "Location not found: " + location);
    }

    public static void addFilesFromTable(ConnectorSession session, TrinoFileSystem fileSystem, HiveMetastoreFactory metastoreFactory, Table targetTable, io.trino.metastore.Table sourceTable, Map<String, String> partitionFilter, RecursiveDirectory recursiveDirectory) {
        HiveMetastore metastore = metastoreFactory.createMetastore(Optional.of(session.getIdentity()));
        PartitionSpec partitionSpec = targetTable.spec();
        Schema schema = targetTable.schema();
        NameMapping nameMapping = MappingUtil.create((Schema)schema);
        HiveStorageFormat storageFormat = HiveMetadata.extractHiveStorageFormat((StorageFormat)sourceTable.getStorage().getStorageFormat());
        String location = sourceTable.getStorage().getLocation();
        try {
            ImmutableList.Builder dataFilesBuilder = ImmutableList.builder();
            if (partitionSpec.isUnpartitioned()) {
                log.debug("Building data files from %s", new Object[]{location});
                dataFilesBuilder.addAll(MigrationUtils.buildDataFiles(fileSystem, recursiveDirectory, storageFormat, location, partitionSpec, Optional.empty(), schema));
            } else {
                ImmutableList partitionNames = partitionFilter == null ? ImmutableList.of() : ImmutableList.of((Object)PARTITION_JOINER.join(partitionFilter));
                Map partitions = metastore.getPartitionsByNames(sourceTable, (List)partitionNames);
                for (Map.Entry partition : partitions.entrySet()) {
                    Storage storage = ((Partition)((Optional)partition.getValue()).orElseThrow(() -> new IllegalArgumentException("Invalid partition: " + (String)partition.getKey()))).getStorage();
                    log.debug("Building data files from partition: %s", new Object[]{partition});
                    HiveStorageFormat partitionStorageFormat = HiveMetadata.extractHiveStorageFormat((StorageFormat)storage.getStorageFormat());
                    PartitionData partitionData = DataFiles.data((PartitionSpec)partitionSpec, (String)((String)partition.getKey()));
                    dataFilesBuilder.addAll(MigrationUtils.buildDataFiles(fileSystem, recursiveDirectory, partitionStorageFormat, storage.getLocation(), partitionSpec, Optional.of(partitionData), schema));
                }
            }
            log.debug("Start new transaction");
            Transaction transaction = targetTable.newTransaction();
            if (!targetTable.properties().containsKey("schema.name-mapping.default")) {
                log.debug("Update default name mapping property");
                transaction.updateProperties().set("schema.name-mapping.default", NameMappingParser.toJson((NameMapping)nameMapping)).commit();
            }
            MigrationUtils.addFiles(session, targetTable, (List<DataFile>)dataFilesBuilder.build());
        }
        catch (Exception e) {
            throw new TrinoException((ErrorCodeSupplier)IcebergErrorCode.ICEBERG_COMMIT_ERROR, "Failed to add files: " + String.valueOf(MoreObjects.firstNonNull((Object)e.getMessage(), (Object)e)), (Throwable)e);
        }
    }

    public static DataFile buildDataFile(String path, long length, Optional<StructLike> partition, PartitionSpec spec, String format, Metrics metrics) {
        DataFiles.Builder dataFile = DataFiles.builder((PartitionSpec)spec).withPath(path).withFormat(format).withFileSizeInBytes(length).withMetrics(metrics);
        partition.ifPresent(arg_0 -> ((DataFiles.Builder)dataFile).withPartition(arg_0));
        return dataFile.build();
    }

    public static void addFiles(ConnectorSession session, Table table, List<DataFile> dataFiles) {
        Schema schema = table.schema();
        Set requiredFields = (Set)schema.columns().stream().filter(Types.NestedField::isRequired).map(Types.NestedField::fieldId).collect(ImmutableSet.toImmutableSet());
        ImmutableSet.Builder existingFilesBuilder = ImmutableSet.builder();
        try (CloseableIterable iterator = table.newScan().planFiles();){
            for (FileScanTask fileScanTask : iterator) {
                DataFile dataFile = (DataFile)fileScanTask.file();
                existingFilesBuilder.add((Object)dataFile.location());
            }
        }
        catch (IOException e) {
            throw new UncheckedIOException(e);
        }
        ImmutableSet existingFiles = existingFilesBuilder.build();
        if (!requiredFields.isEmpty()) {
            for (DataFile dataFile : dataFiles) {
                Map nullValueCounts = (Map)MoreObjects.firstNonNull((Object)dataFile.nullValueCounts(), Map.of());
                for (Integer field : requiredFields) {
                    Long nullCount = (Long)nullValueCounts.get(field);
                    if (nullCount != null && nullCount <= 0L) continue;
                    throw new TrinoException((ErrorCodeSupplier)StandardErrorCode.CONSTRAINT_VIOLATION, "NULL value not allowed for NOT NULL column: " + schema.findField(field.intValue()).name());
                }
            }
        }
        try {
            log.debug("Start new transaction");
            Transaction transaction = table.newTransaction();
            if (!table.properties().containsKey("schema.name-mapping.default")) {
                log.debug("Update default name mapping property");
                transaction.updateProperties().set("schema.name-mapping.default", NameMappingParser.toJson((NameMapping)MappingUtil.create((Schema)schema))).commit();
            }
            log.debug("Append data %d data files", new Object[]{dataFiles.size()});
            AppendFiles appendFiles = IcebergSessionProperties.isMergeManifestsOnWrite(session) ? transaction.newAppend() : transaction.newFastAppend();
            for (DataFile dataFile : dataFiles) {
                if (existingFiles.contains(dataFile.location())) {
                    throw new TrinoException((ErrorCodeSupplier)StandardErrorCode.ALREADY_EXISTS, "File already exists: " + dataFile.location());
                }
                appendFiles.appendFile(dataFile);
            }
            appendFiles.commit();
            transaction.commitTransaction();
            log.debug("Successfully added files to %s table", new Object[]{table.name()});
        }
        catch (Exception e) {
            throw new TrinoException((ErrorCodeSupplier)IcebergErrorCode.ICEBERG_COMMIT_ERROR, "Failed to add files: " + String.valueOf(MoreObjects.firstNonNull((Object)e.getMessage(), (Object)e)), (Throwable)e);
        }
    }

    public static enum RecursiveDirectory {
        TRUE,
        FALSE,
        FAIL;

    }
}

