/*
 * Decompiled with CFR 0.152.
 */
package io.trino.jdbc;

import com.google.common.base.Strings;
import com.google.common.base.Verify;
import com.google.common.collect.ImmutableList;
import com.google.common.collect.ImmutableMap;
import com.google.common.collect.Iterables;
import com.google.common.primitives.Ints;
import io.airlift.log.Logging;
import io.trino.client.ClientTypeSignature;
import io.trino.client.ClientTypeSignatureParameter;
import io.trino.jdbc.BaseTestJdbcResultSet;
import io.trino.jdbc.TestingJdbcUtils;
import io.trino.jdbc.TrinoPreparedStatement;
import io.trino.plugin.blackhole.BlackHolePlugin;
import io.trino.plugin.memory.MemoryPlugin;
import io.trino.server.testing.TestingTrinoServer;
import io.trino.spi.Plugin;
import java.math.BigDecimal;
import java.math.BigInteger;
import java.nio.charset.StandardCharsets;
import java.sql.Array;
import java.sql.Connection;
import java.sql.Date;
import java.sql.DriverManager;
import java.sql.JDBCType;
import java.sql.ParameterMetaData;
import java.sql.PreparedStatement;
import java.sql.ResultSet;
import java.sql.ResultSetMetaData;
import java.sql.SQLException;
import java.sql.Statement;
import java.sql.Time;
import java.sql.Timestamp;
import java.time.Duration;
import java.time.LocalDate;
import java.time.LocalDateTime;
import java.time.LocalTime;
import java.time.OffsetTime;
import java.time.ZoneId;
import java.time.ZoneOffset;
import java.util.Calendar;
import java.util.Iterator;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.TimeZone;
import java.util.stream.LongStream;
import org.assertj.core.api.AbstractStringAssert;
import org.assertj.core.api.AbstractThrowableAssert;
import org.assertj.core.api.Assertions;
import org.assertj.core.api.ObjectAssert;
import org.testng.Assert;
import org.testng.annotations.AfterClass;
import org.testng.annotations.BeforeClass;
import org.testng.annotations.Test;

public class TestJdbcPreparedStatement {
    private static final int HEADER_SIZE_LIMIT = 16384;
    private TestingTrinoServer server;

    @BeforeClass
    public void setup() throws Exception {
        Logging.initialize();
        this.server = TestingTrinoServer.builder().setProperties((Map)ImmutableMap.builder().put((Object)"http-server.max-request-header-size", (Object)String.format("%sB", 16384)).put((Object)"http-server.max-response-header-size", (Object)String.format("%sB", 16384)).buildOrThrow()).build();
        this.server.installPlugin((Plugin)new BlackHolePlugin());
        this.server.installPlugin((Plugin)new MemoryPlugin());
        this.server.createCatalog("blackhole", "blackhole");
        this.server.createCatalog("memory", "memory");
        this.server.waitForNodeRefresh(Duration.ofSeconds(10L));
        try (Connection connection = this.createConnection();
             Statement statement = connection.createStatement();){
            statement.executeUpdate("CREATE SCHEMA blackhole.blackhole");
        }
    }

    @AfterClass(alwaysRun=true)
    public void tearDown() throws Exception {
        this.server.close();
        this.server = null;
    }

    @Test
    public void testExecuteQuery() throws Exception {
        try (Connection connection = this.createConnection();
             PreparedStatement statement = connection.prepareStatement("SELECT ?, ?");){
            statement.setInt(1, 123);
            statement.setString(2, "hello");
            try (ResultSet rs = statement.executeQuery();){
                Assert.assertTrue((boolean)rs.next());
                Assert.assertEquals((int)rs.getInt(1), (int)123);
                Assert.assertEquals((String)rs.getString(2), (String)"hello");
                Assert.assertFalse((boolean)rs.next());
            }
            Assert.assertTrue((boolean)statement.execute());
            rs = statement.getResultSet();
            try {
                Assert.assertTrue((boolean)rs.next());
                Assert.assertEquals((int)rs.getInt(1), (int)123);
                Assert.assertEquals((String)rs.getString(2), (String)"hello");
                Assert.assertFalse((boolean)rs.next());
            }
            finally {
                if (rs != null) {
                    rs.close();
                }
            }
        }
    }

    @Test
    public void testGetMetadata() throws Exception {
        try (Connection connection = this.createConnection("blackhole", "blackhole");){
            try (Statement statement = connection.createStatement();){
                statement.execute("CREATE TABLE test_get_metadata (c_boolean boolean, c_decimal decimal, c_decimal_2 decimal(10,3),c_varchar varchar, c_varchar_2 varchar(10), c_row row(x integer, y array(integer)), c_array array(integer), c_map map(integer, integer))");
            }
            statement = connection.prepareStatement("SELECT * FROM test_get_metadata");
            try {
                ResultSetMetaData metadata = statement.getMetaData();
                Assert.assertEquals((int)metadata.getColumnCount(), (int)8);
                for (int i = 1; i <= metadata.getColumnCount(); ++i) {
                    Assert.assertEquals((String)metadata.getCatalogName(i), (String)"blackhole");
                    Assert.assertEquals((String)metadata.getSchemaName(i), (String)"blackhole");
                    Assert.assertEquals((String)metadata.getTableName(i), (String)"test_get_metadata");
                }
                Assert.assertEquals((String)metadata.getColumnName(1), (String)"c_boolean");
                Assert.assertEquals((String)metadata.getColumnTypeName(1), (String)"boolean");
                Assert.assertEquals((String)metadata.getColumnName(2), (String)"c_decimal");
                Assert.assertEquals((String)metadata.getColumnTypeName(2), (String)"decimal(38,0)");
                Assert.assertEquals((String)metadata.getColumnName(3), (String)"c_decimal_2");
                Assert.assertEquals((String)metadata.getColumnTypeName(3), (String)"decimal(10,3)");
                Assert.assertEquals((String)metadata.getColumnName(4), (String)"c_varchar");
                Assert.assertEquals((String)metadata.getColumnTypeName(4), (String)"varchar");
                Assert.assertEquals((String)metadata.getColumnName(5), (String)"c_varchar_2");
                Assert.assertEquals((String)metadata.getColumnTypeName(5), (String)"varchar(10)");
                Assert.assertEquals((String)metadata.getColumnName(6), (String)"c_row");
                Assert.assertEquals((String)metadata.getColumnTypeName(6), (String)"row");
                Assert.assertEquals((String)metadata.getColumnName(7), (String)"c_array");
                Assert.assertEquals((String)metadata.getColumnTypeName(7), (String)"array");
                Assert.assertEquals((String)metadata.getColumnName(8), (String)"c_map");
                Assert.assertEquals((String)metadata.getColumnTypeName(8), (String)"map");
            }
            finally {
                if (statement != null) {
                    statement.close();
                }
            }
            statement = connection.createStatement();
            try {
                statement.execute("DROP TABLE test_get_metadata");
            }
            finally {
                if (statement != null) {
                    statement.close();
                }
            }
        }
    }

    @Test
    public void testGetParameterMetaData() throws Exception {
        try (Connection connection = this.createConnection("blackhole", "blackhole");){
            try (Statement statement = connection.createStatement();){
                statement.execute("CREATE TABLE test_get_parameterMetaData (c_boolean boolean, c_decimal decimal, c_decimal_2 decimal(10,3),c_varchar varchar, c_varchar_2 varchar(5), c_row row(x integer, y array(integer)), c_array array(integer), c_map map(integer, integer), c_tinyint tinyint, c_integer integer, c_bigint bigint, c_smallint smallint, c_real real, c_double double)");
            }
            statement = connection.prepareStatement("SELECT ? FROM test_get_parameterMetaData WHERE c_boolean = ? AND c_decimal = ? AND c_decimal_2 = ? AND c_varchar = ? AND c_varchar_2 = ? AND c_row = ? AND c_array = ? AND c_map = ? AND c_tinyint = ? AND c_integer = ? AND c_bigint = ? AND c_smallint = ? AND c_real = ? AND c_double = ?");
            try {
                ParameterMetaData parameterMetaData = statement.getParameterMetaData();
                Assert.assertEquals((int)parameterMetaData.getParameterCount(), (int)15);
                Assert.assertEquals((String)parameterMetaData.getParameterClassName(1), (String)"unknown");
                Assert.assertEquals((int)parameterMetaData.getParameterType(1), (int)0);
                Assert.assertEquals((String)parameterMetaData.getParameterTypeName(1), (String)"unknown");
                Assert.assertEquals((int)parameterMetaData.isNullable(1), (int)2);
                Assert.assertFalse((boolean)parameterMetaData.isSigned(1));
                Assert.assertEquals((int)parameterMetaData.getParameterMode(1), (int)0);
                Assert.assertEquals((String)parameterMetaData.getParameterClassName(2), (String)Boolean.class.getName());
                Assert.assertEquals((int)parameterMetaData.getParameterType(2), (int)16);
                Assert.assertEquals((String)parameterMetaData.getParameterTypeName(2), (String)"boolean");
                Assert.assertEquals((int)parameterMetaData.isNullable(2), (int)2);
                Assert.assertFalse((boolean)parameterMetaData.isSigned(2));
                Assert.assertEquals((int)parameterMetaData.getParameterMode(2), (int)0);
                Assert.assertEquals((String)parameterMetaData.getParameterClassName(3), (String)BigDecimal.class.getName());
                Assert.assertEquals((int)parameterMetaData.getParameterType(3), (int)3);
                Assert.assertEquals((String)parameterMetaData.getParameterTypeName(3), (String)"decimal");
                Assert.assertEquals((int)parameterMetaData.isNullable(3), (int)2);
                Assert.assertTrue((boolean)parameterMetaData.isSigned(3));
                Assert.assertEquals((int)parameterMetaData.getParameterMode(3), (int)0);
                Assert.assertEquals((String)parameterMetaData.getParameterClassName(4), (String)BigDecimal.class.getName());
                Assert.assertEquals((int)parameterMetaData.getParameterType(4), (int)3);
                Assert.assertEquals((String)parameterMetaData.getParameterTypeName(4), (String)"decimal");
                Assert.assertEquals((int)parameterMetaData.getPrecision(4), (int)10);
                Assert.assertEquals((int)parameterMetaData.getScale(4), (int)3);
                Assert.assertEquals((int)parameterMetaData.isNullable(4), (int)2);
                Assert.assertTrue((boolean)parameterMetaData.isSigned(4));
                Assert.assertEquals((int)parameterMetaData.getParameterMode(4), (int)0);
                Assert.assertEquals((String)parameterMetaData.getParameterClassName(5), (String)String.class.getName());
                Assert.assertEquals((int)parameterMetaData.getParameterType(5), (int)12);
                Assert.assertEquals((String)parameterMetaData.getParameterTypeName(5), (String)"varchar");
                Assert.assertEquals((int)parameterMetaData.isNullable(5), (int)2);
                Assert.assertFalse((boolean)parameterMetaData.isSigned(5));
                Assert.assertEquals((int)parameterMetaData.getParameterMode(5), (int)0);
                Assert.assertEquals((String)parameterMetaData.getParameterClassName(6), (String)String.class.getName());
                Assert.assertEquals((int)parameterMetaData.getParameterType(6), (int)12);
                Assert.assertEquals((String)parameterMetaData.getParameterTypeName(6), (String)"varchar");
                Assert.assertEquals((int)parameterMetaData.getPrecision(6), (int)5);
                Assert.assertEquals((int)parameterMetaData.isNullable(6), (int)2);
                Assert.assertFalse((boolean)parameterMetaData.isSigned(6));
                Assert.assertEquals((int)parameterMetaData.getParameterMode(6), (int)0);
                Assert.assertEquals((String)parameterMetaData.getParameterClassName(7), (String)String.class.getName());
                Assert.assertEquals((int)parameterMetaData.getParameterType(7), (int)2000);
                Assert.assertEquals((String)parameterMetaData.getParameterTypeName(7), (String)"row");
                Assert.assertEquals((int)parameterMetaData.isNullable(7), (int)2);
                Assert.assertFalse((boolean)parameterMetaData.isSigned(7));
                Assert.assertEquals((int)parameterMetaData.getParameterMode(7), (int)0);
                Assert.assertEquals((String)parameterMetaData.getParameterClassName(8), (String)Array.class.getName());
                Assert.assertEquals((int)parameterMetaData.getParameterType(8), (int)2003);
                Assert.assertEquals((String)parameterMetaData.getParameterTypeName(8), (String)"array");
                Assert.assertEquals((int)parameterMetaData.isNullable(8), (int)2);
                Assert.assertFalse((boolean)parameterMetaData.isSigned(8));
                Assert.assertEquals((int)parameterMetaData.getParameterMode(8), (int)0);
                Assert.assertEquals((String)parameterMetaData.getParameterClassName(9), (String)String.class.getName());
                Assert.assertEquals((int)parameterMetaData.getParameterType(9), (int)2000);
                Assert.assertEquals((String)parameterMetaData.getParameterTypeName(9), (String)"map");
                Assert.assertEquals((int)parameterMetaData.isNullable(9), (int)2);
                Assert.assertFalse((boolean)parameterMetaData.isSigned(9));
                Assert.assertEquals((int)parameterMetaData.getParameterMode(9), (int)0);
                Assert.assertEquals((String)parameterMetaData.getParameterClassName(10), (String)Byte.class.getName());
                Assert.assertEquals((int)parameterMetaData.getParameterType(10), (int)-6);
                Assert.assertEquals((String)parameterMetaData.getParameterTypeName(10), (String)"tinyint");
                Assert.assertEquals((int)parameterMetaData.isNullable(10), (int)2);
                Assert.assertTrue((boolean)parameterMetaData.isSigned(10));
                Assert.assertEquals((int)parameterMetaData.getParameterMode(10), (int)0);
                Assert.assertEquals((String)parameterMetaData.getParameterClassName(11), (String)Integer.class.getName());
                Assert.assertEquals((int)parameterMetaData.getParameterType(11), (int)4);
                Assert.assertEquals((String)parameterMetaData.getParameterTypeName(11), (String)"integer");
                Assert.assertEquals((int)parameterMetaData.isNullable(11), (int)2);
                Assert.assertTrue((boolean)parameterMetaData.isSigned(11));
                Assert.assertEquals((int)parameterMetaData.getParameterMode(11), (int)0);
                Assert.assertEquals((String)parameterMetaData.getParameterClassName(12), (String)Long.class.getName());
                Assert.assertEquals((int)parameterMetaData.getParameterType(12), (int)-5);
                Assert.assertEquals((String)parameterMetaData.getParameterTypeName(12), (String)"bigint");
                Assert.assertEquals((int)parameterMetaData.isNullable(12), (int)2);
                Assert.assertTrue((boolean)parameterMetaData.isSigned(12));
                Assert.assertEquals((int)parameterMetaData.getParameterMode(12), (int)0);
                Assert.assertEquals((String)parameterMetaData.getParameterClassName(13), (String)Short.class.getName());
                Assert.assertEquals((int)parameterMetaData.getParameterType(13), (int)5);
                Assert.assertEquals((String)parameterMetaData.getParameterTypeName(13), (String)"smallint");
                Assert.assertEquals((int)parameterMetaData.isNullable(13), (int)2);
                Assert.assertTrue((boolean)parameterMetaData.isSigned(13));
                Assert.assertEquals((int)parameterMetaData.getParameterMode(13), (int)0);
                Assert.assertEquals((String)parameterMetaData.getParameterClassName(14), (String)Float.class.getName());
                Assert.assertEquals((int)parameterMetaData.getParameterType(14), (int)7);
                Assert.assertEquals((String)parameterMetaData.getParameterTypeName(14), (String)"real");
                Assert.assertEquals((int)parameterMetaData.isNullable(14), (int)2);
                Assert.assertTrue((boolean)parameterMetaData.isSigned(14));
                Assert.assertEquals((int)parameterMetaData.getParameterMode(14), (int)0);
                Assert.assertEquals((String)parameterMetaData.getParameterClassName(15), (String)Double.class.getName());
                Assert.assertEquals((int)parameterMetaData.getParameterType(15), (int)8);
                Assert.assertEquals((String)parameterMetaData.getParameterTypeName(15), (String)"double");
                Assert.assertEquals((int)parameterMetaData.isNullable(15), (int)2);
                Assert.assertTrue((boolean)parameterMetaData.isSigned(15));
                Assert.assertEquals((int)parameterMetaData.getParameterMode(15), (int)0);
            }
            finally {
                if (statement != null) {
                    statement.close();
                }
            }
            statement = connection.createStatement();
            try {
                statement.execute("DROP TABLE test_get_parameterMetaData");
            }
            finally {
                if (statement != null) {
                    statement.close();
                }
            }
        }
    }

    @Test
    public void testGetClientTypeSignatureFromTypeString() {
        ClientTypeSignature actualClientTypeSignature = TrinoPreparedStatement.getClientTypeSignatureFromTypeString((String)"boolean");
        ClientTypeSignature expectedClientTypeSignature = new ClientTypeSignature("boolean", (List)ImmutableList.of());
        Assert.assertEquals((Object)actualClientTypeSignature, (Object)expectedClientTypeSignature);
        actualClientTypeSignature = TrinoPreparedStatement.getClientTypeSignatureFromTypeString((String)"decimal(10,3)");
        expectedClientTypeSignature = new ClientTypeSignature("decimal", (List)ImmutableList.of((Object)ClientTypeSignatureParameter.ofLong((long)10L), (Object)ClientTypeSignatureParameter.ofLong((long)3L)));
        Assert.assertEquals((Object)actualClientTypeSignature, (Object)expectedClientTypeSignature);
        actualClientTypeSignature = TrinoPreparedStatement.getClientTypeSignatureFromTypeString((String)"varchar");
        expectedClientTypeSignature = new ClientTypeSignature("varchar", (List)ImmutableList.of((Object)ClientTypeSignatureParameter.ofLong((long)Integer.MAX_VALUE)));
        Assert.assertEquals((Object)actualClientTypeSignature, (Object)expectedClientTypeSignature);
        actualClientTypeSignature = TrinoPreparedStatement.getClientTypeSignatureFromTypeString((String)"varchar(10)");
        expectedClientTypeSignature = new ClientTypeSignature("varchar", (List)ImmutableList.of((Object)ClientTypeSignatureParameter.ofLong((long)10L)));
        Assert.assertEquals((Object)actualClientTypeSignature, (Object)expectedClientTypeSignature);
        actualClientTypeSignature = TrinoPreparedStatement.getClientTypeSignatureFromTypeString((String)"row(x integer, y array(integer))");
        expectedClientTypeSignature = new ClientTypeSignature("row", (List)ImmutableList.of());
        Assert.assertEquals((Object)actualClientTypeSignature, (Object)expectedClientTypeSignature);
        actualClientTypeSignature = TrinoPreparedStatement.getClientTypeSignatureFromTypeString((String)"array(integer)");
        expectedClientTypeSignature = new ClientTypeSignature("array", (List)ImmutableList.of());
        Assert.assertEquals((Object)actualClientTypeSignature, (Object)expectedClientTypeSignature);
        actualClientTypeSignature = TrinoPreparedStatement.getClientTypeSignatureFromTypeString((String)"map(integer, integer)");
        expectedClientTypeSignature = new ClientTypeSignature("map", (List)ImmutableList.of());
        Assert.assertEquals((Object)actualClientTypeSignature, (Object)expectedClientTypeSignature);
        actualClientTypeSignature = TrinoPreparedStatement.getClientTypeSignatureFromTypeString((String)"timestamp(12) with time zone");
        expectedClientTypeSignature = new ClientTypeSignature("timestamp with time zone", (List)ImmutableList.of((Object)ClientTypeSignatureParameter.ofLong((long)12L)));
        Assert.assertEquals((Object)actualClientTypeSignature, (Object)expectedClientTypeSignature);
        actualClientTypeSignature = TrinoPreparedStatement.getClientTypeSignatureFromTypeString((String)"time(13) with time zone");
        expectedClientTypeSignature = new ClientTypeSignature("time with time zone", (List)ImmutableList.of((Object)ClientTypeSignatureParameter.ofLong((long)13L)));
        Assert.assertEquals((Object)actualClientTypeSignature, (Object)expectedClientTypeSignature);
    }

    @Test
    public void testDeallocate() throws Exception {
        try (Connection connection = this.createConnection();){
            for (int i = 0; i < 200; ++i) {
                try {
                    connection.prepareStatement("SELECT '" + Strings.repeat((String)"a", (int)300) + "'").close();
                    continue;
                }
                catch (Exception e) {
                    throw new RuntimeException("Failed at " + i, e);
                }
            }
        }
    }

    @Test
    public void testCloseIdempotency() throws Exception {
        try (Connection connection = this.createConnection();){
            PreparedStatement statement = connection.prepareStatement("SELECT 123");
            statement.close();
            statement.close();
        }
    }

    @Test
    public void testLargePreparedStatement() throws Exception {
        int elements = 16385;
        try (Connection connection = this.createConnection();
             PreparedStatement statement = connection.prepareStatement("VALUES ?" + Strings.repeat((String)", ?", (int)(elements - 1)));){
            for (int i = 0; i < elements; ++i) {
                statement.setLong(i + 1, i);
            }
            try (ResultSet resultSet = statement.executeQuery();){
                Assertions.assertThat(TestingJdbcUtils.readRows(resultSet).stream().map(Iterables::getOnlyElement)).containsExactlyInAnyOrder(LongStream.range(0L, elements).boxed().toArray());
            }
        }
    }

    @Test
    public void testExecuteUpdate() throws Exception {
        try (Connection connection = this.createConnection("blackhole", "blackhole");){
            try (Statement statement = connection.createStatement();){
                statement.execute("CREATE TABLE test_execute_update (c_boolean boolean, c_bigint bigint, c_double double, c_decimal decimal, c_varchar varchar, c_varbinary varbinary, c_null bigint)");
            }
            statement = connection.prepareStatement("INSERT INTO test_execute_update VALUES (?, ?, ?, ?, ?, ?, ?)");
            try {
                statement.setBoolean(1, true);
                statement.setLong(2, 5L);
                statement.setDouble(3, 7.0);
                statement.setBigDecimal(4, BigDecimal.valueOf(8L));
                statement.setString(5, "abc'xyz");
                statement.setBytes(6, "xyz".getBytes(StandardCharsets.UTF_8));
                statement.setNull(7, -5);
                Assert.assertEquals((int)statement.executeUpdate(), (int)1);
                Assert.assertFalse((boolean)statement.execute());
                Assert.assertEquals((int)statement.getUpdateCount(), (int)1);
                Assert.assertEquals((long)statement.getLargeUpdateCount(), (long)1L);
            }
            finally {
                if (statement != null) {
                    statement.close();
                }
            }
            statement = connection.createStatement();
            try {
                statement.execute("DROP TABLE test_execute_update");
            }
            finally {
                if (statement != null) {
                    statement.close();
                }
            }
        }
    }

    @Test
    public void testExecuteBatch() throws Exception {
        try (Connection connection = this.createConnection("memory", "default");){
            try (Statement statement = connection.createStatement();){
                statement.execute("CREATE TABLE test_execute_batch(c_int integer)");
            }
            try (PreparedStatement preparedStatement = connection.prepareStatement("INSERT INTO test_execute_batch VALUES (?)");){
                Assert.assertEquals((Object)preparedStatement.executeBatch(), (Object)new int[0]);
                for (int i = 0; i < 3; ++i) {
                    preparedStatement.setInt(1, i);
                    preparedStatement.addBatch();
                }
                Assert.assertEquals((Object)preparedStatement.executeBatch(), (Object)new int[]{1, 1, 1});
                try (Statement statement = connection.createStatement();){
                    ResultSet resultSet = statement.executeQuery("SELECT c_int FROM test_execute_batch");
                    Assertions.assertThat(TestingJdbcUtils.readRows(resultSet)).containsExactlyInAnyOrder((Object[])new List[]{TestingJdbcUtils.list(0), TestingJdbcUtils.list(1), TestingJdbcUtils.list(2)});
                }
                Assert.assertEquals((Object)preparedStatement.executeBatch(), (Object)new int[0]);
                preparedStatement.setBoolean(1, true);
                preparedStatement.clearBatch();
                Assert.assertEquals((Object)preparedStatement.executeBatch(), (Object)new int[0]);
                preparedStatement.setInt(1, 1);
                Assert.assertEquals((int)preparedStatement.executeUpdate(), (int)1);
            }
            statement = connection.createStatement();
            try {
                statement.execute("DROP TABLE test_execute_batch");
            }
            finally {
                if (statement != null) {
                    statement.close();
                }
            }
        }
    }

    @Test
    public void testInvalidExecuteBatch() throws Exception {
        try (Connection connection = this.createConnection("blackhole", "blackhole");){
            try (Statement statement = connection.createStatement();){
                statement.execute("CREATE TABLE test_invalid_execute_batch(c_int integer)");
            }
            statement = connection.prepareStatement("INSERT INTO test_invalid_execute_batch VALUES (?)");
            try {
                statement.setInt(1, 1);
                statement.addBatch();
                String message = "Batch prepared statement must be executed using executeBatch method";
                ((AbstractThrowableAssert)Assertions.assertThatThrownBy(((PreparedStatement)statement)::executeQuery).isInstanceOf(SQLException.class)).hasMessage(message);
                ((AbstractThrowableAssert)Assertions.assertThatThrownBy(((PreparedStatement)statement)::executeUpdate).isInstanceOf(SQLException.class)).hasMessage(message);
                ((AbstractThrowableAssert)Assertions.assertThatThrownBy(((PreparedStatement)statement)::executeLargeUpdate).isInstanceOf(SQLException.class)).hasMessage(message);
                ((AbstractThrowableAssert)Assertions.assertThatThrownBy(((PreparedStatement)statement)::execute).isInstanceOf(SQLException.class)).hasMessage(message);
            }
            finally {
                if (statement != null) {
                    statement.close();
                }
            }
            statement = connection.createStatement();
            try {
                statement.execute("DROP TABLE test_invalid_execute_batch");
            }
            finally {
                if (statement != null) {
                    statement.close();
                }
            }
        }
    }

    @Test
    public void testPrepareMultiple() throws Exception {
        try (Connection connection = this.createConnection();
             PreparedStatement statement1 = connection.prepareStatement("SELECT 123");
             PreparedStatement statement2 = connection.prepareStatement("SELECT 456");){
            try (ResultSet rs = statement1.executeQuery();){
                Assert.assertTrue((boolean)rs.next());
                Assert.assertEquals((long)rs.getLong(1), (long)123L);
                Assert.assertFalse((boolean)rs.next());
            }
            rs = statement2.executeQuery();
            try {
                Assert.assertTrue((boolean)rs.next());
                Assert.assertEquals((long)rs.getLong(1), (long)456L);
                Assert.assertFalse((boolean)rs.next());
            }
            finally {
                if (rs != null) {
                    rs.close();
                }
            }
        }
    }

    @Test
    public void testPrepareLarge() throws Exception {
        String sql = String.format("SELECT '%s' = '%s'", Strings.repeat((String)"x", (int)100000), Strings.repeat((String)"y", (int)100000));
        try (Connection connection = this.createConnection();
             PreparedStatement statement = connection.prepareStatement(sql);
             ResultSet rs = statement.executeQuery();){
            Assert.assertTrue((boolean)rs.next());
            Assert.assertFalse((boolean)rs.getBoolean(1));
            Assert.assertFalse((boolean)rs.next());
        }
    }

    @Test
    public void testSetNull() throws Exception {
        this.assertSetNull(16);
        this.assertSetNull(-7, 16);
        this.assertSetNull(-6);
        this.assertSetNull(5);
        this.assertSetNull(4);
        this.assertSetNull(-5);
        this.assertSetNull(7);
        this.assertSetNull(6, 7);
        this.assertSetNull(3);
        this.assertSetNull(2, 3);
        this.assertSetNull(1);
        this.assertSetNull(-15, 1);
        this.assertSetNull(12, 12);
        this.assertSetNull(-9, 12);
        this.assertSetNull(-1, 12);
        this.assertSetNull(12, 12);
        this.assertSetNull(2005, 12);
        this.assertSetNull(2011, 12);
        this.assertSetNull(-3, -3);
        this.assertSetNull(-3);
        this.assertSetNull(2004, -3);
        this.assertSetNull(91);
        this.assertSetNull(92);
        this.assertSetNull(93);
        this.assertSetNull(0);
    }

    private void assertSetNull(int sqlType) throws SQLException {
        this.assertSetNull(sqlType, sqlType);
    }

    private void assertSetNull(int sqlType, int expectedSqlType) throws SQLException {
        try (Connection connection = this.createConnection();
             PreparedStatement statement = connection.prepareStatement("SELECT ?");){
            statement.setNull(1, sqlType);
            try (ResultSet rs = statement.executeQuery();){
                Assert.assertTrue((boolean)rs.next());
                Assert.assertNull((Object)rs.getObject(1));
                Assert.assertTrue((boolean)rs.wasNull());
                Assert.assertFalse((boolean)rs.next());
                Assert.assertEquals((int)rs.getMetaData().getColumnType(1), (int)expectedSqlType);
            }
        }
    }

    @Test
    public void testConvertBoolean() throws SQLException {
        this.assertBind((ps, i) -> ps.setBoolean(i, true)).roundTripsAs(16, true);
        this.assertBind((ps, i) -> ps.setBoolean(i, false)).roundTripsAs(16, false);
        this.assertBind((ps, i) -> ps.setObject(i, true)).roundTripsAs(16, true);
        this.assertBind((ps, i) -> ps.setObject(i, false)).roundTripsAs(16, false);
        Iterator iterator = Ints.asList((int[])new int[]{16, -7}).iterator();
        while (iterator.hasNext()) {
            int type = (Integer)iterator.next();
            this.assertBind((ps, i) -> ps.setObject(i, (Object)true, type)).roundTripsAs(16, true);
            this.assertBind((ps, i) -> ps.setObject(i, (Object)false, type)).roundTripsAs(16, false);
            this.assertBind((ps, i) -> ps.setObject(i, (Object)13, type)).roundTripsAs(16, true);
            this.assertBind((ps, i) -> ps.setObject(i, (Object)0, type)).roundTripsAs(16, false);
            this.assertBind((ps, i) -> ps.setObject(i, (Object)"1", type)).roundTripsAs(16, true);
            this.assertBind((ps, i) -> ps.setObject(i, (Object)"true", type)).roundTripsAs(16, true);
            this.assertBind((ps, i) -> ps.setObject(i, (Object)"0", type)).roundTripsAs(16, false);
            this.assertBind((ps, i) -> ps.setObject(i, (Object)"false", type)).roundTripsAs(16, false);
        }
    }

    @Test
    public void testConvertTinyint() throws SQLException {
        this.assertBind((ps, i) -> ps.setByte(i, (byte)123)).roundTripsAs(-6, (byte)123);
        this.assertBind((ps, i) -> ps.setObject(i, (byte)123)).roundTripsAs(-6, (byte)123);
        this.assertBind((ps, i) -> ps.setObject(i, (Object)123, -6)).roundTripsAs(-6, (byte)123);
        this.assertBind((ps, i) -> ps.setObject(i, (Object)123, -6)).roundTripsAs(-6, (byte)123);
        this.assertBind((ps, i) -> ps.setObject(i, (Object)123, -6)).roundTripsAs(-6, (byte)123);
        this.assertBind((ps, i) -> ps.setObject(i, (Object)123L, -6)).roundTripsAs(-6, (byte)123);
        this.assertBind((ps, i) -> ps.setObject(i, (Object)Float.valueOf(123.9f), -6)).roundTripsAs(-6, (byte)123);
        this.assertBind((ps, i) -> ps.setObject(i, (Object)123.9, -6)).roundTripsAs(-6, (byte)123);
        this.assertBind((ps, i) -> ps.setObject(i, (Object)BigInteger.valueOf(123L), -6)).roundTripsAs(-6, (byte)123);
        this.assertBind((ps, i) -> ps.setObject(i, (Object)BigDecimal.valueOf(123L), -6)).roundTripsAs(-6, (byte)123);
        this.assertBind((ps, i) -> ps.setObject(i, (Object)BigDecimal.valueOf(123.9), -6)).roundTripsAs(-6, (byte)123);
        this.assertBind((ps, i) -> ps.setObject(i, (Object)"123", -6)).roundTripsAs(-6, (byte)123);
        this.assertBind((ps, i) -> ps.setObject(i, (Object)true, -6)).roundTripsAs(-6, (byte)1);
        this.assertBind((ps, i) -> ps.setObject(i, (Object)false, -6)).roundTripsAs(-6, (byte)0);
    }

    @Test
    public void testConvertSmallint() throws SQLException {
        this.assertBind((ps, i) -> ps.setShort(i, (short)123)).roundTripsAs(5, (short)123);
        this.assertBind((ps, i) -> ps.setObject(i, (short)123)).roundTripsAs(5, (short)123);
        this.assertBind((ps, i) -> ps.setObject(i, (Object)123, 5)).roundTripsAs(5, (short)123);
        this.assertBind((ps, i) -> ps.setObject(i, (Object)123, 5)).roundTripsAs(5, (short)123);
        this.assertBind((ps, i) -> ps.setObject(i, (Object)123, 5)).roundTripsAs(5, (short)123);
        this.assertBind((ps, i) -> ps.setObject(i, (Object)123L, 5)).roundTripsAs(5, (short)123);
        this.assertBind((ps, i) -> ps.setObject(i, (Object)Float.valueOf(123.9f), 5)).roundTripsAs(5, (short)123);
        this.assertBind((ps, i) -> ps.setObject(i, (Object)123.9, 5)).roundTripsAs(5, (short)123);
        this.assertBind((ps, i) -> ps.setObject(i, (Object)BigInteger.valueOf(123L), 5)).roundTripsAs(5, (short)123);
        this.assertBind((ps, i) -> ps.setObject(i, (Object)BigDecimal.valueOf(123L), 5)).roundTripsAs(5, (short)123);
        this.assertBind((ps, i) -> ps.setObject(i, (Object)BigDecimal.valueOf(123.9), 5)).roundTripsAs(5, (short)123);
        this.assertBind((ps, i) -> ps.setObject(i, (Object)"123", 5)).roundTripsAs(5, (short)123);
        this.assertBind((ps, i) -> ps.setObject(i, (Object)true, 5)).roundTripsAs(5, (short)1);
        this.assertBind((ps, i) -> ps.setObject(i, (Object)false, 5)).roundTripsAs(5, (short)0);
    }

    @Test
    public void testConvertInteger() throws SQLException {
        this.assertBind((ps, i) -> ps.setInt(i, 123)).roundTripsAs(4, 123);
        this.assertBind((ps, i) -> ps.setObject(i, 123)).roundTripsAs(4, 123);
        this.assertBind((ps, i) -> ps.setObject(i, (Object)123, 4)).roundTripsAs(4, 123);
        this.assertBind((ps, i) -> ps.setObject(i, (Object)123, 4)).roundTripsAs(4, 123);
        this.assertBind((ps, i) -> ps.setObject(i, (Object)123, 4)).roundTripsAs(4, 123);
        this.assertBind((ps, i) -> ps.setObject(i, (Object)123, 4)).roundTripsAs(4, 123);
        this.assertBind((ps, i) -> ps.setObject(i, (Object)123L, 4)).roundTripsAs(4, 123);
        this.assertBind((ps, i) -> ps.setObject(i, (Object)Float.valueOf(123.9f), 4)).roundTripsAs(4, 123);
        this.assertBind((ps, i) -> ps.setObject(i, (Object)123.9, 4)).roundTripsAs(4, 123);
        this.assertBind((ps, i) -> ps.setObject(i, (Object)BigInteger.valueOf(123L), 4)).roundTripsAs(4, 123);
        this.assertBind((ps, i) -> ps.setObject(i, (Object)BigDecimal.valueOf(123L), 4)).roundTripsAs(4, 123);
        this.assertBind((ps, i) -> ps.setObject(i, (Object)BigDecimal.valueOf(123.9), 4)).roundTripsAs(4, 123);
        this.assertBind((ps, i) -> ps.setObject(i, (Object)"123", 4)).roundTripsAs(4, 123);
        this.assertBind((ps, i) -> ps.setObject(i, (Object)true, 4)).roundTripsAs(4, 1);
        this.assertBind((ps, i) -> ps.setObject(i, (Object)false, 4)).roundTripsAs(4, 0);
    }

    @Test
    public void testConvertBigint() throws SQLException {
        this.assertBind((ps, i) -> ps.setLong(i, 123L)).roundTripsAs(-5, 123L);
        this.assertBind((ps, i) -> ps.setObject(i, 123L)).roundTripsAs(-5, 123L);
        this.assertBind((ps, i) -> ps.setObject(i, (Object)123, -5)).roundTripsAs(-5, 123L);
        this.assertBind((ps, i) -> ps.setObject(i, (Object)123, -5)).roundTripsAs(-5, 123L);
        this.assertBind((ps, i) -> ps.setObject(i, (Object)123, -5)).roundTripsAs(-5, 123L);
        this.assertBind((ps, i) -> ps.setObject(i, (Object)123L, -5)).roundTripsAs(-5, 123L);
        this.assertBind((ps, i) -> ps.setObject(i, (Object)Float.valueOf(123.9f), -5)).roundTripsAs(-5, 123L);
        this.assertBind((ps, i) -> ps.setObject(i, (Object)123.9, -5)).roundTripsAs(-5, 123L);
        this.assertBind((ps, i) -> ps.setObject(i, (Object)BigInteger.valueOf(123L), -5)).roundTripsAs(-5, 123L);
        this.assertBind((ps, i) -> ps.setObject(i, (Object)BigDecimal.valueOf(123L), -5)).roundTripsAs(-5, 123L);
        this.assertBind((ps, i) -> ps.setObject(i, (Object)BigDecimal.valueOf(123.9), -5)).roundTripsAs(-5, 123L);
        this.assertBind((ps, i) -> ps.setObject(i, (Object)"123", -5)).roundTripsAs(-5, 123L);
        this.assertBind((ps, i) -> ps.setObject(i, (Object)true, -5)).roundTripsAs(-5, 1L);
        this.assertBind((ps, i) -> ps.setObject(i, (Object)false, -5)).roundTripsAs(-5, 0L);
    }

    @Test
    public void testConvertReal() throws SQLException {
        this.assertBind((ps, i) -> ps.setFloat(i, 4.2f)).roundTripsAs(7, Float.valueOf(4.2f));
        this.assertBind((ps, i) -> ps.setObject(i, Float.valueOf(4.2f))).roundTripsAs(7, Float.valueOf(4.2f));
        Iterator iterator = Ints.asList((int[])new int[]{7, 6}).iterator();
        while (iterator.hasNext()) {
            int type = (Integer)iterator.next();
            this.assertBind((ps, i) -> ps.setObject(i, (Object)123, type)).roundTripsAs(7, Float.valueOf(123.0f));
            this.assertBind((ps, i) -> ps.setObject(i, (Object)123, type)).roundTripsAs(7, Float.valueOf(123.0f));
            this.assertBind((ps, i) -> ps.setObject(i, (Object)123, type)).roundTripsAs(7, Float.valueOf(123.0f));
            this.assertBind((ps, i) -> ps.setObject(i, (Object)123L, type)).roundTripsAs(7, Float.valueOf(123.0f));
            this.assertBind((ps, i) -> ps.setObject(i, (Object)Float.valueOf(123.9f), type)).roundTripsAs(7, Float.valueOf(123.9f));
            this.assertBind((ps, i) -> ps.setObject(i, (Object)123.9, type)).roundTripsAs(7, Float.valueOf(123.9f));
            this.assertBind((ps, i) -> ps.setObject(i, (Object)BigInteger.valueOf(123L), type)).roundTripsAs(7, Float.valueOf(123.0f));
            this.assertBind((ps, i) -> ps.setObject(i, (Object)BigDecimal.valueOf(123L), type)).roundTripsAs(7, Float.valueOf(123.0f));
            this.assertBind((ps, i) -> ps.setObject(i, (Object)BigDecimal.valueOf(123.9), type)).roundTripsAs(7, Float.valueOf(123.9f));
            this.assertBind((ps, i) -> ps.setObject(i, (Object)"4.2", type)).roundTripsAs(7, Float.valueOf(4.2f));
            this.assertBind((ps, i) -> ps.setObject(i, (Object)true, type)).roundTripsAs(7, Float.valueOf(1.0f));
            this.assertBind((ps, i) -> ps.setObject(i, (Object)false, type)).roundTripsAs(7, Float.valueOf(0.0f));
        }
    }

    @Test
    public void testConvertDouble() throws SQLException {
        this.assertBind((ps, i) -> ps.setDouble(i, 4.2)).roundTripsAs(8, 4.2);
        this.assertBind((ps, i) -> ps.setObject(i, 4.2)).roundTripsAs(8, 4.2);
        this.assertBind((ps, i) -> ps.setObject(i, (Object)123, 8)).roundTripsAs(8, 123.0);
        this.assertBind((ps, i) -> ps.setObject(i, (Object)123, 8)).roundTripsAs(8, 123.0);
        this.assertBind((ps, i) -> ps.setObject(i, (Object)123, 8)).roundTripsAs(8, 123.0);
        this.assertBind((ps, i) -> ps.setObject(i, (Object)123L, 8)).roundTripsAs(8, 123.0);
        this.assertBind((ps, i) -> ps.setObject(i, (Object)Float.valueOf(123.9f), 8)).roundTripsAs(8, 123.9f);
        this.assertBind((ps, i) -> ps.setObject(i, (Object)123.9, 8)).roundTripsAs(8, 123.9);
        this.assertBind((ps, i) -> ps.setObject(i, (Object)BigInteger.valueOf(123L), 8)).roundTripsAs(8, 123.0);
        this.assertBind((ps, i) -> ps.setObject(i, (Object)BigDecimal.valueOf(123L), 8)).roundTripsAs(8, 123.0);
        this.assertBind((ps, i) -> ps.setObject(i, (Object)BigDecimal.valueOf(123.9), 8)).roundTripsAs(8, 123.9);
        this.assertBind((ps, i) -> ps.setObject(i, (Object)"4.2", 8)).roundTripsAs(8, 4.2);
        this.assertBind((ps, i) -> ps.setObject(i, (Object)true, 8)).roundTripsAs(8, 1.0);
        this.assertBind((ps, i) -> ps.setObject(i, (Object)false, 8)).roundTripsAs(8, 0.0);
    }

    @Test
    public void testConvertDecimal() throws SQLException {
        this.assertBind((ps, i) -> ps.setBigDecimal(i, BigDecimal.valueOf(123L))).roundTripsAs(3, BigDecimal.valueOf(123L));
        this.assertBind((ps, i) -> ps.setObject(i, BigDecimal.valueOf(123L))).roundTripsAs(3, BigDecimal.valueOf(123L));
        Iterator iterator = Ints.asList((int[])new int[]{3, 2}).iterator();
        while (iterator.hasNext()) {
            int type = (Integer)iterator.next();
            this.assertBind((ps, i) -> ps.setObject(i, (Object)123, type)).roundTripsAs(3, BigDecimal.valueOf(123L));
            this.assertBind((ps, i) -> ps.setObject(i, (Object)123, type)).roundTripsAs(3, BigDecimal.valueOf(123L));
            this.assertBind((ps, i) -> ps.setObject(i, (Object)123, type)).roundTripsAs(3, BigDecimal.valueOf(123L));
            this.assertBind((ps, i) -> ps.setObject(i, (Object)123L, type)).roundTripsAs(3, BigDecimal.valueOf(123L));
            this.assertBind((ps, i) -> ps.setObject(i, (Object)Float.valueOf(123.9f), type)).roundTripsAs(3, BigDecimal.valueOf(123.9f));
            this.assertBind((ps, i) -> ps.setObject(i, (Object)123.9, type)).roundTripsAs(3, BigDecimal.valueOf(123.9));
            this.assertBind((ps, i) -> ps.setObject(i, (Object)BigInteger.valueOf(123L), type)).roundTripsAs(3, BigDecimal.valueOf(123L));
            this.assertBind((ps, i) -> ps.setObject(i, (Object)BigDecimal.valueOf(123L), type)).roundTripsAs(3, BigDecimal.valueOf(123L));
            this.assertBind((ps, i) -> ps.setObject(i, (Object)BigDecimal.valueOf(123.9), type)).roundTripsAs(3, BigDecimal.valueOf(123.9));
            this.assertBind((ps, i) -> ps.setObject(i, (Object)"123", type)).roundTripsAs(3, BigDecimal.valueOf(123L));
            this.assertBind((ps, i) -> ps.setObject(i, (Object)true, type)).roundTripsAs(3, BigDecimal.valueOf(1L));
            this.assertBind((ps, i) -> ps.setObject(i, (Object)false, type)).roundTripsAs(3, BigDecimal.valueOf(0L));
        }
    }

    @Test
    public void testConvertVarchar() throws SQLException {
        this.assertBind((ps, i) -> ps.setString(i, "hello")).roundTripsAs(12, "hello");
        this.assertBind((ps, i) -> ps.setObject(i, "hello")).roundTripsAs(12, "hello");
        String unicodeAndNull = "abc'xyz\u0000\u2603\ud835\udcabtest";
        this.assertBind((ps, i) -> ps.setString(i, unicodeAndNull)).roundTripsAs(12, unicodeAndNull);
        Iterator iterator = Ints.asList((int[])new int[]{1, -15, 12, -9, -1, -16}).iterator();
        while (iterator.hasNext()) {
            int type = (Integer)iterator.next();
            this.assertBind((ps, i) -> ps.setObject(i, (Object)123, type)).roundTripsAs(12, "123");
            this.assertBind((ps, i) -> ps.setObject(i, (Object)123, type)).roundTripsAs(12, "123");
            this.assertBind((ps, i) -> ps.setObject(i, (Object)123, type)).roundTripsAs(12, "123");
            this.assertBind((ps, i) -> ps.setObject(i, (Object)123, type)).roundTripsAs(12, "123");
            this.assertBind((ps, i) -> ps.setObject(i, (Object)123L, type)).roundTripsAs(12, "123");
            this.assertBind((ps, i) -> ps.setObject(i, (Object)Float.valueOf(123.9f), type)).roundTripsAs(12, "123.9");
            this.assertBind((ps, i) -> ps.setObject(i, (Object)123.9, type)).roundTripsAs(12, "123.9");
            this.assertBind((ps, i) -> ps.setObject(i, (Object)BigInteger.valueOf(123L), type)).roundTripsAs(12, "123");
            this.assertBind((ps, i) -> ps.setObject(i, (Object)BigDecimal.valueOf(123L), type)).roundTripsAs(12, "123");
            this.assertBind((ps, i) -> ps.setObject(i, (Object)BigDecimal.valueOf(123.9), type)).roundTripsAs(12, "123.9");
            this.assertBind((ps, i) -> ps.setObject(i, (Object)"hello", type)).roundTripsAs(12, "hello");
            this.assertBind((ps, i) -> ps.setObject(i, (Object)true, type)).roundTripsAs(12, "true");
            this.assertBind((ps, i) -> ps.setObject(i, (Object)false, type)).roundTripsAs(12, "false");
        }
    }

    @Test
    public void testConvertVarbinary() throws SQLException {
        String value = "abc\u0000xyz";
        byte[] bytes = value.getBytes(StandardCharsets.UTF_8);
        this.assertBind((ps, i) -> ps.setBytes(i, bytes)).roundTripsAs(-3, bytes);
        this.assertBind((ps, i) -> ps.setObject(i, bytes)).roundTripsAs(-3, bytes);
        Iterator iterator = Ints.asList((int[])new int[]{-2, -3, -4}).iterator();
        while (iterator.hasNext()) {
            int type = (Integer)iterator.next();
            this.assertBind((ps, i) -> ps.setObject(i, (Object)bytes, type)).roundTripsAs(-3, bytes);
            this.assertBind((ps, i) -> ps.setObject(i, (Object)value, type)).roundTripsAs(-3, bytes);
        }
    }

    @Test
    public void testConvertDate() throws SQLException {
        LocalDate date = LocalDate.of(2001, 5, 6);
        Date sqlDate = Date.valueOf(date);
        java.util.Date javaDate = new java.util.Date(sqlDate.getTime());
        LocalDateTime dateTime = LocalDateTime.of(date, LocalTime.of(12, 34, 56));
        Timestamp sqlTimestamp = Timestamp.valueOf(dateTime);
        this.assertBind((ps, i) -> ps.setDate(i, sqlDate)).resultsIn("date", "DATE '2001-05-06'").roundTripsAs(91, sqlDate);
        this.assertBind((ps, i) -> ps.setObject(i, sqlDate)).resultsIn("date", "DATE '2001-05-06'").roundTripsAs(91, sqlDate);
        this.assertBind((ps, i) -> ps.setObject(i, (Object)sqlDate, 91)).resultsIn("date", "DATE '2001-05-06'").roundTripsAs(91, sqlDate);
        this.assertBind((ps, i) -> ps.setObject(i, (Object)sqlTimestamp, 91)).resultsIn("date", "DATE '2001-05-06'").roundTripsAs(91, sqlDate);
        this.assertBind((ps, i) -> ps.setObject(i, (Object)javaDate, 91)).resultsIn("date", "DATE '2001-05-06'").roundTripsAs(91, sqlDate);
        this.assertBind((ps, i) -> ps.setObject(i, (Object)date, 91)).resultsIn("date", "DATE '2001-05-06'").roundTripsAs(91, sqlDate);
        this.assertBind((ps, i) -> ps.setObject(i, (Object)dateTime, 91)).resultsIn("date", "DATE '2001-05-06'").roundTripsAs(91, sqlDate);
        this.assertBind((ps, i) -> ps.setObject(i, (Object)"2001-05-06", 91)).resultsIn("date", "DATE '2001-05-06'").roundTripsAs(91, sqlDate);
    }

    @Test
    public void testConvertLocalDate() throws SQLException {
        LocalDate date = LocalDate.of(2001, 5, 6);
        this.assertBind((ps, i) -> ps.setObject(i, date)).resultsIn("date", "DATE '2001-05-06'").roundTripsAs(91, Date.valueOf(date));
        this.assertBind((ps, i) -> ps.setObject(i, (Object)date, 91)).resultsIn("date", "DATE '2001-05-06'").roundTripsAs(91, Date.valueOf(date));
        this.assertBind((ps, i) -> ps.setObject(i, (Object)date, 92)).isInvalid("Cannot convert instance of java.time.LocalDate to time");
        this.assertBind((ps, i) -> ps.setObject(i, (Object)date, 2013)).isInvalid("Cannot convert instance of java.time.LocalDate to time with time zone");
        this.assertBind((ps, i) -> ps.setObject(i, (Object)date, 93)).isInvalid("Cannot convert instance of java.time.LocalDate to timestamp");
        this.assertBind((ps, i) -> ps.setObject(i, (Object)date, 2014)).isInvalid("Cannot convert instance of java.time.LocalDate to timestamp with time zone");
        LocalDate jvmGapDate = LocalDate.of(1970, 1, 1);
        TestJdbcPreparedStatement.checkIsGap(ZoneId.systemDefault(), jvmGapDate.atTime(LocalTime.MIDNIGHT));
        BindAssertion assertion = this.assertBind((ps, i) -> ps.setObject(i, jvmGapDate)).resultsIn("date", "DATE '1970-01-01'");
        ((AbstractThrowableAssert)Assertions.assertThatThrownBy(() -> assertion.roundTripsAs(91, Date.valueOf(jvmGapDate))).isInstanceOf(SQLException.class)).hasStackTraceContaining("io.trino.jdbc.TrinoResultSet.getObject").hasMessage("Expected value to be a date but is: 1970-01-01");
        this.assertBind((ps, i) -> ps.setObject(i, (Object)jvmGapDate, 91)).resultsIn("date", "DATE '1970-01-01'");
    }

    @Test
    public void testConvertTime() throws SQLException {
        LocalTime time = LocalTime.of(12, 34, 56);
        Time sqlTime = Time.valueOf(time);
        java.util.Date javaDate = new java.util.Date(sqlTime.getTime());
        LocalDateTime dateTime = LocalDateTime.of(LocalDate.of(2001, 5, 6), time);
        Timestamp sqlTimestamp = Timestamp.valueOf(dateTime);
        this.assertBind((ps, i) -> ps.setTime(i, sqlTime)).resultsIn("time(3)", "TIME '12:34:56.000'").roundTripsAs(92, sqlTime);
        this.assertBind((ps, i) -> ps.setObject(i, sqlTime)).resultsIn("time(3)", "TIME '12:34:56.000'").roundTripsAs(92, sqlTime);
        this.assertBind((ps, i) -> ps.setObject(i, (Object)sqlTime, 92)).resultsIn("time(3)", "TIME '12:34:56.000'").roundTripsAs(92, sqlTime);
        this.assertBind((ps, i) -> ps.setObject(i, (Object)sqlTimestamp, 92)).resultsIn("time(3)", "TIME '12:34:56.000'").roundTripsAs(92, sqlTime);
        this.assertBind((ps, i) -> ps.setObject(i, (Object)javaDate, 92)).resultsIn("time(3)", "TIME '12:34:56.000'").roundTripsAs(92, sqlTime);
        this.assertBind((ps, i) -> ps.setObject(i, (Object)dateTime, 92)).resultsIn("time(0)", "TIME '12:34:56'").roundTripsAs(92, sqlTime);
        this.assertBind((ps, i) -> ps.setObject(i, (Object)"12:34:56", 92)).resultsIn("time(0)", "TIME '12:34:56'").roundTripsAs(92, sqlTime);
        this.assertBind((ps, i) -> ps.setObject(i, (Object)"12:34:56.123", 92)).resultsIn("time(3)", "TIME '12:34:56.123'");
        this.assertBind((ps, i) -> ps.setObject(i, (Object)"12:34:56.123456", 92)).resultsIn("time(6)", "TIME '12:34:56.123456'");
        this.assertBind((ps, i) -> ps.setObject(i, (Object)"12:34:56.123456789", 92)).resultsIn("time(9)", "TIME '12:34:56.123456789'");
        this.assertBind((ps, i) -> ps.setObject(i, (Object)"12:34:56.123456789012", 92)).resultsIn("time(12)", "TIME '12:34:56.123456789012'");
        Time timeWithDecisecond = new Time(sqlTime.getTime() + 100L);
        this.assertBind((ps, i) -> ps.setObject(i, timeWithDecisecond)).resultsIn("time(3)", "TIME '12:34:56.100'").roundTripsAs(92, timeWithDecisecond);
        this.assertBind((ps, i) -> ps.setObject(i, (Object)timeWithDecisecond, 92)).resultsIn("time(3)", "TIME '12:34:56.100'").roundTripsAs(92, timeWithDecisecond);
        Time timeWithMillisecond = new Time(sqlTime.getTime() + 123L);
        this.assertBind((ps, i) -> ps.setObject(i, timeWithMillisecond)).resultsIn("time(3)", "TIME '12:34:56.123'").roundTripsAs(92, timeWithMillisecond);
        this.assertBind((ps, i) -> ps.setObject(i, (Object)timeWithMillisecond, 92)).resultsIn("time(3)", "TIME '12:34:56.123'").roundTripsAs(92, timeWithMillisecond);
    }

    @Test
    public void testConvertTimeWithTimeZone() throws SQLException {
        this.assertBind((ps, i) -> ps.setObject(i, (Object)OffsetTime.of(12, 34, 56, 0, ZoneOffset.UTC), 2013)).resultsIn("time(0) with time zone", "TIME '12:34:56+00:00'").roundTripsAs(2013, BaseTestJdbcResultSet.toSqlTime(LocalTime.of(5, 34, 56)));
        this.assertBind((ps, i) -> ps.setObject(i, OffsetTime.of(12, 34, 56, 0, ZoneOffset.UTC))).resultsIn("time(0) with time zone", "TIME '12:34:56+00:00'");
        this.assertBind((ps, i) -> ps.setObject(i, (Object)OffsetTime.of(12, 34, 56, 0, ZoneOffset.UTC), JDBCType.TIME_WITH_TIMEZONE)).resultsIn("time(0) with time zone", "TIME '12:34:56+00:00'");
        this.assertBind((ps, i) -> ps.setObject(i, (Object)OffsetTime.of(12, 34, 56, 555000000, ZoneOffset.UTC), 2013)).resultsIn("time(3) with time zone", "TIME '12:34:56.555+00:00'").roundTripsAs(2013, BaseTestJdbcResultSet.toSqlTime(LocalTime.of(5, 34, 56, 555000000)));
        this.assertBind((ps, i) -> ps.setObject(i, (Object)OffsetTime.of(12, 34, 56, 555555000, ZoneOffset.UTC), 2013)).resultsIn("time(6) with time zone", "TIME '12:34:56.555555+00:00'").roundTripsAs(2013, BaseTestJdbcResultSet.toSqlTime(LocalTime.of(5, 34, 56, 556000000)));
        this.assertBind((ps, i) -> ps.setObject(i, (Object)OffsetTime.of(12, 34, 56, 555555555, ZoneOffset.UTC), 2013)).resultsIn("time(9) with time zone", "TIME '12:34:56.555555555+00:00'").roundTripsAs(2013, BaseTestJdbcResultSet.toSqlTime(LocalTime.of(5, 34, 56, 556000000)));
        this.assertBind((ps, i) -> ps.setObject(i, (Object)OffsetTime.of(12, 34, 56, 123456789, ZoneOffset.ofHoursMinutes(7, 35)), 2013)).resultsIn("time(9) with time zone", "TIME '12:34:56.123456789+07:35'");
        this.assertBind((ps, i) -> ps.setObject(i, (Object)OffsetTime.of(12, 34, 56, 123456789, ZoneOffset.ofHoursMinutes(-7, -35)), 2013)).resultsIn("time(9) with time zone", "TIME '12:34:56.123456789-07:35'").roundTripsAs(2013, BaseTestJdbcResultSet.toSqlTime(LocalTime.of(13, 9, 56, 123000000)));
        this.assertBind((ps, i) -> ps.setObject(i, (Object)"12:34:56.123 +05:45", 2013)).resultsIn("time(3) with time zone", "TIME '12:34:56.123 +05:45'");
        this.assertBind((ps, i) -> ps.setObject(i, (Object)"12:34:56.123456 +05:45", 2013)).resultsIn("time(6) with time zone", "TIME '12:34:56.123456 +05:45'");
        this.assertBind((ps, i) -> ps.setObject(i, (Object)"12:34:56.123456789 +05:45", 2013)).resultsIn("time(9) with time zone", "TIME '12:34:56.123456789 +05:45'");
        this.assertBind((ps, i) -> ps.setObject(i, (Object)"12:34:56.123456789012 +05:45", 2013)).resultsIn("time(12) with time zone", "TIME '12:34:56.123456789012 +05:45'");
    }

    @Test
    public void testConvertTimestamp() throws SQLException {
        LocalDateTime dateTime = LocalDateTime.of(2001, 5, 6, 12, 34, 56);
        Date sqlDate = Date.valueOf(dateTime.toLocalDate());
        Time sqlTime = Time.valueOf(dateTime.toLocalTime());
        Timestamp sqlTimestamp = Timestamp.valueOf(dateTime);
        Timestamp sameInstantInWarsawZone = Timestamp.valueOf(dateTime.atZone(ZoneId.systemDefault()).withZoneSameInstant(ZoneId.of("Europe/Warsaw")).toLocalDateTime());
        java.util.Date javaDate = java.util.Date.from(dateTime.atZone(ZoneId.systemDefault()).toInstant());
        this.assertBind((ps, i) -> ps.setTimestamp(i, sqlTimestamp)).resultsIn("timestamp(3)", "TIMESTAMP '2001-05-06 12:34:56.000'").roundTripsAs(93, sqlTimestamp);
        this.assertBind((ps, i) -> ps.setTimestamp(i, sqlTimestamp, null)).resultsIn("timestamp(3)", "TIMESTAMP '2001-05-06 12:34:56.000'").roundTripsAs(93, sqlTimestamp);
        this.assertBind((ps, i) -> ps.setTimestamp(i, sqlTimestamp, Calendar.getInstance())).resultsIn("timestamp(3)", "TIMESTAMP '2001-05-06 12:34:56.000'").roundTripsAs(93, sqlTimestamp);
        this.assertBind((ps, i) -> ps.setTimestamp(i, sqlTimestamp, Calendar.getInstance(TimeZone.getTimeZone(ZoneId.of("Europe/Warsaw"))))).resultsIn("timestamp(3)", "TIMESTAMP '2001-05-06 20:34:56.000'").roundTripsAs(93, sameInstantInWarsawZone);
        this.assertBind((ps, i) -> ps.setObject(i, sqlTimestamp)).resultsIn("timestamp(3)", "TIMESTAMP '2001-05-06 12:34:56.000'").roundTripsAs(93, sqlTimestamp);
        this.assertBind((ps, i) -> ps.setObject(i, (Object)sqlDate, 93)).resultsIn("timestamp(3)", "TIMESTAMP '2001-05-06 00:00:00.000'").roundTripsAs(93, new Timestamp(sqlDate.getTime()));
        this.assertBind((ps, i) -> ps.setObject(i, (Object)sqlTime, 93)).resultsIn("timestamp(3)", "TIMESTAMP '1970-01-01 12:34:56.000'").roundTripsAs(93, new Timestamp(sqlTime.getTime()));
        this.assertBind((ps, i) -> ps.setObject(i, (Object)sqlTimestamp, 93)).resultsIn("timestamp(3)", "TIMESTAMP '2001-05-06 12:34:56.000'").roundTripsAs(93, sqlTimestamp);
        this.assertBind((ps, i) -> ps.setObject(i, (Object)javaDate, 93)).resultsIn("timestamp(3)", "TIMESTAMP '2001-05-06 12:34:56.000'").roundTripsAs(93, sqlTimestamp);
        this.assertBind((ps, i) -> ps.setObject(i, (Object)dateTime, 93)).resultsIn("timestamp(0)", "TIMESTAMP '2001-05-06 12:34:56'").roundTripsAs(93, sqlTimestamp);
        this.assertBind((ps, i) -> ps.setObject(i, (Object)"2001-05-06 12:34:56", 93)).resultsIn("timestamp(0)", "TIMESTAMP '2001-05-06 12:34:56'").roundTripsAs(93, sqlTimestamp);
        this.assertBind((ps, i) -> ps.setObject(i, (Object)"2001-05-06 12:34:56.123", 93)).resultsIn("timestamp(3)", "TIMESTAMP '2001-05-06 12:34:56.123'");
        this.assertBind((ps, i) -> ps.setObject(i, (Object)"2001-05-06 12:34:56.123456", 93)).resultsIn("timestamp(6)", "TIMESTAMP '2001-05-06 12:34:56.123456'");
        this.assertBind((ps, i) -> ps.setObject(i, (Object)"2001-05-06 12:34:56.123456789", 93)).resultsIn("timestamp(9)", "TIMESTAMP '2001-05-06 12:34:56.123456789'");
        this.assertBind((ps, i) -> ps.setObject(i, (Object)"2001-05-06 12:34:56.123456789012", 93)).resultsIn("timestamp(12)", "TIMESTAMP '2001-05-06 12:34:56.123456789012'");
        Timestamp timestampWithWithDecisecond = new Timestamp(sqlTimestamp.getTime() + 100L);
        this.assertBind((ps, i) -> ps.setTimestamp(i, timestampWithWithDecisecond)).resultsIn("timestamp(3)", "TIMESTAMP '2001-05-06 12:34:56.100'").roundTripsAs(93, timestampWithWithDecisecond);
        this.assertBind((ps, i) -> ps.setObject(i, timestampWithWithDecisecond)).resultsIn("timestamp(3)", "TIMESTAMP '2001-05-06 12:34:56.100'").roundTripsAs(93, timestampWithWithDecisecond);
        this.assertBind((ps, i) -> ps.setObject(i, (Object)timestampWithWithDecisecond, 93)).resultsIn("timestamp(3)", "TIMESTAMP '2001-05-06 12:34:56.100'").roundTripsAs(93, timestampWithWithDecisecond);
        Timestamp timestampWithMillisecond = new Timestamp(sqlTimestamp.getTime() + 123L);
        this.assertBind((ps, i) -> ps.setTimestamp(i, timestampWithMillisecond)).resultsIn("timestamp(3)", "TIMESTAMP '2001-05-06 12:34:56.123'").roundTripsAs(93, timestampWithMillisecond);
        this.assertBind((ps, i) -> ps.setObject(i, timestampWithMillisecond)).resultsIn("timestamp(3)", "TIMESTAMP '2001-05-06 12:34:56.123'").roundTripsAs(93, timestampWithMillisecond);
        this.assertBind((ps, i) -> ps.setObject(i, (Object)timestampWithMillisecond, 93)).resultsIn("timestamp(3)", "TIMESTAMP '2001-05-06 12:34:56.123'").roundTripsAs(93, timestampWithMillisecond);
    }

    @Test
    public void testConvertTimestampWithTimeZone() throws SQLException {
        this.assertBind((ps, i) -> ps.setObject(i, (Object)"1970-01-01 12:34:56.123 +05:45", 2014)).resultsIn("timestamp(3) with time zone", "TIMESTAMP '1970-01-01 12:34:56.123 +05:45'");
        this.assertBind((ps, i) -> ps.setObject(i, (Object)"1970-01-01 12:34:56.123456 +05:45", 2014)).resultsIn("timestamp(6) with time zone", "TIMESTAMP '1970-01-01 12:34:56.123456 +05:45'");
        this.assertBind((ps, i) -> ps.setObject(i, (Object)"1970-01-01 12:34:56.123456789 +05:45", 2014)).resultsIn("timestamp(9) with time zone", "TIMESTAMP '1970-01-01 12:34:56.123456789 +05:45'");
        this.assertBind((ps, i) -> ps.setObject(i, (Object)"1970-01-01 12:34:56.123456789012 +05:45", 2014)).resultsIn("timestamp(12) with time zone", "TIMESTAMP '1970-01-01 12:34:56.123456789012 +05:45'");
    }

    @Test
    public void testInvalidConversions() throws SQLException {
        this.assertBind((ps, i) -> ps.setObject(i, String.class)).isInvalid("Unsupported object type: java.lang.Class");
        this.assertBind((ps, i) -> ps.setObject(i, String.class, -5)).isInvalid("Cannot convert instance of java.lang.Class to SQL type -5");
        this.assertBind((ps, i) -> ps.setObject(i, (Object)"abc", 5)).isInvalid("Cannot convert instance of java.lang.String to SQL type 5");
    }

    private BindAssertion assertBind(Binder binder) {
        return new BindAssertion(this::createConnection, binder);
    }

    private Connection createConnection() throws SQLException {
        String url = String.format("jdbc:trino://%s", this.server.getAddress());
        return DriverManager.getConnection(url, "test", null);
    }

    private Connection createConnection(String catalog, String schema) throws SQLException {
        String url = String.format("jdbc:trino://%s/%s/%s", this.server.getAddress(), catalog, schema);
        return DriverManager.getConnection(url, "test", null);
    }

    private static void checkIsGap(ZoneId zone, LocalDateTime dateTime) {
        Verify.verify((boolean)TestJdbcPreparedStatement.isGap(zone, dateTime), (String)"Expected %s to be a gap in %s", (Object)dateTime, (Object)zone);
    }

    private static boolean isGap(ZoneId zone, LocalDateTime dateTime) {
        return zone.getRules().getValidOffsets(dateTime).isEmpty();
    }

    private static interface Binder {
        public void bind(PreparedStatement var1, int var2) throws SQLException;
    }

    private static class BindAssertion {
        private final ConnectionFactory connectionFactory;
        private final Binder binder;

        public BindAssertion(ConnectionFactory connectionFactory, Binder binder) {
            this.connectionFactory = Objects.requireNonNull(connectionFactory, "connectionFactory is null");
            this.binder = Objects.requireNonNull(binder, "binder is null");
        }

        public BindAssertion isInvalid(String expectedMessage) throws SQLException {
            try (Connection connection = this.connectionFactory.createConnection();
                 PreparedStatement statement = connection.prepareStatement("SELECT ?");){
                ((AbstractThrowableAssert)Assertions.assertThatThrownBy(() -> this.binder.bind(statement, 1)).isInstanceOf(SQLException.class)).hasMessage(expectedMessage);
            }
            return this;
        }

        public BindAssertion roundTripsAs(int expectedSqlType, Object expectedValue) throws SQLException {
            try (Connection connection = this.connectionFactory.createConnection();
                 PreparedStatement statement = connection.prepareStatement("SELECT ?");){
                this.binder.bind(statement, 1);
                try (ResultSet rs = statement.executeQuery();){
                    Verify.verify((boolean)rs.next(), (String)"no row returned", (Object[])new Object[0]);
                    Assert.assertEquals((Object)rs.getObject(1), (Object)expectedValue);
                    Verify.verify((!rs.next() ? 1 : 0) != 0, (String)"unexpected second row", (Object[])new Object[0]);
                    Assert.assertEquals((int)rs.getMetaData().getColumnType(1), (int)expectedSqlType);
                }
            }
            return this;
        }

        public BindAssertion resultsIn(String type, String expectedValueLiteral) throws SQLException {
            String sql = "SELECT   typeof(bound) type_of_bind,   bound,   CAST(bound AS varchar) bound_as_varchar,   typeof(literal) type_of_literal,   literal,   CAST(literal AS varchar) literal_as_varchar,   bound = literal are_equal FROM (VALUES (?, " + expectedValueLiteral + ")) t(bound, literal)";
            try (Connection connection = this.connectionFactory.createConnection();
                 PreparedStatement statement = connection.prepareStatement(sql);){
                this.binder.bind(statement, 1);
                try (ResultSet rs = statement.executeQuery();){
                    Verify.verify((boolean)rs.next(), (String)"no row returned", (Object[])new Object[0]);
                    ((AbstractStringAssert)Assertions.assertThat((String)rs.getString("type_of_bind")).as("type_of_bind", new Object[0])).isEqualTo(type);
                    ((AbstractStringAssert)Assertions.assertThat((String)rs.getString("type_of_literal")).as("type_of_literal (sanity check)", new Object[0])).isEqualTo(type);
                    ((AbstractStringAssert)Assertions.assertThat((String)rs.getString("bound_as_varchar")).as("bound should cast to VARCHAR the same way as literal " + expectedValueLiteral, new Object[0])).isEqualTo(rs.getString("literal_as_varchar"));
                    ((ObjectAssert)Assertions.assertThat((Object)rs.getObject("are_equal")).as("Expected bound value to be equal to " + expectedValueLiteral, new Object[0])).isEqualTo((Object)true);
                    Verify.verify((!rs.next() ? 1 : 0) != 0, (String)"unexpected second row", (Object[])new Object[0]);
                }
            }
            return this;
        }
    }

    private static interface ConnectionFactory {
        public Connection createConnection() throws SQLException;
    }
}

