/*
 * Decompiled with CFR 0.152.
 */
package io.trino.jdbc;

import com.google.common.base.Verify;
import com.google.common.collect.ImmutableList;
import com.google.common.collect.ImmutableMap;
import com.google.common.math.IntMath;
import io.trino.jdbc.Row;
import io.trino.jdbc.TrinoResultSet;
import java.io.InputStream;
import java.math.BigDecimal;
import java.math.RoundingMode;
import java.nio.charset.StandardCharsets;
import java.sql.Array;
import java.sql.Connection;
import java.sql.Date;
import java.sql.ResultSet;
import java.sql.ResultSetMetaData;
import java.sql.SQLException;
import java.sql.Statement;
import java.sql.Time;
import java.sql.Timestamp;
import java.time.Instant;
import java.time.LocalDate;
import java.time.LocalDateTime;
import java.time.LocalTime;
import java.time.ZoneId;
import java.time.ZonedDateTime;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collections;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.concurrent.TimeUnit;
import org.assertj.core.api.AbstractThrowableAssert;
import org.assertj.core.api.Assertions;
import org.testng.annotations.Test;

public abstract class BaseTestJdbcResultSet {
    protected abstract Connection createConnection() throws SQLException;

    @Test
    public void testDuplicateColumnLabels() throws Exception {
        try (ConnectedStatement connectedStatement = this.newStatement();
             ResultSet rs = connectedStatement.getStatement().executeQuery("SELECT 123 x, 456 x");){
            ResultSetMetaData metadata = rs.getMetaData();
            Assertions.assertThat((int)metadata.getColumnCount()).isEqualTo(2);
            Assertions.assertThat((String)metadata.getColumnName(1)).isEqualTo("x");
            Assertions.assertThat((String)metadata.getColumnName(2)).isEqualTo("x");
            Assertions.assertThat((boolean)rs.next()).isTrue();
            Assertions.assertThat((long)rs.getLong(1)).isEqualTo(123L);
            Assertions.assertThat((long)rs.getLong(2)).isEqualTo(456L);
            Assertions.assertThat((long)rs.getLong("x")).isEqualTo(123L);
        }
    }

    @Test
    public void testPrimitiveTypes() throws Exception {
        try (ConnectedStatement connectedStatement = this.newStatement();){
            this.checkRepresentation(connectedStatement.getStatement(), "123", 4, 123);
            this.checkRepresentation(connectedStatement.getStatement(), "12300000000", -5, 12300000000L);
            this.checkRepresentation(connectedStatement.getStatement(), "REAL '123.45'", 7, Float.valueOf(123.45f));
            this.checkRepresentation(connectedStatement.getStatement(), "1e-1", 8, 0.1);
            this.checkRepresentation(connectedStatement.getStatement(), "1.0E0 / 0.0E0", 8, Double.POSITIVE_INFINITY);
            this.checkRepresentation(connectedStatement.getStatement(), "0.0E0 / 0.0E0", 8, Double.NaN);
            this.checkRepresentation(connectedStatement.getStatement(), "true", 16, true);
            this.checkRepresentation(connectedStatement.getStatement(), "'hello'", 12, (ResultSet rs, int column) -> {
                Assertions.assertThat((Object)rs.getObject(column)).isEqualTo((Object)"hello");
                Assertions.assertThat((InputStream)rs.getAsciiStream(column)).hasBinaryContent("hello".getBytes(StandardCharsets.US_ASCII));
                ((AbstractThrowableAssert)Assertions.assertThatThrownBy(() -> rs.getBinaryStream(column)).isInstanceOf(SQLException.class)).hasMessage("Value is not a byte array: hello");
            });
            this.checkRepresentation(connectedStatement.getStatement(), "CAST(NULL AS VARCHAR)", 12, (ResultSet rs, int column) -> {
                Assertions.assertThat((InputStream)rs.getAsciiStream(column)).isNull();
                Assertions.assertThat((InputStream)rs.getBinaryStream(column)).isNull();
            });
            this.checkRepresentation(connectedStatement.getStatement(), "cast('foo' as char(5))", 1, (ResultSet rs, int column) -> {
                Assertions.assertThat((Object)rs.getObject(column)).isEqualTo((Object)"foo  ");
                Assertions.assertThat((InputStream)rs.getAsciiStream(column)).hasBinaryContent("foo  ".getBytes(StandardCharsets.US_ASCII));
            });
            this.checkRepresentation(connectedStatement.getStatement(), "VARCHAR '123'", 12, (ResultSet rs, int column) -> Assertions.assertThat((long)rs.getLong(column)).isEqualTo(123L));
            this.checkRepresentation(connectedStatement.getStatement(), "VARCHAR ''", 12, (ResultSet rs, int column) -> {
                ((AbstractThrowableAssert)Assertions.assertThatThrownBy(() -> rs.getLong(column)).isInstanceOf(SQLException.class)).hasMessage("Value is not a number: ");
                ((AbstractThrowableAssert)Assertions.assertThatThrownBy(() -> rs.getDouble(column)).isInstanceOf(SQLException.class)).hasMessage("Value is not a number: ");
                ((AbstractThrowableAssert)Assertions.assertThatThrownBy(() -> rs.getBoolean(column)).isInstanceOf(SQLException.class)).hasMessage("Value is not a boolean: ");
                Assertions.assertThat((InputStream)rs.getAsciiStream(column)).isEmpty();
            });
            this.checkRepresentation(connectedStatement.getStatement(), "VARCHAR '123e-1'", 12, (ResultSet rs, int column) -> {
                Assertions.assertThat((double)rs.getDouble(column)).isEqualTo(12.3);
                Assertions.assertThat((long)rs.getLong(column)).isEqualTo(12L);
                Assertions.assertThat((float)rs.getFloat(column)).isEqualTo(12.3f);
                Assertions.assertThat((InputStream)rs.getAsciiStream(column)).hasBinaryContent("123e-1".getBytes(StandardCharsets.US_ASCII));
            });
            this.checkRepresentation(connectedStatement.getStatement(), "DOUBLE '123.456'", 8, (ResultSet rs, int column) -> {
                Assertions.assertThat((double)rs.getDouble(column)).isEqualTo(123.456);
                Assertions.assertThat((long)rs.getLong(column)).isEqualTo(123L);
                Assertions.assertThat((float)rs.getFloat(column)).isEqualTo(123.456f);
                ((AbstractThrowableAssert)Assertions.assertThatThrownBy(() -> rs.getAsciiStream(column)).isInstanceOf(SQLException.class)).hasMessage("Value is not a string: 123.456");
            });
            this.checkRepresentation(connectedStatement.getStatement(), "VARCHAR '123'", 12, (ResultSet rs, int column) -> {
                Assertions.assertThat((double)rs.getDouble(column)).isEqualTo(123.0);
                Assertions.assertThat((long)rs.getLong(column)).isEqualTo(123L);
                Assertions.assertThat((float)rs.getFloat(column)).isEqualTo(123.0f);
                Assertions.assertThat((InputStream)rs.getAsciiStream(column)).hasBinaryContent("123".getBytes(StandardCharsets.US_ASCII));
            });
        }
    }

    @Test
    public void testDecimal() throws Exception {
        try (ConnectedStatement connectedStatement = this.newStatement();){
            this.checkRepresentation(connectedStatement.getStatement(), "0.1", 3, new BigDecimal("0.1"));
            this.checkRepresentation(connectedStatement.getStatement(), "DECIMAL '0.12'", 3, (ResultSet rs, int column) -> {
                Assertions.assertThat((BigDecimal)rs.getBigDecimal(column)).isEqualTo((Object)new BigDecimal("0.12"));
                Assertions.assertThat((double)rs.getDouble(column)).isEqualTo(0.12);
                Assertions.assertThat((long)rs.getLong(column)).isEqualTo(0L);
                Assertions.assertThat((float)rs.getFloat(column)).isEqualTo(0.12f);
            });
            long outsideOfDoubleExactRange = 9223372036854775774L;
            Verify.verify(((long)((double)outsideOfDoubleExactRange) - outsideOfDoubleExactRange != 0L ? 1 : 0) != 0, (String)"outsideOfDoubleExactRange should not be exact-representable as a double", (Object[])new Object[0]);
            this.checkRepresentation(connectedStatement.getStatement(), String.format("DECIMAL '%s'", outsideOfDoubleExactRange), 3, (ResultSet rs, int column) -> {
                Assertions.assertThat((Object)rs.getObject(column)).isEqualTo((Object)new BigDecimal("9223372036854775774"));
                Assertions.assertThat((BigDecimal)rs.getBigDecimal(column)).isEqualTo((Object)new BigDecimal("9223372036854775774"));
                Assertions.assertThat((long)rs.getLong(column)).isEqualTo(9223372036854775774L);
                Assertions.assertThat((double)rs.getDouble(column)).isEqualTo(9.223372036854776E18);
                Assertions.assertThat((String)rs.getString(column)).isEqualTo("9223372036854775774");
            });
            this.checkRepresentation(connectedStatement.getStatement(), "VARCHAR ''", 12, (ResultSet rs, int column) -> ((AbstractThrowableAssert)Assertions.assertThatThrownBy(() -> rs.getBigDecimal(column)).isInstanceOf(SQLException.class)).hasMessage("Value is not a number: "));
            this.checkRepresentation(connectedStatement.getStatement(), "VARCHAR '123a'", 12, (ResultSet rs, int column) -> ((AbstractThrowableAssert)Assertions.assertThatThrownBy(() -> rs.getBigDecimal(column)).isInstanceOf(SQLException.class)).hasMessage("Value is not a number: 123a"));
            this.checkRepresentation(connectedStatement.getStatement(), "VARCHAR '123e-1'", 12, (ResultSet rs, int column) -> Assertions.assertThat((BigDecimal)rs.getBigDecimal(column)).isEqualTo((Object)new BigDecimal("12.3")));
        }
    }

    @Test
    public void testVarbinary() throws Exception {
        try (ConnectedStatement connectedStatement = this.newStatement();){
            this.checkRepresentation(connectedStatement.getStatement(), "X'12345678'", -3, (ResultSet rs, int column) -> {
                byte[] bytes = new byte[]{18, 52, 86, 120};
                Assertions.assertThat((Object)rs.getObject(column)).isEqualTo((Object)bytes);
                Assertions.assertThat((byte[])rs.getObject(column, byte[].class)).isEqualTo((Object)bytes);
                Assertions.assertThat((byte[])rs.getBytes(column)).isEqualTo((Object)bytes);
                ((AbstractThrowableAssert)Assertions.assertThatThrownBy(() -> rs.getLong(column)).isInstanceOf(SQLException.class)).hasMessageStartingWith("Value is not a number: [B@");
                ((AbstractThrowableAssert)Assertions.assertThatThrownBy(() -> rs.getBigDecimal(column)).isInstanceOf(SQLException.class)).hasMessageStartingWith("Value is not a number: [B@");
                Assertions.assertThat((String)rs.getString(column)).isEqualTo("0x12345678");
                Assertions.assertThat((InputStream)rs.getBinaryStream(column)).hasBinaryContent(bytes);
                ((AbstractThrowableAssert)Assertions.assertThatThrownBy(() -> rs.getAsciiStream(column)).isInstanceOf(SQLException.class)).hasMessageStartingWith("Value is not a string: [B@");
            });
            this.checkRepresentation(connectedStatement.getStatement(), "CAST(NULL AS VARBINARY)", -3, (ResultSet rs, int column) -> {
                Assertions.assertThat((InputStream)rs.getAsciiStream(column)).isNull();
                Assertions.assertThat((InputStream)rs.getBinaryStream(column)).isNull();
            });
            this.checkRepresentation(connectedStatement.getStatement(), "X''", -3, (ResultSet rs, int column) -> {
                Assertions.assertThat((InputStream)rs.getBinaryStream(column)).isEmpty();
                ((AbstractThrowableAssert)Assertions.assertThatThrownBy(() -> rs.getAsciiStream(column)).isInstanceOf(SQLException.class)).hasMessageStartingWith("Value is not a string: [B@");
            });
        }
    }

    @Test
    public void testDate() throws Exception {
        try (ConnectedStatement connectedStatement = this.newStatement();){
            this.checkRepresentation(connectedStatement.getStatement(), "DATE '2018-02-13'", 91, (ResultSet rs, int column) -> {
                LocalDate localDate = LocalDate.of(2018, 2, 13);
                Date sqlDate = Date.valueOf(localDate);
                Assertions.assertThat((Object)rs.getObject(column)).isEqualTo((Object)sqlDate);
                Assertions.assertThat((java.util.Date)rs.getObject(column, Date.class)).isEqualTo((Object)sqlDate);
                Assertions.assertThat((LocalDate)rs.getObject(column, LocalDate.class)).isEqualTo((Object)localDate);
                Assertions.assertThat((java.util.Date)rs.getDate(column)).isEqualTo((Object)sqlDate);
                ((AbstractThrowableAssert)Assertions.assertThatThrownBy(() -> rs.getTime(column)).isInstanceOf(IllegalArgumentException.class)).hasMessage("Expected column to be a time type but is date");
                ((AbstractThrowableAssert)Assertions.assertThatThrownBy(() -> rs.getTimestamp(column)).isInstanceOf(IllegalArgumentException.class)).hasMessage("Expected column to be a timestamp type but is date");
                Assertions.assertThat((String)rs.getString(column)).isEqualTo(localDate.toString());
            });
            this.checkRepresentation(connectedStatement.getStatement(), "DATE '0001-01-01'", 91, (ResultSet rs, int column) -> {
                LocalDate localDate = LocalDate.of(1, 1, 1);
                Date sqlDate = Date.valueOf(localDate);
                Assertions.assertThat((Object)rs.getObject(column)).isEqualTo((Object)sqlDate);
                Assertions.assertThat((java.util.Date)rs.getObject(column, Date.class)).isEqualTo((Object)sqlDate);
                Assertions.assertThat((LocalDate)rs.getObject(column, LocalDate.class)).isEqualTo((Object)localDate);
                Assertions.assertThat((java.util.Date)rs.getDate(column)).isEqualTo((Object)sqlDate);
                ((AbstractThrowableAssert)Assertions.assertThatThrownBy(() -> rs.getTime(column)).isInstanceOf(IllegalArgumentException.class)).hasMessage("Expected column to be a time type but is date");
                ((AbstractThrowableAssert)Assertions.assertThatThrownBy(() -> rs.getTimestamp(column)).isInstanceOf(IllegalArgumentException.class)).hasMessage("Expected column to be a timestamp type but is date");
                Assertions.assertThat((String)rs.getString(column)).isEqualTo(localDate.toString());
            });
            this.checkRepresentation(connectedStatement.getStatement(), "DATE '1970-01-01'", 91, (ResultSet rs, int column) -> {
                LocalDate localDate = LocalDate.of(1970, 1, 1);
                Date sqlDate = Date.valueOf(localDate);
                Assertions.assertThat((Object)rs.getObject(column)).isEqualTo((Object)sqlDate);
                Assertions.assertThat((java.util.Date)rs.getObject(column, Date.class)).isEqualTo((Object)sqlDate);
                Assertions.assertThat((LocalDate)rs.getObject(column, LocalDate.class)).isEqualTo((Object)localDate);
                Assertions.assertThat((java.util.Date)rs.getDate(column)).isEqualTo((Object)sqlDate);
                ((AbstractThrowableAssert)Assertions.assertThatThrownBy(() -> rs.getTime(column)).isInstanceOf(IllegalArgumentException.class)).hasMessage("Expected column to be a time type but is date");
                ((AbstractThrowableAssert)Assertions.assertThatThrownBy(() -> rs.getTimestamp(column)).isInstanceOf(IllegalArgumentException.class)).hasMessage("Expected column to be a timestamp type but is date");
                Assertions.assertThat((String)rs.getString(column)).isEqualTo(localDate.toString());
            });
            this.checkRepresentation(connectedStatement.getStatement(), "DATE '1582-10-04'", 91, (ResultSet rs, int column) -> {
                LocalDate localDate = LocalDate.of(1582, 10, 4);
                Date sqlDate = Date.valueOf(localDate);
                Assertions.assertThat((Object)rs.getObject(column)).isEqualTo((Object)sqlDate);
                Assertions.assertThat((java.util.Date)rs.getObject(column, Date.class)).isEqualTo((Object)sqlDate);
                Assertions.assertThat((LocalDate)rs.getObject(column, LocalDate.class)).isEqualTo((Object)localDate);
                Assertions.assertThat((java.util.Date)rs.getDate(column)).isEqualTo((Object)sqlDate);
                ((AbstractThrowableAssert)Assertions.assertThatThrownBy(() -> rs.getTime(column)).isInstanceOf(IllegalArgumentException.class)).hasMessage("Expected column to be a time type but is date");
                ((AbstractThrowableAssert)Assertions.assertThatThrownBy(() -> rs.getTimestamp(column)).isInstanceOf(IllegalArgumentException.class)).hasMessage("Expected column to be a timestamp type but is date");
                Assertions.assertThat((String)rs.getString(column)).isEqualTo(localDate.toString());
            });
            this.checkRepresentation(connectedStatement.getStatement(), "DATE '1582-10-10'", 91, (ResultSet rs, int column) -> {
                LocalDate localDate = LocalDate.of(1582, 10, 10);
                Date sqlDate = Date.valueOf(localDate);
                Assertions.assertThat((Object)rs.getObject(column)).isEqualTo((Object)sqlDate);
                Assertions.assertThat((java.util.Date)rs.getObject(column, Date.class)).isEqualTo((Object)sqlDate);
                Assertions.assertThat((LocalDate)rs.getObject(column, LocalDate.class)).isEqualTo((Object)LocalDate.of(1582, 10, 20));
                Assertions.assertThat((java.util.Date)rs.getDate(column)).isEqualTo((Object)sqlDate);
                ((AbstractThrowableAssert)Assertions.assertThatThrownBy(() -> rs.getTime(column)).isInstanceOf(IllegalArgumentException.class)).hasMessage("Expected column to be a time type but is date");
                ((AbstractThrowableAssert)Assertions.assertThatThrownBy(() -> rs.getTimestamp(column)).isInstanceOf(IllegalArgumentException.class)).hasMessage("Expected column to be a timestamp type but is date");
                Assertions.assertThat((String)rs.getString(column)).isEqualTo(localDate.toString());
            });
            this.checkRepresentation(connectedStatement.getStatement(), "DATE '1582-10-15'", 91, (ResultSet rs, int column) -> {
                LocalDate localDate = LocalDate.of(1582, 10, 15);
                Date sqlDate = Date.valueOf(localDate);
                Assertions.assertThat((Object)rs.getObject(column)).isEqualTo((Object)sqlDate);
                Assertions.assertThat((java.util.Date)rs.getObject(column, Date.class)).isEqualTo((Object)sqlDate);
                Assertions.assertThat((LocalDate)rs.getObject(column, LocalDate.class)).isEqualTo((Object)localDate);
                Assertions.assertThat((java.util.Date)rs.getDate(column)).isEqualTo((Object)sqlDate);
                ((AbstractThrowableAssert)Assertions.assertThatThrownBy(() -> rs.getTime(column)).isInstanceOf(IllegalArgumentException.class)).hasMessage("Expected column to be a time type but is date");
                ((AbstractThrowableAssert)Assertions.assertThatThrownBy(() -> rs.getTimestamp(column)).isInstanceOf(IllegalArgumentException.class)).hasMessage("Expected column to be a timestamp type but is date");
                Assertions.assertThat((String)rs.getString(column)).isEqualTo(localDate.toString());
            });
        }
    }

    @Test
    public void testTime() throws Exception {
        try (ConnectedStatement connectedStatement = this.newStatement();){
            this.checkRepresentation(connectedStatement.getStatement(), "TIME '09:39:05.000'", 92, (ResultSet rs, int column) -> {
                Assertions.assertThat((Object)rs.getObject(column)).isEqualTo((Object)BaseTestJdbcResultSet.toSqlTime(LocalTime.of(9, 39, 5)));
                Assertions.assertThat((java.util.Date)rs.getObject(column, Time.class)).isEqualTo((Object)BaseTestJdbcResultSet.toSqlTime(LocalTime.of(9, 39, 5)));
                ((AbstractThrowableAssert)Assertions.assertThatThrownBy(() -> rs.getDate(column)).isInstanceOf(SQLException.class)).hasMessage("Expected value to be a date but is: 09:39:05.000");
                Assertions.assertThat((java.util.Date)rs.getTime(column)).isEqualTo((Object)Time.valueOf(LocalTime.of(9, 39, 5)));
                ((AbstractThrowableAssert)Assertions.assertThatThrownBy(() -> rs.getTimestamp(column)).isInstanceOf(IllegalArgumentException.class)).hasMessage("Expected column to be a timestamp type but is time(3)");
            });
            this.checkRepresentation(connectedStatement.getStatement(), "TIME '00:39:05'", 92, (ResultSet rs, int column) -> {
                Assertions.assertThat((Object)rs.getObject(column)).isEqualTo((Object)BaseTestJdbcResultSet.toSqlTime(LocalTime.of(0, 39, 5)));
                Assertions.assertThat((java.util.Date)rs.getObject(column, Time.class)).isEqualTo((Object)BaseTestJdbcResultSet.toSqlTime(LocalTime.of(0, 39, 5)));
                ((AbstractThrowableAssert)Assertions.assertThatThrownBy(() -> rs.getDate(column)).isInstanceOf(SQLException.class)).hasMessage("Expected value to be a date but is: 00:39:05");
                Assertions.assertThat((java.util.Date)rs.getTime(column)).isEqualTo((Object)Time.valueOf(LocalTime.of(0, 39, 5)));
                ((AbstractThrowableAssert)Assertions.assertThatThrownBy(() -> rs.getTimestamp(column)).isInstanceOf(IllegalArgumentException.class)).hasMessage("Expected column to be a timestamp type but is time(0)");
            });
            this.checkRepresentation(connectedStatement.getStatement(), "TIME '10:11:12.1235'", 92, (ResultSet rs, int column) -> {
                Assertions.assertThat((Object)rs.getObject(column)).isEqualTo((Object)BaseTestJdbcResultSet.toSqlTime(LocalTime.of(10, 11, 12, 123000000)));
                ((AbstractThrowableAssert)Assertions.assertThatThrownBy(() -> rs.getDate(column)).isInstanceOf(SQLException.class)).hasMessage("Expected value to be a date but is: 10:11:12.1235");
                Assertions.assertThat((java.util.Date)rs.getTime(column)).isEqualTo((Object)BaseTestJdbcResultSet.toSqlTime(LocalTime.of(10, 11, 12, 123000000)));
                ((AbstractThrowableAssert)Assertions.assertThatThrownBy(() -> rs.getTimestamp(column)).isInstanceOf(IllegalArgumentException.class)).hasMessage("Expected column to be a timestamp type but is time(4)");
            });
            this.checkRepresentation(connectedStatement.getStatement(), "TIME '10:59:59.999999999999'", 92, (ResultSet rs, int column) -> {
                Assertions.assertThat((Object)rs.getObject(column)).isEqualTo((Object)BaseTestJdbcResultSet.toSqlTime(LocalTime.of(10, 59, 59, 999000000)));
                ((AbstractThrowableAssert)Assertions.assertThatThrownBy(() -> rs.getDate(column)).isInstanceOf(SQLException.class)).hasMessage("Expected value to be a date but is: 10:59:59.999999999999");
                Assertions.assertThat((java.util.Date)rs.getTime(column)).isEqualTo((Object)BaseTestJdbcResultSet.toSqlTime(LocalTime.of(10, 59, 59, 999000000)));
                ((AbstractThrowableAssert)Assertions.assertThatThrownBy(() -> rs.getTimestamp(column)).isInstanceOf(IllegalArgumentException.class)).hasMessage("Expected column to be a timestamp type but is time(12)");
            });
            this.checkRepresentation(connectedStatement.getStatement(), "TIME '23:59:59.999999999999'", 92, (ResultSet rs, int column) -> {
                Assertions.assertThat((Object)rs.getObject(column)).isEqualTo((Object)BaseTestJdbcResultSet.toSqlTime(LocalTime.of(23, 59, 59, 999000000)));
                ((AbstractThrowableAssert)Assertions.assertThatThrownBy(() -> rs.getDate(column)).isInstanceOf(SQLException.class)).hasMessage("Expected value to be a date but is: 23:59:59.999999999999");
                Assertions.assertThat((java.util.Date)rs.getTime(column)).isEqualTo((Object)BaseTestJdbcResultSet.toSqlTime(LocalTime.of(23, 59, 59, 999000000)));
                ((AbstractThrowableAssert)Assertions.assertThatThrownBy(() -> rs.getTimestamp(column)).isInstanceOf(IllegalArgumentException.class)).hasMessage("Expected column to be a timestamp type but is time(12)");
            });
        }
    }

    @Test
    public void testTimeWithTimeZone() throws Exception {
        try (ConnectedStatement connectedStatement = this.newStatement();){
            this.checkRepresentation(connectedStatement.getStatement(), "TIME '09:39:07 +01:00'", 2013, (ResultSet rs, int column) -> {
                Assertions.assertThat((Object)rs.getObject(column)).isEqualTo((Object)Time.valueOf(LocalTime.of(1, 39, 7)));
                ((AbstractThrowableAssert)Assertions.assertThatThrownBy(() -> rs.getDate(column)).isInstanceOf(SQLException.class)).hasMessage("Expected value to be a date but is: 09:39:07+01:00");
                Assertions.assertThat((java.util.Date)rs.getTime(column)).isEqualTo((Object)Time.valueOf(LocalTime.of(1, 39, 7)));
                ((AbstractThrowableAssert)Assertions.assertThatThrownBy(() -> rs.getTimestamp(column)).isInstanceOf(IllegalArgumentException.class)).hasMessage("Expected column to be a timestamp type but is time(0) with time zone");
            });
            this.checkRepresentation(connectedStatement.getStatement(), "TIME '01:39:07 +01:00'", 2013, (ResultSet rs, int column) -> {
                Time someBogusValue = new Time(Time.valueOf(LocalTime.of(16, 39, 7)).getTime() - TimeUnit.DAYS.toMillis(1L) + TimeUnit.HOURS.toMillis(1L));
                Assertions.assertThat((Object)rs.getObject(column)).isEqualTo((Object)someBogusValue);
                ((AbstractThrowableAssert)Assertions.assertThatThrownBy(() -> rs.getDate(column)).isInstanceOf(SQLException.class)).hasMessage("Expected value to be a date but is: 01:39:07+01:00");
                Assertions.assertThat((java.util.Date)rs.getTime(column)).isEqualTo((Object)someBogusValue);
                ((AbstractThrowableAssert)Assertions.assertThatThrownBy(() -> rs.getTimestamp(column)).isInstanceOf(IllegalArgumentException.class)).hasMessage("Expected column to be a timestamp type but is time(0) with time zone");
            });
            this.checkRepresentation(connectedStatement.getStatement(), "TIME '00:39:07 +01:00'", 2013, (ResultSet rs, int column) -> {
                Time someBogusValue = new Time(Time.valueOf(LocalTime.of(15, 39, 7)).getTime() - TimeUnit.DAYS.toMillis(1L) + TimeUnit.HOURS.toMillis(1L));
                Assertions.assertThat((Object)rs.getObject(column)).isEqualTo((Object)someBogusValue);
                ((AbstractThrowableAssert)Assertions.assertThatThrownBy(() -> rs.getDate(column)).isInstanceOf(SQLException.class)).hasMessage("Expected value to be a date but is: 00:39:07+01:00");
                Assertions.assertThat((java.util.Date)rs.getTime(column)).isEqualTo((Object)someBogusValue);
                ((AbstractThrowableAssert)Assertions.assertThatThrownBy(() -> rs.getTimestamp(column)).isInstanceOf(IllegalArgumentException.class)).hasMessage("Expected column to be a timestamp type but is time(0) with time zone");
            });
        }
    }

    @Test
    public void testTimestamp() throws Exception {
        try (ConnectedStatement connectedStatement = this.newStatement();){
            this.checkRepresentation(connectedStatement.getStatement(), "TIMESTAMP '2018-02-13 13:14:15.123'", 93, (ResultSet rs, int column) -> {
                Assertions.assertThat((Object)rs.getObject(column)).isEqualTo((Object)Timestamp.valueOf(LocalDateTime.of(2018, 2, 13, 13, 14, 15, 123000000)));
                Assertions.assertThat((java.util.Date)rs.getObject(column, Timestamp.class)).isEqualTo((Object)Timestamp.valueOf(LocalDateTime.of(2018, 2, 13, 13, 14, 15, 123000000)));
                ((AbstractThrowableAssert)Assertions.assertThatThrownBy(() -> rs.getDate(column)).isInstanceOf(SQLException.class)).hasMessage("Expected value to be a date but is: 2018-02-13 13:14:15.123");
                ((AbstractThrowableAssert)Assertions.assertThatThrownBy(() -> rs.getTime(column)).isInstanceOf(IllegalArgumentException.class)).hasMessage("Expected column to be a time type but is timestamp(3)");
                Assertions.assertThat((java.util.Date)rs.getTimestamp(column)).isEqualTo((Object)Timestamp.valueOf(LocalDateTime.of(2018, 2, 13, 13, 14, 15, 123000000)));
            });
            this.checkRepresentation(connectedStatement.getStatement(), "TIMESTAMP '2018-02-13 13:14:15.111111111111'", 93, (ResultSet rs, int column) -> {
                Assertions.assertThat((Object)rs.getObject(column)).isEqualTo((Object)Timestamp.valueOf(LocalDateTime.of(2018, 2, 13, 13, 14, 15, 111111111)));
                ((AbstractThrowableAssert)Assertions.assertThatThrownBy(() -> rs.getDate(column)).isInstanceOf(SQLException.class)).hasMessage("Expected value to be a date but is: 2018-02-13 13:14:15.111111111111");
                ((AbstractThrowableAssert)Assertions.assertThatThrownBy(() -> rs.getTime(column)).isInstanceOf(IllegalArgumentException.class)).hasMessage("Expected column to be a time type but is timestamp(12)");
                Assertions.assertThat((java.util.Date)rs.getTimestamp(column)).isEqualTo((Object)Timestamp.valueOf(LocalDateTime.of(2018, 2, 13, 13, 14, 15, 111111111)));
            });
            this.checkRepresentation(connectedStatement.getStatement(), "TIMESTAMP '2018-02-13 13:14:15.555555555555'", 93, (ResultSet rs, int column) -> {
                Assertions.assertThat((Object)rs.getObject(column)).isEqualTo((Object)Timestamp.valueOf(LocalDateTime.of(2018, 2, 13, 13, 14, 15, 555555556)));
                ((AbstractThrowableAssert)Assertions.assertThatThrownBy(() -> rs.getDate(column)).isInstanceOf(SQLException.class)).hasMessage("Expected value to be a date but is: 2018-02-13 13:14:15.555555555555");
                ((AbstractThrowableAssert)Assertions.assertThatThrownBy(() -> rs.getTime(column)).isInstanceOf(IllegalArgumentException.class)).hasMessage("Expected column to be a time type but is timestamp(12)");
                Assertions.assertThat((java.util.Date)rs.getTimestamp(column)).isEqualTo((Object)Timestamp.valueOf(LocalDateTime.of(2018, 2, 13, 13, 14, 15, 555555556)));
            });
            this.checkRepresentation(connectedStatement.getStatement(), "TIMESTAMP '2019-12-31 23:59:59.999999999999'", 93, (ResultSet rs, int column) -> {
                Assertions.assertThat((Object)rs.getObject(column)).isEqualTo((Object)Timestamp.valueOf(LocalDateTime.of(2020, 1, 1, 0, 0, 0, 0)));
                ((AbstractThrowableAssert)Assertions.assertThatThrownBy(() -> rs.getDate(column)).isInstanceOf(SQLException.class)).hasMessage("Expected value to be a date but is: 2019-12-31 23:59:59.999999999999");
                ((AbstractThrowableAssert)Assertions.assertThatThrownBy(() -> rs.getTime(column)).isInstanceOf(IllegalArgumentException.class)).hasMessage("Expected column to be a time type but is timestamp(12)");
                Assertions.assertThat((java.util.Date)rs.getTimestamp(column)).isEqualTo((Object)Timestamp.valueOf(LocalDateTime.of(2020, 1, 1, 0, 0, 0, 0)));
            });
            this.checkRepresentation(connectedStatement.getStatement(), "TIMESTAMP '1957-12-31 23:59:59.999999999999'", 93, (ResultSet rs, int column) -> {
                Assertions.assertThat((Object)rs.getObject(column)).isEqualTo((Object)Timestamp.valueOf(LocalDateTime.of(1958, 1, 1, 0, 0, 0, 0)));
                ((AbstractThrowableAssert)Assertions.assertThatThrownBy(() -> rs.getDate(column)).isInstanceOf(SQLException.class)).hasMessage("Expected value to be a date but is: 1957-12-31 23:59:59.999999999999");
                ((AbstractThrowableAssert)Assertions.assertThatThrownBy(() -> rs.getTime(column)).isInstanceOf(IllegalArgumentException.class)).hasMessage("Expected column to be a time type but is timestamp(12)");
                Assertions.assertThat((java.util.Date)rs.getTimestamp(column)).isEqualTo((Object)Timestamp.valueOf(LocalDateTime.of(1958, 1, 1, 0, 0, 0, 0)));
            });
            this.checkRepresentation(connectedStatement.getStatement(), "TIMESTAMP '0001-01-01 00:00:00'", 93, (ResultSet rs, int column) -> {
                Assertions.assertThat((Object)rs.getObject(column)).isEqualTo((Object)Timestamp.valueOf(LocalDateTime.of(1, 1, 1, 0, 0, 0)));
                ((AbstractThrowableAssert)Assertions.assertThatThrownBy(() -> rs.getDate(column)).isInstanceOf(SQLException.class)).hasMessage("Expected value to be a date but is: 0001-01-01 00:00:00");
                ((AbstractThrowableAssert)Assertions.assertThatThrownBy(() -> rs.getTime(column)).isInstanceOf(IllegalArgumentException.class)).hasMessage("Expected column to be a time type but is timestamp(0)");
                Assertions.assertThat((java.util.Date)rs.getTimestamp(column)).isEqualTo((Object)Timestamp.valueOf(LocalDateTime.of(1, 1, 1, 0, 0, 0)));
            });
            this.checkRepresentation(connectedStatement.getStatement(), "TIMESTAMP '1582-10-04 00:00:00'", 93, (ResultSet rs, int column) -> {
                Assertions.assertThat((Object)rs.getObject(column)).isEqualTo((Object)Timestamp.valueOf(LocalDateTime.of(1582, 10, 4, 0, 0, 0)));
                ((AbstractThrowableAssert)Assertions.assertThatThrownBy(() -> rs.getDate(column)).isInstanceOf(SQLException.class)).hasMessage("Expected value to be a date but is: 1582-10-04 00:00:00");
                ((AbstractThrowableAssert)Assertions.assertThatThrownBy(() -> rs.getTime(column)).isInstanceOf(IllegalArgumentException.class)).hasMessage("Expected column to be a time type but is timestamp(0)");
                Assertions.assertThat((java.util.Date)rs.getTimestamp(column)).isEqualTo((Object)Timestamp.valueOf(LocalDateTime.of(1582, 10, 4, 0, 0, 0)));
            });
            this.checkRepresentation(connectedStatement.getStatement(), "TIMESTAMP '1582-10-10 00:00:00'", 93, (ResultSet rs, int column) -> {
                Assertions.assertThat((Object)rs.getObject(column)).isEqualTo((Object)Timestamp.valueOf(LocalDateTime.of(1582, 10, 10, 0, 0, 0)));
                ((AbstractThrowableAssert)Assertions.assertThatThrownBy(() -> rs.getDate(column)).isInstanceOf(SQLException.class)).hasMessage("Expected value to be a date but is: 1582-10-10 00:00:00");
                ((AbstractThrowableAssert)Assertions.assertThatThrownBy(() -> rs.getTime(column)).isInstanceOf(IllegalArgumentException.class)).hasMessage("Expected column to be a time type but is timestamp(0)");
                Assertions.assertThat((java.util.Date)rs.getTimestamp(column)).isEqualTo((Object)Timestamp.valueOf(LocalDateTime.of(1582, 10, 10, 0, 0, 0)));
            });
            this.checkRepresentation(connectedStatement.getStatement(), "TIMESTAMP '1582-10-15 00:00:00'", 93, (ResultSet rs, int column) -> {
                Assertions.assertThat((Object)rs.getObject(column)).isEqualTo((Object)Timestamp.valueOf(LocalDateTime.of(1582, 10, 15, 0, 0, 0)));
                ((AbstractThrowableAssert)Assertions.assertThatThrownBy(() -> rs.getDate(column)).isInstanceOf(SQLException.class)).hasMessage("Expected value to be a date but is: 1582-10-15 00:00:00");
                ((AbstractThrowableAssert)Assertions.assertThatThrownBy(() -> rs.getTime(column)).isInstanceOf(IllegalArgumentException.class)).hasMessage("Expected column to be a time type but is timestamp(0)");
                Assertions.assertThat((java.util.Date)rs.getTimestamp(column)).isEqualTo((Object)Timestamp.valueOf(LocalDateTime.of(1582, 10, 15, 0, 0, 0)));
            });
            this.checkRepresentation(connectedStatement.getStatement(), "TIMESTAMP '1583-01-01 00:00:00'", 93, (ResultSet rs, int column) -> {
                Assertions.assertThat((Object)rs.getObject(column)).isEqualTo((Object)Timestamp.valueOf(LocalDateTime.of(1583, 1, 1, 0, 0, 0)));
                ((AbstractThrowableAssert)Assertions.assertThatThrownBy(() -> rs.getDate(column)).isInstanceOf(SQLException.class)).hasMessage("Expected value to be a date but is: 1583-01-01 00:00:00");
                ((AbstractThrowableAssert)Assertions.assertThatThrownBy(() -> rs.getTime(column)).isInstanceOf(IllegalArgumentException.class)).hasMessage("Expected column to be a time type but is timestamp(0)");
                Assertions.assertThat((java.util.Date)rs.getTimestamp(column)).isEqualTo((Object)Timestamp.valueOf(LocalDateTime.of(1583, 1, 1, 0, 0, 0)));
            });
            this.checkRepresentation(connectedStatement.getStatement(), "TIMESTAMP '1970-01-01 00:14:15.123'", 93, (ResultSet rs, int column) -> {
                Assertions.assertThat((Object)rs.getObject(column)).isEqualTo((Object)Timestamp.valueOf(LocalDateTime.of(1970, 1, 1, 0, 14, 15, 123000000)));
                ((AbstractThrowableAssert)Assertions.assertThatThrownBy(() -> rs.getDate(column)).isInstanceOf(SQLException.class)).hasMessage("Expected value to be a date but is: 1970-01-01 00:14:15.123");
                ((AbstractThrowableAssert)Assertions.assertThatThrownBy(() -> rs.getTime(column)).isInstanceOf(IllegalArgumentException.class)).hasMessage("Expected column to be a time type but is timestamp(3)");
                Assertions.assertThat((java.util.Date)rs.getTimestamp(column)).isEqualTo((Object)Timestamp.valueOf(LocalDateTime.of(1970, 1, 1, 0, 14, 15, 123000000)));
            });
            this.checkRepresentation(connectedStatement.getStatement(), "TIMESTAMP '123456-01-23 01:23:45.123456789'", 93, (ResultSet rs, int column) -> {
                Assertions.assertThat((Object)rs.getObject(column)).isEqualTo((Object)Timestamp.valueOf(LocalDateTime.of(123456, 1, 23, 1, 23, 45, 123456789)));
                ((AbstractThrowableAssert)Assertions.assertThatThrownBy(() -> rs.getDate(column)).isInstanceOf(SQLException.class)).hasMessage("Expected value to be a date but is: +123456-01-23 01:23:45.123456789");
                ((AbstractThrowableAssert)Assertions.assertThatThrownBy(() -> rs.getTime(column)).isInstanceOf(IllegalArgumentException.class)).hasMessage("Expected column to be a time type but is timestamp(9)");
                Assertions.assertThat((java.util.Date)rs.getTimestamp(column)).isEqualTo((Object)Timestamp.valueOf(LocalDateTime.of(123456, 1, 23, 1, 23, 45, 123456789)));
            });
        }
    }

    @Test
    public void testTimestampWithTimeZone() throws Exception {
        try (ConnectedStatement connectedStatement = this.newStatement();){
            this.checkRepresentation(connectedStatement.getStatement(), "TIMESTAMP '1970-01-01 00:00:00.000 +00:00'", 2014, (ResultSet rs, int column) -> {
                Timestamp timestampForPointInTime = Timestamp.from(Instant.EPOCH);
                Assertions.assertThat((Object)rs.getObject(column)).isEqualTo((Object)timestampForPointInTime);
                Assertions.assertThat((ZonedDateTime)rs.getObject(column, ZonedDateTime.class)).isEqualTo((Object)ZonedDateTime.ofInstant(Instant.EPOCH, ZoneId.of("UTC")));
                ((AbstractThrowableAssert)Assertions.assertThatThrownBy(() -> rs.getDate(column)).isInstanceOf(SQLException.class)).hasMessage("Expected value to be a date but is: 1970-01-01 00:00:00.000 UTC");
                ((AbstractThrowableAssert)Assertions.assertThatThrownBy(() -> rs.getTime(column)).isInstanceOf(IllegalArgumentException.class)).hasMessage("Expected column to be a time type but is timestamp(3) with time zone");
                Assertions.assertThat((java.util.Date)rs.getTimestamp(column)).isEqualTo((Object)timestampForPointInTime);
            });
            this.checkRepresentation(connectedStatement.getStatement(), "TIMESTAMP '2018-02-13 13:14:15.227 Europe/Warsaw'", 2014, (ResultSet rs, int column) -> {
                ZonedDateTime zonedDateTime = ZonedDateTime.of(2018, 2, 13, 13, 14, 15, 227000000, ZoneId.of("Europe/Warsaw"));
                Timestamp timestampForPointInTime = Timestamp.from(zonedDateTime.toInstant());
                Assertions.assertThat((Object)rs.getObject(column)).isEqualTo((Object)timestampForPointInTime);
                Assertions.assertThat((ZonedDateTime)rs.getObject(column, ZonedDateTime.class)).isEqualTo((Object)zonedDateTime);
                ((AbstractThrowableAssert)Assertions.assertThatThrownBy(() -> rs.getDate(column)).isInstanceOf(SQLException.class)).hasMessage("Expected value to be a date but is: 2018-02-13 13:14:15.227 Europe/Warsaw");
                ((AbstractThrowableAssert)Assertions.assertThatThrownBy(() -> rs.getTime(column)).isInstanceOf(IllegalArgumentException.class)).hasMessage("Expected column to be a time type but is timestamp(3) with time zone");
                Assertions.assertThat((java.util.Date)rs.getTimestamp(column)).isEqualTo((Object)timestampForPointInTime);
            });
            this.checkRepresentation(connectedStatement.getStatement(), "TIMESTAMP '2019-12-31 23:59:59.999999999999 Europe/Warsaw'", 2014, (ResultSet rs, int column) -> {
                ZonedDateTime zonedDateTime = ZonedDateTime.of(2020, 1, 1, 0, 0, 0, 0, ZoneId.of("Europe/Warsaw"));
                Timestamp timestampForPointInTime = Timestamp.from(zonedDateTime.toInstant());
                Assertions.assertThat((Object)rs.getObject(column)).isEqualTo((Object)timestampForPointInTime);
                Assertions.assertThat((ZonedDateTime)rs.getObject(column, ZonedDateTime.class)).isEqualTo((Object)zonedDateTime);
                ((AbstractThrowableAssert)Assertions.assertThatThrownBy(() -> rs.getDate(column)).isInstanceOf(SQLException.class)).hasMessage("Expected value to be a date but is: 2019-12-31 23:59:59.999999999999 Europe/Warsaw");
                ((AbstractThrowableAssert)Assertions.assertThatThrownBy(() -> rs.getTime(column)).isInstanceOf(IllegalArgumentException.class)).hasMessage("Expected column to be a time type but is timestamp(12) with time zone");
                Assertions.assertThat((java.util.Date)rs.getTimestamp(column)).isEqualTo((Object)timestampForPointInTime);
            });
            ZoneId jvmZone = ZoneId.systemDefault();
            this.checkRepresentation(connectedStatement.getStatement(), String.format("TIMESTAMP '2019-12-31 23:59:59.999999999999 %s'", jvmZone.getId()), 2014, (ResultSet rs, int column) -> {
                ZonedDateTime zonedDateTime = ZonedDateTime.of(2020, 1, 1, 0, 0, 0, 0, jvmZone);
                Timestamp timestampForPointInTime = Timestamp.from(zonedDateTime.toInstant());
                Assertions.assertThat((Object)rs.getObject(column)).isEqualTo((Object)timestampForPointInTime);
                Assertions.assertThat((ZonedDateTime)rs.getObject(column, ZonedDateTime.class)).isEqualTo((Object)zonedDateTime);
                ((AbstractThrowableAssert)Assertions.assertThatThrownBy(() -> rs.getDate(column)).isInstanceOf(SQLException.class)).hasMessage("Expected value to be a date but is: 2019-12-31 23:59:59.999999999999 America/Bahia_Banderas");
                ((AbstractThrowableAssert)Assertions.assertThatThrownBy(() -> rs.getTime(column)).isInstanceOf(IllegalArgumentException.class)).hasMessage("Expected column to be a time type but is timestamp(12) with time zone");
                Assertions.assertThat((java.util.Date)rs.getTimestamp(column)).isEqualTo((Object)timestampForPointInTime);
            });
            this.checkRepresentation(connectedStatement.getStatement(), "TIMESTAMP '1957-12-31 23:59:59.999999999999 Europe/Warsaw'", 2014, (ResultSet rs, int column) -> {
                ZonedDateTime zonedDateTime = ZonedDateTime.of(1958, 1, 1, 0, 0, 0, 0, ZoneId.of("Europe/Warsaw"));
                Timestamp timestampForPointInTime = Timestamp.from(zonedDateTime.toInstant());
                Assertions.assertThat((Object)rs.getObject(column)).isEqualTo((Object)timestampForPointInTime);
                ((AbstractThrowableAssert)Assertions.assertThatThrownBy(() -> rs.getDate(column)).isInstanceOf(SQLException.class)).hasMessage("Expected value to be a date but is: 1957-12-31 23:59:59.999999999999 Europe/Warsaw");
                ((AbstractThrowableAssert)Assertions.assertThatThrownBy(() -> rs.getTime(column)).isInstanceOf(IllegalArgumentException.class)).hasMessage("Expected column to be a time type but is timestamp(12) with time zone");
                Assertions.assertThat((java.util.Date)rs.getTimestamp(column)).isEqualTo((Object)timestampForPointInTime);
            });
            this.checkRepresentation(connectedStatement.getStatement(), "TIMESTAMP '1970-01-01 09:14:15.227 Europe/Warsaw'", 2014, (ResultSet rs, int column) -> {
                ZonedDateTime zonedDateTime = ZonedDateTime.of(1970, 1, 1, 9, 14, 15, 227000000, ZoneId.of("Europe/Warsaw"));
                Timestamp timestampForPointInTime = Timestamp.from(zonedDateTime.toInstant());
                Assertions.assertThat((Object)rs.getObject(column)).isEqualTo((Object)timestampForPointInTime);
                Assertions.assertThat((ZonedDateTime)rs.getObject(column, ZonedDateTime.class)).isEqualTo((Object)zonedDateTime);
                ((AbstractThrowableAssert)Assertions.assertThatThrownBy(() -> rs.getDate(column)).isInstanceOf(SQLException.class)).hasMessage("Expected value to be a date but is: 1970-01-01 09:14:15.227 Europe/Warsaw");
                ((AbstractThrowableAssert)Assertions.assertThatThrownBy(() -> rs.getTime(column)).isInstanceOf(IllegalArgumentException.class)).hasMessage("Expected column to be a time type but is timestamp(3) with time zone");
                Assertions.assertThat((java.util.Date)rs.getTimestamp(column)).isEqualTo((Object)timestampForPointInTime);
            });
            this.checkRepresentation(connectedStatement.getStatement(), "TIMESTAMP '1970-01-01 00:14:15.227 Europe/Warsaw'", 2014, (ResultSet rs, int column) -> {
                ZonedDateTime zonedDateTime = ZonedDateTime.of(1970, 1, 1, 0, 14, 15, 227000000, ZoneId.of("Europe/Warsaw"));
                Timestamp timestampForPointInTime = Timestamp.from(zonedDateTime.toInstant());
                Assertions.assertThat((Object)rs.getObject(column)).isEqualTo((Object)timestampForPointInTime);
                Assertions.assertThat((ZonedDateTime)rs.getObject(column, ZonedDateTime.class)).isEqualTo((Object)zonedDateTime);
                ((AbstractThrowableAssert)Assertions.assertThatThrownBy(() -> rs.getDate(column)).isInstanceOf(SQLException.class)).hasMessage("Expected value to be a date but is: 1970-01-01 00:14:15.227 Europe/Warsaw");
                ((AbstractThrowableAssert)Assertions.assertThatThrownBy(() -> rs.getTime(column)).isInstanceOf(IllegalArgumentException.class)).hasMessage("Expected column to be a time type but is timestamp(3) with time zone");
                Assertions.assertThat((java.util.Date)rs.getTimestamp(column)).isEqualTo((Object)timestampForPointInTime);
            });
            this.checkRepresentation(connectedStatement.getStatement(), "TIMESTAMP '12345-01-23 01:23:45.123456789 Europe/Warsaw'", 2014, (ResultSet rs, int column) -> {
                ZonedDateTime zonedDateTime = ZonedDateTime.of(12345, 1, 23, 1, 23, 45, 123456789, ZoneId.of("Europe/Warsaw"));
                Timestamp timestampForPointInTime = Timestamp.from(zonedDateTime.toInstant());
                Assertions.assertThat((Object)rs.getObject(column)).isEqualTo((Object)timestampForPointInTime);
                Assertions.assertThat((ZonedDateTime)rs.getObject(column, ZonedDateTime.class)).isEqualTo((Object)zonedDateTime);
                ((AbstractThrowableAssert)Assertions.assertThatThrownBy(() -> rs.getDate(column)).isInstanceOf(SQLException.class)).hasMessage("Expected value to be a date but is: +12345-01-23 01:23:45.123456789 Europe/Warsaw");
                ((AbstractThrowableAssert)Assertions.assertThatThrownBy(() -> rs.getTime(column)).isInstanceOf(IllegalArgumentException.class)).hasMessage("Expected column to be a time type but is timestamp(9) with time zone");
                Assertions.assertThat((java.util.Date)rs.getTimestamp(column)).isEqualTo((Object)timestampForPointInTime);
            });
        }
    }

    @Test
    public void testIpAddress() throws Exception {
        try (ConnectedStatement connectedStatement = this.newStatement();){
            this.checkRepresentation(connectedStatement.getStatement(), "IPADDRESS '1.2.3.4'", 2000, "1.2.3.4");
        }
    }

    @Test
    public void testUuid() throws Exception {
        try (ConnectedStatement connectedStatement = this.newStatement();){
            this.checkRepresentation(connectedStatement.getStatement(), "UUID '0397e63b-2b78-4b7b-9c87-e085fa225dd8'", 2000, "0397e63b-2b78-4b7b-9c87-e085fa225dd8");
        }
    }

    @Test
    public void testArray() throws Exception {
        try (ConnectedStatement connectedStatement = this.newStatement();){
            this.checkRepresentation(connectedStatement.getStatement(), "ARRAY[1, 2]", 2003, (ResultSet rs, int column) -> {
                Array array = rs.getArray(column);
                Assertions.assertThat((Object)array.getArray()).isEqualTo((Object)new Object[]{1, 2});
                Assertions.assertThat((int)array.getBaseType()).isEqualTo(4);
                Assertions.assertThat((String)array.getBaseTypeName()).isEqualTo("integer");
                array = (Array)rs.getObject(column);
                Assertions.assertThat((Object)array.getArray()).isEqualTo((Object)new Object[]{1, 2});
                Assertions.assertThat((int)array.getBaseType()).isEqualTo(4);
                Assertions.assertThat((String)array.getBaseTypeName()).isEqualTo("integer");
                Assertions.assertThat((List)rs.getObject(column, List.class)).isEqualTo((Object)ImmutableList.of((Object)1, (Object)2));
            });
            this.checkArrayRepresentation(connectedStatement.getStatement(), "1", 4, "integer");
            this.checkArrayRepresentation(connectedStatement.getStatement(), "BIGINT '1'", -5, "bigint");
            this.checkArrayRepresentation(connectedStatement.getStatement(), "REAL '42.123'", 7, "real");
            this.checkArrayRepresentation(connectedStatement.getStatement(), "DOUBLE '42.123'", 8, "double");
            this.checkArrayRepresentation(connectedStatement.getStatement(), "42.123", 3, "decimal(5,3)");
            this.checkArrayRepresentation(connectedStatement.getStatement(), "TIMESTAMP '2017-01-02 09:00:00.123'", 93, "timestamp(3)");
            this.checkArrayRepresentation(connectedStatement.getStatement(), "TIMESTAMP '2017-01-02 09:00:00.123456789'", 93, "timestamp(9)");
            this.checkArrayRepresentation(connectedStatement.getStatement(), "TIMESTAMP '2017-01-02 09:00:00.123 Europe/Warsaw'", 2014, "timestamp(3) with time zone");
            this.checkArrayRepresentation(connectedStatement.getStatement(), "TIMESTAMP '2017-01-02 09:00:00.123456789 Europe/Warsaw'", 2014, "timestamp(9) with time zone");
            this.checkRepresentation(connectedStatement.getStatement(), "ARRAY[NULL, ARRAY[NULL, BIGINT '1', 2]]", 2003, (ResultSet rs, int column) -> {
                Array array = rs.getArray(column);
                Assertions.assertThat((Object)array.getArray()).isEqualTo((Object)new Object[]{null, Arrays.asList(null, 1L, 2L)});
                Assertions.assertThat((int)array.getBaseType()).isEqualTo(2003);
                Assertions.assertThat((String)array.getBaseTypeName()).isEqualTo("array(bigint)");
                array = (Array)rs.getObject(column);
                Assertions.assertThat((Object)array.getArray()).isEqualTo((Object)new Object[]{null, Arrays.asList(null, 1L, 2L)});
                Assertions.assertThat((int)array.getBaseType()).isEqualTo(2003);
                Assertions.assertThat((String)array.getBaseTypeName()).isEqualTo("array(bigint)");
                Assertions.assertThat((List)rs.getObject(column, List.class)).isEqualTo(Arrays.asList(null, Arrays.asList(null, 1L, 2L)));
            });
            this.checkRepresentation(connectedStatement.getStatement(), "ARRAY[map(ARRAY['k1', 'k2'], ARRAY[42, NULL])]", 2003, (ResultSet rs, int column) -> {
                HashMap<String, Integer> element = new HashMap<String, Integer>();
                element.put("k1", 42);
                element.put("k2", null);
                Array array = rs.getArray(column);
                Assertions.assertThat((Object)array.getArray()).isEqualTo((Object)new Object[]{element});
                Assertions.assertThat((int)array.getBaseType()).isEqualTo(2000);
                Assertions.assertThat((String)array.getBaseTypeName()).isEqualTo("map(varchar(2),integer)");
                array = (Array)rs.getObject(column);
                Assertions.assertThat((Object)array.getArray()).isEqualTo((Object)new Object[]{element});
                Assertions.assertThat((int)array.getBaseType()).isEqualTo(2000);
                Assertions.assertThat((String)array.getBaseTypeName()).isEqualTo("map(varchar(2),integer)");
                Assertions.assertThat((List)rs.getObject(column, List.class)).isEqualTo((Object)ImmutableList.of(element));
            });
            this.checkRepresentation(connectedStatement.getStatement(), "ARRAY[CAST(ROW(42, 'Trino') AS row(a_bigint bigint, a_varchar varchar(17)))]", 2003, (ResultSet rs, int column) -> {
                Row element = Row.builder().addField("a_bigint", (Object)42L).addField("a_varchar", (Object)"Trino").build();
                Array array = rs.getArray(column);
                Assertions.assertThat((Object)array.getArray()).isEqualTo((Object)new Object[]{element});
                Assertions.assertThat((int)array.getBaseType()).isEqualTo(2000);
                Assertions.assertThat((String)array.getBaseTypeName()).isEqualTo("row(a_bigint bigint,a_varchar varchar(17))");
                array = (Array)rs.getObject(column);
                Assertions.assertThat((Object)array.getArray()).isEqualTo((Object)new Object[]{element});
                Assertions.assertThat((int)array.getBaseType()).isEqualTo(2000);
                Assertions.assertThat((String)array.getBaseTypeName()).isEqualTo("row(a_bigint bigint,a_varchar varchar(17))");
                Assertions.assertThat((List)rs.getObject(column, List.class)).isEqualTo((Object)ImmutableList.of((Object)element));
            });
        }
    }

    private void checkArrayRepresentation(Statement statement, String elementExpression, int elementSqlType, String elementTypeName) throws Exception {
        Object element = this.getObjectRepresentation(statement.getConnection(), elementExpression);
        this.checkRepresentation(statement, String.format("ARRAY[NULL, %s]", elementExpression), 2003, (ResultSet rs, int column) -> {
            Array array = rs.getArray(column);
            Assertions.assertThat((Object)array.getArray()).isEqualTo((Object)new Object[]{null, element});
            Assertions.assertThat((int)array.getBaseType()).isEqualTo(elementSqlType);
            Assertions.assertThat((String)array.getBaseTypeName()).isEqualTo(elementTypeName);
            array = (Array)rs.getObject(column);
            Assertions.assertThat((Object)array.getArray()).isEqualTo((Object)new Object[]{null, element});
            Assertions.assertThat((int)array.getBaseType()).isEqualTo(elementSqlType);
            Assertions.assertThat((String)array.getBaseTypeName()).isEqualTo(elementTypeName);
            Assertions.assertThat((List)rs.getObject(column, List.class)).isEqualTo(Arrays.asList(null, element));
        });
    }

    @Test
    public void testMap() throws Exception {
        try (ConnectedStatement connectedStatement = this.newStatement();){
            this.checkRepresentation(connectedStatement.getStatement(), "map(ARRAY['k1', 'k2'], ARRAY[BIGINT '42', -117])", 2000, (ResultSet rs, int column) -> {
                Assertions.assertThat((Object)rs.getObject(column)).isEqualTo((Object)ImmutableMap.of((Object)"k1", (Object)42L, (Object)"k2", (Object)-117L));
                Assertions.assertThat((Map)rs.getObject(column, Map.class)).isEqualTo((Object)ImmutableMap.of((Object)"k1", (Object)42L, (Object)"k2", (Object)-117L));
            });
            this.checkRepresentation(connectedStatement.getStatement(), "map(ARRAY['k1', 'k2'], ARRAY[42, NULL])", 2000, (ResultSet rs, int column) -> {
                HashMap<String, Integer> expected = new HashMap<String, Integer>();
                expected.put("k1", 42);
                expected.put("k2", null);
                Assertions.assertThat((Object)rs.getObject(column)).isEqualTo(expected);
                Assertions.assertThat((Map)rs.getObject(column, Map.class)).isEqualTo(expected);
            });
            this.checkRepresentation(connectedStatement.getStatement(), "map(ARRAY['k1', 'k2'], ARRAY[CAST(ROW(42) AS row(a integer)), NULL])", 2000, (ResultSet rs, int column) -> {
                HashMap<String, Row> expected = new HashMap<String, Row>();
                expected.put("k1", Row.builder().addField("a", (Object)42).build());
                expected.put("k2", null);
                Assertions.assertThat((Object)rs.getObject(column)).isEqualTo(expected);
                Assertions.assertThat((Map)rs.getObject(column, Map.class)).isEqualTo(expected);
            });
        }
    }

    @Test
    public void testRow() throws Exception {
        try (ConnectedStatement connectedStatement = this.newStatement();){
            this.checkRepresentation(connectedStatement.getStatement(), "CAST(ROW(42, 'Trino') AS row(a_bigint bigint, a_varchar varchar(17)))", 2000, (ResultSet rs, int column) -> {
                Assertions.assertThat((Object)rs.getObject(column)).isEqualTo((Object)Row.builder().addField("a_bigint", (Object)42L).addField("a_varchar", (Object)"Trino").build());
                Assertions.assertThat((Map)rs.getObject(column, Map.class)).isEqualTo((Object)ImmutableMap.of((Object)"a_bigint", (Object)42L, (Object)"a_varchar", (Object)"Trino"));
            });
            this.checkRepresentation(connectedStatement.getStatement(), "CAST(ROW(42, 'Trino') AS row(a_bigint bigint, varchar(17)))", 2000, (ResultSet rs, int column) -> {
                Assertions.assertThat((Object)rs.getObject(column)).isEqualTo((Object)Row.builder().addField("a_bigint", (Object)42L).addUnnamedField((Object)"Trino").build());
                Assertions.assertThat((Map)rs.getObject(column, Map.class)).isEqualTo((Object)ImmutableMap.of((Object)"a_bigint", (Object)42L, (Object)"field1", (Object)"Trino"));
            });
            this.checkRepresentation(connectedStatement.getStatement(), "ROW(42, 'Trino')", 2000, (ResultSet rs, int column) -> {
                Assertions.assertThat((Object)rs.getObject(column)).isEqualTo((Object)Row.builder().addUnnamedField((Object)42).addUnnamedField((Object)"Trino").build());
                Assertions.assertThat((Map)rs.getObject(column, Map.class)).isEqualTo((Object)ImmutableMap.of((Object)"field0", (Object)42, (Object)"field1", (Object)"Trino"));
            });
            this.checkRepresentation(connectedStatement.getStatement(), "CAST(ROW(42, 'Trino') AS row(field1 integer, varchar(17)))", 2000, (ResultSet rs, int column) -> {
                Assertions.assertThat((Object)rs.getObject(column)).isEqualTo((Object)Row.builder().addField("field1", (Object)42).addUnnamedField((Object)"Trino").build());
                ((AbstractThrowableAssert)Assertions.assertThatThrownBy(() -> rs.getObject(column, Map.class)).isInstanceOf(SQLException.class)).hasMessageMatching("Duplicate field name: field1");
            });
            this.checkRepresentation(connectedStatement.getStatement(), "CAST(ROW(NULL, NULL) AS row(field1 integer, varchar(17)))", 2000, (ResultSet rs, int column) -> {
                Assertions.assertThat((Object)rs.getObject(column)).isEqualTo((Object)Row.builder().addField("field1", null).addUnnamedField(null).build());
                ((AbstractThrowableAssert)Assertions.assertThatThrownBy(() -> rs.getObject(column, Map.class)).isInstanceOf(SQLException.class)).hasMessageMatching("Duplicate field name: field1");
            });
            this.checkRepresentation(connectedStatement.getStatement(), "ROW(ROW(ROW(42)))", 2000, (ResultSet rs, int column) -> Assertions.assertThat((Object)rs.getObject(column)).isEqualTo((Object)Row.builder().addUnnamedField((Object)Row.builder().addUnnamedField((Object)Row.builder().addUnnamedField((Object)42).build()).build()).build()));
            this.checkRepresentation(connectedStatement.getStatement(), "CAST(ROW(ROW(ROW(42))) AS row(a row(b row(c integer))))", 2000, (ResultSet rs, int column) -> Assertions.assertThat((Object)rs.getObject(column)).isEqualTo((Object)Row.builder().addField("a", (Object)Row.builder().addField("b", (Object)Row.builder().addField("c", (Object)42).build()).build()).build()));
            this.checkRepresentation(connectedStatement.getStatement(), "CAST(   ROW(ARRAY[NULL, map(ARRAY['k1', 'k2'], ARRAY[NULL, ROW(42)])]) AS   row(\"outer\" array(map(varchar, row(leaf integer)))))", 2000, (ResultSet rs, int column) -> {
                Map<String, Object> map = new HashMap<String, Row>();
                map.put("k1", null);
                map.put("k2", Row.builder().addField("leaf", (Object)42).build());
                map = Collections.unmodifiableMap(map);
                List<HashMap<String, Object>> array = new ArrayList<HashMap<String, Row>>();
                array.add(null);
                array.add((HashMap<String, Object>)map);
                array = Collections.unmodifiableList(array);
                Assertions.assertThat((Object)rs.getObject(column)).isEqualTo((Object)Row.builder().addField("outer", array).build());
                Assertions.assertThat((Map)rs.getObject(column, Map.class)).isEqualTo((Object)ImmutableMap.of((Object)"outer", array));
            });
        }
    }

    private void checkRepresentation(Statement statement, String expression, int expectedSqlType, Object expectedRepresentation) throws SQLException {
        this.checkRepresentation(statement, expression, expectedSqlType, (ResultSet rs, int column) -> {
            Assertions.assertThat((Object)rs.getObject(column)).isEqualTo(expectedRepresentation);
            Assertions.assertThat(rs.getObject(column, expectedRepresentation.getClass())).isEqualTo(expectedRepresentation);
        });
    }

    private void checkRepresentation(Statement statement, String expression, int expectedSqlType, ResultAssertion assertion) throws SQLException {
        try (ResultSet rs = statement.executeQuery("SELECT " + expression);){
            ResultSetMetaData metadata = rs.getMetaData();
            Assertions.assertThat((int)metadata.getColumnCount()).isEqualTo(1);
            Assertions.assertThat((int)metadata.getColumnType(1)).isEqualTo(expectedSqlType);
            Assertions.assertThat((boolean)rs.next()).isTrue();
            assertion.accept(rs, 1);
            Assertions.assertThat((boolean)rs.next()).isFalse();
        }
    }

    private Object getObjectRepresentation(Connection connection, String expression) throws SQLException {
        try (Statement statement = connection.createStatement();){
            Object object;
            block12: {
                ResultSet rs = statement.executeQuery("SELECT " + expression);
                try {
                    ResultSetMetaData metadata = rs.getMetaData();
                    Assertions.assertThat((int)metadata.getColumnCount()).isEqualTo(1);
                    Assertions.assertThat((boolean)rs.next()).isTrue();
                    Object object2 = rs.getObject(1);
                    Assertions.assertThat((boolean)rs.next()).isFalse();
                    object = object2;
                    if (rs == null) break block12;
                }
                catch (Throwable throwable) {
                    if (rs != null) {
                        try {
                            rs.close();
                        }
                        catch (Throwable throwable2) {
                            throwable.addSuppressed(throwable2);
                        }
                    }
                    throw throwable;
                }
                rs.close();
            }
            return object;
        }
    }

    @Test
    public void testStatsExtraction() throws Exception {
        try (ConnectedStatement connectedStatement = this.newStatement();
             TrinoResultSet rs = (TrinoResultSet)connectedStatement.getStatement().executeQuery("SELECT 123 x, 456 x");){
            Assertions.assertThat((Object)rs.getStats()).isNotNull();
            Assertions.assertThat((boolean)rs.next()).isTrue();
            Assertions.assertThat((Object)rs.getStats()).isNotNull();
            Assertions.assertThat((boolean)rs.next()).isFalse();
            Assertions.assertThat((Object)rs.getStats()).isNotNull();
        }
    }

    @Test
    public void testMaxRowsUnset() throws Exception {
        try (ConnectedStatement connectedStatement = this.newStatement();){
            this.assertMaxRowsLimit(connectedStatement.getStatement(), 0);
            this.assertMaxRowsResult(connectedStatement.getStatement(), 7L);
        }
    }

    @Test
    public void testMaxRowsUnlimited() throws Exception {
        try (ConnectedStatement connectedStatement = this.newStatement();){
            this.assertMaxRowsLimit(connectedStatement.getStatement(), 0);
            connectedStatement.getStatement().setMaxRows(0);
            this.assertMaxRowsLimit(connectedStatement.getStatement(), 0);
            this.assertMaxRowsResult(connectedStatement.getStatement(), 7L);
        }
    }

    @Test
    public void testMaxRowsLimited() throws Exception {
        try (ConnectedStatement connectedStatement = this.newStatement();){
            this.assertMaxRowsLimit(connectedStatement.getStatement(), 0);
            connectedStatement.getStatement().setMaxRows(4);
            this.assertMaxRowsLimit(connectedStatement.getStatement(), 4);
            this.assertMaxRowsResult(connectedStatement.getStatement(), 4L);
        }
    }

    @Test
    public void testMaxRowsLimitLargerThanResult() throws Exception {
        try (ConnectedStatement connectedStatement = this.newStatement();){
            this.assertMaxRowsLimit(connectedStatement.getStatement(), 0);
            connectedStatement.getStatement().setMaxRows(10);
            this.assertMaxRowsLimit(connectedStatement.getStatement(), 10);
            this.assertMaxRowsResult(connectedStatement.getStatement(), 7L);
        }
    }

    @Test
    public void testLargeMaxRowsUnlimited() throws Exception {
        try (ConnectedStatement connectedStatement = this.newStatement();){
            this.assertMaxRowsLimit(connectedStatement.getStatement(), 0);
            connectedStatement.getStatement().setLargeMaxRows(0L);
            this.assertMaxRowsLimit(connectedStatement.getStatement(), 0);
            this.assertMaxRowsResult(connectedStatement.getStatement(), 7L);
        }
    }

    @Test
    public void testLargeMaxRowsLimited() throws Exception {
        try (ConnectedStatement connectedStatement = this.newStatement();){
            this.assertMaxRowsLimit(connectedStatement.getStatement(), 0);
            connectedStatement.getStatement().setLargeMaxRows(4L);
            this.assertMaxRowsLimit(connectedStatement.getStatement(), 4);
            this.assertMaxRowsResult(connectedStatement.getStatement(), 4L);
        }
    }

    @Test
    public void testLargeMaxRowsLimitLargerThanResult() throws Exception {
        try (ConnectedStatement connectedStatement = this.newStatement();){
            long limit = 0x4FFFFFFF6L;
            connectedStatement.getStatement().setLargeMaxRows(limit);
            Assertions.assertThat((long)connectedStatement.getStatement().getLargeMaxRows()).isEqualTo(limit);
            this.assertMaxRowsResult(connectedStatement.getStatement(), 7L);
        }
    }

    private void assertMaxRowsLimit(Statement statement, int expectedLimit) throws SQLException {
        Assertions.assertThat((int)statement.getMaxRows()).isEqualTo(expectedLimit);
        Assertions.assertThat((long)statement.getLargeMaxRows()).isEqualTo((long)expectedLimit);
    }

    private void assertMaxRowsResult(Statement statement, long expectedCount) throws SQLException {
        try (ResultSet rs = statement.executeQuery("SELECT * FROM (VALUES (1), (2), (3), (4), (5), (6), (7)) AS x (a)");){
            Assertions.assertThat((long)BaseTestJdbcResultSet.countRows(rs)).isEqualTo(expectedCount);
        }
    }

    @Test
    public void testMaxRowsExceedsLimit() throws Exception {
        try (ConnectedStatement connectedStatement = this.newStatement();){
            connectedStatement.getStatement().setLargeMaxRows(0x4FFFFFFF6L);
            ((AbstractThrowableAssert)Assertions.assertThatThrownBy(connectedStatement.getStatement()::getMaxRows).isInstanceOf(SQLException.class)).hasMessage("Max rows exceeds limit of 2147483647");
        }
    }

    @Test
    public void testGetStatement() throws Exception {
        try (ConnectedStatement connectedStatement = this.newStatement();
             ResultSet rs = connectedStatement.getStatement().executeQuery("SELECT * FROM (VALUES (1), (2), (3))");){
            Assertions.assertThat((Object)rs.getStatement()).isEqualTo((Object)connectedStatement.getStatement());
        }
    }

    @Test
    public void testGetRow() throws Exception {
        try (ConnectedStatement connectedStatement = this.newStatement();
             ResultSet rs = connectedStatement.getStatement().executeQuery("SELECT * FROM (VALUES (1), (2), (3))");){
            Assertions.assertThat((int)rs.getRow()).isEqualTo(0);
            int currentRow = 0;
            while (rs.next()) {
                Assertions.assertThat((int)rs.getRow()).isEqualTo(++currentRow);
            }
            Assertions.assertThat((int)rs.getRow()).isEqualTo(0);
        }
    }

    @Test
    public void testGetRowException() throws Exception {
        try (ConnectedStatement connectedStatement = this.newStatement();){
            ResultSet rs = connectedStatement.getStatement().executeQuery("SELECT * FROM (VALUES (1), (2), (3))");
            rs.close();
            ((AbstractThrowableAssert)Assertions.assertThatThrownBy(rs::getRow).isInstanceOf(SQLException.class)).hasMessage("ResultSet is closed");
        }
    }

    private static long countRows(ResultSet rs) throws SQLException {
        long count = 0L;
        while (rs.next()) {
            ++count;
        }
        return count;
    }

    protected ConnectedStatement newStatement() {
        return new ConnectedStatement();
    }

    static Time toSqlTime(LocalTime localTime) {
        return new Time(Time.valueOf(localTime).getTime() + (long)IntMath.divide((int)localTime.getNano(), (int)1000000, (RoundingMode)RoundingMode.UNNECESSARY));
    }

    protected class ConnectedStatement
    implements AutoCloseable {
        private final Connection connection;
        private final Statement statement;

        public ConnectedStatement() {
            try {
                this.connection = BaseTestJdbcResultSet.this.createConnection();
                this.statement = this.connection.createStatement();
            }
            catch (Exception e) {
                throw new RuntimeException(e);
            }
        }

        @Override
        public void close() throws SQLException {
            this.statement.close();
            this.connection.close();
        }

        public Statement getStatement() {
            return this.statement;
        }
    }

    @FunctionalInterface
    private static interface ResultAssertion {
        public void accept(ResultSet var1, int var2) throws SQLException;
    }
}

