/*
 * Decompiled with CFR 0.152.
 */
package io.trino.jdbc;

import com.google.common.base.Verify;
import com.google.common.collect.ImmutableList;
import com.google.common.collect.Iterables;
import io.airlift.log.Logger;
import io.airlift.log.Logging;
import io.trino.jdbc.TrinoConnection;
import io.trino.server.testing.TestingTrinoServer;
import io.trino.util.AutoCloseableCloser;
import java.io.Closeable;
import java.sql.Connection;
import java.sql.Date;
import java.sql.DriverManager;
import java.sql.JDBCType;
import java.sql.PreparedStatement;
import java.sql.ResultSet;
import java.sql.SQLException;
import java.sql.Statement;
import java.sql.Time;
import java.sql.Timestamp;
import java.time.Instant;
import java.time.LocalDate;
import java.time.LocalDateTime;
import java.time.LocalTime;
import java.time.ZoneId;
import java.time.ZoneOffset;
import java.time.ZonedDateTime;
import java.util.ArrayList;
import java.util.Calendar;
import java.util.List;
import java.util.Objects;
import java.util.Optional;
import java.util.TimeZone;
import oracle.jdbc.OracleType;
import org.assertj.core.api.AbstractIntegerAssert;
import org.assertj.core.api.AbstractStringAssert;
import org.assertj.core.api.Assertions;
import org.junit.jupiter.api.AfterAll;
import org.junit.jupiter.api.BeforeAll;
import org.junit.jupiter.api.Test;
import org.junit.jupiter.api.TestInstance;
import org.junit.jupiter.api.parallel.Execution;
import org.junit.jupiter.api.parallel.ExecutionMode;
import org.testcontainers.containers.OracleContainer;
import org.testcontainers.containers.PostgreSQLContainer;

@TestInstance(value=TestInstance.Lifecycle.PER_CLASS)
@Execution(value=ExecutionMode.SAME_THREAD)
public class TestJdbcVendorCompatibility {
    private static final String OTHER_TIMEZONE = "Asia/Kathmandu";
    private Logger log;
    private TestingTrinoServer server;
    private List<ReferenceDriver> referenceDrivers;

    @BeforeAll
    public void setupServer() {
        ((AbstractStringAssert)Assertions.assertThat((String)OTHER_TIMEZONE).describedAs("We need a timezone different from the default JVM one", new Object[0])).isNotEqualTo((Object)TimeZone.getDefault().getID());
        Logging.initialize();
        this.log = Logger.get(TestJdbcVendorCompatibility.class);
        this.server = TestingTrinoServer.create();
        this.referenceDrivers = new ArrayList<ReferenceDriver>();
        this.referenceDrivers.add(new PostgresqlReferenceDriver());
        this.referenceDrivers.add(new OracleReferenceDriver());
    }

    @AfterAll
    public void tearDownServer() throws Exception {
        try (AutoCloseableCloser closer = AutoCloseableCloser.create();){
            if (this.referenceDrivers != null) {
                this.referenceDrivers.forEach(arg_0 -> ((AutoCloseableCloser)closer).register(arg_0));
                this.referenceDrivers.clear();
            }
            if (this.server != null) {
                closer.register((AutoCloseable)this.server);
                this.server = null;
            }
        }
    }

    @Test
    public void testVarbinary() throws Exception {
        try (Connection connection = this.createConnection();
             Statement statement = connection.createStatement();
             ConnectionSetup connectionSetup = new ConnectionSetup(this.referenceDrivers);){
            this.checkRepresentation(connection, statement, "X'12345678'", (List<String>)ImmutableList.of((Object)"bytea E'\\\\x12345678'", (Object)"hextoraw('12345678')"), JDBCType.VARBINARY, Optional.empty(), (rs, reference, column) -> {
                Assertions.assertThat((byte[])rs.getBytes(column)).isEqualTo((Object)new byte[]{18, 52, 86, 120});
                Assertions.assertThat((byte[])rs.getBytes(column)).isEqualTo((Object)reference.getBytes(column));
                Assertions.assertThat((Object)rs.getObject(column)).isEqualTo(reference.getObject(column));
                Assertions.assertThat((String)rs.getString(column).replaceFirst("^0x", "")).isEqualTo(reference.getString(column).replaceFirst("^\\\\x", ""));
            });
        }
    }

    @Test
    public void testDate() throws Exception {
        this.testDate(Optional.empty());
        this.testDate(Optional.of("UTC"));
        this.testDate(Optional.of("Europe/Warsaw"));
        this.testDate(Optional.of("America/Denver"));
        this.testDate(Optional.of(ZoneId.systemDefault().getId()));
    }

    private void testDate(Optional<String> sessionTimezoneId) throws Exception {
        try (Connection connection = this.createConnection();
             Statement statement = connection.createStatement();
             ConnectionSetup connectionSetup = new ConnectionSetup(this.referenceDrivers);){
            this.checkRepresentation(connection, statement, "DATE '2018-02-13'", JDBCType.DATE, sessionTimezoneId, (rs, reference, column) -> {
                Assertions.assertThat((java.util.Date)rs.getDate(column)).isEqualTo((Object)reference.getDate(column));
                Assertions.assertThat((java.util.Date)rs.getDate(column)).isEqualTo((Object)Date.valueOf(LocalDate.of(2018, 2, 13)));
                Assertions.assertThat((java.util.Date)rs.getDate(column, this.getCalendar())).isEqualTo((Object)reference.getDate(column, this.getCalendar()));
                Assertions.assertThat((java.util.Date)rs.getDate(column, this.getCalendar())).isEqualTo((Object)new Date(LocalDate.of(2018, 2, 13).atStartOfDay(this.getZoneId()).toInstant().toEpochMilli()));
            });
        }
    }

    @Test
    public void testTimestamp() throws Exception {
        this.testTimestamp(Optional.empty());
        this.testTimestamp(Optional.of("UTC"));
        this.testTimestamp(Optional.of("Europe/Warsaw"));
        this.testTimestamp(Optional.of("America/Denver"));
        this.testTimestamp(Optional.of(ZoneId.systemDefault().getId()));
    }

    private void testTimestamp(Optional<String> sessionTimezoneId) throws Exception {
        try (Connection connection = this.createConnection();
             Statement statement = connection.createStatement();
             ConnectionSetup connectionSetup = new ConnectionSetup(this.referenceDrivers);){
            this.checkRepresentation(connection, statement, "TIMESTAMP '2018-02-13 13:14:15.123'", JDBCType.TIMESTAMP, sessionTimezoneId, (rs, reference, column) -> {
                Assertions.assertThat((java.util.Date)rs.getTimestamp(column)).isEqualTo((Object)reference.getTimestamp(column));
                Assertions.assertThat((java.util.Date)rs.getTimestamp(column)).isEqualTo((Object)Timestamp.valueOf(LocalDateTime.of(2018, 2, 13, 13, 14, 15, 123000000)));
                Assertions.assertThat((java.util.Date)rs.getTimestamp(column, this.getCalendar())).isEqualTo((Object)reference.getTimestamp(column, this.getCalendar()));
                Assertions.assertThat((java.util.Date)rs.getTimestamp(column, this.getCalendar())).isEqualTo((Object)new Timestamp(LocalDateTime.of(2018, 2, 13, 13, 14, 15, 123000000).atZone(this.getZoneId()).toInstant().toEpochMilli()));
            });
        }
    }

    @Test
    public void testTimestampWithTimeZone() throws Exception {
        this.testTimestampWithTimeZone(Optional.empty());
        this.testTimestampWithTimeZone(Optional.of("UTC"));
        this.testTimestampWithTimeZone(Optional.of("Europe/Warsaw"));
        this.testTimestampWithTimeZone(Optional.of("America/Denver"));
        this.testTimestampWithTimeZone(Optional.of(ZoneId.systemDefault().getId()));
    }

    private void testTimestampWithTimeZone(Optional<String> sessionTimezoneId) throws Exception {
        try (Connection connection = this.createConnection();
             Statement statement = connection.createStatement();
             ConnectionSetup connectionSetup = new ConnectionSetup(this.referenceDrivers);){
            this.checkRepresentation(connection, statement, "TIMESTAMP '1970-01-01 00:00:00.000 +00:00'", (List<String>)ImmutableList.of((Object)"TIMESTAMP WITH TIME ZONE '1970-01-01 00:00:00.000 +00:00'", (Object)"from_tz(TIMESTAMP '1970-01-01 00:00:00.000', '+00:00')"), JDBCType.TIMESTAMP_WITH_TIMEZONE, sessionTimezoneId, (rs, reference, column) -> {
                Timestamp timestampForPointInTime = Timestamp.from(Instant.EPOCH);
                Assertions.assertThat((long)rs.getTimestamp(column).getTime()).isEqualTo(reference.getTimestamp(column).getTime());
                Assertions.assertThat((java.util.Date)rs.getTimestamp(column)).isEqualTo((Object)reference.getTimestamp(column));
                Assertions.assertThat((java.util.Date)rs.getTimestamp(column)).isEqualTo((Object)timestampForPointInTime);
                Assertions.assertThat((long)rs.getTimestamp(column, this.getCalendar()).getTime()).isEqualTo(reference.getTimestamp(column, this.getCalendar()).getTime());
                Assertions.assertThat((java.util.Date)rs.getTimestamp(column, this.getCalendar())).isEqualTo((Object)reference.getTimestamp(column, this.getCalendar()));
                Assertions.assertThat((java.util.Date)rs.getTimestamp(column, this.getCalendar())).isEqualTo((Object)timestampForPointInTime);
            });
            this.checkRepresentation(connection, statement, "TIMESTAMP '2018-02-13 13:14:15.123 +03:15'", (List<String>)ImmutableList.of((Object)"TIMESTAMP WITH TIME ZONE '2018-02-13 13:14:15.123 +03:15'", (Object)"from_tz(TIMESTAMP '2018-02-13 13:14:15.123', '+03:15')"), JDBCType.TIMESTAMP_WITH_TIMEZONE, sessionTimezoneId, (rs, reference, column) -> {
                Timestamp timestampForPointInTime = Timestamp.from(ZonedDateTime.of(2018, 2, 13, 13, 14, 15, 123000000, ZoneOffset.ofHoursMinutes(3, 15)).toInstant());
                Assertions.assertThat((long)rs.getTimestamp(column).getTime()).isEqualTo(reference.getTimestamp(column).getTime());
                Assertions.assertThat((java.util.Date)rs.getTimestamp(column)).isEqualTo((Object)reference.getTimestamp(column));
                Assertions.assertThat((java.util.Date)rs.getTimestamp(column)).isEqualTo((Object)timestampForPointInTime);
                Assertions.assertThat((long)rs.getTimestamp(column, this.getCalendar()).getTime()).isEqualTo(reference.getTimestamp(column, this.getCalendar()).getTime());
                Assertions.assertThat((java.util.Date)rs.getTimestamp(column, this.getCalendar())).isEqualTo((Object)reference.getTimestamp(column, this.getCalendar()));
                Assertions.assertThat((java.util.Date)rs.getTimestamp(column, this.getCalendar())).isEqualTo((Object)timestampForPointInTime);
            });
            this.checkRepresentation(connection, statement, "TIMESTAMP '2018-02-13 13:14:15.123 Europe/Warsaw'", (List<String>)ImmutableList.of((Object)"TIMESTAMP WITH TIME ZONE '2018-02-13 13:14:15.123 Europe/Warsaw'", (Object)"from_tz(TIMESTAMP '2018-02-13 13:14:15.123', 'Europe/Warsaw')"), JDBCType.TIMESTAMP_WITH_TIMEZONE, sessionTimezoneId, (rs, reference, column) -> {
                Timestamp timestampForPointInTime = Timestamp.from(ZonedDateTime.of(2018, 2, 13, 13, 14, 15, 123000000, ZoneId.of("Europe/Warsaw")).toInstant());
                Assertions.assertThat((long)rs.getTimestamp(column).getTime()).isEqualTo(reference.getTimestamp(column).getTime());
                Assertions.assertThat((java.util.Date)rs.getTimestamp(column)).isEqualTo((Object)reference.getTimestamp(column));
                Assertions.assertThat((java.util.Date)rs.getTimestamp(column)).isEqualTo((Object)timestampForPointInTime);
                Assertions.assertThat((long)rs.getTimestamp(column, this.getCalendar()).getTime()).isEqualTo(reference.getTimestamp(column, this.getCalendar()).getTime());
                Assertions.assertThat((java.util.Date)rs.getTimestamp(column, this.getCalendar())).isEqualTo((Object)reference.getTimestamp(column, this.getCalendar()));
                Assertions.assertThat((java.util.Date)rs.getTimestamp(column, this.getCalendar())).isEqualTo((Object)timestampForPointInTime);
            });
        }
    }

    @Test
    public void testTime() throws Exception {
        this.testTime(Optional.empty());
        this.testTime(Optional.of("UTC"));
        this.testTime(Optional.of("Europe/Warsaw"));
        this.testTime(Optional.of("America/Denver"));
        this.testTime(Optional.of(ZoneId.systemDefault().getId()));
    }

    private void testTime(Optional<String> sessionTimezoneId) throws Exception {
        try (Connection connection = this.createConnection();
             Statement statement = connection.createStatement();
             ConnectionSetup connectionSetup = new ConnectionSetup(this.referenceDrivers);){
            this.checkRepresentation(connection, statement, "TIME '09:39:05'", JDBCType.TIME, sessionTimezoneId, (rs, reference, column) -> {
                Assertions.assertThat((java.util.Date)rs.getTime(column)).isEqualTo((Object)reference.getTime(column));
                Assertions.assertThat((java.util.Date)rs.getTime(column)).isEqualTo((Object)Time.valueOf(LocalTime.of(9, 39, 5)));
                Assertions.assertThat((java.util.Date)rs.getTime(column, this.getCalendar())).isEqualTo((Object)reference.getTime(column, this.getCalendar()));
                Assertions.assertThat((java.util.Date)rs.getTime(column, this.getCalendar())).isEqualTo((Object)new Time(LocalDate.of(1970, 1, 1).atTime(LocalTime.of(9, 39, 5)).atZone(this.getZoneId()).toInstant().toEpochMilli()));
            });
        }
    }

    @Test
    public void testDateRoundTrip() throws Exception {
        this.testDateRoundTrip(Optional.empty());
        this.testDateRoundTrip(Optional.of("UTC"));
        this.testDateRoundTrip(Optional.of("Europe/Warsaw"));
        this.testDateRoundTrip(Optional.of("America/Denver"));
        this.testDateRoundTrip(Optional.of(ZoneId.systemDefault().getId()));
    }

    private void testDateRoundTrip(Optional<String> sessionTimezoneId) throws SQLException {
        try (Connection connection = this.createConnection();){
            LocalDate date = LocalDate.of(2001, 5, 6);
            Date sqlDate = Date.valueOf(date);
            java.util.Date javaDate = new java.util.Date(sqlDate.getTime());
            LocalDateTime dateTime = LocalDateTime.of(date, LocalTime.of(12, 34, 56));
            Timestamp sqlTimestamp = Timestamp.valueOf(dateTime);
            this.assertParameter(connection, sqlDate, sessionTimezoneId, (ps, i) -> ps.setDate(i, sqlDate));
            this.assertParameter(connection, sqlDate, sessionTimezoneId, (ps, i) -> ps.setObject(i, sqlDate));
            this.assertParameter(connection, sqlDate, sessionTimezoneId, (ps, i) -> ps.setObject(i, (Object)sqlDate, 91));
            this.assertParameter(connection, sqlDate, sessionTimezoneId, (ps, i) -> ps.setObject(i, (Object)sqlTimestamp, 91));
            this.assertParameter(connection, sqlDate, sessionTimezoneId, (ps, i) -> ps.setObject(i, (Object)javaDate, 91));
            this.assertParameter(connection, sqlDate, sessionTimezoneId, (ps, i) -> ps.setObject(i, (Object)date, 91));
            this.assertParameter(connection, sqlDate, sessionTimezoneId, (ps, i) -> ps.setObject(i, (Object)dateTime, 91));
            this.assertParameter(connection, sqlDate, sessionTimezoneId, (ps, i) -> ps.setObject(i, (Object)"2001-05-06", 91));
        }
    }

    @Test
    public void testTimestampRoundTrip() throws Exception {
        this.testTimestampRoundTrip(Optional.empty());
        this.testTimestampRoundTrip(Optional.of("UTC"));
        this.testTimestampRoundTrip(Optional.of("Europe/Warsaw"));
        this.testTimestampRoundTrip(Optional.of("America/Denver"));
        this.testTimestampRoundTrip(Optional.of(ZoneId.systemDefault().getId()));
    }

    private void testTimestampRoundTrip(Optional<String> sessionTimezoneId) throws SQLException {
        try (Connection connection = this.createConnection();){
            LocalDateTime dateTime = LocalDateTime.of(2001, 5, 6, 12, 34, 56);
            Date sqlDate = Date.valueOf(dateTime.toLocalDate());
            Time sqlTime = Time.valueOf(dateTime.toLocalTime());
            Timestamp sqlTimestamp = Timestamp.valueOf(dateTime);
            Timestamp sameInstantInWarsawZone = Timestamp.valueOf(dateTime.atZone(ZoneId.systemDefault()).withZoneSameInstant(ZoneId.of("Europe/Warsaw")).toLocalDateTime());
            java.util.Date javaDate = java.util.Date.from(dateTime.atZone(ZoneId.systemDefault()).toInstant());
            this.assertParameter(connection, sqlTimestamp, sessionTimezoneId, (ps, i) -> ps.setTimestamp(i, sqlTimestamp));
            this.assertParameter(connection, sqlTimestamp, sessionTimezoneId, (ps, i) -> ps.setTimestamp(i, sqlTimestamp, null));
            this.assertParameter(connection, sqlTimestamp, sessionTimezoneId, (ps, i) -> ps.setTimestamp(i, sqlTimestamp, Calendar.getInstance()));
            this.assertParameter(connection, sameInstantInWarsawZone, sessionTimezoneId, (ps, i) -> ps.setTimestamp(i, sqlTimestamp, Calendar.getInstance(TimeZone.getTimeZone(ZoneId.of("Europe/Warsaw")))));
            this.assertParameter(connection, sqlTimestamp, sessionTimezoneId, (ps, i) -> ps.setObject(i, sqlTimestamp));
            this.assertParameter(connection, new Timestamp(sqlDate.getTime()), sessionTimezoneId, (ps, i) -> ps.setObject(i, (Object)sqlDate, 93));
            this.assertParameter(connection, new Timestamp(sqlTime.getTime()), sessionTimezoneId, (ps, i) -> ps.setObject(i, (Object)sqlTime, 93));
            this.assertParameter(connection, sqlTimestamp, sessionTimezoneId, (ps, i) -> ps.setObject(i, (Object)sqlTimestamp, 93));
            this.assertParameter(connection, sqlTimestamp, sessionTimezoneId, (ps, i) -> ps.setObject(i, (Object)javaDate, 93));
            this.assertParameter(connection, sqlTimestamp, sessionTimezoneId, (ps, i) -> ps.setObject(i, (Object)dateTime, 93));
            this.assertParameter(connection, sqlTimestamp, sessionTimezoneId, (ps, i) -> ps.setObject(i, (Object)"2001-05-06 12:34:56", 93));
        }
    }

    private Connection createConnection() throws SQLException {
        return DriverManager.getConnection("jdbc:trino://" + this.server.getAddress(), "test", null);
    }

    private void assertParameter(Connection connection, Object expectedValue, Optional<String> sessionTimezoneId, Binder binder) throws SQLException {
        sessionTimezoneId.ifPresent(arg_0 -> ((TrinoConnection)connection.unwrap(TrinoConnection.class)).setTimeZoneId(arg_0));
        try (PreparedStatement statement = connection.prepareStatement("SELECT ?");){
            binder.bind(statement, 1);
            try (ResultSet rs = statement.executeQuery();){
                Assertions.assertThat((boolean)rs.next()).isTrue();
                Assertions.assertThat((Object)expectedValue).isEqualTo(rs.getObject(1));
                Assertions.assertThat((boolean)rs.next()).isFalse();
            }
        }
    }

    private void checkRepresentation(Connection connection, Statement statement, String expression, JDBCType type, Optional<String> sessionTimezoneId, ResultAssertion assertion) throws Exception {
        List referenceDriversExpressions = (List)this.referenceDrivers.stream().map(driver -> driver.supports(type) ? expression : "").collect(ImmutableList.toImmutableList());
        this.checkRepresentation(connection, statement, expression, referenceDriversExpressions, type, sessionTimezoneId, assertion);
    }

    private void checkRepresentation(Connection connection, Statement statement, String trinoExpression, List<String> referenceDriversExpressions, JDBCType type, Optional<String> sessionTimezoneId, ResultAssertion assertion) throws Exception {
        Verify.verify((referenceDriversExpressions.size() == this.referenceDrivers.size() ? 1 : 0) != 0, (String)"Wrong referenceDriversExpressions list size", (Object[])new Object[0]);
        int tests = 0;
        ArrayList<AssertionError> failures = new ArrayList<AssertionError>();
        for (int i = 0; i < this.referenceDrivers.size(); ++i) {
            ReferenceDriver driver = this.referenceDrivers.get(i);
            String referenceExpression = referenceDriversExpressions.get(i);
            if (!driver.supports(type)) {
                Verify.verify((boolean)referenceExpression.isEmpty(), (String)"referenceExpression must be empty for %s so that the test code clearly indicates which cases are actually tested", (Object)driver);
                continue;
            }
            ++tests;
            this.log.info("Checking behavior against %s using expression: %s", new Object[]{driver, referenceExpression});
            try {
                Verify.verify((!referenceExpression.isEmpty() ? 1 : 0) != 0, (String)"referenceExpression is empty", (Object[])new Object[0]);
                this.checkRepresentation(connection, statement, trinoExpression, referenceExpression, type, sessionTimezoneId, driver, assertion);
                continue;
            }
            catch (AssertionError | RuntimeException e) {
                String message = String.format("Failure when checking behavior against %s", driver);
                this.log.error((Throwable)e, "%s", new Object[]{message});
                failures.add(new AssertionError(message, (Throwable)e));
            }
        }
        Verify.verify((tests > 0 ? 1 : 0) != 0, (String)"No reference driver found supporting %s", (Object)type);
        if (!failures.isEmpty()) {
            if (failures.size() == 1 && tests == 1) {
                throw (AssertionError)Iterables.getOnlyElement(failures);
            }
            AssertionError error = new AssertionError((Object)String.format("Test failed for %s reference drivers out of %s applicable", failures.size(), tests));
            failures.forEach(arg_0 -> error.addSuppressed(arg_0));
            throw error;
        }
    }

    private void checkRepresentation(Connection connection, Statement statement, String trinoExpression, String referenceExpression, JDBCType type, Optional<String> sessionTimezoneId, ReferenceDriver reference, ResultAssertion assertion) throws Exception {
        try (ResultSet trinoResultSet = this.trinoQuery(connection, statement, trinoExpression, sessionTimezoneId);
             ResultSet referenceResultSet = reference.query(referenceExpression, sessionTimezoneId);){
            Assertions.assertThat((boolean)trinoResultSet.next()).isTrue();
            Assertions.assertThat((boolean)referenceResultSet.next()).isTrue();
            assertion.accept(trinoResultSet, referenceResultSet, 1);
            ((AbstractIntegerAssert)Assertions.assertThat((int)trinoResultSet.getMetaData().getColumnType(1)).as("Trino declared SQL type", new Object[0])).isEqualTo((Object)type.getVendorTypeNumber());
            ((AbstractIntegerAssert)Assertions.assertThat((int)referenceResultSet.getMetaData().getColumnType(1)).as("Reference driver's declared SQL type for " + type, new Object[0])).isEqualTo(reference.expectedDeclaredJdbcType(type));
            Assertions.assertThat((boolean)trinoResultSet.next()).isFalse();
            Assertions.assertThat((boolean)referenceResultSet.next()).isFalse();
        }
    }

    private ResultSet trinoQuery(Connection connection, Statement statement, String expression, Optional<String> sessionTimezoneId) throws Exception {
        sessionTimezoneId.ifPresent(arg_0 -> ((TrinoConnection)connection.unwrap(TrinoConnection.class)).setTimeZoneId(arg_0));
        return statement.executeQuery("SELECT " + expression);
    }

    private Calendar getCalendar() {
        return Calendar.getInstance(TimeZone.getTimeZone(ZoneId.of(OTHER_TIMEZONE)));
    }

    private ZoneId getZoneId() {
        return ZoneId.of(this.getCalendar().getTimeZone().getID());
    }

    private static class PostgresqlReferenceDriver
    implements ReferenceDriver {
        private final PostgreSQLContainer<?> postgresqlContainer;
        private Connection connection;
        private Statement statement;
        private Optional<Optional<String>> timezoneSet = Optional.empty();

        PostgresqlReferenceDriver() {
            this.postgresqlContainer = new PostgreSQLContainer("postgres:15");
            this.postgresqlContainer.start();
        }

        @Override
        public ResultSet query(String expression, Optional<String> timezoneId) throws Exception {
            Verify.verify((!this.timezoneSet.isPresent() || Objects.equals(this.timezoneSet.get(), timezoneId) ? 1 : 0) != 0, (String)"Cannot set time zone %s while %s set previously", timezoneId, this.timezoneSet);
            this.timezoneSet = Optional.of(timezoneId);
            if (timezoneId.isPresent()) {
                this.statement.execute(String.format("SET SESSION TIME ZONE '%s'", timezoneId.get()));
            }
            return this.statement.executeQuery(String.format("SELECT %s", expression));
        }

        @Override
        public boolean supports(JDBCType type) {
            return true;
        }

        @Override
        public int expectedDeclaredJdbcType(JDBCType type) {
            switch (type) {
                case TIMESTAMP_WITH_TIMEZONE: {
                    return 93;
                }
                case VARBINARY: {
                    return -2;
                }
            }
            return type.getVendorTypeNumber();
        }

        @Override
        public void setUp() {
            try {
                this.connection = this.postgresqlContainer.createConnection("");
                this.statement = this.connection.createStatement();
                this.timezoneSet = Optional.empty();
            }
            catch (SQLException e) {
                throw new RuntimeException(e);
            }
        }

        @Override
        public void tearDown() throws Exception {
            this.statement.close();
            this.connection.close();
        }

        @Override
        public void close() {
            this.postgresqlContainer.stop();
        }

        public String toString() {
            return "[postgresql]";
        }
    }

    private static class OracleReferenceDriver
    implements ReferenceDriver {
        private final OracleContainer oracleServer;
        private Connection connection;
        private Statement statement;
        private Optional<Optional<String>> timezoneSet = Optional.empty();

        OracleReferenceDriver() {
            this.oracleServer = new OracleContainer("gvenzl/oracle-xe:11.2.0.2-full").usingSid();
            this.oracleServer.start();
        }

        @Override
        public ResultSet query(String expression, Optional<String> timezoneId) throws Exception {
            Verify.verify((!this.timezoneSet.isPresent() || Objects.equals(this.timezoneSet.get(), timezoneId) ? 1 : 0) != 0, (String)"Cannot set time zone %s while %s set previously", timezoneId, this.timezoneSet);
            this.timezoneSet = Optional.of(timezoneId);
            if (timezoneId.isPresent()) {
                this.statement.execute(String.format("ALTER SESSION SET TIME_ZONE='%s'", timezoneId.get()));
            }
            return this.statement.executeQuery(String.format("SELECT %s FROM dual", expression));
        }

        @Override
        public boolean supports(JDBCType type) {
            return type != JDBCType.TIME;
        }

        @Override
        public int expectedDeclaredJdbcType(JDBCType type) {
            switch (type) {
                case DATE: {
                    return 93;
                }
                case TIMESTAMP_WITH_TIMEZONE: {
                    return OracleType.TIMESTAMP_WITH_TIME_ZONE.getVendorTypeNumber();
                }
            }
            return type.getVendorTypeNumber();
        }

        @Override
        public void setUp() {
            try {
                this.connection = DriverManager.getConnection(this.oracleServer.getJdbcUrl(), this.oracleServer.getUsername(), this.oracleServer.getPassword());
                this.statement = this.connection.createStatement();
                this.timezoneSet = Optional.empty();
            }
            catch (SQLException e) {
                throw new RuntimeException(e);
            }
        }

        @Override
        public void tearDown() throws Exception {
            this.statement.close();
            this.connection.close();
        }

        @Override
        public void close() {
            this.oracleServer.stop();
        }

        public String toString() {
            return "[oracle]";
        }
    }

    private class ConnectionSetup
    implements Closeable {
        private final List<ReferenceDriver> drivers;

        public ConnectionSetup(List<ReferenceDriver> drivers) {
            this.drivers = drivers;
            for (ReferenceDriver driver : drivers) {
                driver.setUp();
            }
        }

        @Override
        public void close() {
            for (ReferenceDriver driver : this.drivers) {
                try {
                    driver.tearDown();
                }
                catch (Exception e) {
                    TestJdbcVendorCompatibility.this.log.warn((Throwable)e, "Failed to close reference JDBC driver %s; continuing", new Object[]{driver});
                }
            }
        }
    }

    @FunctionalInterface
    private static interface ResultAssertion {
        public void accept(ResultSet var1, ResultSet var2, int var3) throws Exception;
    }

    private static interface Binder {
        public void bind(PreparedStatement var1, int var2) throws SQLException;
    }

    private static interface ReferenceDriver
    extends Closeable {
        public ResultSet query(String var1, Optional<String> var2) throws Exception;

        public boolean supports(JDBCType var1);

        public int expectedDeclaredJdbcType(JDBCType var1);

        public void setUp();

        public void tearDown() throws Exception;

        @Override
        public void close();
    }
}

