/*
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package io.trino.jdbc.\$internal.client.spooling;

import io.trino.jdbc.\$internal.jackson.annotation.JsonCreator;
import io.trino.jdbc.\$internal.jackson.annotation.JsonIgnore;
import io.trino.jdbc.\$internal.jackson.annotation.JsonInclude;
import io.trino.jdbc.\$internal.jackson.annotation.JsonProperty;
import io.trino.jdbc.\$internal.guava.collect.ImmutableList;
import io.trino.jdbc.\$internal.client.QueryData;
import io.trino.jdbc.\$internal.client.QueryDataDecoder;
import io.trino.jdbc.\$internal.client.RawQueryData;

import java.io.ByteArrayInputStream;
import java.io.IOException;
import java.io.InputStream;
import java.io.UncheckedIOException;
import java.util.List;
import java.util.Map;

import static io.trino.jdbc.\$internal.guava.base.MoreObjects.toStringHelper;
import static io.trino.jdbc.\$internal.guava.collect.Iterables.concat;
import static io.trino.jdbc.\$internal.guava.collect.Iterables.transform;
import static io.trino.jdbc.\$internal.guava.collect.Iterables.unmodifiableIterable;
import static java.lang.String.format;
import static java.util.Objects.requireNonNull;

public class EncodedQueryData
        implements QueryData
{
    private final String encodingId;
    private final DataAttributes metadata;
    private final List<Segment> segments;

    @JsonCreator
    public EncodedQueryData(
            @JsonProperty("encodingId") String encodingId,
            @JsonProperty("metadata") Map<String, Object> metadata,
            @JsonProperty("segments") List<Segment> segments)
    {
        this(encodingId, new DataAttributes(metadata), segments);
    }

    public EncodedQueryData(String encodingId, DataAttributes metadata, List<Segment> segments)
    {
        this.encodingId = requireNonNull(encodingId, "encodingId is null");
        this.metadata = requireNonNull(metadata, "metadata is null");
        this.segments = ImmutableList.copyOf(requireNonNull(segments, "segments is null"));
    }

    @JsonProperty("segments")
    public List<Segment> getSegments()
    {
        return segments;
    }

    @JsonProperty("encodingId")
    public String getEncodingId()
    {
        return encodingId;
    }

    @JsonInclude(JsonInclude.Include.NON_EMPTY)
    @JsonProperty("metadata")
    public Map<String, Object> getJsonMetadata()
    {
        return metadata.attributes;
    }

    @JsonIgnore
    public DataAttributes getMetadata()
    {
        return metadata;
    }

    @Override
    public Iterable<List<Object>> getData()
    {
        throw new UnsupportedOperationException("EncodedQueryData required decoding via matching QueryDataDecoder");
    }

    public QueryData toRawData(QueryDataDecoder decoder, SegmentLoader segmentLoader)
    {
        if (!decoder.encodingId().equals(encodingId)) {
            throw new IllegalArgumentException(format("Invalid decoder supplied, expected %s, got %s", encodingId, decoder.encodingId()));
        }

        return RawQueryData.of(unmodifiableIterable(concat(transform(segments, segment -> {
            if (segment instanceof InlineSegment) {
                InlineSegment inline = (InlineSegment) segment;
                try {
                    return decoder.decode(new ByteArrayInputStream(inline.getData()), inline.getMetadata());
                }
                catch (IOException e) {
                    throw new UncheckedIOException(e);
                }
            }

            if (segment instanceof SpooledSegment) {
                SpooledSegment spooled = (SpooledSegment) segment;
                try (InputStream stream = segmentLoader.load(spooled)) {
                    return decoder.decode(stream, segment.getMetadata());
                }
                catch (IOException e) {
                    throw new UncheckedIOException(e);
                }
            }

            throw new IllegalArgumentException("Unexpected segment type: " + segment.getClass().getSimpleName());
        }))));
    }

    @Override
    public String toString()
    {
        return toStringHelper(this)
                .add("encodingId", encodingId)
                .add("segments", segments)
                .add("metadata", metadata.attributes.keySet())
                .toString();
    }

    public static Builder builder(String format)
    {
        return new Builder(format);
    }

    public static class Builder
    {
        private final String encodingId;
        private final ImmutableList.Builder<Segment> segments = ImmutableList.builder();
        private DataAttributes metadata = DataAttributes.empty();

        private Builder(String encodingId)
        {
            this.encodingId = requireNonNull(encodingId, "encodingId is null");
        }

        public Builder withSegment(Segment segment)
        {
            this.segments.add(segment);
            return this;
        }

        public Builder withAttributes(DataAttributes attributes)
        {
            this.metadata = requireNonNull(attributes, "attributes is null");
            return this;
        }

        public EncodedQueryData build()
        {
            return new EncodedQueryData(encodingId, metadata, segments.build());
        }
    }
}
