/*
 * Decompiled with CFR 0.152.
 */
package io.trino.jdbc;

import com.google.common.base.Joiner;
import com.google.common.collect.ImmutableList;
import com.google.common.collect.ImmutableMap;
import com.google.common.collect.ImmutableSet;
import com.google.inject.Binder;
import com.google.inject.Module;
import com.google.inject.Scopes;
import com.google.inject.multibindings.Multibinder;
import io.airlift.concurrent.Threads;
import io.airlift.log.Logging;
import io.airlift.testing.Closeables;
import io.trino.client.ClientSelectedRole;
import io.trino.jdbc.TrinoConnection;
import io.trino.jdbc.TrinoResultSet;
import io.trino.metadata.MetadataUtil;
import io.trino.plugin.blackhole.BlackHolePlugin;
import io.trino.plugin.hive.HivePlugin;
import io.trino.server.testing.TestingTrinoServer;
import io.trino.spi.Plugin;
import io.trino.spi.connector.ConnectorSession;
import io.trino.spi.connector.ConnectorTableMetadata;
import io.trino.spi.connector.ConnectorTransactionHandle;
import io.trino.spi.connector.InMemoryRecordSet;
import io.trino.spi.connector.RecordCursor;
import io.trino.spi.connector.SchemaTableName;
import io.trino.spi.connector.SystemTable;
import io.trino.spi.predicate.TupleDomain;
import io.trino.spi.type.Type;
import io.trino.spi.type.VarcharType;
import io.trino.testing.assertions.Assert;
import java.io.Closeable;
import java.sql.Connection;
import java.sql.DriverManager;
import java.sql.PreparedStatement;
import java.sql.ResultSet;
import java.sql.SQLException;
import java.sql.SQLFeatureNotSupportedException;
import java.sql.Statement;
import java.util.List;
import java.util.Map;
import java.util.Optional;
import java.util.Set;
import java.util.UUID;
import java.util.concurrent.ExecutorService;
import java.util.concurrent.Executors;
import java.util.concurrent.Future;
import java.util.stream.IntStream;
import org.assertj.core.api.AbstractCollectionAssert;
import org.assertj.core.api.AbstractThrowableAssert;
import org.assertj.core.api.Assertions;
import org.assertj.core.api.Fail;
import org.assertj.core.api.ListAssert;
import org.junit.jupiter.api.AfterAll;
import org.junit.jupiter.api.BeforeAll;
import org.junit.jupiter.api.Disabled;
import org.junit.jupiter.api.Test;
import org.junit.jupiter.api.TestInstance;
import org.junit.jupiter.api.Timeout;
import org.junit.jupiter.api.parallel.Execution;
import org.junit.jupiter.api.parallel.ExecutionMode;

@TestInstance(value=TestInstance.Lifecycle.PER_CLASS)
@Execution(value=ExecutionMode.CONCURRENT)
public class TestJdbcConnection {
    private final ExecutorService executor = Executors.newCachedThreadPool(Threads.daemonThreadsNamed((String)this.getClass().getName()));
    private TestingTrinoServer server;

    @BeforeAll
    public void setupServer() throws Exception {
        Logging.initialize();
        Module systemTables = binder -> Multibinder.newSetBinder((Binder)binder, SystemTable.class).addBinding().to(ExtraCredentialsSystemTable.class).in(Scopes.SINGLETON);
        this.server = TestingTrinoServer.builder().setAdditionalModule(systemTables).build();
        this.server.installPlugin((Plugin)new HivePlugin());
        this.server.createCatalog("hive", "hive", (Map)ImmutableMap.builder().put((Object)"hive.metastore", (Object)"file").put((Object)"hive.metastore.catalog.dir", (Object)this.server.getBaseDataDir().resolve("hive").toAbsolutePath().toString()).put((Object)"hive.security", (Object)"sql-standard").put((Object)"fs.hadoop.enabled", (Object)"true").buildOrThrow());
        this.server.installPlugin((Plugin)new BlackHolePlugin());
        this.server.createCatalog("blackhole", "blackhole", (Map)ImmutableMap.of());
        try (Connection connection = this.createConnection();
             Statement statement = connection.createStatement();){
            statement.execute("SET ROLE admin IN hive");
            statement.execute("CREATE SCHEMA default");
            statement.execute("CREATE SCHEMA fruit");
            statement.execute("CREATE TABLE blackhole.default.devzero(dummy bigint) WITH (split_count = 100000, pages_per_split = 100000, rows_per_page = 10000)");
            statement.execute("CREATE TABLE blackhole.default.delay(dummy bigint) WITH (split_count = 1, pages_per_split = 1, rows_per_page = 1, page_processing_delay = '60s')");
        }
    }

    @AfterAll
    public void tearDown() throws Exception {
        Closeable[] closeableArray = new Closeable[2];
        closeableArray[0] = this.server;
        closeableArray[1] = this.executor::shutdownNow;
        Closeables.closeAll((Closeable[])closeableArray);
        this.server = null;
    }

    @Test
    public void testAutocommit() throws SQLException {
        try (Connection connection = this.createConnection();){
            Assertions.assertThat((boolean)connection.getAutoCommit()).isTrue();
            connection.setAutoCommit(false);
            Assertions.assertThat((boolean)connection.getAutoCommit()).isFalse();
            connection.setAutoCommit(true);
            Assertions.assertThat((boolean)connection.getAutoCommit()).isTrue();
        }
    }

    @Test
    public void testCommit() throws SQLException {
        try (Connection connection = this.createConnection();){
            connection.setAutoCommit(false);
            try (Statement statement = connection.createStatement();){
                statement.execute("CREATE TABLE test_commit (x bigint)");
            }
            try (Connection otherConnection = this.createConnection();){
                Assertions.assertThat(TestJdbcConnection.listTables(otherConnection)).doesNotContain((Object[])new String[]{"test_commit"});
            }
            connection.commit();
        }
        connection = this.createConnection();
        try {
            Assertions.assertThat(TestJdbcConnection.listTables(connection)).contains((Object[])new String[]{"test_commit"});
        }
        finally {
            if (connection != null) {
                connection.close();
            }
        }
    }

    @Test
    public void testImmediateCommit() throws SQLException {
        try (Connection connection = this.createConnection();){
            connection.setAutoCommit(false);
            connection.commit();
        }
    }

    @Test
    public void testRollback() throws SQLException {
        try (Connection connection = this.createConnection();){
            connection.setAutoCommit(false);
            try (Statement statement = connection.createStatement();){
                statement.execute("CREATE TABLE test_rollback (x bigint)");
            }
            try (Connection otherConnection = this.createConnection();){
                Assertions.assertThat(TestJdbcConnection.listTables(otherConnection)).doesNotContain((Object[])new String[]{"test_rollback"});
            }
            connection.rollback();
        }
        connection = this.createConnection();
        try {
            Assertions.assertThat(TestJdbcConnection.listTables(connection)).doesNotContain((Object[])new String[]{"test_rollback"});
        }
        finally {
            if (connection != null) {
                connection.close();
            }
        }
    }

    @Test
    public void testImmediateRollback() throws SQLException {
        try (Connection connection = this.createConnection();){
            connection.setAutoCommit(false);
            connection.rollback();
        }
    }

    @Test
    public void testUse() throws SQLException {
        try (Connection connection = this.createConnection();){
            Assertions.assertThat((String)connection.getCatalog()).isEqualTo("hive");
            Assertions.assertThat((String)connection.getSchema()).isEqualTo("default");
            try (Statement statement = connection.createStatement();){
                statement.execute("USE fruit");
            }
            Assertions.assertThat((String)connection.getCatalog()).isEqualTo("hive");
            Assertions.assertThat((String)connection.getSchema()).isEqualTo("fruit");
            statement = connection.createStatement();
            try {
                statement.execute("USE system.runtime");
            }
            finally {
                if (statement != null) {
                    statement.close();
                }
            }
            Assertions.assertThat((String)connection.getCatalog()).isEqualTo("system");
            Assertions.assertThat((String)connection.getSchema()).isEqualTo("runtime");
            statement = connection.createStatement();
            try {
                Assertions.assertThatThrownBy(() -> statement.execute("USE abc.xyz")).hasMessageEndingWith("Catalog 'abc' not found");
            }
            finally {
                if (statement != null) {
                    statement.close();
                }
            }
            statement = connection.createStatement();
            try {
                Assertions.assertThatThrownBy(() -> statement.execute("USE hive.xyz")).hasMessageEndingWith("Schema does not exist: hive.xyz");
            }
            finally {
                if (statement != null) {
                    statement.close();
                }
            }
            Assertions.assertThat((String)connection.getCatalog()).isEqualTo("system");
            Assertions.assertThat((String)connection.getSchema()).isEqualTo("runtime");
            Assertions.assertThat(TestJdbcConnection.listTables(connection)).contains((Object[])new String[]{"nodes"});
            Assertions.assertThat(TestJdbcConnection.listTables(connection)).contains((Object[])new String[]{"queries"});
            Assertions.assertThat(TestJdbcConnection.listTables(connection)).contains((Object[])new String[]{"tasks"});
        }
    }

    @Test
    public void testSession() throws SQLException {
        try (Connection connection = this.createConnection();){
            ((AbstractCollectionAssert)Assertions.assertThat(TestJdbcConnection.listSession(connection)).contains((Object[])new String[]{"join_distribution_type|AUTOMATIC|AUTOMATIC"})).contains((Object[])new String[]{"exchange_compression_codec|NONE|NONE"});
            try (Statement statement = connection.createStatement();){
                statement.execute("SET SESSION join_distribution_type = 'BROADCAST'");
            }
            ((AbstractCollectionAssert)Assertions.assertThat(TestJdbcConnection.listSession(connection)).contains((Object[])new String[]{"join_distribution_type|BROADCAST|AUTOMATIC"})).contains((Object[])new String[]{"exchange_compression_codec|NONE|NONE"});
            statement = connection.createStatement();
            try {
                statement.execute("SET SESSION exchange_compression_codec = 'LZ4'");
            }
            finally {
                if (statement != null) {
                    statement.close();
                }
            }
            ((AbstractCollectionAssert)Assertions.assertThat(TestJdbcConnection.listSession(connection)).contains((Object[])new String[]{"join_distribution_type|BROADCAST|AUTOMATIC"})).contains((Object[])new String[]{"exchange_compression_codec|LZ4|NONE"});
            statement = connection.createStatement();
            try {
                statement.execute("SET ROLE admin IN hive");
            }
            finally {
                if (statement != null) {
                    statement.close();
                }
            }
            for (String part : ImmutableList.of((Object)",", (Object)"=", (Object)":", (Object)"|", (Object)"/", (Object)"\\", (Object)"'", (Object)"\\'", (Object)"''", (Object)"\"", (Object)"\\\"", (Object)"[", (Object[])new String[]{"]"})) {
                String value = String.format("my-table-%s-name", part);
                try {
                    try (Statement statement = connection.createStatement();){
                        statement.execute(String.format("SET SESSION spatial_partitioning_table_name = '%s'", value.replace("'", "''")));
                    }
                    ((AbstractCollectionAssert)((AbstractCollectionAssert)Assertions.assertThat(TestJdbcConnection.listSession(connection)).contains((Object[])new String[]{"join_distribution_type|BROADCAST|AUTOMATIC"})).contains((Object[])new String[]{"exchange_compression_codec|LZ4|NONE"})).contains((Object[])new String[]{String.format("spatial_partitioning_table_name|%s|", value)});
                }
                catch (Exception e) {
                    Fail.fail((String)String.format("Failed to set session property value to [%s]", value), (Throwable)e);
                }
            }
        }
    }

    @Test
    public void testApplicationName() throws SQLException {
        try (Connection connection = this.createConnection();){
            TestJdbcConnection.assertConnectionSource(connection, "trino-jdbc");
        }
        connection = this.createConnection();
        try {
            connection.setClientInfo("ApplicationName", "testing");
            TestJdbcConnection.assertConnectionSource(connection, "testing");
        }
        finally {
            if (connection != null) {
                connection.close();
            }
        }
        connection = this.createConnection("applicationNamePrefix=fruit:");
        try {
            TestJdbcConnection.assertConnectionSource(connection, "fruit:");
        }
        finally {
            if (connection != null) {
                connection.close();
            }
        }
        connection = this.createConnection("applicationNamePrefix=fruit:");
        try {
            connection.setClientInfo("ApplicationName", "testing");
            TestJdbcConnection.assertConnectionSource(connection, "fruit:testing");
        }
        finally {
            if (connection != null) {
                connection.close();
            }
        }
    }

    @Test
    public void testSource() throws SQLException {
        try (Connection connection = this.createConnection("source=testing");){
            TestJdbcConnection.assertConnectionSource(connection, "testing");
        }
        connection = this.createConnection("source=testing&applicationNamePrefix=fruit:");
        try {
            TestJdbcConnection.assertConnectionSource(connection, "testing");
        }
        finally {
            if (connection != null) {
                connection.close();
            }
        }
        connection = this.createConnection("source=testing");
        try {
            connection.setClientInfo("ApplicationName", "testingApplicationName");
            TestJdbcConnection.assertConnectionSource(connection, "testing");
        }
        finally {
            if (connection != null) {
                connection.close();
            }
        }
        connection = this.createConnection("source=testing&applicationNamePrefix=fruit:");
        try {
            connection.setClientInfo("ApplicationName", "testingApplicationName");
            TestJdbcConnection.assertConnectionSource(connection, "testing");
        }
        finally {
            if (connection != null) {
                connection.close();
            }
        }
        connection = this.createConnection();
        try {
            TestJdbcConnection.assertConnectionSource(connection, "trino-jdbc");
        }
        finally {
            if (connection != null) {
                connection.close();
            }
        }
    }

    @Test
    public void testExtraCredentials() throws SQLException {
        try (Connection connection = this.createConnection("extraCredentials=test.token.foo:bar;test.token.abc:xyz;colon:-::-");){
            ImmutableMap expectedCredentials = ImmutableMap.builder().put((Object)"test.token.foo", (Object)"bar").put((Object)"test.token.abc", (Object)"xyz").put((Object)"colon", (Object)"-::-").buildOrThrow();
            TrinoConnection trinoConnection = connection.unwrap(TrinoConnection.class);
            Assertions.assertThat((Map)trinoConnection.getExtraCredentials()).isEqualTo((Object)expectedCredentials);
            Assertions.assertThat(TestJdbcConnection.listExtraCredentials(connection)).isEqualTo((Object)expectedCredentials);
        }
    }

    @Test
    public void testClientInfoParameter() throws SQLException {
        try (Connection connection = this.createConnection("clientInfo=hello%20world");){
            Assertions.assertThat((String)connection.getClientInfo("ClientInfo")).isEqualTo("hello world");
        }
    }

    @Test
    public void testClientTags() throws SQLException {
        try (Connection connection = this.createConnection("clientTags=c2,c3");){
            Assertions.assertThat((String)connection.getClientInfo("ClientTags")).isEqualTo("c2,c3");
        }
    }

    @Test
    public void testTraceToken() throws SQLException {
        try (Connection connection = this.createConnection("traceToken=trace%20me");){
            Assertions.assertThat((String)connection.getClientInfo("TraceToken")).isEqualTo("trace me");
        }
    }

    @Test
    public void testRole() throws SQLException {
        this.testRole("admin", new ClientSelectedRole(ClientSelectedRole.Type.ROLE, Optional.of("admin")), (ImmutableSet<String>)ImmutableSet.of((Object)"public", (Object)"admin"));
    }

    @Test
    public void testAllRole() throws SQLException {
        this.testRole("all", new ClientSelectedRole(ClientSelectedRole.Type.ALL, Optional.empty()), (ImmutableSet<String>)ImmutableSet.of((Object)"public"));
    }

    @Test
    public void testNoneRole() throws SQLException {
        this.testRole("none", new ClientSelectedRole(ClientSelectedRole.Type.NONE, Optional.empty()), (ImmutableSet<String>)ImmutableSet.of((Object)"public"));
    }

    private void testRole(String roleParameterValue, ClientSelectedRole clientSelectedRole, ImmutableSet<String> currentRoles) throws SQLException {
        try (Connection connection = this.createConnection("roles=hive:" + roleParameterValue);){
            TrinoConnection trinoConnection = connection.unwrap(TrinoConnection.class);
            Assertions.assertThat((Map)trinoConnection.getRoles()).isEqualTo((Object)ImmutableMap.of((Object)"hive", (Object)clientSelectedRole));
            Assertions.assertThat(TestJdbcConnection.listCurrentRoles(connection)).isEqualTo(currentRoles);
        }
    }

    @Test
    public void testSessionProperties() throws SQLException {
        try (Connection connection = this.createConnection("roles=hive:admin&sessionProperties=hive.hive_views_legacy_translation:true;execution_policy:all-at-once");){
            TrinoConnection trinoConnection = connection.unwrap(TrinoConnection.class);
            Assertions.assertThat((Map)trinoConnection.getSessionProperties()).extractingByKeys((Object[])new String[]{"hive.hive_views_legacy_translation", "execution_policy"}).containsExactly((Object[])new String[]{"true", "all-at-once"});
            Assertions.assertThat(TestJdbcConnection.listSession(connection)).containsAll((Iterable)ImmutableSet.of((Object)"execution_policy|all-at-once|phased", (Object)"hive.hive_views_legacy_translation|true|false"));
        }
    }

    @Test
    public void testSessionUser() throws SQLException {
        try (Connection connection = this.createConnection();){
            Assertions.assertThat((String)this.getSingleStringColumn(connection, "select current_user")).isEqualTo("admin");
            TrinoConnection trinoConnection = connection.unwrap(TrinoConnection.class);
            String impersonatedUser = "alice";
            trinoConnection.setSessionUser(impersonatedUser);
            Assertions.assertThat((String)this.getSingleStringColumn(connection, "select current_user")).isEqualTo(impersonatedUser);
            trinoConnection.clearSessionUser();
            Assertions.assertThat((String)this.getSingleStringColumn(connection, "select current_user")).isEqualTo("admin");
        }
    }

    @Test
    public void testPreparedStatementCreationOption() throws SQLException {
        String sql = "SELECT 123";
        try (Connection connection = this.createConnection();){
            try (PreparedStatement statement = connection.prepareStatement(sql, 2);){
                Assertions.assertThat((Object)statement).isNotNull();
            }
            ((AbstractThrowableAssert)Assertions.assertThatThrownBy(() -> connection.prepareStatement(sql, 1)).isInstanceOf(SQLFeatureNotSupportedException.class)).hasMessage("Auto generated keys must be NO_GENERATED_KEYS");
        }
    }

    @Test
    @Timeout(value=60L)
    @Disabled
    public void testConcurrentCancellationOnConnectionClose() throws Exception {
        this.testConcurrentCancellationOnConnectionClose(true);
        this.testConcurrentCancellationOnConnectionClose(false);
    }

    private void testConcurrentCancellationOnConnectionClose(boolean autoCommit) throws Exception {
        String sql = "SELECT * FROM blackhole.default.delay -- test cancellation " + UUID.randomUUID();
        Connection connection = this.createConnection();
        connection.setAutoCommit(autoCommit);
        List futures = (List)IntStream.range(0, 10).mapToObj(i -> this.executor.submit(() -> {
            try (Statement statement = connection.createStatement();
                 ResultSet resultSet = statement.executeQuery(sql);){
                while (resultSet.next()) {
                }
            }
            return null;
        })).collect(ImmutableList.toImmutableList());
        Assert.assertEventually(() -> {
            futures.forEach(TestJdbcConnection::assertThatFutureIsBlocked);
            Assertions.assertThat(this.listQueryStatuses(sql)).hasSize(futures.size());
        });
        connection.close();
        futures.forEach(future -> Assertions.assertThatThrownBy(future::get).isNotNull());
        ((ListAssert)Assertions.assertThat(this.listQueryErrorCodes(sql)).hasSize(futures.size())).allMatch(errorCode -> "TRANSACTION_ALREADY_ABORTED".equals(errorCode) || "USER_CANCELED".equals(errorCode));
    }

    private Connection createConnection() throws SQLException {
        return this.createConnection("");
    }

    private Connection createConnection(String extra) throws SQLException {
        String url = String.format("jdbc:trino://%s/hive/default?%s", this.server.getAddress(), extra);
        return DriverManager.getConnection(url, "admin", null);
    }

    private static Set<String> listTables(Connection connection) throws SQLException {
        ImmutableSet.Builder set = ImmutableSet.builder();
        try (Statement statement = connection.createStatement();
             ResultSet rs = statement.executeQuery("SHOW TABLES");){
            while (rs.next()) {
                set.add((Object)rs.getString(1));
            }
        }
        return set.build();
    }

    private static Set<String> listSession(Connection connection) throws SQLException {
        ImmutableSet.Builder set = ImmutableSet.builder();
        try (Statement statement = connection.createStatement();
             ResultSet rs = statement.executeQuery("SHOW SESSION");){
            while (rs.next()) {
                set.add((Object)Joiner.on((char)'|').join((Object)rs.getString(1), (Object)rs.getString(2), new Object[]{rs.getString(3)}));
            }
        }
        return set.build();
    }

    private static Map<String, String> listExtraCredentials(Connection connection) throws SQLException {
        ImmutableMap.Builder builder = ImmutableMap.builder();
        try (Statement statement = connection.createStatement();
             ResultSet rs = statement.executeQuery("SELECT * FROM system.test.extra_credentials");){
            while (rs.next()) {
                builder.put((Object)rs.getString("name"), (Object)rs.getString("value"));
            }
        }
        return builder.buildOrThrow();
    }

    private static Set<String> listCurrentRoles(Connection connection) throws SQLException {
        ImmutableSet.Builder builder = ImmutableSet.builder();
        try (Statement statement = connection.createStatement();
             ResultSet rs = statement.executeQuery("SHOW CURRENT ROLES IN hive");){
            while (rs.next()) {
                builder.add((Object)rs.getString("role"));
            }
        }
        return builder.build();
    }

    private List<String> listQueryStatuses(String sql) {
        return this.listSingleStringColumn(String.format("SELECT state FROM system.runtime.queries WHERE query = '%s'", sql));
    }

    private List<String> listQueryErrorCodes(String sql) {
        return this.listSingleStringColumn(String.format("SELECT error_code FROM system.runtime.queries WHERE query = '%s'", sql));
    }

    private List<String> listSingleStringColumn(String sql) {
        ImmutableList.Builder statuses = ImmutableList.builder();
        try (Connection connection = this.createConnection();
             Statement statement = connection.createStatement();
             ResultSet resultSet = statement.executeQuery(sql);){
            Assertions.assertThat((int)resultSet.getMetaData().getColumnCount()).isOne();
            Assertions.assertThat((int)resultSet.getMetaData().getColumnType(1)).isEqualTo(12);
            while (resultSet.next()) {
                statuses.add((Object)resultSet.getString(1));
            }
        }
        catch (SQLException e) {
            throw new RuntimeException(e);
        }
        return statuses.build();
    }

    private String getSingleStringColumn(Connection connection, String sql) throws SQLException {
        try (Statement statement = connection.createStatement();){
            String string;
            block12: {
                ResultSet resultSet = statement.executeQuery(sql);
                try {
                    Assertions.assertThat((int)resultSet.getMetaData().getColumnCount()).isOne();
                    Assertions.assertThat((boolean)resultSet.next()).isTrue();
                    String result = resultSet.getString(1);
                    Assertions.assertThat((boolean)resultSet.next()).isFalse();
                    string = result;
                    if (resultSet == null) break block12;
                }
                catch (Throwable throwable) {
                    if (resultSet != null) {
                        try {
                            resultSet.close();
                        }
                        catch (Throwable throwable2) {
                            throwable.addSuppressed(throwable2);
                        }
                    }
                    throw throwable;
                }
                resultSet.close();
            }
            return string;
        }
    }

    private static void assertConnectionSource(Connection connection, String expectedSource) throws SQLException {
        String queryId;
        ResultSet rs;
        try (Statement statement = connection.createStatement();){
            rs = statement.executeQuery("SELECT 123");
            try {
                queryId = rs.unwrap(TrinoResultSet.class).getQueryId();
            }
            finally {
                if (rs != null) {
                    rs.close();
                }
            }
        }
        statement = connection.prepareStatement("SELECT source FROM system.runtime.queries WHERE query_id = ?");
        try {
            statement.setString(1, queryId);
            rs = statement.executeQuery();
            try {
                Assertions.assertThat((boolean)rs.next()).isTrue();
                Assertions.assertThat((String)rs.getString("source")).isEqualTo(expectedSource);
                Assertions.assertThat((boolean)rs.next()).isFalse();
            }
            finally {
                if (rs != null) {
                    rs.close();
                }
            }
        }
        finally {
            if (statement != null) {
                statement.close();
            }
        }
    }

    static void assertThatFutureIsBlocked(Future<?> future) {
        if (!future.isDone()) {
            return;
        }
        try {
            future.get();
        }
        catch (Exception e) {
            throw new RuntimeException(e);
        }
        Fail.fail((String)"Expecting future to be blocked");
    }

    private static class ExtraCredentialsSystemTable
    implements SystemTable {
        private static final SchemaTableName NAME = new SchemaTableName("test", "extra_credentials");
        private static final ConnectorTableMetadata METADATA = MetadataUtil.TableMetadataBuilder.tableMetadataBuilder((SchemaTableName)NAME).column("name", (Type)VarcharType.createUnboundedVarcharType()).column("value", (Type)VarcharType.createUnboundedVarcharType()).build();

        private ExtraCredentialsSystemTable() {
        }

        public SystemTable.Distribution getDistribution() {
            return SystemTable.Distribution.ALL_NODES;
        }

        public ConnectorTableMetadata getTableMetadata() {
            return METADATA;
        }

        public RecordCursor cursor(ConnectorTransactionHandle transactionHandle, ConnectorSession session, TupleDomain<Integer> constraint) {
            InMemoryRecordSet.Builder table = InMemoryRecordSet.builder((ConnectorTableMetadata)METADATA);
            session.getIdentity().getExtraCredentials().forEach((xva$0, xva$1) -> table.addRow(new Object[]{xva$0, xva$1}));
            return table.build().cursor();
        }
    }
}

