/*
 * Decompiled with CFR 0.152.
 */
package io.trino.jdbc;

import com.google.common.collect.ImmutableList;
import com.google.common.collect.ImmutableMap;
import com.google.common.collect.Iterators;
import io.airlift.concurrent.Threads;
import io.trino.client.Warning;
import io.trino.execution.warnings.WarningCollectorConfig;
import io.trino.jdbc.TrinoSqlWarning;
import io.trino.plugin.blackhole.BlackHolePlugin;
import io.trino.server.testing.TestingTrinoServer;
import io.trino.spi.Plugin;
import io.trino.spi.TrinoWarning;
import io.trino.spi.WarningCode;
import io.trino.testing.TestingWarningCollector;
import io.trino.testing.TestingWarningCollectorConfig;
import java.sql.Connection;
import java.sql.DriverManager;
import java.sql.ResultSet;
import java.sql.SQLException;
import java.sql.SQLWarning;
import java.sql.Statement;
import java.util.HashSet;
import java.util.Iterator;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Set;
import java.util.concurrent.ExecutorService;
import java.util.concurrent.Executors;
import java.util.concurrent.Future;
import org.assertj.core.api.Assertions;
import org.junit.jupiter.api.AfterAll;
import org.junit.jupiter.api.BeforeAll;
import org.junit.jupiter.api.Test;
import org.junit.jupiter.api.TestInstance;
import org.junit.jupiter.api.parallel.Execution;
import org.junit.jupiter.api.parallel.ExecutionMode;

@TestInstance(value=TestInstance.Lifecycle.PER_CLASS)
@Execution(value=ExecutionMode.SAME_THREAD)
public class TestJdbcWarnings {
    private static final int PRELOADED_WARNINGS = 5;
    private TestingTrinoServer server;
    private final ExecutorService executor = Executors.newCachedThreadPool(Threads.daemonThreadsNamed((String)(this.getClass().getSimpleName() + "-%s")));

    @BeforeAll
    public void setupServer() throws Exception {
        this.server = TestingTrinoServer.builder().setProperties((Map)ImmutableMap.builder().put((Object)"testing-warning-collector.add-warnings", (Object)"true").put((Object)"testing-warning-collector.preloaded-warnings", (Object)String.valueOf(5)).buildOrThrow()).build();
        this.server.installPlugin((Plugin)new BlackHolePlugin());
        this.server.createCatalog("blackhole", "blackhole");
        try (Connection connection = this.createConnection();
             Statement statement = connection.createStatement();){
            statement.execute("CREATE SCHEMA blackhole.blackhole");
            statement.executeUpdate("CREATE TABLE slow_table (x int) WITH (   split_count = 1,    pages_per_split = 5,    rows_per_page = 3,    page_processing_delay = '1s')");
        }
    }

    @AfterAll
    public void tearDownServer() throws Exception {
        this.server.close();
        this.server = null;
        this.executor.shutdownNow();
    }

    @Test
    public void testStatementWarnings() throws SQLException {
        try (Connection connection = this.createConnection();
             Statement statement = connection.createStatement();){
            Assertions.assertThat((boolean)statement.execute("CREATE SCHEMA blackhole.test_schema")).isFalse();
            SQLWarning warning = statement.getWarnings();
            Assertions.assertThat((Object)warning).isNotNull();
            TestingWarningCollectorConfig warningCollectorConfig = new TestingWarningCollectorConfig().setPreloadedWarnings(5);
            TestingWarningCollector warningCollector = new TestingWarningCollector(new WarningCollectorConfig(), warningCollectorConfig);
            List expectedWarnings = warningCollector.getWarnings();
            TestJdbcWarnings.assertStartsWithExpectedWarnings(warning, TestJdbcWarnings.fromTrinoWarnings(expectedWarnings));
            statement.clearWarnings();
            Assertions.assertThat((Object)statement.getWarnings()).isNull();
        }
    }

    @Test
    public void testLongRunningStatement() throws Exception {
        try (Connection connection = this.createConnection();
             Statement statement = connection.createStatement();){
            Future<Object> future = this.executor.submit(() -> {
                statement.execute("CREATE TABLE test_long_running AS SELECT * FROM slow_table");
                return null;
            });
            TestJdbcWarnings.assertStatementWarnings(statement, future);
            statement.execute("DROP TABLE test_long_running");
        }
    }

    @Test
    public void testLongRunningQuery() throws Exception {
        try (Connection connection = this.createConnection();
             Statement statement = connection.createStatement();){
            Future<Object> future = this.executor.submit(() -> {
                ResultSet resultSet = statement.executeQuery("SELECT * FROM slow_table");
                while (resultSet.next()) {
                }
                return null;
            });
            TestJdbcWarnings.assertStatementWarnings(statement, future);
        }
    }

    @Test
    public void testExecuteQueryWarnings() throws SQLException {
        try (Connection connection = this.createConnection();
             Statement statement = connection.createStatement();
             ResultSet rs = statement.executeQuery("SELECT a FROM (VALUES 1, 2, 3) t(a)");){
            Assertions.assertThat((Object)statement.getConnection().getWarnings()).isNull();
            HashSet<WarningEntry> currentWarnings = new HashSet<WarningEntry>();
            TestJdbcWarnings.assertWarnings(rs.getWarnings(), currentWarnings);
            while (rs.next()) {
                TestJdbcWarnings.assertWarnings(statement.getWarnings(), currentWarnings);
            }
            TestingWarningCollectorConfig warningCollectorConfig = new TestingWarningCollectorConfig().setPreloadedWarnings(5).setAddWarnings(true);
            TestingWarningCollector warningCollector = new TestingWarningCollector(new WarningCollectorConfig(), warningCollectorConfig);
            List expectedWarnings = warningCollector.getWarnings();
            for (TrinoWarning trinoWarning : expectedWarnings) {
                Assertions.assertThat(currentWarnings).contains((Object[])new WarningEntry[]{new WarningEntry((Throwable)TestJdbcWarnings.toTrinoSqlWarning(trinoWarning))});
            }
        }
    }

    @Test
    public void testSqlWarning() throws SQLException {
        ImmutableList.Builder builder = ImmutableList.builder();
        for (int i = 0; i < 3; ++i) {
            builder.add((Object)new TrinoWarning(new WarningCode(i, "CODE_" + i), "warning message " + i));
        }
        ImmutableList warnings = builder.build();
        SQLWarning warning = TestJdbcWarnings.fromTrinoWarnings((List<TrinoWarning>)warnings);
        Assertions.assertThat((int)Iterators.size(warning.iterator())).isEqualTo(warnings.size());
        TestJdbcWarnings.assertWarningsEqual(warning, (SQLWarning)TestJdbcWarnings.toTrinoSqlWarning((TrinoWarning)warnings.get(0)));
        TestJdbcWarnings.assertWarningsEqual(warning.getNextWarning(), (SQLWarning)TestJdbcWarnings.toTrinoSqlWarning((TrinoWarning)warnings.get(1)));
        TestJdbcWarnings.assertWarningsEqual(warning.getNextWarning().getNextWarning(), (SQLWarning)TestJdbcWarnings.toTrinoSqlWarning((TrinoWarning)warnings.get(2)));
    }

    private static void assertStatementWarnings(Statement statement, Future<?> future) throws Exception {
        while (!future.isDone() && statement.getWarnings() == null) {
            Thread.sleep(100L);
        }
        HashSet<WarningEntry> warnings = new HashSet<WarningEntry>();
        SQLWarning warning = statement.getWarnings();
        Assertions.assertThat((boolean)warnings.add(new WarningEntry(warning))).isTrue();
        while (warning.getNextWarning() != null) {
            warning = warning.getNextWarning();
            Assertions.assertThat((boolean)warnings.add(new WarningEntry(warning))).isTrue();
        }
        int initialSize = warnings.size();
        Assertions.assertThat((int)initialSize).isGreaterThanOrEqualTo(6);
        while (!future.isDone()) {
            if (warning.getNextWarning() == null) {
                Thread.sleep(100L);
                continue;
            }
            warning = warning.getNextWarning();
            Assertions.assertThat((boolean)warnings.add(new WarningEntry(warning))).isTrue();
        }
        int finalSize = warnings.size();
        Assertions.assertThat((int)finalSize).isGreaterThan(initialSize);
        future.get();
    }

    private static SQLWarning fromTrinoWarnings(List<TrinoWarning> warnings) {
        TrinoSqlWarning first;
        Objects.requireNonNull(warnings, "warnings is null");
        Assertions.assertThat(warnings).isNotEmpty();
        Iterator<TrinoWarning> iterator = warnings.iterator();
        Object current = first = TestJdbcWarnings.toTrinoSqlWarning(iterator.next());
        while (iterator.hasNext()) {
            current.setNextWarning((SQLWarning)TestJdbcWarnings.toTrinoSqlWarning(iterator.next()));
            current = current.getNextWarning();
        }
        return first;
    }

    private static TrinoSqlWarning toTrinoSqlWarning(TrinoWarning warning) {
        return new TrinoSqlWarning(TestJdbcWarnings.toClientWarning(warning));
    }

    private static Warning toClientWarning(TrinoWarning warning) {
        WarningCode code = warning.getWarningCode();
        return new Warning(new Warning.Code(code.getCode(), code.getName()), warning.getMessage());
    }

    private static void assertWarningsEqual(SQLWarning actual, SQLWarning expected) {
        Assertions.assertThat((String)actual.getMessage()).isEqualTo(expected.getMessage());
        Assertions.assertThat((String)actual.getSQLState()).isEqualTo(expected.getSQLState());
        Assertions.assertThat((int)actual.getErrorCode()).isEqualTo(expected.getErrorCode());
    }

    private static void addWarnings(Set<WarningEntry> currentWarnings, SQLWarning newWarning) {
        if (newWarning == null) {
            return;
        }
        for (Throwable warning : newWarning) {
            WarningEntry entry = new WarningEntry(warning);
            currentWarnings.add(entry);
        }
    }

    private Connection createConnection() throws SQLException {
        String url = String.format("jdbc:trino://%s/blackhole/blackhole", this.server.getAddress());
        return DriverManager.getConnection(url, "test", null);
    }

    private static void assertWarnings(SQLWarning warning, Set<WarningEntry> currentWarnings) {
        if (warning == null) {
            return;
        }
        int previousSize = currentWarnings.size();
        TestJdbcWarnings.addWarnings(currentWarnings, warning);
        Assertions.assertThat((currentWarnings.size() >= previousSize ? 1 : 0) != 0).isTrue();
    }

    private static void assertStartsWithExpectedWarnings(SQLWarning warning, SQLWarning expected) {
        Assertions.assertThat((Object)expected).isNotNull();
        Assertions.assertThat((Object)warning).isNotNull();
        while (true) {
            TestJdbcWarnings.assertWarningsEqual(warning, expected);
            warning = warning.getNextWarning();
            expected = expected.getNextWarning();
            if (expected == null) {
                return;
            }
            Assertions.assertThat((Object)warning).isNotNull();
        }
    }

    private static class WarningEntry {
        public final int vendorCode;
        public final String sqlState;
        public final String message;

        public WarningEntry(Throwable throwable) {
            Objects.requireNonNull(throwable, "throwable is null");
            Assertions.assertThat((boolean)(throwable instanceof SQLWarning)).isTrue();
            SQLWarning warning = (SQLWarning)throwable;
            this.vendorCode = warning.getErrorCode();
            this.sqlState = Objects.requireNonNull(warning.getSQLState(), "SQLState is null");
            this.message = Objects.requireNonNull(warning.getMessage(), "message is null");
        }

        public boolean equals(Object other) {
            if (this == other) {
                return true;
            }
            if (!(other instanceof WarningEntry)) {
                return false;
            }
            WarningEntry that = (WarningEntry)other;
            return this.vendorCode == that.vendorCode && this.sqlState.equals(that.sqlState) && this.message.equals(that.message);
        }

        public int hashCode() {
            return Objects.hash(this.vendorCode, this.sqlState, this.message);
        }
    }
}

