/*
 * Decompiled with CFR 0.152.
 */
package io.trino.jdbc;

import com.google.common.collect.ImmutableMap;
import com.google.common.io.Files;
import com.google.common.io.Resources;
import io.airlift.log.Logging;
import io.airlift.security.pem.PemReader;
import io.jsonwebtoken.JwtBuilder;
import io.jsonwebtoken.Jwts;
import io.jsonwebtoken.security.Keys;
import io.jsonwebtoken.security.SecretKeyBuilder;
import io.trino.plugin.tpch.TpchPlugin;
import io.trino.server.security.jwt.JwtUtil;
import io.trino.server.testing.TestingTrinoServer;
import io.trino.spi.Plugin;
import java.io.File;
import java.net.URL;
import java.nio.charset.Charset;
import java.nio.charset.StandardCharsets;
import java.security.Key;
import java.security.PrivateKey;
import java.sql.Connection;
import java.sql.DriverManager;
import java.sql.ResultSet;
import java.sql.SQLException;
import java.sql.Statement;
import java.util.Base64;
import java.util.Map;
import java.util.Optional;
import java.util.Properties;
import javax.crypto.SecretKey;
import org.assertj.core.api.AbstractThrowableAssert;
import org.assertj.core.api.AbstractUrlAssert;
import org.assertj.core.api.Assertions;
import org.junit.jupiter.api.AfterAll;
import org.junit.jupiter.api.BeforeAll;
import org.junit.jupiter.api.Test;
import org.junit.jupiter.api.TestInstance;
import org.junit.jupiter.api.parallel.Execution;
import org.junit.jupiter.api.parallel.ExecutionMode;

@TestInstance(value=TestInstance.Lifecycle.PER_CLASS)
@Execution(value=ExecutionMode.CONCURRENT)
public class TestTrinoDriverAuth {
    private static final String TEST_CATALOG = "test_catalog";
    private TestingTrinoServer server;
    private Key defaultKey;
    private Key hmac222;
    private PrivateKey privateKey33;

    @BeforeAll
    public void setup() throws Exception {
        Logging.initialize();
        URL resource = this.getClass().getClassLoader().getResource("33.privateKey");
        ((AbstractUrlAssert)Assertions.assertThat((URL)resource).describedAs("key directory not found", new Object[0])).isNotNull();
        File keyDir = new File(resource.toURI()).getAbsoluteFile().getParentFile();
        this.defaultKey = Keys.hmacShaKeyFor((byte[])Base64.getMimeDecoder().decode(Files.asCharSource((File)new File(keyDir, "default-key.key"), (Charset)StandardCharsets.US_ASCII).read().getBytes(StandardCharsets.US_ASCII)));
        this.hmac222 = Keys.hmacShaKeyFor((byte[])Base64.getMimeDecoder().decode(Files.asCharSource((File)new File(keyDir, "222.key"), (Charset)StandardCharsets.US_ASCII).read().getBytes(StandardCharsets.US_ASCII)));
        this.privateKey33 = PemReader.loadPrivateKey((File)new File(keyDir, "33.privateKey"), Optional.empty());
        this.server = TestingTrinoServer.builder().setProperties((Map)ImmutableMap.builder().put((Object)"http-server.authentication.type", (Object)"JWT").put((Object)"http-server.authentication.jwt.key-file", (Object)new File(keyDir, "${KID}.key").getPath()).put((Object)"http-server.https.enabled", (Object)"true").put((Object)"http-server.https.keystore.path", (Object)new File(Resources.getResource((String)"localhost.keystore").toURI()).getPath()).put((Object)"http-server.https.keystore.key", (Object)"changeit").buildOrThrow()).build();
        this.server.installPlugin((Plugin)new TpchPlugin());
        this.server.createCatalog(TEST_CATALOG, "tpch");
    }

    @AfterAll
    public void teardown() throws Exception {
        this.server.close();
        this.server = null;
    }

    @Test
    public void testSuccessDefaultKey() throws Exception {
        String accessToken = JwtUtil.newJwtBuilder().subject("test").signWith(this.defaultKey).compact();
        try (Connection connection = this.createConnection((Map<String, String>)ImmutableMap.of((Object)"accessToken", (Object)accessToken));
             Statement statement = connection.createStatement();){
            Assertions.assertThat((boolean)statement.execute("SELECT 123")).isTrue();
            ResultSet rs = statement.getResultSet();
            Assertions.assertThat((boolean)rs.next()).isTrue();
            Assertions.assertThat((long)rs.getLong(1)).isEqualTo(123L);
            Assertions.assertThat((boolean)rs.next()).isFalse();
        }
    }

    @Test
    public void testSuccessHmac() throws Exception {
        String accessToken = ((JwtBuilder)((JwtBuilder.BuilderHeader)JwtUtil.newJwtBuilder().subject("test").header().keyId("222")).and()).signWith(this.hmac222).compact();
        try (Connection connection = this.createConnection((Map<String, String>)ImmutableMap.of((Object)"accessToken", (Object)accessToken));
             Statement statement = connection.createStatement();){
            Assertions.assertThat((boolean)statement.execute("SELECT 123")).isTrue();
            ResultSet rs = statement.getResultSet();
            Assertions.assertThat((boolean)rs.next()).isTrue();
            Assertions.assertThat((long)rs.getLong(1)).isEqualTo(123L);
            Assertions.assertThat((boolean)rs.next()).isFalse();
        }
    }

    @Test
    public void testSuccessPublicKey() throws Exception {
        String accessToken = ((JwtBuilder)((JwtBuilder.BuilderHeader)JwtUtil.newJwtBuilder().subject("test").header().keyId("33")).and()).signWith((Key)this.privateKey33).compact();
        try (Connection connection = this.createConnection((Map<String, String>)ImmutableMap.of((Object)"accessToken", (Object)accessToken));
             Statement statement = connection.createStatement();){
            Assertions.assertThat((boolean)statement.execute("SELECT 123")).isTrue();
            ResultSet rs = statement.getResultSet();
            Assertions.assertThat((boolean)rs.next()).isTrue();
            Assertions.assertThat((long)rs.getLong(1)).isEqualTo(123L);
            Assertions.assertThat((boolean)rs.next()).isFalse();
        }
    }

    @Test
    public void testFailedNoToken() {
        ((AbstractThrowableAssert)Assertions.assertThatThrownBy(() -> {
            try (Connection connection = this.createConnection((Map<String, String>)ImmutableMap.of());
                 Statement statement = connection.createStatement();){
                statement.execute("SELECT 123");
            }
        }).isInstanceOf(SQLException.class)).hasMessage("Authentication failed: Unauthorized");
    }

    @Test
    public void testFailedUnsigned() throws Exception {
        String accessToken = JwtUtil.newJwtBuilder().subject("test").compact();
        try (Connection connection = this.createConnection((Map<String, String>)ImmutableMap.of((Object)"accessToken", (Object)accessToken));
             Statement statement = connection.createStatement();){
            ((AbstractThrowableAssert)Assertions.assertThatThrownBy(() -> statement.execute("SELECT 123")).isInstanceOf(SQLException.class)).hasMessageContaining("Authentication failed: Unsecured JWSs (those with an 'alg' (Algorithm) header value of 'none') are disallowed by default");
        }
    }

    @Test
    public void testFailedBadHmacSignature() throws Exception {
        SecretKey badKey = (SecretKey)((SecretKeyBuilder)Jwts.SIG.HS512.key()).build();
        String accessToken = JwtUtil.newJwtBuilder().subject("test").signWith((Key)badKey).compact();
        try (Connection connection = this.createConnection((Map<String, String>)ImmutableMap.of((Object)"accessToken", (Object)accessToken));
             Statement statement = connection.createStatement();){
            ((AbstractThrowableAssert)Assertions.assertThatThrownBy(() -> statement.execute("SELECT 123")).isInstanceOf(SQLException.class)).hasMessageContaining("Authentication failed: JWT signature does not match locally computed signature. JWT validity cannot be asserted and should not be trusted.");
        }
    }

    @Test
    public void testFailedWrongPublicKey() {
        ((AbstractThrowableAssert)Assertions.assertThatThrownBy(() -> {
            String accessToken = ((JwtBuilder)((JwtBuilder.BuilderHeader)JwtUtil.newJwtBuilder().subject("test").header().keyId("42")).and()).signWith((Key)this.privateKey33).compact();
            try (Connection connection = this.createConnection((Map<String, String>)ImmutableMap.of((Object)"accessToken", (Object)accessToken));
                 Statement statement = connection.createStatement();){
                statement.execute("SELECT 123");
            }
        }).isInstanceOf(SQLException.class)).hasMessageMatching("Authentication failed: JWT signature does not match.*");
    }

    @Test
    public void testFailedUnknownPublicKey() {
        ((AbstractThrowableAssert)Assertions.assertThatThrownBy(() -> {
            String accessToken = ((JwtBuilder)((JwtBuilder.BuilderHeader)JwtUtil.newJwtBuilder().subject("test").header().keyId("unknown")).and()).signWith((Key)this.privateKey33).compact();
            try (Connection connection = this.createConnection((Map<String, String>)ImmutableMap.of((Object)"accessToken", (Object)accessToken));
                 Statement statement = connection.createStatement();){
                statement.execute("SELECT 123");
            }
        }).isInstanceOf(SQLException.class)).hasMessage("Authentication failed: Unknown signing key ID");
    }

    @Test
    public void testSuccessFullSslVerification() throws Exception {
        String accessToken = ((JwtBuilder)((JwtBuilder.BuilderHeader)JwtUtil.newJwtBuilder().subject("test").header().keyId("33")).and()).signWith((Key)this.privateKey33).compact();
        try (Connection connection = this.createConnection((Map<String, String>)ImmutableMap.of((Object)"accessToken", (Object)accessToken, (Object)"SSLVerification", (Object)"FULL"));
             Statement statement = connection.createStatement();){
            Assertions.assertThat((boolean)statement.execute("SELECT 123")).isTrue();
            ResultSet rs = statement.getResultSet();
            Assertions.assertThat((boolean)rs.next()).isTrue();
            Assertions.assertThat((long)rs.getLong(1)).isEqualTo(123L);
            Assertions.assertThat((boolean)rs.next()).isFalse();
        }
    }

    @Test
    public void testSuccessFullSslVerificationAlternateHostname() throws Exception {
        String accessToken = ((JwtBuilder)((JwtBuilder.BuilderHeader)JwtUtil.newJwtBuilder().subject("test").header().keyId("33")).and()).signWith((Key)this.privateKey33).compact();
        String url = String.format("jdbc:trino://127.0.0.1:%s", this.server.getHttpsAddress().getPort());
        Properties properties = new Properties();
        properties.setProperty("SSL", "true");
        properties.setProperty("SSLVerification", "FULL");
        properties.setProperty("SSLTrustStorePath", new File(Resources.getResource((String)"localhost.truststore").toURI()).getPath());
        properties.setProperty("SSLTrustStorePassword", "changeit");
        properties.setProperty("accessToken", accessToken);
        properties.setProperty("hostnameInCertificate", "localhost");
        try (Connection connection = DriverManager.getConnection(url, properties);
             Statement statement = connection.createStatement();){
            Assertions.assertThat((boolean)statement.execute("SELECT 123")).isTrue();
            ResultSet rs = statement.getResultSet();
            Assertions.assertThat((boolean)rs.next()).isTrue();
            Assertions.assertThat((long)rs.getLong(1)).isEqualTo(123L);
            Assertions.assertThat((boolean)rs.next()).isFalse();
        }
    }

    @Test
    public void testFailedFullSslVerificationAlternateHostnameNotProvided() throws Exception {
        String accessToken = ((JwtBuilder)((JwtBuilder.BuilderHeader)JwtUtil.newJwtBuilder().subject("test").header().keyId("33")).and()).signWith((Key)this.privateKey33).compact();
        String url = String.format("jdbc:trino://127.0.0.1:%s", this.server.getHttpsAddress().getPort());
        Properties properties = new Properties();
        properties.setProperty("SSL", "true");
        properties.setProperty("SSLVerification", "FULL");
        properties.setProperty("SSLTrustStorePath", new File(Resources.getResource((String)"localhost.truststore").toURI()).getPath());
        properties.setProperty("SSLTrustStorePassword", "changeit");
        properties.setProperty("accessToken", accessToken);
        try (Connection connection = DriverManager.getConnection(url, properties);
             Statement statement = connection.createStatement();){
            ((AbstractThrowableAssert)Assertions.assertThatThrownBy(() -> statement.execute("SELECT 123")).isInstanceOf(SQLException.class)).hasMessageContaining("Error executing query: javax.net.ssl.SSLPeerUnverifiedException: Hostname 127.0.0.1 not verified");
        }
    }

    @Test
    public void testFailedCaSslVerificationAlternateHostname() {
        String accessToken = ((JwtBuilder)((JwtBuilder.BuilderHeader)JwtUtil.newJwtBuilder().subject("test").header().keyId("33")).and()).signWith((Key)this.privateKey33).compact();
        String url = String.format("jdbc:trino://127.0.0.1:%s", this.server.getHttpsAddress().getPort());
        Properties properties = new Properties();
        properties.setProperty("SSL", "true");
        properties.setProperty("SSLVerification", "CA");
        properties.setProperty("accessToken", accessToken);
        properties.setProperty("hostnameInCertificate", "localhost");
        ((AbstractThrowableAssert)Assertions.assertThatThrownBy(() -> DriverManager.getConnection(url, properties)).isInstanceOf(SQLException.class)).hasMessage("Connection property hostnameInCertificate requires SSLVerification to be set to FULL");
    }

    @Test
    public void testFailedNoneSslVerificationAlternateHostname() {
        String accessToken = ((JwtBuilder)((JwtBuilder.BuilderHeader)JwtUtil.newJwtBuilder().subject("test").header().keyId("33")).and()).signWith((Key)this.privateKey33).compact();
        String url = String.format("jdbc:trino://127.0.0.1:%s", this.server.getHttpsAddress().getPort());
        Properties properties = new Properties();
        properties.setProperty("SSL", "true");
        properties.setProperty("SSLVerification", "NONE");
        properties.setProperty("accessToken", accessToken);
        properties.setProperty("hostnameInCertificate", "localhost");
        ((AbstractThrowableAssert)Assertions.assertThatThrownBy(() -> DriverManager.getConnection(url, properties)).isInstanceOf(SQLException.class)).hasMessage("Connection property hostnameInCertificate requires SSLVerification to be set to FULL");
    }

    @Test
    public void testSuccessCaSslVerification() throws Exception {
        String accessToken = ((JwtBuilder)((JwtBuilder.BuilderHeader)JwtUtil.newJwtBuilder().subject("test").header().keyId("33")).and()).signWith((Key)this.privateKey33).compact();
        try (Connection connection = this.createConnection((Map<String, String>)ImmutableMap.of((Object)"accessToken", (Object)accessToken, (Object)"SSLVerification", (Object)"CA"));
             Statement statement = connection.createStatement();){
            Assertions.assertThat((boolean)statement.execute("SELECT 123")).isTrue();
            ResultSet rs = statement.getResultSet();
            Assertions.assertThat((boolean)rs.next()).isTrue();
            Assertions.assertThat((long)rs.getLong(1)).isEqualTo(123L);
            Assertions.assertThat((boolean)rs.next()).isFalse();
        }
    }

    @Test
    public void testFailedFullSslVerificationWithoutSSL() {
        ((AbstractThrowableAssert)Assertions.assertThatThrownBy(() -> this.createBasicConnection((Map<String, String>)ImmutableMap.of((Object)"SSLVerification", (Object)"FULL"))).isInstanceOf(SQLException.class)).hasMessage("Connection property SSLVerification requires TLS/SSL to be enabled");
    }

    @Test
    public void testFailedCaSslVerificationWithoutSSL() {
        ((AbstractThrowableAssert)Assertions.assertThatThrownBy(() -> this.createBasicConnection((Map<String, String>)ImmutableMap.of((Object)"SSLVerification", (Object)"CA"))).isInstanceOf(SQLException.class)).hasMessage("Connection property SSLVerification requires TLS/SSL to be enabled");
    }

    @Test
    public void testFailedNoneSslVerificationWithSSL() {
        ((AbstractThrowableAssert)Assertions.assertThatThrownBy(() -> this.createConnection((Map<String, String>)ImmutableMap.of((Object)"SSLVerification", (Object)"NONE"))).isInstanceOf(SQLException.class)).hasMessageContaining("Connection property SSLTrustStorePath cannot be set if SSLVerification is set to NONE");
    }

    @Test
    public void testFailedNoneSslVerificationWithSSLUnsigned() throws Exception {
        try (Connection connection = this.createBasicConnection((Map<String, String>)ImmutableMap.of((Object)"SSL", (Object)"true", (Object)"SSLVerification", (Object)"NONE"));
             Statement statement = connection.createStatement();){
            ((AbstractThrowableAssert)Assertions.assertThatThrownBy(() -> statement.execute("SELECT 123")).isInstanceOf(SQLException.class)).hasMessage("Authentication failed: Unauthorized");
        }
    }

    private Connection createConnection(Map<String, String> additionalProperties) throws Exception {
        String url = String.format("jdbc:trino://localhost:%s", this.server.getHttpsAddress().getPort());
        Properties properties = new Properties();
        properties.setProperty("SSL", "true");
        properties.setProperty("SSLTrustStorePath", new File(Resources.getResource((String)"localhost.truststore").toURI()).getPath());
        properties.setProperty("SSLTrustStorePassword", "changeit");
        additionalProperties.forEach(properties::setProperty);
        return DriverManager.getConnection(url, properties);
    }

    private Connection createBasicConnection(Map<String, String> additionalProperties) throws SQLException {
        String url = String.format("jdbc:trino://localhost:%s", this.server.getHttpsAddress().getPort());
        Properties properties = new Properties();
        additionalProperties.forEach(properties::setProperty);
        return DriverManager.getConnection(url, properties);
    }
}

