/*
 * Decompiled with CFR 0.152.
 */
package io.trino.plugin.kafka.encoder.raw;

import com.google.common.base.Joiner;
import com.google.common.base.Preconditions;
import com.google.common.collect.ImmutableList;
import com.google.common.collect.ImmutableSet;
import io.trino.plugin.base.io.ByteBuffers;
import io.trino.plugin.kafka.encoder.AbstractRowEncoder;
import io.trino.plugin.kafka.encoder.EncoderColumnHandle;
import io.trino.spi.connector.ConnectorSession;
import io.trino.spi.type.BigintType;
import io.trino.spi.type.BooleanType;
import io.trino.spi.type.DoubleType;
import io.trino.spi.type.IntegerType;
import io.trino.spi.type.RealType;
import io.trino.spi.type.SmallintType;
import io.trino.spi.type.TinyintType;
import io.trino.spi.type.Type;
import io.trino.spi.type.VarcharType;
import java.nio.ByteBuffer;
import java.nio.charset.StandardCharsets;
import java.util.Arrays;
import java.util.List;
import java.util.Locale;
import java.util.Optional;
import java.util.Set;
import java.util.regex.Matcher;
import java.util.regex.Pattern;

public class RawRowEncoder
extends AbstractRowEncoder {
    private static final Pattern MAPPING_PATTERN = Pattern.compile("(\\d+)(?::(\\d+))?");
    private static final Set<Type> SUPPORTED_PRIMITIVE_TYPES = ImmutableSet.of((Object)BigintType.BIGINT, (Object)IntegerType.INTEGER, (Object)SmallintType.SMALLINT, (Object)TinyintType.TINYINT, (Object)DoubleType.DOUBLE, (Object)RealType.REAL, (Object[])new Type[]{BooleanType.BOOLEAN});
    public static final String NAME = "raw";
    private final List<ColumnMapping> columnMappings;
    private final ByteBuffer buffer;

    public RawRowEncoder(ConnectorSession session, List<EncoderColumnHandle> columnHandles) {
        super(session, columnHandles);
        for (EncoderColumnHandle handle : this.columnHandles) {
            Preconditions.checkArgument((boolean)RawRowEncoder.isSupportedType(handle.getType()), (String)"Unsupported column type '%s' for column '%s'", (Object)handle.getType().getDisplayName(), (Object)handle.getName());
            Preconditions.checkArgument((handle.getFormatHint() == null ? 1 : 0) != 0, (String)"Unexpected format hint '%s' defined for column '%s'", (Object)handle.getFormatHint(), (Object)handle.getName());
        }
        this.columnMappings = (List)this.columnHandles.stream().map(ColumnMapping::new).collect(ImmutableList.toImmutableList());
        for (ColumnMapping mapping : this.columnMappings) {
            if (mapping.getLength() == mapping.getFieldType().getSize() || mapping.getType() instanceof VarcharType) continue;
            throw new IndexOutOfBoundsException(String.format("Mapping length '%s' is not equal to expected length '%s' for column '%s'", mapping.getLength(), mapping.getFieldType().getSize(), mapping.getName()));
        }
        int position = 0;
        for (ColumnMapping mapping : this.columnMappings) {
            Preconditions.checkArgument((mapping.getStart() == position ? 1 : 0) != 0, (Object)String.format("Start mapping '%s' for column '%s' does not equal expected mapping '%s'", mapping.getStart(), mapping.getName(), position));
            Preconditions.checkArgument((mapping.getEnd() > mapping.getStart() ? 1 : 0) != 0, (Object)String.format("End mapping '%s' for column '%s' is less than or equal to start '%s'", mapping.getEnd(), mapping.getName(), mapping.getStart()));
            position += mapping.getLength();
        }
        this.buffer = ByteBuffer.allocate(position);
    }

    private static boolean isSupportedType(Type type) {
        return type instanceof VarcharType || SUPPORTED_PRIMITIVE_TYPES.contains(type);
    }

    @Override
    protected void appendLong(long value) {
        this.buffer.putLong(value);
    }

    @Override
    protected void appendInt(int value) {
        this.buffer.putInt(value);
    }

    @Override
    protected void appendShort(short value) {
        this.buffer.putShort(value);
    }

    @Override
    protected void appendByte(byte value) {
        this.buffer.put(value);
    }

    @Override
    protected void appendDouble(double value) {
        this.buffer.putDouble(value);
    }

    @Override
    protected void appendFloat(float value) {
        this.buffer.putFloat(value);
    }

    @Override
    protected void appendBoolean(boolean value) {
        this.buffer.put((byte)(value ? 1 : 0));
    }

    @Override
    protected void appendString(String value) {
        byte[] valueBytes = value.getBytes(StandardCharsets.UTF_8);
        Preconditions.checkArgument((valueBytes.length == this.columnMappings.get(this.currentColumnIndex).getLength() ? 1 : 0) != 0, (Object)String.format("length '%s' of message '%s' for column '%s' does not equal expected length '%s'", valueBytes.length, value, ((EncoderColumnHandle)this.columnHandles.get(this.currentColumnIndex)).getName(), this.columnMappings.get(this.currentColumnIndex).getLength()));
        this.buffer.put(valueBytes, 0, valueBytes.length);
    }

    @Override
    protected void appendByteBuffer(ByteBuffer value) {
        byte[] valueBytes = ByteBuffers.getWrappedBytes((ByteBuffer)value);
        Preconditions.checkArgument((valueBytes.length == this.columnMappings.get(this.currentColumnIndex).getLength() ? 1 : 0) != 0, (Object)String.format("length '%s' of message for column '%s' does not equal expected length '%s'", valueBytes.length, ((EncoderColumnHandle)this.columnHandles.get(this.currentColumnIndex)).getName(), this.columnMappings.get(this.currentColumnIndex).getLength()));
        this.buffer.put(valueBytes, 0, valueBytes.length);
    }

    @Override
    public byte[] toByteArray() {
        Preconditions.checkArgument((this.currentColumnIndex == this.columnHandles.size() ? 1 : 0) != 0, (Object)String.format("Missing %d columns", this.columnHandles.size() - this.currentColumnIndex + 1));
        this.resetColumnIndex();
        this.buffer.clear();
        return this.buffer.array();
    }

    private static class ColumnMapping {
        private final String name;
        private final Type type;
        private final FieldType fieldType;
        private final int start;
        private final int end;

        public ColumnMapping(EncoderColumnHandle columnHandle) {
            this.name = columnHandle.getName();
            this.type = columnHandle.getType();
            this.fieldType = ColumnMapping.parseFieldType(columnHandle.getDataFormat(), this.name);
            ColumnMapping.checkFieldType(this.name, this.type, this.fieldType);
            Optional<String> mapping = Optional.ofNullable(columnHandle.getMapping());
            if (mapping.isPresent()) {
                Matcher mappingMatcher = MAPPING_PATTERN.matcher(mapping.get());
                if (!mappingMatcher.matches()) {
                    throw new IllegalArgumentException(String.format("Invalid mapping for column '%s'", this.name));
                }
                if (mappingMatcher.group(2) != null) {
                    this.start = ColumnMapping.parseOffset(mappingMatcher.group(1), "start", this.name);
                    this.end = ColumnMapping.parseOffset(mappingMatcher.group(2), "end", this.name);
                } else {
                    this.start = ColumnMapping.parseOffset(mappingMatcher.group(1), "start", this.name);
                    this.end = this.start + this.fieldType.getSize();
                }
            } else {
                throw new IllegalArgumentException(String.format("No mapping defined for column '%s'", this.name));
            }
        }

        private static int parseOffset(String group, String offsetName, String columnName) {
            try {
                return Integer.parseInt(group);
            }
            catch (NumberFormatException e) {
                throw new IllegalArgumentException(String.format("Unable to parse '%s' offset for column '%s'", offsetName, columnName), e);
            }
        }

        private static FieldType parseFieldType(String dataFormat, String columnName) {
            try {
                if (dataFormat != null && !dataFormat.equals("")) {
                    return FieldType.valueOf(dataFormat.toUpperCase(Locale.ENGLISH));
                }
                return FieldType.BYTE;
            }
            catch (IllegalArgumentException e) {
                throw new IllegalArgumentException(String.format("Invalid dataFormat '%s' for column '%s'", dataFormat, columnName));
            }
        }

        private static void checkFieldType(String columnName, Type columnType, FieldType fieldType) {
            if (columnType == BigintType.BIGINT) {
                ColumnMapping.checkFieldTypeOneOf(fieldType, columnName, columnType, FieldType.BYTE, FieldType.SHORT, FieldType.INT, FieldType.LONG);
            } else if (columnType == IntegerType.INTEGER) {
                ColumnMapping.checkFieldTypeOneOf(fieldType, columnName, columnType, FieldType.BYTE, FieldType.SHORT, FieldType.INT);
            } else if (columnType == SmallintType.SMALLINT) {
                ColumnMapping.checkFieldTypeOneOf(fieldType, columnName, columnType, FieldType.BYTE, FieldType.SHORT);
            } else if (columnType == TinyintType.TINYINT) {
                ColumnMapping.checkFieldTypeOneOf(fieldType, columnName, columnType, FieldType.BYTE);
            } else if (columnType == BooleanType.BOOLEAN) {
                ColumnMapping.checkFieldTypeOneOf(fieldType, columnName, columnType, FieldType.BYTE, FieldType.SHORT, FieldType.INT, FieldType.LONG);
            } else if (columnType == DoubleType.DOUBLE) {
                ColumnMapping.checkFieldTypeOneOf(fieldType, columnName, columnType, FieldType.DOUBLE, FieldType.FLOAT);
            } else if (columnType instanceof VarcharType) {
                ColumnMapping.checkFieldTypeOneOf(fieldType, columnName, columnType, FieldType.BYTE);
            }
        }

        private static void checkFieldTypeOneOf(FieldType declaredFieldType, String columnName, Type columnType, FieldType ... allowedFieldTypes) {
            Preconditions.checkArgument((boolean)Arrays.asList(allowedFieldTypes).contains((Object)declaredFieldType), (Object)String.format("Wrong dataformat '%s' specified for column '%s'; %s type implies use of %s", declaredFieldType.name(), columnName, columnType, Joiner.on((String)"/").join((Object[])allowedFieldTypes)));
        }

        public String getName() {
            return this.name;
        }

        public Type getType() {
            return this.type;
        }

        public int getStart() {
            return this.start;
        }

        public int getEnd() {
            return this.end;
        }

        public FieldType getFieldType() {
            return this.fieldType;
        }

        public int getLength() {
            return this.end - this.start;
        }
    }

    private static enum FieldType {
        BYTE(8),
        SHORT(16),
        INT(32),
        LONG(64),
        FLOAT(32),
        DOUBLE(64);

        private final int size;

        private FieldType(int bitSize) {
            this.size = bitSize / 8;
        }

        public int getSize() {
            return this.size;
        }
    }
}

