/*
 * Decompiled with CFR 0.152.
 */
package io.trino.plugin.kafka.schema.confluent;

import com.google.common.base.Preconditions;
import com.google.common.collect.ImmutableList;
import com.google.common.collect.ImmutableSet;
import com.google.common.collect.Iterables;
import io.trino.spi.type.ArrayType;
import io.trino.spi.type.BigintType;
import io.trino.spi.type.BooleanType;
import io.trino.spi.type.DoubleType;
import io.trino.spi.type.IntegerType;
import io.trino.spi.type.MapType;
import io.trino.spi.type.RealType;
import io.trino.spi.type.RowType;
import io.trino.spi.type.Type;
import io.trino.spi.type.TypeManager;
import io.trino.spi.type.VarbinaryType;
import io.trino.spi.type.VarcharType;
import java.util.List;
import java.util.Objects;
import java.util.Optional;
import java.util.Set;
import org.apache.avro.Schema;
import org.apache.avro.SchemaFormatter;

public class AvroSchemaConverter {
    private static final SchemaFormatter JSON_PRETTY_FORMATTER = SchemaFormatter.getInstance((String)"json/pretty");
    public static final String DUMMY_FIELD_NAME = "$empty_field_marker";
    public static final RowType DUMMY_ROW_TYPE = RowType.from((List)ImmutableList.of((Object)new RowType.Field(Optional.of("$empty_field_marker"), (Type)BooleanType.BOOLEAN)));
    private static final Set<Schema.Type> INTEGRAL_TYPES = ImmutableSet.of((Object)Schema.Type.INT, (Object)Schema.Type.LONG);
    private static final Set<Schema.Type> DECIMAL_TYPES = ImmutableSet.of((Object)Schema.Type.FLOAT, (Object)Schema.Type.DOUBLE);
    private static final Set<Schema.Type> STRING_TYPES = ImmutableSet.of((Object)Schema.Type.STRING, (Object)Schema.Type.ENUM);
    private static final Set<Schema.Type> BINARY_TYPES = ImmutableSet.of((Object)Schema.Type.BYTES, (Object)Schema.Type.FIXED);
    private final TypeManager typeManager;
    private final EmptyFieldStrategy emptyFieldStrategy;

    public AvroSchemaConverter(TypeManager typeManager, EmptyFieldStrategy emptyFieldStrategy) {
        this.typeManager = Objects.requireNonNull(typeManager, "typeManager is null");
        this.emptyFieldStrategy = Objects.requireNonNull(emptyFieldStrategy, "emptyFieldStrategy is null");
    }

    public List<Type> convertAvroSchema(Schema schema) {
        Objects.requireNonNull(schema, "schema is null");
        List<Type> types = schema.getType().equals((Object)Schema.Type.RECORD) ? this.convertRecordSchema(schema) : this.convertSimpleSchema(schema);
        Preconditions.checkState((!types.isEmpty() ? 1 : 0) != 0, (String)"Schema has no valid fields: '%s'", (Object)schema);
        return types;
    }

    private List<Type> convertRecordSchema(Schema schema) {
        Preconditions.checkState((boolean)schema.getType().equals((Object)Schema.Type.RECORD), (Object)"schema is not an avro record");
        ImmutableList.Builder builder = ImmutableList.builder();
        for (Schema.Field field : schema.getFields()) {
            this.convert(field.schema()).ifPresent(arg_0 -> ((ImmutableList.Builder)builder).add(arg_0));
        }
        return builder.build();
    }

    private List<Type> convertSimpleSchema(Schema schema) {
        Preconditions.checkState((!schema.getType().equals((Object)Schema.Type.RECORD) ? 1 : 0) != 0, (Object)"Unexpected type for simple schema, cannot be a record");
        return (List)this.convert(schema).stream().collect(ImmutableList.toImmutableList());
    }

    private Optional<Type> convert(Schema schema) {
        switch (schema.getType()) {
            case INT: {
                return Optional.of(IntegerType.INTEGER);
            }
            case LONG: {
                return Optional.of(BigintType.BIGINT);
            }
            case BOOLEAN: {
                return Optional.of(BooleanType.BOOLEAN);
            }
            case FLOAT: {
                return Optional.of(RealType.REAL);
            }
            case DOUBLE: {
                return Optional.of(DoubleType.DOUBLE);
            }
            case ENUM: 
            case STRING: {
                return Optional.of(VarcharType.VARCHAR);
            }
            case BYTES: 
            case FIXED: {
                return Optional.of(VarbinaryType.VARBINARY);
            }
            case UNION: {
                return this.convertUnion(schema);
            }
            case ARRAY: {
                return this.convertArray(schema);
            }
            case MAP: {
                return this.convertMap(schema);
            }
            case RECORD: {
                return this.convertRecord(schema);
            }
        }
        throw new UnsupportedOperationException(String.format("Type %s not supported", schema.getType()));
    }

    private Optional<Type> convertUnion(Schema schema) {
        Preconditions.checkArgument((boolean)schema.getType().equals((Object)Schema.Type.UNION), (Object)"schema is not a union schema");
        Set types = (Set)schema.getTypes().stream().map(Schema::getType).collect(ImmutableSet.toImmutableSet());
        if (types.contains(Schema.Type.NULL)) {
            return this.convertUnion(Schema.createUnion((List)((List)schema.getTypes().stream().filter(type -> type.getType() != Schema.Type.NULL).collect(ImmutableList.toImmutableList()))));
        }
        if (schema.getTypes().size() == 1) {
            return this.convert((Schema)Iterables.getOnlyElement((Iterable)schema.getTypes()));
        }
        if (INTEGRAL_TYPES.containsAll(types)) {
            return Optional.of(BigintType.BIGINT);
        }
        if (DECIMAL_TYPES.containsAll(types)) {
            return Optional.of(DoubleType.DOUBLE);
        }
        if (STRING_TYPES.containsAll(types)) {
            return Optional.of(VarcharType.VARCHAR);
        }
        if (BINARY_TYPES.containsAll(types)) {
            return Optional.of(VarbinaryType.VARBINARY);
        }
        throw new UnsupportedOperationException(String.format("Incompatible UNION type: '%s'", JSON_PRETTY_FORMATTER.format(schema)));
    }

    private Optional<Type> convertArray(Schema schema) {
        Preconditions.checkArgument((schema.getType() == Schema.Type.ARRAY ? 1 : 0) != 0, (Object)"schema is not an ARRAY");
        return this.convert(schema.getElementType()).map(ArrayType::new);
    }

    private Optional<Type> convertMap(Schema schema) {
        Preconditions.checkArgument((schema.getType() == Schema.Type.MAP ? 1 : 0) != 0, (Object)"schema is not a MAP");
        return this.convert(schema.getValueType()).map(this::createMapType);
    }

    private Type createMapType(Type valueType) {
        VarcharType keyType = VarcharType.VARCHAR;
        return new MapType((Type)keyType, valueType, this.typeManager.getTypeOperators());
    }

    private Optional<Type> convertRecord(Schema schema) {
        Preconditions.checkArgument((schema.getType() == Schema.Type.RECORD ? 1 : 0) != 0, (Object)"schema is not a RECORD");
        List fields = (List)schema.getFields().stream().map(field -> this.convert(field.schema()).map(type -> new RowType.Field(Optional.ofNullable(field.name()), type))).filter(Optional::isPresent).map(Optional::get).collect(ImmutableList.toImmutableList());
        if (fields.isEmpty()) {
            switch (this.emptyFieldStrategy.ordinal()) {
                case 0: {
                    return Optional.empty();
                }
                case 1: {
                    return Optional.of(DUMMY_ROW_TYPE);
                }
                case 2: {
                    throw new IllegalStateException(String.format("Struct type has no valid fields for schema: '%s'", schema));
                }
            }
            throw new IllegalStateException(String.format("Unknown emptyFieldStrategy '%s'", new Object[]{this.emptyFieldStrategy}));
        }
        return Optional.of(RowType.from((List)fields));
    }

    public static enum EmptyFieldStrategy {
        IGNORE,
        MARK,
        FAIL;

    }
}

