/*
 * Decompiled with CFR 0.152.
 */
package io.trino.execution.buffer;

import com.google.common.annotations.VisibleForTesting;
import com.google.common.base.Preconditions;
import com.google.common.collect.ImmutableList;
import com.google.common.util.concurrent.ListenableFuture;
import io.airlift.units.DataSize;
import io.trino.execution.StateMachine;
import io.trino.execution.buffer.BufferInfo;
import io.trino.execution.buffer.BufferResult;
import io.trino.execution.buffer.BufferState;
import io.trino.execution.buffer.ClientBuffer;
import io.trino.execution.buffer.OutputBuffer;
import io.trino.execution.buffer.OutputBufferInfo;
import io.trino.execution.buffer.OutputBufferMemoryManager;
import io.trino.execution.buffer.OutputBuffers;
import io.trino.execution.buffer.SerializedPage;
import io.trino.execution.buffer.SerializedPageReference;
import io.trino.memory.context.LocalMemoryContext;
import java.util.Collection;
import java.util.List;
import java.util.Objects;
import java.util.concurrent.Executor;
import java.util.concurrent.atomic.AtomicLong;
import java.util.function.Supplier;

public class PartitionedOutputBuffer
implements OutputBuffer {
    private final StateMachine<BufferState> state;
    private final OutputBuffers outputBuffers;
    private final OutputBufferMemoryManager memoryManager;
    private final SerializedPageReference.PagesReleasedListener onPagesReleased;
    private final List<ClientBuffer> partitions;
    private final AtomicLong totalPagesAdded = new AtomicLong();
    private final AtomicLong totalRowsAdded = new AtomicLong();

    public PartitionedOutputBuffer(String taskInstanceId, StateMachine<BufferState> state, OutputBuffers outputBuffers, DataSize maxBufferSize, Supplier<LocalMemoryContext> systemMemoryContextSupplier, Executor notificationExecutor) {
        this.state = Objects.requireNonNull(state, "state is null");
        Objects.requireNonNull(outputBuffers, "outputBuffers is null");
        Preconditions.checkArgument((outputBuffers.getType() == OutputBuffers.BufferType.PARTITIONED ? 1 : 0) != 0, (Object)"Expected a PARTITIONED output buffer descriptor");
        Preconditions.checkArgument((boolean)outputBuffers.isNoMoreBufferIds(), (Object)"Expected a final output buffer descriptor");
        this.outputBuffers = outputBuffers;
        this.memoryManager = new OutputBufferMemoryManager(Objects.requireNonNull(maxBufferSize, "maxBufferSize is null").toBytes(), Objects.requireNonNull(systemMemoryContextSupplier, "systemMemoryContextSupplier is null"), Objects.requireNonNull(notificationExecutor, "notificationExecutor is null"));
        this.onPagesReleased = SerializedPageReference.PagesReleasedListener.forOutputBufferMemoryManager(this.memoryManager);
        ImmutableList.Builder partitions = ImmutableList.builder();
        for (OutputBuffers.OutputBufferId bufferId : outputBuffers.getBuffers().keySet()) {
            ClientBuffer partition = new ClientBuffer(taskInstanceId, bufferId, this.onPagesReleased);
            partitions.add((Object)partition);
        }
        this.partitions = partitions.build();
        state.compareAndSet(BufferState.OPEN, BufferState.NO_MORE_BUFFERS);
        state.compareAndSet(BufferState.NO_MORE_PAGES, BufferState.FLUSHING);
        this.checkFlushComplete();
    }

    @Override
    public void addStateChangeListener(StateMachine.StateChangeListener<BufferState> stateChangeListener) {
        this.state.addStateChangeListener(stateChangeListener);
    }

    @Override
    public boolean isFinished() {
        return this.state.get() == BufferState.FINISHED;
    }

    @Override
    public double getUtilization() {
        return this.memoryManager.getUtilization();
    }

    @Override
    public boolean isOverutilized() {
        return this.memoryManager.isOverutilized();
    }

    @Override
    public OutputBufferInfo getInfo() {
        BufferState state = this.state.get();
        int totalBufferedPages = 0;
        ImmutableList.Builder infos = ImmutableList.builderWithExpectedSize((int)this.partitions.size());
        for (ClientBuffer partition : this.partitions) {
            BufferInfo bufferInfo = partition.getInfo();
            infos.add((Object)bufferInfo);
            totalBufferedPages = (int)((long)totalBufferedPages + bufferInfo.getPageBufferInfo().getBufferedPages());
        }
        return new OutputBufferInfo("PARTITIONED", state, state.canAddBuffers(), state.canAddPages(), this.memoryManager.getBufferedBytes(), totalBufferedPages, this.totalRowsAdded.get(), this.totalPagesAdded.get(), (List<BufferInfo>)infos.build());
    }

    @Override
    public void setOutputBuffers(OutputBuffers newOutputBuffers) {
        Objects.requireNonNull(newOutputBuffers, "newOutputBuffers is null");
        if (this.state.get().isTerminal() || this.outputBuffers.getVersion() >= newOutputBuffers.getVersion()) {
            return;
        }
        this.outputBuffers.checkValidTransition(newOutputBuffers);
    }

    @Override
    public ListenableFuture<?> isFull() {
        return this.memoryManager.getBufferBlockedFuture();
    }

    @Override
    public void enqueue(List<SerializedPage> pages) {
        Preconditions.checkState((this.partitions.size() == 1 ? 1 : 0) != 0, (Object)"Expected exactly one partition");
        this.enqueue(0, pages);
    }

    @Override
    public void enqueue(int partitionNumber, List<SerializedPage> pages) {
        Objects.requireNonNull(pages, "pages is null");
        if (!this.state.get().canAddPages()) {
            return;
        }
        ImmutableList.Builder references = ImmutableList.builderWithExpectedSize((int)pages.size());
        long bytesAdded = 0L;
        long rowCount = 0L;
        for (SerializedPage page : pages) {
            bytesAdded += page.getRetainedSizeInBytes();
            rowCount += (long)page.getPositionCount();
            references.add((Object)new SerializedPageReference(page, 1));
        }
        ImmutableList serializedPageReferences = references.build();
        this.totalRowsAdded.addAndGet(rowCount);
        this.totalPagesAdded.addAndGet(serializedPageReferences.size());
        this.memoryManager.updateMemoryUsage(bytesAdded);
        this.partitions.get(partitionNumber).enqueuePages((Collection<SerializedPageReference>)serializedPageReferences);
        SerializedPageReference.dereferencePages((List<SerializedPageReference>)serializedPageReferences, this.onPagesReleased);
    }

    @Override
    public ListenableFuture<BufferResult> get(OutputBuffers.OutputBufferId outputBufferId, long startingSequenceId, DataSize maxSize) {
        Objects.requireNonNull(outputBufferId, "outputBufferId is null");
        Preconditions.checkArgument((maxSize.toBytes() > 0L ? 1 : 0) != 0, (Object)"maxSize must be at least 1 byte");
        return this.partitions.get(outputBufferId.getId()).getPages(startingSequenceId, maxSize);
    }

    @Override
    public void acknowledge(OutputBuffers.OutputBufferId outputBufferId, long sequenceId) {
        Objects.requireNonNull(outputBufferId, "bufferId is null");
        this.partitions.get(outputBufferId.getId()).acknowledgePages(sequenceId);
    }

    @Override
    public void abort(OutputBuffers.OutputBufferId bufferId) {
        Objects.requireNonNull(bufferId, "bufferId is null");
        this.partitions.get(bufferId.getId()).destroy();
        this.checkFlushComplete();
    }

    @Override
    public void setNoMorePages() {
        this.state.compareAndSet(BufferState.OPEN, BufferState.NO_MORE_PAGES);
        this.state.compareAndSet(BufferState.NO_MORE_BUFFERS, BufferState.FLUSHING);
        this.memoryManager.setNoBlockOnFull();
        this.partitions.forEach(ClientBuffer::setNoMorePages);
        this.checkFlushComplete();
    }

    @Override
    public void destroy() {
        if (this.state.setIf(BufferState.FINISHED, oldState -> !oldState.isTerminal())) {
            this.partitions.forEach(ClientBuffer::destroy);
            this.memoryManager.setNoBlockOnFull();
            this.forceFreeMemory();
        }
    }

    @Override
    public void fail() {
        if (this.state.setIf(BufferState.FAILED, oldState -> !oldState.isTerminal())) {
            this.memoryManager.setNoBlockOnFull();
            this.forceFreeMemory();
        }
    }

    @Override
    public long getPeakMemoryUsage() {
        return this.memoryManager.getPeakMemoryUsage();
    }

    @VisibleForTesting
    void forceFreeMemory() {
        this.memoryManager.close();
    }

    private void checkFlushComplete() {
        if (this.state.get() != BufferState.FLUSHING && this.state.get() != BufferState.NO_MORE_BUFFERS) {
            return;
        }
        if (this.partitions.stream().allMatch(ClientBuffer::isDestroyed)) {
            this.destroy();
        }
    }

    @VisibleForTesting
    OutputBufferMemoryManager getMemoryManager() {
        return this.memoryManager;
    }
}

