/*
 * Decompiled with CFR 0.152.
 */
package io.trino.operator;

import com.google.common.annotations.VisibleForTesting;
import com.google.common.base.Preconditions;
import com.google.common.base.Verify;
import io.trino.array.LongBigArray;
import io.trino.operator.RowIdComparisonHashStrategy;
import io.trino.operator.RowReference;
import io.trino.operator.TopNPeerGroupLookup;
import io.trino.util.HeapTraversal;
import io.trino.util.LongBigArrayFIFOQueue;
import java.util.Objects;
import java.util.function.LongConsumer;
import javax.annotation.Nullable;
import org.openjdk.jol.info.ClassLayout;

public class GroupedTopNRankAccumulator {
    private static final long INSTANCE_SIZE = ClassLayout.parseClass(GroupedTopNRankAccumulator.class).instanceSize();
    private static final long UNKNOWN_INDEX = -1L;
    private static final long NULL_GROUP_ID = -1L;
    private final GroupIdToHeapBuffer groupIdToHeapBuffer = new GroupIdToHeapBuffer();
    private final HeapNodeBuffer heapNodeBuffer = new HeapNodeBuffer();
    private final PeerGroupBuffer peerGroupBuffer = new PeerGroupBuffer();
    private final HeapTraversal heapTraversal = new HeapTraversal();
    private final TopNPeerGroupLookup peerGroupLookup;
    private final RowIdComparisonHashStrategy strategy;
    private final int topN;
    private final LongConsumer rowIdEvictionListener;

    public GroupedTopNRankAccumulator(RowIdComparisonHashStrategy strategy, int topN, LongConsumer rowIdEvictionListener) {
        this.strategy = Objects.requireNonNull(strategy, "rowComparisonStrategy is null");
        this.peerGroupLookup = new TopNPeerGroupLookup(10000L, strategy, -1L, -1L);
        Preconditions.checkArgument((topN > 0 ? 1 : 0) != 0, (Object)"topN must be greater than zero");
        this.topN = topN;
        this.rowIdEvictionListener = Objects.requireNonNull(rowIdEvictionListener, "rowIdEvictionListener is null");
    }

    public long sizeOf() {
        return INSTANCE_SIZE + this.groupIdToHeapBuffer.sizeOf() + this.heapNodeBuffer.sizeOf() + this.peerGroupBuffer.sizeOf() + this.heapTraversal.sizeOf() + this.peerGroupLookup.sizeOf();
    }

    public boolean add(long groupId, RowReference rowReference) {
        long peerHeapNodeIndex = this.peerGroupLookup.get(groupId, rowReference);
        if (peerHeapNodeIndex != -1L) {
            this.directPeerGroupInsert(groupId, peerHeapNodeIndex, rowReference.allocateRowId());
            if (this.calculateRootRank(groupId) > (long)this.topN) {
                this.heapPop(groupId, this.rowIdEvictionListener);
            }
            return true;
        }
        this.groupIdToHeapBuffer.allocateGroupIfNeeded(groupId);
        if (this.groupIdToHeapBuffer.getHeapValueCount(groupId) < (long)this.topN) {
            long newPeerGroupIndex = this.peerGroupBuffer.allocateNewNode(rowReference.allocateRowId(), -1L);
            this.heapInsert(groupId, newPeerGroupIndex, 1L);
            return true;
        }
        if (rowReference.compareTo(this.strategy, this.peekRootRowId(groupId)) < 0) {
            long newPeerGroupIndex = this.peerGroupBuffer.allocateNewNode(rowReference.allocateRowId(), -1L);
            if (this.calculateRootRank(groupId) < (long)this.topN) {
                this.heapInsert(groupId, newPeerGroupIndex, 1L);
            } else {
                this.heapPopAndInsert(groupId, newPeerGroupIndex, 1L, this.rowIdEvictionListener);
            }
            return true;
        }
        return false;
    }

    public long drainTo(long groupId, LongBigArray rowIdOutput, LongBigArray rankingOutput) {
        long valueCount = this.groupIdToHeapBuffer.getHeapValueCount(groupId);
        rowIdOutput.ensureCapacity(valueCount);
        rankingOutput.ensureCapacity(valueCount);
        for (long insertionIndex = valueCount - 1L; insertionIndex >= 0L; --insertionIndex) {
            long heapRootNodeIndex = this.groupIdToHeapBuffer.getHeapRootNodeIndex(groupId);
            Verify.verify((heapRootNodeIndex != -1L ? 1 : 0) != 0);
            long peerGroupIndex = this.heapNodeBuffer.getPeerGroupIndex(heapRootNodeIndex);
            Verify.verify((peerGroupIndex != -1L ? 1 : 0) != 0, (String)"Peer group should have at least one value", (Object[])new Object[0]);
            long rank = this.calculateRootRank(groupId);
            do {
                rowIdOutput.set(insertionIndex, this.peerGroupBuffer.getRowId(peerGroupIndex));
                rankingOutput.set(insertionIndex, rank);
            } while ((peerGroupIndex = this.peerGroupBuffer.getNextPeerIndex(peerGroupIndex)) != -1L);
            this.heapPop(groupId, null);
        }
        return valueCount;
    }

    public long drainTo(long groupId, LongBigArray rowIdOutput) {
        long valueCount = this.groupIdToHeapBuffer.getHeapValueCount(groupId);
        rowIdOutput.ensureCapacity(valueCount);
        for (long insertionIndex = valueCount - 1L; insertionIndex >= 0L; --insertionIndex) {
            long heapRootNodeIndex = this.groupIdToHeapBuffer.getHeapRootNodeIndex(groupId);
            Verify.verify((heapRootNodeIndex != -1L ? 1 : 0) != 0);
            long peerGroupIndex = this.heapNodeBuffer.getPeerGroupIndex(heapRootNodeIndex);
            Verify.verify((peerGroupIndex != -1L ? 1 : 0) != 0, (String)"Peer group should have at least one value", (Object[])new Object[0]);
            do {
                rowIdOutput.set(insertionIndex, this.peerGroupBuffer.getRowId(peerGroupIndex));
            } while ((peerGroupIndex = this.peerGroupBuffer.getNextPeerIndex(peerGroupIndex)) != -1L);
            this.heapPop(groupId, null);
        }
        return valueCount;
    }

    private long calculateRootRank(long groupId) {
        long heapValueCount = this.groupIdToHeapBuffer.getHeapValueCount(groupId);
        long heapRootIndex = this.groupIdToHeapBuffer.getHeapRootNodeIndex(groupId);
        Preconditions.checkArgument((heapRootIndex != -1L ? 1 : 0) != 0, (Object)"Group does not have a root");
        long rootPeerGroupCount = this.heapNodeBuffer.getPeerGroupCount(heapRootIndex);
        return heapValueCount - rootPeerGroupCount + 1L;
    }

    private void directPeerGroupInsert(long groupId, long heapNodeIndex, long rowId) {
        long existingPeerGroupIndex = this.heapNodeBuffer.getPeerGroupIndex(heapNodeIndex);
        long newPeerGroupIndex = this.peerGroupBuffer.allocateNewNode(rowId, existingPeerGroupIndex);
        this.heapNodeBuffer.setPeerGroupIndex(heapNodeIndex, newPeerGroupIndex);
        this.heapNodeBuffer.incrementPeerGroupCount(heapNodeIndex);
        this.groupIdToHeapBuffer.incrementHeapValueCount(groupId);
    }

    private long peekRootRowId(long groupId) {
        long heapRootNodeIndex = this.groupIdToHeapBuffer.getHeapRootNodeIndex(groupId);
        Preconditions.checkArgument((heapRootNodeIndex != -1L ? 1 : 0) != 0, (Object)"Group has nothing to peek");
        return this.peerGroupBuffer.getRowId(this.heapNodeBuffer.getPeerGroupIndex(heapRootNodeIndex));
    }

    private long getChildIndex(long heapNodeIndex, HeapTraversal.Child child) {
        return child == HeapTraversal.Child.LEFT ? this.heapNodeBuffer.getLeftChildHeapIndex(heapNodeIndex) : this.heapNodeBuffer.getRightChildHeapIndex(heapNodeIndex);
    }

    private void setChildIndex(long heapNodeIndex, HeapTraversal.Child child, long newChildIndex) {
        if (child == HeapTraversal.Child.LEFT) {
            this.heapNodeBuffer.setLeftChildHeapIndex(heapNodeIndex, newChildIndex);
        } else {
            this.heapNodeBuffer.setRightChildHeapIndex(heapNodeIndex, newChildIndex);
        }
    }

    private void heapPop(long groupId, @Nullable LongConsumer contextEvictionListener) {
        long heapRootNodeIndex = this.groupIdToHeapBuffer.getHeapRootNodeIndex(groupId);
        Preconditions.checkArgument((heapRootNodeIndex != -1L ? 1 : 0) != 0, (Object)"Group ID has an empty heap");
        long lastHeapNodeIndex = this.heapDetachLastInsertionLeaf(groupId);
        long lastPeerGroupIndex = this.heapNodeBuffer.getPeerGroupIndex(lastHeapNodeIndex);
        long lastPeerGroupCount = this.heapNodeBuffer.getPeerGroupCount(lastHeapNodeIndex);
        if (lastHeapNodeIndex == heapRootNodeIndex) {
            this.dropHeapNodePeerGroup(groupId, lastHeapNodeIndex, contextEvictionListener);
        } else {
            this.heapPopAndInsert(groupId, lastPeerGroupIndex, lastPeerGroupCount, contextEvictionListener);
        }
        this.heapNodeBuffer.deallocate(lastHeapNodeIndex);
    }

    private long heapDetachLastInsertionLeaf(long groupId) {
        long heapRootNodeIndex = this.groupIdToHeapBuffer.getHeapRootNodeIndex(groupId);
        long heapSize = this.groupIdToHeapBuffer.getHeapSize(groupId);
        long previousNodeIndex = -1L;
        HeapTraversal.Child childPosition = null;
        long currentNodeIndex = heapRootNodeIndex;
        this.heapTraversal.resetWithPathTo(heapSize);
        while (!this.heapTraversal.isTarget()) {
            previousNodeIndex = currentNodeIndex;
            childPosition = this.heapTraversal.nextChild();
            Verify.verify(((currentNodeIndex = this.getChildIndex(currentNodeIndex, childPosition)) != -1L ? 1 : 0) != 0, (String)"Target node must exist", (Object[])new Object[0]);
        }
        if (previousNodeIndex == -1L) {
            this.groupIdToHeapBuffer.setHeapRootNodeIndex(groupId, -1L);
            this.groupIdToHeapBuffer.setHeapValueCount(groupId, 0L);
            this.groupIdToHeapBuffer.setHeapSize(groupId, 0L);
        } else {
            this.setChildIndex(previousNodeIndex, childPosition, -1L);
            this.groupIdToHeapBuffer.addHeapValueCount(groupId, -this.heapNodeBuffer.getPeerGroupCount(currentNodeIndex));
            this.groupIdToHeapBuffer.addHeapSize(groupId, -1L);
        }
        return currentNodeIndex;
    }

    private void heapInsert(long groupId, long newPeerGroupIndex, long newPeerGroupCount) {
        long newCanonicalRowId = this.peerGroupBuffer.getRowId(newPeerGroupIndex);
        long heapRootNodeIndex = this.groupIdToHeapBuffer.getHeapRootNodeIndex(groupId);
        if (heapRootNodeIndex == -1L) {
            heapRootNodeIndex = this.heapNodeBuffer.allocateNewNode(newPeerGroupIndex, newPeerGroupCount);
            Verify.verify((this.peerGroupLookup.put(groupId, newCanonicalRowId, heapRootNodeIndex) == -1L ? 1 : 0) != 0);
            this.groupIdToHeapBuffer.setHeapRootNodeIndex(groupId, heapRootNodeIndex);
            this.groupIdToHeapBuffer.setHeapValueCount(groupId, newPeerGroupCount);
            this.groupIdToHeapBuffer.setHeapSize(groupId, 1L);
            return;
        }
        long previousHeapNodeIndex = -1L;
        HeapTraversal.Child childPosition = null;
        long currentHeapNodeIndex = heapRootNodeIndex;
        boolean swapped = false;
        this.groupIdToHeapBuffer.addHeapValueCount(groupId, newPeerGroupCount);
        this.groupIdToHeapBuffer.incrementHeapSize(groupId);
        this.heapTraversal.resetWithPathTo(this.groupIdToHeapBuffer.getHeapSize(groupId));
        while (!this.heapTraversal.isTarget()) {
            long peerGroupIndex = this.heapNodeBuffer.getPeerGroupIndex(currentHeapNodeIndex);
            long currentCanonicalRowId = this.peerGroupBuffer.getRowId(peerGroupIndex);
            if (swapped || this.strategy.compare(newCanonicalRowId, currentCanonicalRowId) > 0) {
                long peerGroupCount = this.heapNodeBuffer.getPeerGroupCount(currentHeapNodeIndex);
                this.heapNodeBuffer.setPeerGroupIndex(currentHeapNodeIndex, newPeerGroupIndex);
                this.heapNodeBuffer.setPeerGroupCount(currentHeapNodeIndex, newPeerGroupCount);
                this.peerGroupLookup.put(groupId, newCanonicalRowId, currentHeapNodeIndex);
                newPeerGroupIndex = peerGroupIndex;
                newPeerGroupCount = peerGroupCount;
                newCanonicalRowId = currentCanonicalRowId;
                swapped = true;
            }
            previousHeapNodeIndex = currentHeapNodeIndex;
            childPosition = this.heapTraversal.nextChild();
            currentHeapNodeIndex = this.getChildIndex(currentHeapNodeIndex, childPosition);
        }
        Verify.verify((previousHeapNodeIndex != -1L && childPosition != null ? 1 : 0) != 0, (String)"heap must have at least one node before starting traversal", (Object[])new Object[0]);
        Verify.verify((currentHeapNodeIndex == -1L ? 1 : 0) != 0, (String)"New child shouldn't exist yet", (Object[])new Object[0]);
        long newHeapNodeIndex = this.heapNodeBuffer.allocateNewNode(newPeerGroupIndex, newPeerGroupCount);
        this.peerGroupLookup.put(groupId, newCanonicalRowId, newHeapNodeIndex);
        this.setChildIndex(previousHeapNodeIndex, childPosition, newHeapNodeIndex);
    }

    private void heapPopAndInsert(long groupId, long newPeerGroupIndex, long newPeerGroupCount, @Nullable LongConsumer contextEvictionListener) {
        long maxChildNodeIndex;
        long heapRootNodeIndex = this.groupIdToHeapBuffer.getHeapRootNodeIndex(groupId);
        Preconditions.checkState((heapRootNodeIndex != -1L ? 1 : 0) != 0, (Object)"popAndInsert() requires at least a root node");
        this.groupIdToHeapBuffer.addHeapValueCount(groupId, newPeerGroupCount - this.heapNodeBuffer.getPeerGroupCount(heapRootNodeIndex));
        this.dropHeapNodePeerGroup(groupId, heapRootNodeIndex, contextEvictionListener);
        long newCanonicalRowId = this.peerGroupBuffer.getRowId(newPeerGroupIndex);
        long currentNodeIndex = heapRootNodeIndex;
        while ((maxChildNodeIndex = this.heapNodeBuffer.getLeftChildHeapIndex(currentNodeIndex)) != -1L) {
            long rightChildPeerGroupIndex;
            long rightChildCanonicalRowId;
            long maxChildPeerGroupIndex = this.heapNodeBuffer.getPeerGroupIndex(maxChildNodeIndex);
            long maxChildCanonicalRowId = this.peerGroupBuffer.getRowId(maxChildPeerGroupIndex);
            long rightChildNodeIndex = this.heapNodeBuffer.getRightChildHeapIndex(currentNodeIndex);
            if (rightChildNodeIndex != -1L && this.strategy.compare(rightChildCanonicalRowId = this.peerGroupBuffer.getRowId(rightChildPeerGroupIndex = this.heapNodeBuffer.getPeerGroupIndex(rightChildNodeIndex)), maxChildCanonicalRowId) > 0) {
                maxChildNodeIndex = rightChildNodeIndex;
                maxChildPeerGroupIndex = rightChildPeerGroupIndex;
                maxChildCanonicalRowId = rightChildCanonicalRowId;
            }
            if (this.strategy.compare(newCanonicalRowId, maxChildCanonicalRowId) >= 0) break;
            this.heapNodeBuffer.setPeerGroupIndex(currentNodeIndex, maxChildPeerGroupIndex);
            this.heapNodeBuffer.setPeerGroupCount(currentNodeIndex, this.heapNodeBuffer.getPeerGroupCount(maxChildNodeIndex));
            this.peerGroupLookup.put(groupId, maxChildCanonicalRowId, currentNodeIndex);
            currentNodeIndex = maxChildNodeIndex;
        }
        this.heapNodeBuffer.setPeerGroupIndex(currentNodeIndex, newPeerGroupIndex);
        this.heapNodeBuffer.setPeerGroupCount(currentNodeIndex, newPeerGroupCount);
        this.peerGroupLookup.put(groupId, newCanonicalRowId, currentNodeIndex);
    }

    private void dropHeapNodePeerGroup(long groupId, long heapNodeIndex, @Nullable LongConsumer contextEvictionListener) {
        long peerGroupIndex = this.heapNodeBuffer.getPeerGroupIndex(heapNodeIndex);
        Preconditions.checkState((peerGroupIndex != -1L ? 1 : 0) != 0, (Object)"Heap node must have at least one peer group");
        long rowId = this.peerGroupBuffer.getRowId(peerGroupIndex);
        long nextPeerIndex = this.peerGroupBuffer.getNextPeerIndex(peerGroupIndex);
        this.peerGroupBuffer.deallocate(peerGroupIndex);
        Verify.verify((this.peerGroupLookup.remove(groupId, rowId) == heapNodeIndex ? 1 : 0) != 0);
        if (contextEvictionListener != null) {
            contextEvictionListener.accept(rowId);
        }
        peerGroupIndex = nextPeerIndex;
        while (peerGroupIndex != -1L) {
            rowId = this.peerGroupBuffer.getRowId(peerGroupIndex);
            nextPeerIndex = this.peerGroupBuffer.getNextPeerIndex(peerGroupIndex);
            this.peerGroupBuffer.deallocate(peerGroupIndex);
            if (contextEvictionListener != null) {
                contextEvictionListener.accept(rowId);
            }
            peerGroupIndex = nextPeerIndex;
        }
    }

    @VisibleForTesting
    void verifyIntegrity() {
        long totalHeapNodes = 0L;
        long totalValueCount = 0L;
        for (long groupId = 0L; groupId < this.groupIdToHeapBuffer.getTotalGroups(); ++groupId) {
            long heapSize = this.groupIdToHeapBuffer.getHeapSize(groupId);
            long heapValueCount = this.groupIdToHeapBuffer.getHeapValueCount(groupId);
            long rootNodeIndex = this.groupIdToHeapBuffer.getHeapRootNodeIndex(groupId);
            Verify.verify((rootNodeIndex == -1L || this.calculateRootRank(rootNodeIndex) <= (long)this.topN ? 1 : 0) != 0, (String)"Max heap has more values than needed", (Object[])new Object[0]);
            IntegrityStats integrityStats = this.verifyHeapIntegrity(groupId, rootNodeIndex);
            Verify.verify((integrityStats.getPeerGroupCount() == heapSize ? 1 : 0) != 0, (String)"Recorded heap size does not match actual heap size", (Object[])new Object[0]);
            totalHeapNodes += integrityStats.getPeerGroupCount();
            Verify.verify((integrityStats.getValueCount() == heapValueCount ? 1 : 0) != 0, (String)"Recorded value count does not match actual value count", (Object[])new Object[0]);
            totalValueCount += integrityStats.getValueCount();
        }
        Verify.verify((totalHeapNodes == this.heapNodeBuffer.getActiveNodeCount() ? 1 : 0) != 0, (String)"Failed to deallocate some unused nodes", (Object[])new Object[0]);
        Verify.verify((totalHeapNodes == this.peerGroupLookup.size() ? 1 : 0) != 0, (String)"Peer group lookup does not have the right number of entries", (Object[])new Object[0]);
        Verify.verify((totalValueCount == this.peerGroupBuffer.getActiveNodeCount() ? 1 : 0) != 0, (String)"Failed to deallocate some unused nodes", (Object[])new Object[0]);
    }

    private IntegrityStats verifyHeapIntegrity(long groupId, long heapNodeIndex) {
        if (heapNodeIndex == -1L) {
            return new IntegrityStats(0L, 0L, 0L);
        }
        long peerGroupIndex = this.heapNodeBuffer.getPeerGroupIndex(heapNodeIndex);
        long peerGroupCount = this.heapNodeBuffer.getPeerGroupCount(heapNodeIndex);
        long leftChildHeapIndex = this.heapNodeBuffer.getLeftChildHeapIndex(heapNodeIndex);
        long rightChildHeapIndex = this.heapNodeBuffer.getRightChildHeapIndex(heapNodeIndex);
        long actualPeerGroupCount = 0L;
        long rowId = -1L;
        do {
            long previousRowId = rowId;
            rowId = this.peerGroupBuffer.getRowId(peerGroupIndex);
            if (++actualPeerGroupCount >= 2L) {
                Verify.verify((boolean)this.strategy.equals(rowId, previousRowId), (String)"Row value does not belong in peer group", (Object[])new Object[0]);
            }
            Verify.verify((this.peerGroupLookup.get(groupId, rowId) == heapNodeIndex ? 1 : 0) != 0, (String)"Mismatch between peer group and lookup mapping", (Object[])new Object[0]);
        } while ((peerGroupIndex = this.peerGroupBuffer.getNextPeerIndex(peerGroupIndex)) != -1L);
        Verify.verify((actualPeerGroupCount == peerGroupCount ? 1 : 0) != 0, (String)"Recorded peer group count does not match actual", (Object[])new Object[0]);
        if (leftChildHeapIndex != -1L) {
            Verify.verify((this.strategy.compare(rowId, this.peerGroupBuffer.getRowId(this.heapNodeBuffer.getPeerGroupIndex(leftChildHeapIndex))) > 0 ? 1 : 0) != 0, (String)"Max heap invariant violated", (Object[])new Object[0]);
        }
        if (rightChildHeapIndex != -1L) {
            Verify.verify((leftChildHeapIndex != -1L ? 1 : 0) != 0, (String)"Left should always be inserted before right", (Object[])new Object[0]);
            Verify.verify((this.strategy.compare(rowId, this.peerGroupBuffer.getRowId(this.heapNodeBuffer.getPeerGroupIndex(rightChildHeapIndex))) > 0 ? 1 : 0) != 0, (String)"Max heap invariant violated", (Object[])new Object[0]);
        }
        IntegrityStats leftIntegrityStats = this.verifyHeapIntegrity(groupId, leftChildHeapIndex);
        IntegrityStats rightIntegrityStats = this.verifyHeapIntegrity(groupId, rightChildHeapIndex);
        Verify.verify((Math.abs(leftIntegrityStats.getMaxDepth() - rightIntegrityStats.getMaxDepth()) <= 1L ? 1 : 0) != 0, (String)"Heap not balanced", (Object[])new Object[0]);
        return new IntegrityStats(Math.max(leftIntegrityStats.getMaxDepth(), rightIntegrityStats.getMaxDepth()) + 1L, leftIntegrityStats.getPeerGroupCount() + rightIntegrityStats.getPeerGroupCount() + 1L, leftIntegrityStats.getValueCount() + rightIntegrityStats.getValueCount() + peerGroupCount);
    }

    private static class PeerGroupBuffer {
        private static final long INSTANCE_SIZE = ClassLayout.parseClass(PeerGroupBuffer.class).instanceSize();
        private static final int POSITIONS_PER_ENTRY = 2;
        private static final int NEXT_PEER_INDEX_OFFSET = 1;
        private final LongBigArray buffer = new LongBigArray();
        private final LongBigArrayFIFOQueue emptySlots = new LongBigArrayFIFOQueue();
        private long capacity;

        private PeerGroupBuffer() {
        }

        public long allocateNewNode(long rowId, long nextPeerIndex) {
            long newPeerIndex;
            if (!this.emptySlots.isEmpty()) {
                newPeerIndex = this.emptySlots.dequeueLong();
            } else {
                newPeerIndex = this.capacity++;
                this.buffer.ensureCapacity(this.capacity * 2L);
            }
            this.setRowId(newPeerIndex, rowId);
            this.setNextPeerIndex(newPeerIndex, nextPeerIndex);
            return newPeerIndex;
        }

        public void deallocate(long index) {
            this.emptySlots.enqueue(index);
        }

        public long getActiveNodeCount() {
            return this.capacity - this.emptySlots.longSize();
        }

        public long getRowId(long index) {
            return this.buffer.get(index * 2L);
        }

        public void setRowId(long index, long rowId) {
            this.buffer.set(index * 2L, rowId);
        }

        public long getNextPeerIndex(long index) {
            return this.buffer.get(index * 2L + 1L);
        }

        public void setNextPeerIndex(long index, long nextPeerIndex) {
            this.buffer.set(index * 2L + 1L, nextPeerIndex);
        }

        public long sizeOf() {
            return INSTANCE_SIZE + this.buffer.sizeOf() + this.emptySlots.sizeOf();
        }
    }

    private static class HeapNodeBuffer {
        private static final long INSTANCE_SIZE = ClassLayout.parseClass(HeapNodeBuffer.class).instanceSize();
        private static final int POSITIONS_PER_ENTRY = 4;
        private static final int PEER_GROUP_COUNT_OFFSET = 1;
        private static final int LEFT_CHILD_HEAP_INDEX_OFFSET = 2;
        private static final int RIGHT_CHILD_HEAP_INDEX_OFFSET = 3;
        private final LongBigArray buffer = new LongBigArray();
        private final LongBigArrayFIFOQueue emptySlots = new LongBigArrayFIFOQueue();
        private long capacity;

        private HeapNodeBuffer() {
        }

        public long allocateNewNode(long peerGroupIndex, long peerGroupCount) {
            long newHeapIndex;
            if (!this.emptySlots.isEmpty()) {
                newHeapIndex = this.emptySlots.dequeueLong();
            } else {
                newHeapIndex = this.capacity++;
                this.buffer.ensureCapacity(this.capacity * 4L);
            }
            this.setPeerGroupIndex(newHeapIndex, peerGroupIndex);
            this.setPeerGroupCount(newHeapIndex, peerGroupCount);
            this.setLeftChildHeapIndex(newHeapIndex, -1L);
            this.setRightChildHeapIndex(newHeapIndex, -1L);
            return newHeapIndex;
        }

        public void deallocate(long index) {
            this.emptySlots.enqueue(index);
        }

        public long getActiveNodeCount() {
            return this.capacity - this.emptySlots.longSize();
        }

        public long getPeerGroupIndex(long index) {
            return this.buffer.get(index * 4L);
        }

        public void setPeerGroupIndex(long index, long peerGroupIndex) {
            this.buffer.set(index * 4L, peerGroupIndex);
        }

        public long getPeerGroupCount(long index) {
            return this.buffer.get(index * 4L + 1L);
        }

        public void setPeerGroupCount(long index, long peerGroupCount) {
            this.buffer.set(index * 4L + 1L, peerGroupCount);
        }

        public void incrementPeerGroupCount(long index) {
            this.buffer.increment(index * 4L + 1L);
        }

        public void addPeerGroupCount(long index, long delta) {
            this.buffer.add(index * 4L + 1L, delta);
        }

        public long getLeftChildHeapIndex(long index) {
            return this.buffer.get(index * 4L + 2L);
        }

        public void setLeftChildHeapIndex(long index, long childHeapIndex) {
            this.buffer.set(index * 4L + 2L, childHeapIndex);
        }

        public long getRightChildHeapIndex(long index) {
            return this.buffer.get(index * 4L + 3L);
        }

        public void setRightChildHeapIndex(long index, long childHeapIndex) {
            this.buffer.set(index * 4L + 3L, childHeapIndex);
        }

        public long sizeOf() {
            return INSTANCE_SIZE + this.buffer.sizeOf() + this.emptySlots.sizeOf();
        }
    }

    private static class GroupIdToHeapBuffer {
        private static final long INSTANCE_SIZE = ClassLayout.parseClass(GroupIdToHeapBuffer.class).instanceSize();
        private static final int METRICS_POSITIONS_PER_ENTRY = 2;
        private static final int METRICS_HEAP_SIZE_OFFSET = 1;
        private final LongBigArray heapIndexBuffer = new LongBigArray(-1L);
        private final LongBigArray metricsBuffer = new LongBigArray(0L);
        private long totalGroups;

        private GroupIdToHeapBuffer() {
        }

        public void allocateGroupIfNeeded(long groupId) {
            this.totalGroups = Math.max(groupId + 1L, this.totalGroups);
            this.heapIndexBuffer.ensureCapacity(this.totalGroups);
            this.metricsBuffer.ensureCapacity(this.totalGroups * 2L);
        }

        public long getTotalGroups() {
            return this.totalGroups;
        }

        public long getHeapRootNodeIndex(long groupId) {
            return this.heapIndexBuffer.get(groupId);
        }

        public void setHeapRootNodeIndex(long groupId, long heapNodeIndex) {
            this.heapIndexBuffer.set(groupId, heapNodeIndex);
        }

        public long getHeapValueCount(long groupId) {
            return this.metricsBuffer.get(groupId * 2L);
        }

        public void setHeapValueCount(long groupId, long count) {
            this.metricsBuffer.set(groupId * 2L, count);
        }

        public void addHeapValueCount(long groupId, long delta) {
            this.metricsBuffer.add(groupId * 2L, delta);
        }

        public void incrementHeapValueCount(long groupId) {
            this.metricsBuffer.increment(groupId * 2L);
        }

        public long getHeapSize(long groupId) {
            return this.metricsBuffer.get(groupId * 2L + 1L);
        }

        public void setHeapSize(long groupId, long size) {
            this.metricsBuffer.set(groupId * 2L + 1L, size);
        }

        public void addHeapSize(long groupId, long delta) {
            this.metricsBuffer.add(groupId * 2L + 1L, delta);
        }

        public void incrementHeapSize(long groupId) {
            this.metricsBuffer.increment(groupId * 2L + 1L);
        }

        public long sizeOf() {
            return INSTANCE_SIZE + this.heapIndexBuffer.sizeOf() + this.metricsBuffer.sizeOf();
        }
    }

    private static class IntegrityStats {
        private final long maxDepth;
        private final long peerGroupCount;
        private final long valueCount;

        public IntegrityStats(long maxDepth, long peerGroupCount, long valueCount) {
            this.maxDepth = maxDepth;
            this.peerGroupCount = peerGroupCount;
            this.valueCount = valueCount;
        }

        public long getMaxDepth() {
            return this.maxDepth;
        }

        public long getPeerGroupCount() {
            return this.peerGroupCount;
        }

        public long getValueCount() {
            return this.valueCount;
        }
    }
}

