/*
 * Decompiled with CFR 0.152.
 */
package io.trino.server.security.oauth2;

import com.google.common.collect.ImmutableList;
import com.google.common.collect.ImmutableMap;
import com.google.common.io.Resources;
import com.google.inject.Module;
import io.airlift.http.client.HttpClient;
import io.airlift.http.client.HttpClientConfig;
import io.airlift.http.client.jetty.JettyHttpClient;
import io.airlift.log.Level;
import io.airlift.log.Logging;
import io.airlift.testing.Closeables;
import io.jsonwebtoken.Claims;
import io.jsonwebtoken.Jws;
import io.jsonwebtoken.Jwts;
import io.jsonwebtoken.SignatureAlgorithm;
import io.jsonwebtoken.SigningKeyResolver;
import io.jsonwebtoken.impl.DefaultClaims;
import io.trino.client.OkHttpUtil;
import io.trino.server.security.jwt.JwkService;
import io.trino.server.security.jwt.JwkSigningKeyResolver;
import io.trino.server.security.jwt.JwtAuthenticatorConfig;
import io.trino.server.security.oauth2.TestingHydraIdentityProvider;
import io.trino.server.security.oauth2.TokenEndpointAuthMethod;
import io.trino.server.testing.TestingTrinoServer;
import io.trino.server.ui.OAuth2WebUiAuthenticationFilter;
import io.trino.server.ui.WebUiModule;
import java.io.IOException;
import java.net.CookieHandler;
import java.net.CookieManager;
import java.net.CookieStore;
import java.net.HttpCookie;
import java.net.MalformedURLException;
import java.net.URI;
import java.net.URL;
import java.security.Key;
import java.security.KeyPairGenerator;
import java.security.NoSuchAlgorithmException;
import java.time.Duration;
import java.time.Instant;
import java.util.List;
import java.util.Map;
import java.util.Optional;
import java.util.UUID;
import javax.ws.rs.core.Response;
import okhttp3.Cookie;
import okhttp3.CookieJar;
import okhttp3.HttpUrl;
import okhttp3.JavaNetCookieJar;
import okhttp3.OkHttpClient;
import okhttp3.Request;
import okhttp3.Response;
import org.assertj.core.api.Assertions;
import org.testng.Assert;
import org.testng.annotations.AfterClass;
import org.testng.annotations.BeforeClass;
import org.testng.annotations.Test;

public class TestOAuth2WebUiAuthenticationFilter {
    private static final Duration TTL_ACCESS_TOKEN_IN_SECONDS = Duration.ofSeconds(5L);
    private static final String TRINO_CLIENT_ID = "trino-client";
    private static final String TRINO_CLIENT_SECRET = "trino-secret";
    private static final String TRINO_AUDIENCE = "https://trino/ui";
    private static final String ADDITIONAL_AUDIENCE = "https://external-service.com";
    private static final String TRUSTED_CLIENT_ID = "trusted-client";
    private static final String TRUSTED_CLIENT_SECRET = "trusted-secret";
    private static final String UNTRUSTED_CLIENT_ID = "untrusted-client";
    private static final String UNTRUSTED_CLIENT_SECRET = "untrusted-secret";
    private static final String UNTRUSTED_CLIENT_AUDIENCE = "https://untrusted.com";
    private final Logging logging = Logging.initialize();
    private final OkHttpClient httpClient;
    private TestingHydraIdentityProvider hydraIdP;
    private TestingTrinoServer server;
    private URI serverUri;
    private URI uiUri;

    public TestOAuth2WebUiAuthenticationFilter() {
        OkHttpClient.Builder httpClientBuilder = new OkHttpClient.Builder();
        OkHttpUtil.setupInsecureSsl((OkHttpClient.Builder)httpClientBuilder);
        httpClientBuilder.followRedirects(false);
        this.httpClient = httpClientBuilder.build();
    }

    @BeforeClass
    public void setup() throws Exception {
        this.logging.setLevel(OAuth2WebUiAuthenticationFilter.class.getName(), Level.DEBUG);
        this.hydraIdP = new TestingHydraIdentityProvider(TTL_ACCESS_TOKEN_IN_SECONDS);
        this.hydraIdP.start();
        String idpUrl = "https://localhost:" + this.hydraIdP.getAuthPort();
        this.server = TestingTrinoServer.builder().setCoordinator(true).setAdditionalModule((Module)new WebUiModule()).setProperties((Map)ImmutableMap.builder().put((Object)"web-ui.enabled", (Object)"true").put((Object)"web-ui.authentication.type", (Object)"oauth2").put((Object)"http-server.https.enabled", (Object)"true").put((Object)"http-server.https.keystore.path", (Object)Resources.getResource((String)"cert/localhost.pem").getPath()).put((Object)"http-server.https.keystore.key", (Object)"").put((Object)"http-server.authentication.oauth2.auth-url", (Object)(idpUrl + "/oauth2/auth")).put((Object)"http-server.authentication.oauth2.token-url", (Object)(idpUrl + "/oauth2/token")).put((Object)"http-server.authentication.oauth2.jwks-url", (Object)(idpUrl + "/.well-known/jwks.json")).put((Object)"http-server.authentication.oauth2.client-id", (Object)TRINO_CLIENT_ID).put((Object)"http-server.authentication.oauth2.client-secret", (Object)TRINO_CLIENT_SECRET).put((Object)"http-server.authentication.oauth2.audience", (Object)TRINO_AUDIENCE).put((Object)"http-server.authentication.oauth2.user-mapping.pattern", (Object)"(.*)(@.*)?").put((Object)"oauth2-jwk.http-client.trust-store-path", (Object)Resources.getResource((String)"cert/localhost.pem").getPath()).build()).build();
        this.server.waitForNodeRefresh(Duration.ofSeconds(10L));
        this.serverUri = this.server.getHttpsBaseUrl();
        this.uiUri = this.serverUri.resolve("/ui/");
        this.hydraIdP.createClient(TRINO_CLIENT_ID, TRINO_CLIENT_SECRET, TokenEndpointAuthMethod.CLIENT_SECRET_BASIC, (List<String>)ImmutableList.of((Object)TRINO_AUDIENCE, (Object)ADDITIONAL_AUDIENCE), this.serverUri + "/oauth2/callback");
        this.hydraIdP.createClient(TRUSTED_CLIENT_ID, TRUSTED_CLIENT_SECRET, TokenEndpointAuthMethod.CLIENT_SECRET_BASIC, (List<String>)ImmutableList.of((Object)TRINO_AUDIENCE), this.serverUri + "/oauth2/callback");
        this.hydraIdP.createClient(UNTRUSTED_CLIENT_ID, UNTRUSTED_CLIENT_SECRET, TokenEndpointAuthMethod.CLIENT_SECRET_BASIC, (List<String>)ImmutableList.of((Object)UNTRUSTED_CLIENT_AUDIENCE), "https://untrusted.com/callback");
    }

    @AfterClass(alwaysRun=true)
    public void tearDown() throws Exception {
        this.logging.clearLevel(OAuth2WebUiAuthenticationFilter.class.getName());
        Closeables.closeAll((AutoCloseable[])new AutoCloseable[]{this.server, this.hydraIdP});
    }

    @Test
    public void testUnauthorizedApiCall() throws IOException {
        try (Response response = this.httpClient.newCall(this.apiCall().build()).execute();){
            this.assertUnauthorizedResponse(response);
        }
    }

    @Test
    public void testUnauthorizedUICall() throws IOException {
        try (Response response = this.httpClient.newCall(this.uiCall().build()).execute();){
            this.assertRedirectResponse(response);
        }
    }

    @Test
    public void testUnsignedToken() throws NoSuchAlgorithmException, IOException {
        KeyPairGenerator keyGenerator = KeyPairGenerator.getInstance("RSA");
        SignatureAlgorithm signatureAlgorithm = SignatureAlgorithm.RS256;
        keyGenerator.initialize(4096);
        long now = Instant.now().getEpochSecond();
        String token = Jwts.builder().setHeaderParam("alg", (Object)"RS256").setHeaderParam("kid", (Object)"public:f467aa08-1c1b-4cde-ba45-84b0ef5d2ba8").setHeaderParam("typ", (Object)"JWT").setClaims((Claims)new DefaultClaims((Map)ImmutableMap.builder().put((Object)"aud", (Object)ImmutableList.of()).put((Object)"client_id", (Object)TRINO_CLIENT_ID).put((Object)"exp", (Object)(now + 60L)).put((Object)"iat", (Object)now).put((Object)"iss", (Object)"https://hydra:4444/").put((Object)"jti", (Object)UUID.randomUUID()).put((Object)"nbf", (Object)now).put((Object)"scp", (Object)ImmutableList.of((Object)"openid")).put((Object)"sub", (Object)"foo@bar.com").build())).signWith(signatureAlgorithm, (Key)keyGenerator.generateKeyPair().getPrivate()).compact();
        try (Response response = this.httpClientWithOAuth2Cookie(token).newCall(this.uiCall().build()).execute();){
            this.assertRedirectResponse(response);
        }
    }

    @Test
    public void testTokenWithInvalidAudience() throws IOException {
        String token = this.hydraIdP.getToken(UNTRUSTED_CLIENT_ID, UNTRUSTED_CLIENT_SECRET, (List<String>)ImmutableList.of((Object)UNTRUSTED_CLIENT_AUDIENCE));
        try (Response response = this.httpClientWithOAuth2Cookie(token).newCall(this.uiCall().build()).execute();){
            this.assertUnauthorizedResponse(response);
        }
    }

    @Test
    public void testTokenFromTrustedClient() throws IOException {
        String token = this.hydraIdP.getToken(TRUSTED_CLIENT_ID, TRUSTED_CLIENT_SECRET, (List<String>)ImmutableList.of((Object)TRINO_AUDIENCE));
        this.assertUICallWithCookie(token);
    }

    @Test
    public void testTokenWithMultipleAudiences() throws IOException {
        String token = this.hydraIdP.getToken(TRINO_CLIENT_ID, TRINO_CLIENT_SECRET, (List<String>)ImmutableList.of((Object)TRINO_AUDIENCE, (Object)ADDITIONAL_AUDIENCE));
        this.assertUICallWithCookie(token);
    }

    @Test
    public void testSuccessfulFlow() throws Exception {
        CookieManager cookieManager = new CookieManager();
        CookieStore cookieStore = cookieManager.getCookieStore();
        OkHttpClient.Builder httpClientBuilder = new OkHttpClient.Builder();
        OkHttpUtil.setupInsecureSsl((OkHttpClient.Builder)httpClientBuilder);
        OkHttpClient httpClient = httpClientBuilder.followRedirects(true).cookieJar((CookieJar)new JavaNetCookieJar((CookieHandler)cookieManager)).build();
        Assertions.assertThat(cookieStore.get(this.uiUri)).isEmpty();
        Response response = httpClient.newCall(new Request.Builder().url(this.serverUri + "/ui/").get().build()).execute();
        Assert.assertEquals((int)response.code(), (int)200);
        Assert.assertEquals((String)response.request().url().toString(), (String)this.uiUri.toString());
        Optional<HttpCookie> oauth2Cookie = cookieStore.get(this.uiUri).stream().filter(cookie -> cookie.getName().equals("__Secure-Trino-OAuth2-Token")).findFirst();
        Assertions.assertThat(oauth2Cookie).isNotEmpty();
        this.assertTrinoCookie(oauth2Cookie.get());
        this.assertUICallWithCookie(oauth2Cookie.get().getValue());
    }

    @Test
    public void testExpiredAccessToken() throws Exception {
        String token = this.hydraIdP.getToken(TRINO_CLIENT_ID, TRINO_CLIENT_SECRET, (List<String>)ImmutableList.of((Object)TRINO_AUDIENCE));
        this.assertUICallWithCookie(token);
        Thread.sleep(TTL_ACCESS_TOKEN_IN_SECONDS.plusSeconds(1L).toMillis());
        try (Response response = this.httpClientWithOAuth2Cookie(token).newCall(this.uiCall().build()).execute();){
            this.assertRedirectResponse(response);
        }
    }

    private Request.Builder uiCall() {
        return new Request.Builder().url(this.serverUri.resolve("/ui/").toString()).get();
    }

    private Request.Builder apiCall() {
        return new Request.Builder().url(this.serverUri.resolve("/ui/api/cluster").toString()).get();
    }

    private void assertTrinoCookie(HttpCookie cookie) {
        Assertions.assertThat((String)cookie.getName()).isEqualTo("__Secure-Trino-OAuth2-Token");
        Assertions.assertThat((String)cookie.getDomain()).isIn(new Object[]{"127.0.0.1", "::1"});
        Assertions.assertThat((String)cookie.getPath()).isEqualTo("/ui/");
        Assertions.assertThat((boolean)cookie.getSecure()).isTrue();
        Assertions.assertThat((boolean)cookie.isHttpOnly()).isTrue();
        Assertions.assertThat((String)cookie.getValue()).isNotBlank();
        Jws jwt = Jwts.parser().setSigningKeyResolver((SigningKeyResolver)new JwkSigningKeyResolver(new JwkService(new JwtAuthenticatorConfig().setKeyFile("https://localhost:" + this.hydraIdP.getAuthPort() + "/.well-known/jwks.json"), (HttpClient)new JettyHttpClient(new HttpClientConfig().setTrustStorePath(Resources.getResource((String)"cert/localhost.pem").getPath()))))).parseClaimsJws(cookie.getValue());
        Assertions.assertThat((long)cookie.getMaxAge()).isLessThanOrEqualTo(TTL_ACCESS_TOKEN_IN_SECONDS.getSeconds());
        this.assertAccessToken((Jws<Claims>)jwt);
    }

    private void assertAccessToken(Jws<Claims> jwt) {
        Assertions.assertThat((String)((Claims)jwt.getBody()).getSubject()).isEqualTo("foo@bar.com");
        Assertions.assertThat((Object)((Claims)jwt.getBody()).get((Object)"client_id")).isEqualTo((Object)TRINO_CLIENT_ID);
        Assertions.assertThat((String)((Claims)jwt.getBody()).getIssuer()).isEqualTo("https://localhost:4444/");
    }

    private void assertUICallWithCookie(String cookieValue) throws IOException {
        OkHttpClient httpClient = this.httpClientWithOAuth2Cookie(cookieValue);
        try (Response response = httpClient.newCall(this.uiCall().build()).execute();){
            Assertions.assertThat((int)response.code()).isEqualTo(Response.Status.OK.getStatusCode());
        }
    }

    private OkHttpClient httpClientWithOAuth2Cookie(final String cookieValue) {
        OkHttpClient.Builder httpClientBuilder = new OkHttpClient.Builder();
        OkHttpUtil.setupInsecureSsl((OkHttpClient.Builder)httpClientBuilder);
        httpClientBuilder.followRedirects(false);
        httpClientBuilder.cookieJar(new CookieJar(){

            public void saveFromResponse(HttpUrl url, List<Cookie> cookies) {
            }

            public List<Cookie> loadForRequest(HttpUrl url) {
                if (url.encodedPath().equals("/ui/")) {
                    return ImmutableList.of((Object)new Cookie.Builder().domain(TestOAuth2WebUiAuthenticationFilter.this.serverUri.getHost()).path("/ui/").name("__Secure-Trino-OAuth2-Token").value(cookieValue).httpOnly().secure().build());
                }
                return ImmutableList.of();
            }
        });
        return httpClientBuilder.build();
    }

    private void assertRedirectResponse(Response response) throws MalformedURLException {
        Assertions.assertThat((int)response.code()).isEqualTo(Response.Status.SEE_OTHER.getStatusCode());
        this.assertRedirectUrl(response.header("Location"));
    }

    private void assertUnauthorizedResponse(Response response) throws IOException {
        Assertions.assertThat((int)response.code()).isEqualTo(Response.Status.UNAUTHORIZED.getStatusCode());
        Assertions.assertThat((Object)response.body()).isNotNull();
        Assertions.assertThat((String)response.body().string()).isEqualTo("Unauthorized");
    }

    private void assertRedirectUrl(String redirectUrl) throws MalformedURLException {
        Assertions.assertThat((String)redirectUrl).isNotNull();
        URL location = new URL(redirectUrl);
        HttpUrl url = HttpUrl.parse((String)redirectUrl);
        Assertions.assertThat((Object)url).isNotNull();
        Assertions.assertThat((String)location.getProtocol()).isEqualTo("https");
        Assertions.assertThat((String)location.getHost()).isEqualTo("localhost");
        Assertions.assertThat((int)location.getPort()).isEqualTo(this.hydraIdP.getAuthPort());
        Assertions.assertThat((String)location.getPath()).isEqualTo("/oauth2/auth");
        Assertions.assertThat((List)url.queryParameterValues("response_type")).isEqualTo((Object)ImmutableList.of((Object)"code"));
        Assertions.assertThat((List)url.queryParameterValues("scope")).isEqualTo((Object)ImmutableList.of((Object)"openid"));
        Assertions.assertThat((List)url.queryParameterValues("redirect_uri")).isEqualTo((Object)ImmutableList.of((Object)(this.serverUri + "/oauth2/callback")));
        Assertions.assertThat((List)url.queryParameterValues("client_id")).isEqualTo((Object)ImmutableList.of((Object)TRINO_CLIENT_ID));
        Assertions.assertThat((List)url.queryParameterValues("state")).isNotNull();
    }
}

