/*
 * Decompiled with CFR 0.152.
 */
package io.trino.dispatcher;

import com.google.common.base.Preconditions;
import com.google.common.collect.ImmutableList;
import com.google.common.util.concurrent.AbstractFuture;
import com.google.common.util.concurrent.Futures;
import com.google.common.util.concurrent.ListenableFuture;
import io.trino.Session;
import io.trino.dispatcher.DispatchExecutor;
import io.trino.dispatcher.DispatchInfo;
import io.trino.dispatcher.DispatchQuery;
import io.trino.dispatcher.DispatchQueryFactory;
import io.trino.dispatcher.FailedDispatchQuery;
import io.trino.dispatcher.FailedDispatchQueryFactory;
import io.trino.execution.ManagedQueryExecution;
import io.trino.execution.QueryIdGenerator;
import io.trino.execution.QueryInfo;
import io.trino.execution.QueryManagerConfig;
import io.trino.execution.QueryManagerStats;
import io.trino.execution.QueryPreparer;
import io.trino.execution.QueryState;
import io.trino.execution.QueryTracker;
import io.trino.execution.resourcegroups.ResourceGroupManager;
import io.trino.metadata.SessionPropertyManager;
import io.trino.security.AccessControl;
import io.trino.server.BasicQueryInfo;
import io.trino.server.SessionContext;
import io.trino.server.SessionPropertyDefaults;
import io.trino.server.SessionSupplier;
import io.trino.server.protocol.Slug;
import io.trino.spi.ErrorCodeSupplier;
import io.trino.spi.QueryId;
import io.trino.spi.StandardErrorCode;
import io.trino.spi.TrinoException;
import io.trino.spi.resourcegroups.SelectionContext;
import io.trino.spi.resourcegroups.SelectionCriteria;
import io.trino.transaction.TransactionManager;
import io.trino.util.StatementUtils;
import java.util.List;
import java.util.Objects;
import java.util.Optional;
import java.util.concurrent.Executor;
import java.util.concurrent.ScheduledExecutorService;
import javax.annotation.PostConstruct;
import javax.annotation.PreDestroy;
import javax.inject.Inject;
import org.weakref.jmx.Flatten;
import org.weakref.jmx.Managed;

public class DispatchManager {
    private final QueryIdGenerator queryIdGenerator;
    private final QueryPreparer queryPreparer;
    private final ResourceGroupManager<?> resourceGroupManager;
    private final DispatchQueryFactory dispatchQueryFactory;
    private final FailedDispatchQueryFactory failedDispatchQueryFactory;
    private final TransactionManager transactionManager;
    private final AccessControl accessControl;
    private final SessionSupplier sessionSupplier;
    private final SessionPropertyDefaults sessionPropertyDefaults;
    private final SessionPropertyManager sessionPropertyManager;
    private final int maxQueryLength;
    private final Executor dispatchExecutor;
    private final QueryTracker<DispatchQuery> queryTracker;
    private final QueryManagerStats stats = new QueryManagerStats();

    @Inject
    public DispatchManager(QueryIdGenerator queryIdGenerator, QueryPreparer queryPreparer, ResourceGroupManager<?> resourceGroupManager, DispatchQueryFactory dispatchQueryFactory, FailedDispatchQueryFactory failedDispatchQueryFactory, TransactionManager transactionManager, AccessControl accessControl, SessionSupplier sessionSupplier, SessionPropertyDefaults sessionPropertyDefaults, SessionPropertyManager sessionPropertyManager, QueryManagerConfig queryManagerConfig, DispatchExecutor dispatchExecutor) {
        this.queryIdGenerator = Objects.requireNonNull(queryIdGenerator, "queryIdGenerator is null");
        this.queryPreparer = Objects.requireNonNull(queryPreparer, "queryPreparer is null");
        this.resourceGroupManager = Objects.requireNonNull(resourceGroupManager, "resourceGroupManager is null");
        this.dispatchQueryFactory = Objects.requireNonNull(dispatchQueryFactory, "dispatchQueryFactory is null");
        this.failedDispatchQueryFactory = Objects.requireNonNull(failedDispatchQueryFactory, "failedDispatchQueryFactory is null");
        this.transactionManager = Objects.requireNonNull(transactionManager, "transactionManager is null");
        this.accessControl = Objects.requireNonNull(accessControl, "accessControl is null");
        this.sessionSupplier = Objects.requireNonNull(sessionSupplier, "sessionSupplier is null");
        this.sessionPropertyDefaults = Objects.requireNonNull(sessionPropertyDefaults, "sessionPropertyDefaults is null");
        this.sessionPropertyManager = sessionPropertyManager;
        Objects.requireNonNull(queryManagerConfig, "queryManagerConfig is null");
        this.maxQueryLength = queryManagerConfig.getMaxQueryLength();
        this.dispatchExecutor = Objects.requireNonNull(dispatchExecutor, "dispatchExecutor is null").getExecutor();
        this.queryTracker = new QueryTracker(queryManagerConfig, (ScheduledExecutorService)dispatchExecutor.getScheduledExecutor());
    }

    @PostConstruct
    public void start() {
        this.queryTracker.start();
    }

    @PreDestroy
    public void stop() {
        this.queryTracker.stop();
    }

    @Managed
    @Flatten
    public QueryManagerStats getStats() {
        return this.stats;
    }

    public QueryId createQueryId() {
        return this.queryIdGenerator.createNextQueryId();
    }

    public ListenableFuture<Void> createQuery(QueryId queryId, Slug slug, SessionContext sessionContext, String query) {
        Objects.requireNonNull(queryId, "queryId is null");
        Objects.requireNonNull(sessionContext, "sessionContext is null");
        Objects.requireNonNull(query, "query is null");
        Preconditions.checkArgument((!query.isEmpty() ? 1 : 0) != 0, (Object)"query must not be empty string");
        Preconditions.checkArgument((boolean)this.queryTracker.tryGetQuery(queryId).isEmpty(), (String)"query %s already exists", (Object)queryId);
        DispatchQueryCreationFuture queryCreationFuture = new DispatchQueryCreationFuture();
        this.dispatchExecutor.execute(() -> {
            try {
                this.createQueryInternal(queryId, slug, sessionContext, query, this.resourceGroupManager);
            }
            finally {
                queryCreationFuture.set(null);
            }
        });
        return queryCreationFuture;
    }

    private <C> void createQueryInternal(QueryId queryId, Slug slug, SessionContext sessionContext, String query, ResourceGroupManager<C> resourceGroupManager) {
        Session session = null;
        QueryPreparer.PreparedQuery preparedQuery = null;
        try {
            if (query.length() > this.maxQueryLength) {
                int queryLength = query.length();
                query = query.substring(0, this.maxQueryLength);
                throw new TrinoException((ErrorCodeSupplier)StandardErrorCode.QUERY_TEXT_TOO_LARGE, String.format("Query text length (%s) exceeds the maximum length (%s)", queryLength, this.maxQueryLength));
            }
            session = this.sessionSupplier.createSession(queryId, sessionContext);
            this.accessControl.checkCanExecuteQuery(sessionContext.getIdentity());
            preparedQuery = this.queryPreparer.prepareQuery(session, query);
            Optional<String> queryType = StatementUtils.getQueryType(preparedQuery.getStatement()).map(Enum::name);
            SelectionContext<C> selectionContext = resourceGroupManager.selectGroup(new SelectionCriteria(sessionContext.getIdentity().getPrincipal().isPresent(), sessionContext.getIdentity().getUser(), sessionContext.getIdentity().getGroups(), Optional.ofNullable(sessionContext.getSource()), sessionContext.getClientTags(), sessionContext.getResourceEstimates(), queryType));
            session = this.sessionPropertyDefaults.newSessionWithDefaultProperties(session, queryType, selectionContext.getResourceGroupId());
            this.transactionManager.activateTransaction(session, StatementUtils.isTransactionControlStatement(preparedQuery.getStatement()), this.accessControl);
            DispatchQuery dispatchQuery = this.dispatchQueryFactory.createDispatchQuery(session, query, preparedQuery, slug, selectionContext.getResourceGroupId());
            boolean queryAdded = this.queryCreated(dispatchQuery);
            if (queryAdded && !dispatchQuery.isDone()) {
                try {
                    resourceGroupManager.submit(dispatchQuery, selectionContext, this.dispatchExecutor);
                }
                catch (Throwable e) {
                    dispatchQuery.fail(e);
                }
            }
        }
        catch (Throwable throwable) {
            if (session == null) {
                session = Session.builder(this.sessionPropertyManager).setQueryId(queryId).setIdentity(sessionContext.getIdentity()).setSource(sessionContext.getSource()).build();
            }
            Optional<String> preparedSql = Optional.ofNullable(preparedQuery).flatMap(QueryPreparer.PreparedQuery::getPrepareSql);
            FailedDispatchQuery failedDispatchQuery = this.failedDispatchQueryFactory.createFailedDispatchQuery(session, query, preparedSql, Optional.empty(), throwable);
            this.queryCreated(failedDispatchQuery);
        }
    }

    private boolean queryCreated(DispatchQuery dispatchQuery) {
        boolean queryAdded = this.queryTracker.addQuery(dispatchQuery);
        if (queryAdded) {
            dispatchQuery.addStateChangeListener(newState -> {
                if (newState.isDone()) {
                    this.queryTracker.expireQuery(dispatchQuery.getQueryId());
                }
            });
            this.stats.trackQueryStats(dispatchQuery);
        }
        return queryAdded;
    }

    public ListenableFuture<Void> waitForDispatched(QueryId queryId) {
        return this.queryTracker.tryGetQuery(queryId).map(dispatchQuery -> {
            dispatchQuery.recordHeartbeat();
            return dispatchQuery.getDispatchedFuture();
        }).orElseGet(Futures::immediateVoidFuture);
    }

    public List<BasicQueryInfo> getQueries() {
        return (List)this.queryTracker.getAllQueries().stream().map(ManagedQueryExecution::getBasicQueryInfo).collect(ImmutableList.toImmutableList());
    }

    @Managed
    public long getQueuedQueries() {
        return this.queryTracker.getAllQueries().stream().filter(query -> query.getState() == QueryState.QUEUED).count();
    }

    @Managed
    public long getRunningQueries() {
        return this.queryTracker.getAllQueries().stream().filter(query -> query.getState() == QueryState.RUNNING && !query.getBasicQueryInfo().getQueryStats().isFullyBlocked()).count();
    }

    public boolean isQueryRegistered(QueryId queryId) {
        return this.queryTracker.tryGetQuery(queryId).isPresent();
    }

    public DispatchQuery getQuery(QueryId queryId) {
        return this.queryTracker.getQuery(queryId);
    }

    public BasicQueryInfo getQueryInfo(QueryId queryId) {
        return this.queryTracker.getQuery(queryId).getBasicQueryInfo();
    }

    public Optional<QueryInfo> getFullQueryInfo(QueryId queryId) {
        return this.queryTracker.tryGetQuery(queryId).map(ManagedQueryExecution::getFullQueryInfo);
    }

    public Optional<DispatchInfo> getDispatchInfo(QueryId queryId) {
        return this.queryTracker.tryGetQuery(queryId).map(dispatchQuery -> {
            dispatchQuery.recordHeartbeat();
            return dispatchQuery.getDispatchInfo();
        });
    }

    public void cancelQuery(QueryId queryId) {
        this.queryTracker.tryGetQuery(queryId).ifPresent(DispatchQuery::cancel);
    }

    public void failQuery(QueryId queryId, Throwable cause) {
        Objects.requireNonNull(cause, "cause is null");
        this.queryTracker.tryGetQuery(queryId).ifPresent(query -> query.fail(cause));
    }

    private static class DispatchQueryCreationFuture
    extends AbstractFuture<Void> {
        private DispatchQueryCreationFuture() {
        }

        protected boolean set(Void value) {
            return super.set((Object)value);
        }

        protected boolean setException(Throwable throwable) {
            return super.setException(throwable);
        }

        public boolean cancel(boolean mayInterruptIfRunning) {
            return false;
        }
    }
}

