/*
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package io.trino.operator;

import com.google.common.util.concurrent.ListenableFuture;
import com.google.common.util.concurrent.SettableFuture;

import javax.annotation.concurrent.GuardedBy;
import javax.annotation.concurrent.ThreadSafe;

import static com.google.common.base.Preconditions.checkArgument;
import static com.google.common.base.Preconditions.checkState;

@ThreadSafe
public class ReferenceCount
{
    private final SettableFuture<Void> freeFuture = SettableFuture.create();

    @GuardedBy("this")
    private int count;

    public ReferenceCount(int initialCount)
    {
        checkArgument(initialCount >= 0, "initialCount must not be negative, got %s", initialCount);
        count = initialCount;
        if (initialCount == 0) {
            freeFuture.set(null);
        }
    }

    public ListenableFuture<Void> getFreeFuture()
    {
        return freeFuture;
    }

    public synchronized void retain()
    {
        checkState(!freeFuture.isDone(), "Reference has already been freed");
        count++;
    }

    public synchronized void release()
    {
        checkState(!freeFuture.isDone(), "Reference has already been freed");
        count--;
        if (count == 0) {
            freeFuture.set(null);
        }
    }
}
