/*
 * Decompiled with CFR 0.152.
 */
package io.trino.memory;

import com.google.common.annotations.VisibleForTesting;
import com.google.common.base.Preconditions;
import com.google.common.base.Verify;
import com.google.common.collect.ImmutableMap;
import com.google.common.util.concurrent.ListenableFuture;
import io.airlift.stats.GcMonitor;
import io.airlift.units.DataSize;
import io.trino.ExceededMemoryLimitException;
import io.trino.ExceededSpillLimitException;
import io.trino.Session;
import io.trino.execution.TaskId;
import io.trino.execution.TaskStateMachine;
import io.trino.memory.MemoryPool;
import io.trino.memory.QueryContextVisitor;
import io.trino.memory.context.AggregatedMemoryContext;
import io.trino.memory.context.MemoryReservationHandler;
import io.trino.memory.context.MemoryTrackingContext;
import io.trino.operator.Operator;
import io.trino.operator.TaskContext;
import io.trino.spi.QueryId;
import io.trino.spiller.SpillSpaceTracker;
import java.util.Comparator;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.concurrent.ConcurrentHashMap;
import java.util.concurrent.Executor;
import java.util.concurrent.ScheduledExecutorService;
import java.util.function.BiFunction;
import java.util.function.BiPredicate;
import java.util.stream.Collectors;
import javax.annotation.concurrent.GuardedBy;
import javax.annotation.concurrent.ThreadSafe;

@ThreadSafe
public class QueryContext {
    private static final long GUARANTEED_MEMORY = DataSize.of((long)1L, (DataSize.Unit)DataSize.Unit.MEGABYTE).toBytes();
    private final QueryId queryId;
    private final GcMonitor gcMonitor;
    private final Executor notificationExecutor;
    private final ScheduledExecutorService yieldExecutor;
    private final long maxSpill;
    private final SpillSpaceTracker spillSpaceTracker;
    private final Map<TaskId, TaskContext> taskContexts = new ConcurrentHashMap<TaskId, TaskContext>();
    private volatile boolean memoryLimitsInitialized;
    @GuardedBy(value="this")
    private long maxUserMemory;
    private final MemoryTrackingContext queryMemoryContext;
    private final MemoryPool memoryPool;
    @GuardedBy(value="this")
    private long spillUsed;

    public QueryContext(QueryId queryId, DataSize maxUserMemory, MemoryPool memoryPool, GcMonitor gcMonitor, Executor notificationExecutor, ScheduledExecutorService yieldExecutor, DataSize maxSpill, SpillSpaceTracker spillSpaceTracker) {
        this(queryId, maxUserMemory, memoryPool, GUARANTEED_MEMORY, gcMonitor, notificationExecutor, yieldExecutor, maxSpill, spillSpaceTracker);
    }

    public QueryContext(QueryId queryId, DataSize maxUserMemory, MemoryPool memoryPool, long guaranteedMemory, GcMonitor gcMonitor, Executor notificationExecutor, ScheduledExecutorService yieldExecutor, DataSize maxSpill, SpillSpaceTracker spillSpaceTracker) {
        this.queryId = Objects.requireNonNull(queryId, "queryId is null");
        this.maxUserMemory = Objects.requireNonNull(maxUserMemory, "maxUserMemory is null").toBytes();
        this.memoryPool = Objects.requireNonNull(memoryPool, "memoryPool is null");
        this.gcMonitor = Objects.requireNonNull(gcMonitor, "gcMonitor is null");
        this.notificationExecutor = Objects.requireNonNull(notificationExecutor, "notificationExecutor is null");
        this.yieldExecutor = Objects.requireNonNull(yieldExecutor, "yieldExecutor is null");
        this.maxSpill = Objects.requireNonNull(maxSpill, "maxSpill is null").toBytes();
        this.spillSpaceTracker = Objects.requireNonNull(spillSpaceTracker, "spillSpaceTracker is null");
        this.queryMemoryContext = new MemoryTrackingContext(AggregatedMemoryContext.newRootAggregatedMemoryContext((MemoryReservationHandler)new QueryMemoryReservationHandler(this::updateUserMemory, this::tryUpdateUserMemory), (long)guaranteedMemory), AggregatedMemoryContext.newRootAggregatedMemoryContext((MemoryReservationHandler)new QueryMemoryReservationHandler(this::updateRevocableMemory, this::tryReserveMemoryNotSupported), (long)0L));
    }

    public boolean isMemoryLimitsInitialized() {
        return this.memoryLimitsInitialized;
    }

    public synchronized void initializeMemoryLimits(boolean resourceOverCommit, long maxUserMemory) {
        Preconditions.checkArgument((maxUserMemory >= 0L ? 1 : 0) != 0, (String)"maxUserMemory must be >= 0, found: %s", (long)maxUserMemory);
        this.maxUserMemory = resourceOverCommit ? this.memoryPool.getMaxBytes() : maxUserMemory;
        this.memoryLimitsInitialized = true;
    }

    @VisibleForTesting
    MemoryTrackingContext getQueryMemoryContext() {
        return this.queryMemoryContext;
    }

    @VisibleForTesting
    public synchronized long getMaxUserMemory() {
        return this.maxUserMemory;
    }

    public QueryId getQueryId() {
        return this.queryId;
    }

    private synchronized ListenableFuture<Void> updateUserMemory(String allocationTag, long delta) {
        if (delta >= 0L) {
            this.enforceUserMemoryLimit(this.queryMemoryContext.getUserMemory(), delta, this.maxUserMemory);
            ListenableFuture<Void> future = this.memoryPool.reserve(this.queryId, allocationTag, delta);
            if (future.isDone()) {
                return Operator.NOT_BLOCKED;
            }
            return future;
        }
        this.memoryPool.free(this.queryId, allocationTag, -delta);
        return Operator.NOT_BLOCKED;
    }

    private synchronized ListenableFuture<Void> updateRevocableMemory(String allocationTag, long delta) {
        if (delta >= 0L) {
            ListenableFuture<Void> future = this.memoryPool.reserveRevocable(this.queryId, delta);
            if (future.isDone()) {
                return Operator.NOT_BLOCKED;
            }
            return future;
        }
        this.memoryPool.freeRevocable(this.queryId, -delta);
        return Operator.NOT_BLOCKED;
    }

    public synchronized ListenableFuture<Void> reserveSpill(long bytes) {
        Preconditions.checkArgument((bytes >= 0L ? 1 : 0) != 0, (Object)"bytes is negative");
        if (this.spillUsed + bytes > this.maxSpill) {
            throw ExceededSpillLimitException.exceededPerQueryLocalLimit(DataSize.succinctBytes((long)this.maxSpill));
        }
        ListenableFuture<Void> future = this.spillSpaceTracker.reserve(bytes);
        this.spillUsed += bytes;
        return future;
    }

    private synchronized boolean tryUpdateUserMemory(String allocationTag, long delta) {
        if (delta <= 0L) {
            ListenableFuture<Void> future = this.updateUserMemory(allocationTag, delta);
            if (delta < 0L) {
                Verify.verify((boolean)future.isDone(), (String)"future should be done", (Object[])new Object[0]);
            }
            return true;
        }
        if (this.queryMemoryContext.getUserMemory() + delta > this.maxUserMemory) {
            return false;
        }
        return this.memoryPool.tryReserve(this.queryId, allocationTag, delta);
    }

    public synchronized void freeSpill(long bytes) {
        Preconditions.checkArgument((this.spillUsed - bytes >= 0L ? 1 : 0) != 0, (Object)"tried to free more memory than is reserved");
        this.spillUsed -= bytes;
        this.spillSpaceTracker.free(bytes);
    }

    public synchronized MemoryPool getMemoryPool() {
        return this.memoryPool;
    }

    public TaskContext addTaskContext(TaskStateMachine taskStateMachine, Session session, Runnable notifyStatusChanged, boolean perOperatorCpuTimerEnabled, boolean cpuTimerEnabled) {
        TaskContext taskContext = TaskContext.createTaskContext(this, taskStateMachine, this.gcMonitor, this.notificationExecutor, this.yieldExecutor, session, this.queryMemoryContext.newMemoryTrackingContext(), notifyStatusChanged, perOperatorCpuTimerEnabled, cpuTimerEnabled);
        this.taskContexts.put(taskStateMachine.getTaskId(), taskContext);
        return taskContext;
    }

    public <C, R> R accept(QueryContextVisitor<C, R> visitor, C context) {
        return visitor.visitQueryContext(this, context);
    }

    public <C, R> List<R> acceptChildren(QueryContextVisitor<C, R> visitor, C context) {
        return this.taskContexts.values().stream().map(taskContext -> taskContext.accept(visitor, context)).collect(Collectors.toList());
    }

    public TaskContext getTaskContextByTaskId(TaskId taskId) {
        TaskContext taskContext = this.taskContexts.get(taskId);
        return (TaskContext)Verify.verifyNotNull((Object)taskContext, (String)"task does not exist", (Object[])new Object[0]);
    }

    private boolean tryReserveMemoryNotSupported(String allocationTag, long bytes) {
        throw new UnsupportedOperationException("tryReserveMemory is not supported");
    }

    @GuardedBy(value="this")
    private void enforceUserMemoryLimit(long allocated, long delta, long maxMemory) {
        if (allocated + delta > maxMemory) {
            throw ExceededMemoryLimitException.exceededLocalUserMemoryLimit(DataSize.succinctBytes((long)maxMemory), this.getAdditionalFailureInfo(allocated, delta));
        }
    }

    @GuardedBy(value="this")
    private String getAdditionalFailureInfo(long allocated, long delta) {
        Map<String, Long> queryAllocations = this.memoryPool.getTaggedMemoryAllocations().get(this.queryId);
        String additionalInfo = String.format("Allocated: %s, Delta: %s", DataSize.succinctBytes((long)allocated), DataSize.succinctBytes((long)delta));
        if (queryAllocations == null) {
            return additionalInfo;
        }
        String topConsumers = ((ImmutableMap)queryAllocations.entrySet().stream().sorted(Map.Entry.comparingByValue(Comparator.reverseOrder())).limit(3L).filter(e -> (Long)e.getValue() >= 0L).collect(ImmutableMap.toImmutableMap(Map.Entry::getKey, e -> DataSize.succinctBytes((long)((Long)e.getValue()))))).toString();
        return String.format("%s, Top Consumers: %s", additionalInfo, topConsumers);
    }

    private static class QueryMemoryReservationHandler
    implements MemoryReservationHandler {
        private final BiFunction<String, Long, ListenableFuture<Void>> reserveMemoryFunction;
        private final BiPredicate<String, Long> tryReserveMemoryFunction;

        public QueryMemoryReservationHandler(BiFunction<String, Long, ListenableFuture<Void>> reserveMemoryFunction, BiPredicate<String, Long> tryReserveMemoryFunction) {
            this.reserveMemoryFunction = Objects.requireNonNull(reserveMemoryFunction, "reserveMemoryFunction is null");
            this.tryReserveMemoryFunction = Objects.requireNonNull(tryReserveMemoryFunction, "tryReserveMemoryFunction is null");
        }

        public ListenableFuture<Void> reserveMemory(String allocationTag, long delta) {
            return this.reserveMemoryFunction.apply(allocationTag, delta);
        }

        public boolean tryReserveMemory(String allocationTag, long delta) {
            return this.tryReserveMemoryFunction.test(allocationTag, delta);
        }
    }
}

