/*
 * Decompiled with CFR 0.152.
 */
package io.trino.operator.project;

import com.google.common.annotations.VisibleForTesting;
import com.google.common.base.Preconditions;
import com.google.common.base.Verify;
import com.google.common.collect.ImmutableList;
import io.airlift.slice.SizeOf;
import io.trino.array.ReferenceCountMap;
import io.trino.memory.context.LocalMemoryContext;
import io.trino.operator.DriverYieldSignal;
import io.trino.operator.PageUtils;
import io.trino.operator.Work;
import io.trino.operator.WorkProcessor;
import io.trino.operator.project.DictionaryAwarePageFilter;
import io.trino.operator.project.DictionaryAwarePageProjection;
import io.trino.operator.project.InputPageProjection;
import io.trino.operator.project.PageFilter;
import io.trino.operator.project.PageProjection;
import io.trino.operator.project.SelectedPositions;
import io.trino.spi.Page;
import io.trino.spi.block.Block;
import io.trino.spi.block.DictionaryBlock;
import io.trino.spi.block.DictionaryId;
import io.trino.spi.connector.ConnectorSession;
import io.trino.sql.gen.ExpressionProfiler;
import java.util.HashMap;
import java.util.Iterator;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Optional;
import java.util.OptionalInt;
import java.util.function.Function;
import javax.annotation.concurrent.NotThreadSafe;

@NotThreadSafe
public class PageProcessor {
    public static final int MAX_BATCH_SIZE = 8192;
    static final int MAX_PAGE_SIZE_IN_BYTES = 0x400000;
    static final int MIN_PAGE_SIZE_IN_BYTES = 0x100000;
    private final ExpressionProfiler expressionProfiler;
    private final DictionarySourceIdFunction dictionarySourceIdFunction = new DictionarySourceIdFunction();
    private final Optional<PageFilter> filter;
    private final List<PageProjection> projections;
    private int projectBatchSize;

    @VisibleForTesting
    public PageProcessor(Optional<PageFilter> filter, List<? extends PageProjection> projections, OptionalInt initialBatchSize) {
        this(filter, projections, initialBatchSize, new ExpressionProfiler());
    }

    @VisibleForTesting
    public PageProcessor(Optional<PageFilter> filter, List<? extends PageProjection> projections, OptionalInt initialBatchSize, ExpressionProfiler expressionProfiler) {
        this.filter = Objects.requireNonNull(filter, "filter is null").map(pageFilter -> {
            if (pageFilter.getInputChannels().size() == 1 && pageFilter.isDeterministic()) {
                return new DictionaryAwarePageFilter((PageFilter)pageFilter);
            }
            return pageFilter;
        });
        this.projections = (List)Objects.requireNonNull(projections, "projections is null").stream().map(projection -> {
            if (projection.getInputChannels().size() == 1 && projection.isDeterministic()) {
                return new DictionaryAwarePageProjection((PageProjection)projection, this.dictionarySourceIdFunction, projection instanceof InputPageProjection);
            }
            return projection;
        }).collect(ImmutableList.toImmutableList());
        this.projectBatchSize = initialBatchSize.orElse(1);
        this.expressionProfiler = Objects.requireNonNull(expressionProfiler, "expressionProfiler is null");
    }

    public PageProcessor(Optional<PageFilter> filter, List<? extends PageProjection> projections) {
        this(filter, projections, OptionalInt.of(1));
    }

    public Iterator<Optional<Page>> process(ConnectorSession session, DriverYieldSignal yieldSignal, LocalMemoryContext memoryContext, Page page) {
        return this.process(session, yieldSignal, memoryContext, page, false);
    }

    public Iterator<Optional<Page>> process(ConnectorSession session, DriverYieldSignal yieldSignal, LocalMemoryContext memoryContext, Page page, boolean avoidPageMaterialization) {
        WorkProcessor<Page> processor = this.createWorkProcessor(session, yieldSignal, memoryContext, page, avoidPageMaterialization);
        return processor.yieldingIterator();
    }

    public WorkProcessor<Page> createWorkProcessor(ConnectorSession session, DriverYieldSignal yieldSignal, LocalMemoryContext memoryContext, Page page, boolean avoidPageMaterialization) {
        this.dictionarySourceIdFunction.reset();
        if (page.getPositionCount() == 0) {
            return WorkProcessor.of(new Page[0]);
        }
        if (this.filter.isPresent()) {
            SelectedPositions selectedPositions = this.filter.get().filter(session, this.filter.get().getInputChannels().getInputChannels(page));
            if (selectedPositions.isEmpty()) {
                return WorkProcessor.of(new Page[0]);
            }
            if (this.projections.isEmpty()) {
                return WorkProcessor.of(new Page(selectedPositions.size()));
            }
            if (selectedPositions.size() != page.getPositionCount()) {
                return WorkProcessor.create(new ProjectSelectedPositions(session, yieldSignal, memoryContext, page, selectedPositions, avoidPageMaterialization));
            }
        } else if (this.projections.isEmpty()) {
            return WorkProcessor.of(new Page(page.getPositionCount()));
        }
        return WorkProcessor.create(new ProjectSelectedPositions(session, yieldSignal, memoryContext, page, SelectedPositions.positionsRange(0, page.getPositionCount()), avoidPageMaterialization));
    }

    @VisibleForTesting
    public List<PageProjection> getProjections() {
        return this.projections;
    }

    private static class ProcessBatchResult {
        private final ProcessBatchState state;
        private final Page page;

        private ProcessBatchResult(ProcessBatchState state, Page page) {
            this.state = state;
            this.page = page;
        }

        public static ProcessBatchResult processBatchYield() {
            return new ProcessBatchResult(ProcessBatchState.YIELD, null);
        }

        public static ProcessBatchResult processBatchTooLarge() {
            return new ProcessBatchResult(ProcessBatchState.PAGE_TOO_LARGE, null);
        }

        public static ProcessBatchResult processBatchSuccess(Page page) {
            return new ProcessBatchResult(ProcessBatchState.SUCCESS, Objects.requireNonNull(page));
        }

        public boolean isYieldFinish() {
            return this.state == ProcessBatchState.YIELD;
        }

        public boolean isPageTooLarge() {
            return this.state == ProcessBatchState.PAGE_TOO_LARGE;
        }

        public boolean isSuccess() {
            return this.state == ProcessBatchState.SUCCESS;
        }

        public Page getPage() {
            Verify.verify((this.state == ProcessBatchState.SUCCESS ? 1 : 0) != 0);
            return (Page)Verify.verifyNotNull((Object)this.page);
        }

        private static enum ProcessBatchState {
            YIELD,
            PAGE_TOO_LARGE,
            SUCCESS;

        }
    }

    @NotThreadSafe
    private static class DictionarySourceIdFunction
    implements Function<DictionaryBlock, DictionaryId> {
        private final Map<DictionaryId, DictionaryId> dictionarySourceIds = new HashMap<DictionaryId, DictionaryId>();

        private DictionarySourceIdFunction() {
        }

        @Override
        public DictionaryId apply(DictionaryBlock block) {
            return this.dictionarySourceIds.computeIfAbsent(block.getDictionarySourceId(), ignored -> DictionaryId.randomDictionaryId());
        }

        public void reset() {
            this.dictionarySourceIds.clear();
        }
    }

    private class ProjectSelectedPositions
    implements WorkProcessor.Process<Page> {
        private final ConnectorSession session;
        private final DriverYieldSignal yieldSignal;
        private final LocalMemoryContext memoryContext;
        private final boolean avoidPageMaterialization;
        private Page page;
        private final Block[] previouslyComputedResults;
        private SelectedPositions selectedPositions;
        private long retainedSizeInBytes;
        private boolean lastComputeYielded;
        private int lastComputeBatchSize;
        private Work<Block> pageProjectWork;
        private int outputPagePositions = -1;
        private long outputPageSizeInBytes;

        private ProjectSelectedPositions(ConnectorSession session, DriverYieldSignal yieldSignal, LocalMemoryContext memoryContext, Page page, SelectedPositions selectedPositions, boolean avoidPageMaterialization) {
            Preconditions.checkArgument((!selectedPositions.isEmpty() ? 1 : 0) != 0, (Object)"selectedPositions is empty");
            this.session = session;
            this.yieldSignal = yieldSignal;
            this.page = page;
            this.memoryContext = memoryContext;
            this.avoidPageMaterialization = avoidPageMaterialization;
            this.selectedPositions = selectedPositions;
            this.previouslyComputedResults = new Block[PageProcessor.this.projections.size()];
        }

        @Override
        public WorkProcessor.ProcessState<Page> process() {
            int i;
            ProcessBatchResult result;
            int batchSize;
            if (this.avoidPageMaterialization && this.outputPagePositions != -1) {
                this.updateBatchSize(this.outputPagePositions, this.outputPageSizeInBytes);
                this.outputPagePositions = -1;
            }
            while (true) {
                if (this.selectedPositions.isEmpty()) {
                    Verify.verify((!this.lastComputeYielded ? 1 : 0) != 0);
                    return WorkProcessor.ProcessState.finished();
                }
                if (this.lastComputeYielded) {
                    Verify.verify((this.lastComputeBatchSize > 0 ? 1 : 0) != 0);
                    batchSize = this.lastComputeBatchSize;
                    this.lastComputeYielded = false;
                    this.lastComputeBatchSize = 0;
                } else {
                    batchSize = Math.min(this.selectedPositions.size(), PageProcessor.this.projectBatchSize);
                }
                result = this.processBatch(batchSize);
                if (result.isYieldFinish()) {
                    this.lastComputeYielded = true;
                    this.lastComputeBatchSize = batchSize;
                    this.updateRetainedSize();
                    return WorkProcessor.ProcessState.yielded();
                }
                if (!result.isPageTooLarge()) break;
                Verify.verify((batchSize > 1 ? 1 : 0) != 0);
                PageProcessor.this.projectBatchSize /= 2;
            }
            Verify.verify((boolean)result.isSuccess());
            Page resultPage = result.getPage();
            if (!this.avoidPageMaterialization) {
                this.updateBatchSize(resultPage.getPositionCount(), resultPage.getSizeInBytes());
            } else {
                this.outputPagePositions = resultPage.getPositionCount();
                this.outputPageSizeInBytes = 0L;
                PageUtils.recordMaterializedBytes(resultPage, sizeInBytes -> this.outputPageSizeInBytes += sizeInBytes);
            }
            this.selectedPositions = this.selectedPositions.subRange(batchSize, this.selectedPositions.size());
            for (i = 0; i < this.previouslyComputedResults.length; ++i) {
                this.previouslyComputedResults[i] = this.previouslyComputedResults[i] != null && this.previouslyComputedResults[i].getPositionCount() > batchSize ? this.previouslyComputedResults[i].getRegion(batchSize, this.previouslyComputedResults[i].getPositionCount() - batchSize) : null;
            }
            if (!this.selectedPositions.isEmpty()) {
                this.updateRetainedSize();
            } else {
                this.page = null;
                for (i = 0; i < this.previouslyComputedResults.length; ++i) {
                    this.previouslyComputedResults[i] = null;
                }
                this.memoryContext.setBytes(0L);
            }
            return WorkProcessor.ProcessState.ofResult(resultPage);
        }

        private void updateBatchSize(int positionCount, long pageSize) {
            if (positionCount > 1 && (pageSize > 0x400000L || PageProcessor.this.expressionProfiler.isExpressionExpensive())) {
                PageProcessor.this.projectBatchSize /= 2;
            }
            if (pageSize < 0x100000L && PageProcessor.this.projectBatchSize < 8192 && !PageProcessor.this.expressionProfiler.isExpressionExpensive()) {
                PageProcessor.this.projectBatchSize *= 2;
            }
        }

        private void updateRetainedSize() {
            this.retainedSizeInBytes = (long)Page.INSTANCE_SIZE + SizeOf.sizeOfObjectArray((int)this.page.getChannelCount());
            ReferenceCountMap referenceCountMap = new ReferenceCountMap();
            for (int channel = 0; channel < this.page.getChannelCount(); ++channel) {
                Block block = this.page.getBlock(channel);
                if (!block.isLoaded()) continue;
                block.retainedBytesForEachPart((object, size) -> {
                    if (referenceCountMap.incrementAndGet(object) == 1) {
                        this.retainedSizeInBytes += size;
                    }
                });
            }
            for (Block previouslyComputedResult : this.previouslyComputedResults) {
                if (previouslyComputedResult == null) continue;
                previouslyComputedResult.retainedBytesForEachPart((object, size) -> {
                    if (referenceCountMap.incrementAndGet(object) == 1) {
                        this.retainedSizeInBytes += size;
                    }
                });
            }
            this.memoryContext.setBytes(this.retainedSizeInBytes);
        }

        private ProcessBatchResult processBatch(int batchSize) {
            Block[] blocks = new Block[PageProcessor.this.projections.size()];
            int pageSize = 0;
            SelectedPositions positionsBatch = this.selectedPositions.subRange(0, batchSize);
            for (int i = 0; i < PageProcessor.this.projections.size(); ++i) {
                if (this.yieldSignal.isSet()) {
                    return ProcessBatchResult.processBatchYield();
                }
                if (positionsBatch.size() > 1 && pageSize > 0x400000) {
                    return ProcessBatchResult.processBatchTooLarge();
                }
                PageProjection projection = PageProcessor.this.projections.get(i);
                if (this.previouslyComputedResults[i] != null && this.previouslyComputedResults[i].getPositionCount() >= batchSize) {
                    blocks[i] = this.previouslyComputedResults[i].getRegion(0, batchSize);
                } else {
                    if (this.pageProjectWork == null) {
                        PageProcessor.this.expressionProfiler.start();
                        this.pageProjectWork = projection.project(this.session, this.yieldSignal, projection.getInputChannels().getInputChannels(this.page), positionsBatch);
                        PageProcessor.this.expressionProfiler.stop(positionsBatch.size());
                    }
                    if (!this.pageProjectWork.process()) {
                        return ProcessBatchResult.processBatchYield();
                    }
                    this.previouslyComputedResults[i] = this.pageProjectWork.getResult();
                    this.pageProjectWork = null;
                    blocks[i] = this.previouslyComputedResults[i];
                }
                if (this.avoidPageMaterialization) continue;
                blocks[i] = blocks[i].getLoadedBlock();
                pageSize = (int)((long)pageSize + blocks[i].getSizeInBytes());
            }
            return ProcessBatchResult.processBatchSuccess(new Page(positionsBatch.size(), blocks));
        }
    }
}

