/*
 * Decompiled with CFR 0.152.
 */
package io.trino.execution.scheduler;

import com.google.common.base.Preconditions;
import com.google.common.base.Ticker;
import com.google.common.collect.ImmutableList;
import com.google.common.collect.ImmutableMap;
import com.google.common.util.concurrent.Futures;
import io.airlift.concurrent.MoreFutures;
import io.airlift.testing.TestingTicker;
import io.airlift.units.DataSize;
import io.airlift.units.Duration;
import io.trino.Session;
import io.trino.client.NodeVersion;
import io.trino.connector.CatalogName;
import io.trino.execution.StageId;
import io.trino.execution.TaskId;
import io.trino.execution.scheduler.BinPackingNodeAllocatorService;
import io.trino.execution.scheduler.NodeAllocator;
import io.trino.execution.scheduler.NodeRequirements;
import io.trino.memory.MemoryInfo;
import io.trino.metadata.InMemoryNodeManager;
import io.trino.metadata.InternalNode;
import io.trino.metadata.InternalNodeManager;
import io.trino.spi.HostAddress;
import io.trino.spi.memory.MemoryPoolInfo;
import io.trino.testing.TestingSession;
import io.trino.testing.assertions.Assert;
import java.net.URI;
import java.time.temporal.ChronoUnit;
import java.util.Arrays;
import java.util.List;
import java.util.Map;
import java.util.Optional;
import java.util.Set;
import java.util.concurrent.ConcurrentHashMap;
import java.util.concurrent.Future;
import java.util.concurrent.TimeUnit;
import org.assertj.core.api.Assertions;
import org.testng.annotations.AfterMethod;
import org.testng.annotations.Test;

@Test(singleThreaded=true)
public class TestBinPackingNodeAllocator {
    private static final Session SESSION = TestingSession.testSessionBuilder().build();
    private static final HostAddress NODE_1_ADDRESS = HostAddress.fromParts((String)"127.0.0.1", (int)8080);
    private static final HostAddress NODE_2_ADDRESS = HostAddress.fromParts((String)"127.0.0.1", (int)8081);
    private static final HostAddress NODE_3_ADDRESS = HostAddress.fromParts((String)"127.0.0.1", (int)8082);
    private static final HostAddress NODE_4_ADDRESS = HostAddress.fromParts((String)"127.0.0.1", (int)8083);
    private static final InternalNode NODE_1 = new InternalNode("node-1", URI.create("local://" + NODE_1_ADDRESS), NodeVersion.UNKNOWN, false);
    private static final InternalNode NODE_2 = new InternalNode("node-2", URI.create("local://" + NODE_2_ADDRESS), NodeVersion.UNKNOWN, false);
    private static final InternalNode NODE_3 = new InternalNode("node-3", URI.create("local://" + NODE_3_ADDRESS), NodeVersion.UNKNOWN, false);
    private static final InternalNode NODE_4 = new InternalNode("node-4", URI.create("local://" + NODE_4_ADDRESS), NodeVersion.UNKNOWN, false);
    private static final CatalogName CATALOG_1 = new CatalogName("catalog1");
    private static final CatalogName CATALOG_2 = new CatalogName("catalog2");
    private static final List<CatalogName> ALL_CATALOGS = ImmutableList.of((Object)CATALOG_1, (Object)CATALOG_2);
    private static final NodeRequirements REQ_32 = new NodeRequirements(Optional.empty(), Set.of(), DataSize.of((long)32L, (DataSize.Unit)DataSize.Unit.GIGABYTE));
    private static final NodeRequirements REQ_20 = new NodeRequirements(Optional.empty(), Set.of(), DataSize.of((long)16L, (DataSize.Unit)DataSize.Unit.GIGABYTE));
    private static final NodeRequirements REQ_16 = new NodeRequirements(Optional.empty(), Set.of(), DataSize.of((long)16L, (DataSize.Unit)DataSize.Unit.GIGABYTE));
    private static final NodeRequirements REQ_1 = new NodeRequirements(Optional.empty(), Set.of(), DataSize.of((long)1L, (DataSize.Unit)DataSize.Unit.GIGABYTE));
    private static final NodeRequirements REQ_NODE_1_32 = new NodeRequirements(Optional.empty(), Set.of(NODE_1_ADDRESS), DataSize.of((long)32L, (DataSize.Unit)DataSize.Unit.GIGABYTE));
    private static final NodeRequirements REQ_NODE_2_32 = new NodeRequirements(Optional.empty(), Set.of(NODE_2_ADDRESS), DataSize.of((long)32L, (DataSize.Unit)DataSize.Unit.GIGABYTE));
    private static final NodeRequirements REQ_CATALOG_1_32 = new NodeRequirements(Optional.of(CATALOG_1), Set.of(), DataSize.of((long)32L, (DataSize.Unit)DataSize.Unit.GIGABYTE));
    private static final long TEST_TIMEOUT = 2500L;
    private BinPackingNodeAllocatorService nodeAllocatorService;
    private ConcurrentHashMap<String, Optional<MemoryInfo>> workerMemoryInfos;
    private final TestingTicker ticker = new TestingTicker();

    private void setupNodeAllocatorService(InMemoryNodeManager nodeManager) {
        this.setupNodeAllocatorService(nodeManager, DataSize.ofBytes((long)0L));
    }

    private void setupNodeAllocatorService(InMemoryNodeManager nodeManager, DataSize taskRuntimeMemoryEstimationOverhead) {
        this.shutdownNodeAllocatorService();
        this.workerMemoryInfos = new ConcurrentHashMap();
        MemoryInfo memoryInfo = this.buildWorkerMemoryInfo(DataSize.ofBytes((long)0L), (Map<TaskId, DataSize>)ImmutableMap.of());
        this.workerMemoryInfos.put(NODE_1.getNodeIdentifier(), Optional.of(memoryInfo));
        this.workerMemoryInfos.put(NODE_2.getNodeIdentifier(), Optional.of(memoryInfo));
        this.workerMemoryInfos.put(NODE_3.getNodeIdentifier(), Optional.of(memoryInfo));
        this.workerMemoryInfos.put(NODE_4.getNodeIdentifier(), Optional.of(memoryInfo));
        this.nodeAllocatorService = new BinPackingNodeAllocatorService((InternalNodeManager)nodeManager, () -> this.workerMemoryInfos, false, java.time.Duration.of(1L, ChronoUnit.MINUTES), taskRuntimeMemoryEstimationOverhead, (Ticker)this.ticker);
        this.nodeAllocatorService.start();
    }

    private void updateWorkerUsedMemory(InternalNode node, DataSize usedMemory, Map<TaskId, DataSize> taskMemoryUsage) {
        this.workerMemoryInfos.put(node.getNodeIdentifier(), Optional.of(this.buildWorkerMemoryInfo(usedMemory, taskMemoryUsage)));
    }

    private MemoryInfo buildWorkerMemoryInfo(DataSize usedMemory, Map<TaskId, DataSize> taskMemoryUsage) {
        return new MemoryInfo(4, new MemoryPoolInfo(DataSize.of((long)64L, (DataSize.Unit)DataSize.Unit.GIGABYTE).toBytes(), usedMemory.toBytes(), 0L, (Map)ImmutableMap.of(), (Map)ImmutableMap.of(), (Map)ImmutableMap.of(), (Map)taskMemoryUsage.entrySet().stream().collect(ImmutableMap.toImmutableMap(entry -> ((TaskId)entry.getKey()).toString(), entry -> ((DataSize)entry.getValue()).toBytes())), (Map)ImmutableMap.of()));
    }

    @AfterMethod(alwaysRun=true)
    public void shutdownNodeAllocatorService() {
        if (this.nodeAllocatorService != null) {
            this.nodeAllocatorService.stop();
        }
        this.nodeAllocatorService = null;
    }

    @Test(timeOut=2500L)
    public void testAllocateSimple() throws Exception {
        InMemoryNodeManager nodeManager = this.testingNodeManager(this.basicNodesMap(NODE_1, NODE_2));
        this.setupNodeAllocatorService(nodeManager);
        try (NodeAllocator nodeAllocator = this.nodeAllocatorService.getNodeAllocator(SESSION);){
            NodeAllocator.NodeLease acquire1 = nodeAllocator.acquire(REQ_32);
            this.assertAcquired(acquire1, NODE_1);
            NodeAllocator.NodeLease acquire2 = nodeAllocator.acquire(REQ_32);
            this.assertAcquired(acquire2, NODE_2);
            NodeAllocator.NodeLease acquire3 = nodeAllocator.acquire(REQ_32);
            this.assertAcquired(acquire3, NODE_1);
            NodeAllocator.NodeLease acquire4 = nodeAllocator.acquire(REQ_32);
            this.assertAcquired(acquire4, NODE_2);
            NodeAllocator.NodeLease acquire5 = nodeAllocator.acquire(REQ_32);
            this.assertNotAcquired(acquire5);
            acquire2.release();
            TestBinPackingNodeAllocator.assertEventually(() -> {
                this.assertAcquired(acquire5);
                org.testng.Assert.assertEquals((Object)acquire5.getNode().get(), (Object)NODE_2);
            });
            NodeAllocator.NodeLease acquire6 = nodeAllocator.acquire(REQ_32);
            this.assertNotAcquired(acquire6);
            this.addNode(nodeManager, NODE_3);
            this.nodeAllocatorService.processPendingAcquires();
            TestBinPackingNodeAllocator.assertEventually(() -> {
                this.assertAcquired(acquire6);
                org.testng.Assert.assertEquals((Object)acquire6.getNode().get(), (Object)NODE_3);
            });
        }
    }

    @Test(timeOut=2500L)
    public void testAllocateDifferentSizes() throws Exception {
        InMemoryNodeManager nodeManager = this.testingNodeManager(this.basicNodesMap(NODE_1, NODE_2));
        this.setupNodeAllocatorService(nodeManager);
        try (NodeAllocator nodeAllocator = this.nodeAllocatorService.getNodeAllocator(SESSION);){
            NodeAllocator.NodeLease acquire1 = nodeAllocator.acquire(REQ_32);
            this.assertAcquired(acquire1, NODE_1);
            NodeAllocator.NodeLease acquire2 = nodeAllocator.acquire(REQ_32);
            this.assertAcquired(acquire2, NODE_2);
            NodeAllocator.NodeLease acquire3 = nodeAllocator.acquire(REQ_16);
            this.assertAcquired(acquire3, NODE_1);
            NodeAllocator.NodeLease acquire4 = nodeAllocator.acquire(REQ_16);
            this.assertAcquired(acquire4, NODE_2);
            NodeAllocator.NodeLease acquire5 = nodeAllocator.acquire(REQ_16);
            this.assertAcquired(acquire5, NODE_1);
            NodeAllocator.NodeLease acquire6 = nodeAllocator.acquire(REQ_16);
            this.assertAcquired(acquire6, NODE_2);
            NodeAllocator.NodeLease acquire7 = nodeAllocator.acquire(REQ_32);
            this.assertNotAcquired(acquire7);
            NodeAllocator.NodeLease acquire8 = nodeAllocator.acquire(REQ_16);
            this.assertNotAcquired(acquire8);
            acquire3.release();
            this.assertNotAcquired(acquire7);
            this.assertNotAcquired(acquire8);
            acquire4.release();
            this.assertAcquired(acquire8);
            acquire5.release();
            this.assertAcquired(acquire7);
        }
    }

    @Test(timeOut=2500L)
    public void testAllocateDifferentSizesOpportunisticAcquisition() {
        InMemoryNodeManager nodeManager = this.testingNodeManager(this.basicNodesMap(NODE_1, NODE_2));
        this.setupNodeAllocatorService(nodeManager);
        try (NodeAllocator nodeAllocator = this.nodeAllocatorService.getNodeAllocator(SESSION);){
            NodeAllocator.NodeLease acquire1 = nodeAllocator.acquire(REQ_32);
            this.assertAcquired(acquire1, NODE_1);
            NodeAllocator.NodeLease acquire2 = nodeAllocator.acquire(REQ_32);
            this.assertAcquired(acquire2, NODE_2);
            NodeAllocator.NodeLease acquire3 = nodeAllocator.acquire(REQ_16);
            this.assertAcquired(acquire3, NODE_1);
            NodeAllocator.NodeLease acquire4 = nodeAllocator.acquire(REQ_16);
            this.assertAcquired(acquire4, NODE_2);
            NodeAllocator.NodeLease acquire5 = nodeAllocator.acquire(REQ_16);
            this.assertAcquired(acquire5, NODE_1);
            NodeAllocator.NodeLease acquire6 = nodeAllocator.acquire(REQ_16);
            this.assertAcquired(acquire6, NODE_2);
            NodeAllocator.NodeLease acquire7 = nodeAllocator.acquire(REQ_32);
            this.assertNotAcquired(acquire7);
            NodeAllocator.NodeLease acquire8 = nodeAllocator.acquire(REQ_16);
            this.assertNotAcquired(acquire8);
            acquire2.release();
            this.assertAcquired(acquire7);
            acquire1.release();
            this.assertAcquired(acquire8);
        }
    }

    @Test(timeOut=2500L)
    public void testAllocateReleaseBeforeAcquired() {
        InMemoryNodeManager nodeManager = this.testingNodeManager(this.basicNodesMap(NODE_1));
        this.setupNodeAllocatorService(nodeManager);
        try (NodeAllocator nodeAllocator = this.nodeAllocatorService.getNodeAllocator(SESSION);){
            NodeAllocator.NodeLease acquire1 = nodeAllocator.acquire(REQ_32);
            this.assertAcquired(acquire1, NODE_1);
            NodeAllocator.NodeLease acquire2 = nodeAllocator.acquire(REQ_32);
            this.assertAcquired(acquire2, NODE_1);
            NodeAllocator.NodeLease acquire3 = nodeAllocator.acquire(REQ_32);
            this.assertNotAcquired(acquire3);
            NodeAllocator.NodeLease acquire4 = nodeAllocator.acquire(REQ_32);
            this.assertNotAcquired(acquire4);
            acquire3.release();
            this.assertNotAcquired(acquire4);
            acquire2.release();
            TestBinPackingNodeAllocator.assertEventually(() -> this.assertAcquired(acquire4, NODE_1));
        }
    }

    @Test(timeOut=2500L)
    public void testNoMatchingNodeAvailable() {
        InMemoryNodeManager nodeManager = this.testingNodeManager((Map<InternalNode, List<CatalogName>>)this.nodesMapBuilder().put((Object)NODE_1, (Object)ImmutableList.of((Object)CATALOG_2)).buildOrThrow());
        this.setupNodeAllocatorService(nodeManager);
        try (NodeAllocator nodeAllocator = this.nodeAllocatorService.getNodeAllocator(SESSION);){
            NodeAllocator.NodeLease acquireNoMatching = nodeAllocator.acquire(REQ_CATALOG_1_32.withMemory(DataSize.of((long)64L, (DataSize.Unit)DataSize.Unit.GIGABYTE)));
            this.assertNotAcquired(acquireNoMatching);
            this.ticker.increment(59L, TimeUnit.SECONDS);
            this.nodeAllocatorService.processPendingAcquires();
            this.assertNotAcquired(acquireNoMatching);
            this.ticker.increment(2L, TimeUnit.SECONDS);
            this.nodeAllocatorService.processPendingAcquires();
            Assertions.assertThatThrownBy(() -> Futures.getUnchecked((Future)acquireNoMatching.getNode())).hasMessageContaining("No nodes available to run query");
            this.addNode(nodeManager, NODE_2, CATALOG_1);
            NodeAllocator.NodeLease acquire1 = nodeAllocator.acquire(REQ_CATALOG_1_32.withMemory(DataSize.of((long)64L, (DataSize.Unit)DataSize.Unit.GIGABYTE)));
            this.assertAcquired(acquire1, NODE_2);
            NodeAllocator.NodeLease acquire2 = nodeAllocator.acquire(REQ_CATALOG_1_32.withMemory(DataSize.of((long)64L, (DataSize.Unit)DataSize.Unit.GIGABYTE)));
            this.assertNotAcquired(acquire2);
            nodeManager.removeNode(NODE_2);
            this.nodeAllocatorService.processPendingAcquires();
            this.ticker.increment(61L, TimeUnit.SECONDS);
            this.nodeAllocatorService.processPendingAcquires();
            TestBinPackingNodeAllocator.assertEventually(() -> {
                org.testng.Assert.assertFalse((boolean)acquire2.getNode().isCancelled());
                org.testng.Assert.assertTrue((boolean)acquire2.getNode().isDone());
                Assertions.assertThatThrownBy(() -> MoreFutures.getFutureValue((Future)acquire2.getNode())).hasMessage("No nodes available to run query");
            });
        }
    }

    @Test(timeOut=2500L)
    public void testRemoveAcquiredNode() {
        InMemoryNodeManager nodeManager = this.testingNodeManager(this.basicNodesMap(NODE_1));
        this.setupNodeAllocatorService(nodeManager);
        try (NodeAllocator nodeAllocator = this.nodeAllocatorService.getNodeAllocator(SESSION);){
            NodeAllocator.NodeLease acquire1 = nodeAllocator.acquire(REQ_32);
            this.assertAcquired(acquire1, NODE_1);
            nodeManager.removeNode(NODE_1);
            acquire1.release();
        }
    }

    @Test(timeOut=2500L)
    public void testAllocateNodeWithAddressRequirements() {
        InMemoryNodeManager nodeManager = this.testingNodeManager(this.basicNodesMap(NODE_1, NODE_2));
        this.setupNodeAllocatorService(nodeManager);
        try (NodeAllocator nodeAllocator = this.nodeAllocatorService.getNodeAllocator(SESSION);){
            NodeAllocator.NodeLease acquire1 = nodeAllocator.acquire(REQ_NODE_2_32);
            this.assertAcquired(acquire1, NODE_2);
            NodeAllocator.NodeLease acquire2 = nodeAllocator.acquire(REQ_NODE_2_32);
            this.assertAcquired(acquire2, NODE_2);
            NodeAllocator.NodeLease acquire3 = nodeAllocator.acquire(REQ_NODE_2_32);
            this.assertNotAcquired(acquire3);
            NodeAllocator.NodeLease acquire4 = nodeAllocator.acquire(REQ_NODE_1_32);
            this.assertAcquired(acquire4, NODE_1);
            acquire1.release();
            TestBinPackingNodeAllocator.assertEventually(() -> this.assertAcquired(acquire3));
        }
    }

    @Test(timeOut=2500L)
    public void testAllocateNotEnoughRuntimeMemory() {
        InMemoryNodeManager nodeManager = this.testingNodeManager(this.basicNodesMap(NODE_1, NODE_2));
        this.setupNodeAllocatorService(nodeManager);
        try (NodeAllocator nodeAllocator = this.nodeAllocatorService.getNodeAllocator(SESSION);){
            NodeAllocator.NodeLease acquire1 = nodeAllocator.acquire(REQ_32);
            this.assertAcquired(acquire1, NODE_1);
            acquire1.attachTaskId(this.taskId(1));
            this.updateWorkerUsedMemory(NODE_1, DataSize.of((long)33L, (DataSize.Unit)DataSize.Unit.GIGABYTE), (Map<TaskId, DataSize>)ImmutableMap.of((Object)this.taskId(1), (Object)DataSize.of((long)33L, (DataSize.Unit)DataSize.Unit.GIGABYTE)));
            this.nodeAllocatorService.refreshNodePoolMemoryInfos();
            NodeAllocator.NodeLease acquire2 = nodeAllocator.acquire(REQ_32);
            this.assertAcquired(acquire2, NODE_2);
            acquire2.attachTaskId(this.taskId(2));
            NodeAllocator.NodeLease acquire3 = nodeAllocator.acquire(REQ_32);
            this.assertAcquired(acquire3, NODE_2);
            acquire3.attachTaskId(this.taskId(3));
            NodeAllocator.NodeLease acquire4 = nodeAllocator.acquire(REQ_16);
            this.assertAcquired(acquire4, NODE_1);
            acquire4.attachTaskId(this.taskId(4));
            NodeAllocator.NodeLease acquire5 = nodeAllocator.acquire(REQ_16);
            this.assertNotAcquired(acquire5);
            NodeAllocator.NodeLease acquire6 = nodeAllocator.acquire(REQ_1);
            this.assertNotAcquired(acquire6);
            this.updateWorkerUsedMemory(NODE_1, DataSize.of((long)32L, (DataSize.Unit)DataSize.Unit.GIGABYTE), (Map<TaskId, DataSize>)ImmutableMap.of((Object)this.taskId(1), (Object)DataSize.of((long)32L, (DataSize.Unit)DataSize.Unit.GIGABYTE)));
            this.nodeAllocatorService.refreshNodePoolMemoryInfos();
            this.nodeAllocatorService.processPendingAcquires();
            this.assertAcquired(acquire5, NODE_1);
            acquire5.attachTaskId(this.taskId(5));
            this.assertNotAcquired(acquire6);
        }
    }

    @Test(timeOut=2500L)
    public void testAllocateRuntimeMemoryDiscrepancies() {
        NodeAllocator.NodeLease acquire2;
        NodeAllocator.NodeLease acquire1;
        InMemoryNodeManager nodeManager = this.testingNodeManager(this.basicNodesMap(NODE_1));
        this.setupNodeAllocatorService(nodeManager);
        try (NodeAllocator nodeAllocator = this.nodeAllocatorService.getNodeAllocator(SESSION);){
            acquire1 = nodeAllocator.acquire(REQ_32);
            this.assertAcquired(acquire1, NODE_1);
            acquire1.attachTaskId(this.taskId(1));
            this.updateWorkerUsedMemory(NODE_1, DataSize.of((long)33L, (DataSize.Unit)DataSize.Unit.GIGABYTE), (Map<TaskId, DataSize>)ImmutableMap.of((Object)this.taskId(1), (Object)DataSize.of((long)4L, (DataSize.Unit)DataSize.Unit.GIGABYTE)));
            this.nodeAllocatorService.refreshNodePoolMemoryInfos();
            acquire2 = nodeAllocator.acquire(REQ_32);
            this.assertNotAcquired(acquire2);
        }
        this.setupNodeAllocatorService(nodeManager);
        nodeAllocator = this.nodeAllocatorService.getNodeAllocator(SESSION);
        try {
            acquire1 = nodeAllocator.acquire(REQ_32);
            this.assertAcquired(acquire1, NODE_1);
            acquire1.attachTaskId(this.taskId(1));
            this.updateWorkerUsedMemory(NODE_1, DataSize.of((long)4L, (DataSize.Unit)DataSize.Unit.GIGABYTE), (Map<TaskId, DataSize>)ImmutableMap.of((Object)this.taskId(1), (Object)DataSize.of((long)33L, (DataSize.Unit)DataSize.Unit.GIGABYTE)));
            this.nodeAllocatorService.refreshNodePoolMemoryInfos();
            acquire2 = nodeAllocator.acquire(REQ_32);
            this.assertNotAcquired(acquire2);
        }
        finally {
            if (nodeAllocator != null) {
                nodeAllocator.close();
            }
        }
        this.setupNodeAllocatorService(nodeManager);
        nodeAllocator = this.nodeAllocatorService.getNodeAllocator(SESSION);
        try {
            acquire1 = nodeAllocator.acquire(REQ_32);
            this.assertAcquired(acquire1, NODE_1);
            acquire1.attachTaskId(this.taskId(1));
            this.updateWorkerUsedMemory(NODE_1, DataSize.of((long)33L, (DataSize.Unit)DataSize.Unit.GIGABYTE), (Map<TaskId, DataSize>)ImmutableMap.of());
            this.nodeAllocatorService.refreshNodePoolMemoryInfos();
            acquire2 = nodeAllocator.acquire(REQ_32);
            this.assertNotAcquired(acquire2);
        }
        finally {
            if (nodeAllocator != null) {
                nodeAllocator.close();
            }
        }
    }

    @Test(timeOut=2500L)
    public void testSpaceReservedOnPrimaryNodeIfNoNodeWithEnoughRuntimeMemoryAvailable() {
        InMemoryNodeManager nodeManager = this.testingNodeManager(this.basicNodesMap(NODE_1, NODE_2));
        this.setupNodeAllocatorService(nodeManager);
        try (NodeAllocator nodeAllocator = this.nodeAllocatorService.getNodeAllocator(SESSION);){
            NodeAllocator.NodeLease acquire1 = nodeAllocator.acquire(REQ_32);
            this.assertAcquired(acquire1, NODE_1);
            acquire1.attachTaskId(this.taskId(1));
            NodeAllocator.NodeLease acquire2 = nodeAllocator.acquire(REQ_16);
            this.assertAcquired(acquire2, NODE_2);
            acquire2.attachTaskId(this.taskId(2));
            this.updateWorkerUsedMemory(NODE_1, DataSize.of((long)40L, (DataSize.Unit)DataSize.Unit.GIGABYTE), (Map<TaskId, DataSize>)ImmutableMap.of((Object)this.taskId(1), (Object)DataSize.of((long)40L, (DataSize.Unit)DataSize.Unit.GIGABYTE)));
            this.updateWorkerUsedMemory(NODE_2, DataSize.of((long)41L, (DataSize.Unit)DataSize.Unit.GIGABYTE), (Map<TaskId, DataSize>)ImmutableMap.of((Object)this.taskId(2), (Object)DataSize.of((long)41L, (DataSize.Unit)DataSize.Unit.GIGABYTE)));
            this.nodeAllocatorService.refreshNodePoolMemoryInfos();
            NodeAllocator.NodeLease acquire3 = nodeAllocator.acquire(REQ_32);
            this.assertNotAcquired(acquire3);
            NodeAllocator.NodeLease acquire4 = nodeAllocator.acquire(REQ_20);
            this.assertAcquired(acquire4, NODE_1);
            acquire4.attachTaskId(this.taskId(2));
        }
    }

    @Test(timeOut=2500L)
    public void testAllocateWithRuntimeMemoryEstimateOverhead() {
        InMemoryNodeManager nodeManager = this.testingNodeManager(this.basicNodesMap(NODE_1));
        this.setupNodeAllocatorService(nodeManager, DataSize.of((long)4L, (DataSize.Unit)DataSize.Unit.GIGABYTE));
        try (NodeAllocator nodeAllocator = this.nodeAllocatorService.getNodeAllocator(SESSION);){
            NodeAllocator.NodeLease acquire1 = nodeAllocator.acquire(REQ_32);
            this.assertAcquired(acquire1, NODE_1);
            acquire1.attachTaskId(this.taskId(1));
            this.updateWorkerUsedMemory(NODE_1, DataSize.of((long)30L, (DataSize.Unit)DataSize.Unit.GIGABYTE), (Map<TaskId, DataSize>)ImmutableMap.of((Object)this.taskId(1), (Object)DataSize.of((long)30L, (DataSize.Unit)DataSize.Unit.GIGABYTE)));
            this.nodeAllocatorService.refreshNodePoolMemoryInfos();
            NodeAllocator.NodeLease acquire2 = nodeAllocator.acquire(REQ_32);
            this.assertNotAcquired(acquire2);
            this.updateWorkerUsedMemory(NODE_1, DataSize.of((long)28L, (DataSize.Unit)DataSize.Unit.GIGABYTE), (Map<TaskId, DataSize>)ImmutableMap.of((Object)this.taskId(1), (Object)DataSize.of((long)28L, (DataSize.Unit)DataSize.Unit.GIGABYTE)));
            this.nodeAllocatorService.refreshNodePoolMemoryInfos();
            this.nodeAllocatorService.processPendingAcquires();
            this.assertAcquired(acquire2, NODE_1);
        }
    }

    @Test
    public void testStressAcquireRelease() {
        InMemoryNodeManager nodeManager = this.testingNodeManager(this.basicNodesMap(NODE_1));
        this.setupNodeAllocatorService(nodeManager, DataSize.of((long)4L, (DataSize.Unit)DataSize.Unit.GIGABYTE));
        try (NodeAllocator nodeAllocator = this.nodeAllocatorService.getNodeAllocator(SESSION);){
            for (int i = 0; i < 10000000; ++i) {
                NodeAllocator.NodeLease lease = nodeAllocator.acquire(REQ_32);
                lease.release();
            }
        }
    }

    private TaskId taskId(int partition) {
        return new TaskId(new StageId("test_query", 0), partition, 0);
    }

    private InMemoryNodeManager testingNodeManager(Map<InternalNode, List<CatalogName>> nodeMap) {
        InMemoryNodeManager nodeManager = new InMemoryNodeManager();
        for (Map.Entry<InternalNode, List<CatalogName>> entry : nodeMap.entrySet()) {
            InternalNode node = entry.getKey();
            List<CatalogName> catalogs = entry.getValue();
            for (CatalogName catalog : catalogs) {
                nodeManager.addNode(catalog, new InternalNode[]{node});
            }
        }
        return nodeManager;
    }

    private Map<InternalNode, List<CatalogName>> basicNodesMap(InternalNode ... nodes) {
        return (Map)Arrays.stream(nodes).collect(ImmutableMap.toImmutableMap(node -> node, node -> ALL_CATALOGS));
    }

    private ImmutableMap.Builder<InternalNode, List<CatalogName>> nodesMapBuilder() {
        return ImmutableMap.builder();
    }

    private void addNode(InMemoryNodeManager nodeManager, InternalNode node) {
        this.addNode(nodeManager, node, ALL_CATALOGS);
    }

    private void addNode(InMemoryNodeManager nodeManager, InternalNode node, CatalogName ... catalogs) {
        this.addNode(nodeManager, node, (List<CatalogName>)ImmutableList.copyOf(Arrays.asList(catalogs)));
    }

    private void addNode(InMemoryNodeManager nodeManager, InternalNode node, List<CatalogName> catalogs) {
        Preconditions.checkArgument((!catalogs.isEmpty() ? 1 : 0) != 0, (Object)"no catalogs specified");
        for (CatalogName catalog : catalogs) {
            nodeManager.addNode(catalog, new InternalNode[]{node});
        }
    }

    private void assertAcquired(NodeAllocator.NodeLease lease, InternalNode node) {
        this.assertAcquired(lease, Optional.of(node));
    }

    private void assertAcquired(NodeAllocator.NodeLease lease) {
        this.assertAcquired(lease, Optional.empty());
    }

    private void assertAcquired(NodeAllocator.NodeLease lease, Optional<InternalNode> expectedNode) {
        TestBinPackingNodeAllocator.assertEventually(() -> {
            org.testng.Assert.assertFalse((boolean)lease.getNode().isCancelled(), (String)"node lease cancelled");
            org.testng.Assert.assertTrue((boolean)lease.getNode().isDone(), (String)"node lease not acquired");
            if (expectedNode.isPresent()) {
                org.testng.Assert.assertEquals((Object)lease.getNode().get(), expectedNode.get());
            }
        });
    }

    private void assertNotAcquired(NodeAllocator.NodeLease lease) {
        org.testng.Assert.assertFalse((boolean)lease.getNode().isCancelled(), (String)"node lease cancelled");
        org.testng.Assert.assertFalse((boolean)lease.getNode().isDone(), (String)"node lease acquired");
        this.nodeAllocatorService.processPendingAcquires();
        org.testng.Assert.assertFalse((boolean)lease.getNode().isCancelled(), (String)"node lease cancelled");
        org.testng.Assert.assertFalse((boolean)lease.getNode().isDone(), (String)"node lease acquired");
    }

    private static void assertEventually(ThrowingRunnable assertion) {
        Assert.assertEventually((Duration)new Duration(2500.0, TimeUnit.MILLISECONDS), (Duration)new Duration(10.0, TimeUnit.MILLISECONDS), () -> {
            try {
                assertion.run();
            }
            catch (Exception e) {
                throw new RuntimeException(e);
            }
        });
    }

    static interface ThrowingRunnable {
        public void run() throws Exception;
    }
}

