/*
 * Decompiled with CFR 0.152.
 */
package io.trino.execution;

import com.google.common.base.Preconditions;
import com.google.common.collect.ImmutableList;
import io.airlift.log.Logger;
import io.airlift.units.Duration;
import io.trino.Session;
import io.trino.SystemSessionProperties;
import io.trino.execution.QueryManagerConfig;
import io.trino.spi.ErrorCodeSupplier;
import io.trino.spi.QueryId;
import io.trino.spi.StandardErrorCode;
import io.trino.spi.TrinoException;
import java.util.Collection;
import java.util.NoSuchElementException;
import java.util.Objects;
import java.util.Optional;
import java.util.Queue;
import java.util.concurrent.ConcurrentHashMap;
import java.util.concurrent.ConcurrentMap;
import java.util.concurrent.LinkedBlockingQueue;
import java.util.concurrent.ScheduledExecutorService;
import java.util.concurrent.ScheduledFuture;
import java.util.concurrent.TimeUnit;
import javax.annotation.concurrent.GuardedBy;
import javax.annotation.concurrent.ThreadSafe;
import org.joda.time.DateTime;
import org.joda.time.ReadableInstant;

@ThreadSafe
public class QueryTracker<T extends TrackedQuery> {
    private static final Logger log = Logger.get(QueryTracker.class);
    private final int maxQueryHistory;
    private final Duration minQueryExpireAge;
    private final ConcurrentMap<QueryId, T> queries = new ConcurrentHashMap<QueryId, T>();
    private final Queue<T> expirationQueue = new LinkedBlockingQueue<T>();
    private final Duration clientTimeout;
    private final ScheduledExecutorService queryManagementExecutor;
    @GuardedBy(value="this")
    private ScheduledFuture<?> backgroundTask;

    public QueryTracker(QueryManagerConfig queryManagerConfig, ScheduledExecutorService queryManagementExecutor) {
        this.minQueryExpireAge = queryManagerConfig.getMinQueryExpireAge();
        this.maxQueryHistory = queryManagerConfig.getMaxQueryHistory();
        this.clientTimeout = queryManagerConfig.getClientTimeout();
        this.queryManagementExecutor = Objects.requireNonNull(queryManagementExecutor, "queryManagementExecutor is null");
    }

    public synchronized void start() {
        Preconditions.checkState((this.backgroundTask == null ? 1 : 0) != 0, (Object)"QueryTracker already started");
        this.backgroundTask = this.queryManagementExecutor.scheduleWithFixedDelay(() -> {
            try {
                this.failAbandonedQueries();
            }
            catch (Throwable e) {
                log.error(e, "Error cancelling abandoned queries");
            }
            try {
                this.enforceTimeLimits();
            }
            catch (Throwable e) {
                log.error(e, "Error enforcing query timeout limits");
            }
            try {
                this.removeExpiredQueries();
            }
            catch (Throwable e) {
                log.error(e, "Error removing expired queries");
            }
            try {
                this.pruneExpiredQueries();
            }
            catch (Throwable e) {
                log.error(e, "Error pruning expired queries");
            }
        }, 1L, 1L, TimeUnit.SECONDS);
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    public void stop() {
        QueryTracker queryTracker = this;
        synchronized (queryTracker) {
            if (this.backgroundTask != null) {
                this.backgroundTask.cancel(true);
            }
        }
        boolean queryCancelled = false;
        for (TrackedQuery trackedQuery : this.queries.values()) {
            if (trackedQuery.isDone()) continue;
            log.info("Server shutting down. Query %s has been cancelled", new Object[]{trackedQuery.getQueryId()});
            trackedQuery.fail(new TrinoException((ErrorCodeSupplier)StandardErrorCode.SERVER_SHUTTING_DOWN, "Server is shutting down. Query " + trackedQuery.getQueryId() + " has been cancelled"));
            queryCancelled = true;
        }
        if (queryCancelled) {
            try {
                TimeUnit.SECONDS.sleep(5L);
            }
            catch (InterruptedException e) {
                Thread.currentThread().interrupt();
            }
        }
    }

    public Collection<T> getAllQueries() {
        return ImmutableList.copyOf(this.queries.values());
    }

    public T getQuery(QueryId queryId) throws NoSuchElementException {
        return (T)((TrackedQuery)this.tryGetQuery(queryId).orElseThrow(() -> new NoSuchElementException(queryId.toString())));
    }

    public Optional<T> tryGetQuery(QueryId queryId) {
        Objects.requireNonNull(queryId, "queryId is null");
        return Optional.ofNullable((TrackedQuery)this.queries.get(queryId));
    }

    public boolean addQuery(T execution) {
        return this.queries.putIfAbsent(execution.getQueryId(), execution) == null;
    }

    public void expireQuery(QueryId queryId) {
        this.tryGetQuery(queryId).ifPresent(this.expirationQueue::add);
    }

    private void enforceTimeLimits() {
        for (TrackedQuery query : this.queries.values()) {
            if (query.isDone()) continue;
            Duration queryMaxRunTime = SystemSessionProperties.getQueryMaxRunTime(query.getSession());
            Duration queryMaxExecutionTime = SystemSessionProperties.getQueryMaxExecutionTime(query.getSession());
            Duration queryMaxPlanningTime = SystemSessionProperties.getQueryMaxPlanningTime(query.getSession());
            Optional<DateTime> executionStartTime = query.getExecutionStartTime();
            Optional<Duration> planningTime = query.getPlanningTime();
            DateTime createTime = query.getCreateTime();
            if (executionStartTime.isPresent() && executionStartTime.get().plus(queryMaxExecutionTime.toMillis()).isBeforeNow()) {
                query.fail(new TrinoException((ErrorCodeSupplier)StandardErrorCode.EXCEEDED_TIME_LIMIT, "Query exceeded the maximum execution time limit of " + queryMaxExecutionTime));
            }
            planningTime.filter(duration -> duration.compareTo(queryMaxPlanningTime) > 0).ifPresent(ignored -> query.fail(new TrinoException((ErrorCodeSupplier)StandardErrorCode.EXCEEDED_TIME_LIMIT, "Query exceeded the maximum planning time limit of " + queryMaxPlanningTime)));
            if (!createTime.plus(queryMaxRunTime.toMillis()).isBeforeNow()) continue;
            query.fail(new TrinoException((ErrorCodeSupplier)StandardErrorCode.EXCEEDED_TIME_LIMIT, "Query exceeded maximum time limit of " + queryMaxRunTime));
        }
    }

    private void pruneExpiredQueries() {
        if (this.expirationQueue.size() <= this.maxQueryHistory) {
            return;
        }
        int count = 0;
        for (TrackedQuery query : this.expirationQueue) {
            if (this.expirationQueue.size() - count <= this.maxQueryHistory) break;
            query.pruneInfo();
            ++count;
        }
    }

    private void removeExpiredQueries() {
        DateTime timeHorizon = DateTime.now().minus(this.minQueryExpireAge.toMillis());
        while (this.expirationQueue.size() > this.maxQueryHistory) {
            TrackedQuery query = (TrackedQuery)this.expirationQueue.peek();
            if (query == null) {
                return;
            }
            Optional<DateTime> endTime = query.getEndTime();
            if (endTime.isEmpty()) continue;
            if (endTime.get().isAfter((ReadableInstant)timeHorizon)) {
                return;
            }
            QueryId queryId = query.getQueryId();
            log.debug("Remove query %s", new Object[]{queryId});
            this.queries.remove(queryId);
            this.expirationQueue.remove(query);
        }
    }

    private void failAbandonedQueries() {
        for (TrackedQuery query : this.queries.values()) {
            try {
                if (query.isDone() || !this.isAbandoned(query)) continue;
                log.info("Failing abandoned query %s", new Object[]{query.getQueryId()});
                query.fail(new TrinoException((ErrorCodeSupplier)StandardErrorCode.ABANDONED_QUERY, String.format("Query %s was abandoned by the client, as it may have exited or stopped checking for query results. Query results have not been accessed since %s: currentTime %s", query.getQueryId(), query.getLastHeartbeat(), DateTime.now())));
            }
            catch (RuntimeException e) {
                log.error((Throwable)e, "Exception failing abandoned query %s", new Object[]{query.getQueryId()});
            }
        }
    }

    private boolean isAbandoned(T query) {
        DateTime oldestAllowedHeartbeat = DateTime.now().minus(this.clientTimeout.toMillis());
        DateTime lastHeartbeat = query.getLastHeartbeat();
        return lastHeartbeat != null && lastHeartbeat.isBefore((ReadableInstant)oldestAllowedHeartbeat);
    }

    public static interface TrackedQuery {
        public QueryId getQueryId();

        public boolean isDone();

        public Session getSession();

        public DateTime getCreateTime();

        public Optional<DateTime> getExecutionStartTime();

        public Optional<Duration> getPlanningTime();

        public DateTime getLastHeartbeat();

        public Optional<DateTime> getEndTime();

        public void fail(Throwable var1);

        public void pruneInfo();
    }
}

