/*
 * Decompiled with CFR 0.152.
 */
package io.trino.operator;

import com.google.common.base.Preconditions;
import com.google.common.collect.AbstractIterator;
import com.google.common.util.concurrent.ListenableFuture;
import io.trino.operator.WorkProcessor;
import java.util.Comparator;
import java.util.Iterator;
import java.util.Objects;
import java.util.Optional;
import java.util.PriorityQueue;
import java.util.function.BooleanSupplier;
import java.util.function.Consumer;
import java.util.function.Function;
import java.util.function.Supplier;
import javax.annotation.Nullable;

public final class WorkProcessorUtils {
    private WorkProcessorUtils() {
    }

    static <T> Iterator<T> iteratorFrom(final WorkProcessor<T> processor) {
        Objects.requireNonNull(processor, "processor is null");
        return new AbstractIterator<T>(){
            final Iterator<Optional<T>> yieldingIterator;
            {
                this.yieldingIterator = WorkProcessorUtils.yieldingIteratorFrom(processor);
            }

            protected T computeNext() {
                if (!this.yieldingIterator.hasNext()) {
                    return this.endOfData();
                }
                return this.yieldingIterator.next().orElseThrow(() -> new IllegalStateException("Cannot iterate over yielding WorkProcessor"));
            }
        };
    }

    static <T> Iterator<Optional<T>> yieldingIteratorFrom(WorkProcessor<T> processor) {
        return new YieldingIterator<T>(processor);
    }

    static <T> WorkProcessor<T> fromIterator(Iterator<T> iterator) {
        Objects.requireNonNull(iterator, "iterator is null");
        return WorkProcessorUtils.create(() -> {
            if (!iterator.hasNext()) {
                return WorkProcessor.ProcessState.finished();
            }
            return WorkProcessor.ProcessState.ofResult(iterator.next());
        });
    }

    static <T> WorkProcessor<T> mergeSorted(Iterable<WorkProcessor<T>> processorIterable, Comparator<T> comparator) {
        Objects.requireNonNull(comparator, "comparator is null");
        final Iterator<WorkProcessor<T>> processorIterator = processorIterable.iterator();
        Preconditions.checkArgument((boolean)processorIterator.hasNext(), (Object)"There must be at least one base processor");
        final PriorityQueue<ElementAndProcessor> queue = new PriorityQueue<ElementAndProcessor>(2, Comparator.comparing(ElementAndProcessor::getElement, comparator));
        return WorkProcessorUtils.create(new WorkProcessor.Process<T>(){
            WorkProcessor<T> processor;
            {
                this.processor = Objects.requireNonNull((WorkProcessor)processorIterator.next());
            }

            @Override
            public WorkProcessor.ProcessState<T> process() {
                while (true) {
                    if (this.processor.process()) {
                        if (!this.processor.isFinished()) {
                            queue.add(new ElementAndProcessor(this.processor.getResult(), this.processor));
                        }
                    } else {
                        if (this.processor.isBlocked()) {
                            return WorkProcessor.ProcessState.blocked(this.processor.getBlockedFuture());
                        }
                        return WorkProcessor.ProcessState.yielded();
                    }
                    if (!processorIterator.hasNext()) break;
                    this.processor = Objects.requireNonNull((WorkProcessor)processorIterator.next());
                }
                if (queue.isEmpty()) {
                    return WorkProcessor.ProcessState.finished();
                }
                ElementAndProcessor elementAndProcessor = (ElementAndProcessor)queue.poll();
                this.processor = elementAndProcessor.getProcessor();
                return WorkProcessor.ProcessState.ofResult(elementAndProcessor.getElement());
            }
        });
    }

    static <T> WorkProcessor<T> yielding(WorkProcessor<T> processor, BooleanSupplier yieldSignal) {
        return WorkProcessor.create(new YieldingProcess<T>(processor, yieldSignal));
    }

    static <T> WorkProcessor<T> blocking(WorkProcessor<T> processor, Supplier<ListenableFuture<Void>> futureSupplier) {
        return WorkProcessor.create(new BlockingProcess<T>(processor, futureSupplier));
    }

    static <T> WorkProcessor<T> processEntryMonitor(WorkProcessor<T> processor, Runnable monitor) {
        Objects.requireNonNull(processor, "processor is null");
        Objects.requireNonNull(monitor, "monitor is null");
        return WorkProcessor.create(() -> {
            monitor.run();
            return WorkProcessorUtils.getNextState(processor);
        });
    }

    static <T> WorkProcessor<T> processStateMonitor(WorkProcessor<T> processor, Consumer<WorkProcessor.ProcessState<T>> monitor) {
        Objects.requireNonNull(processor, "processor is null");
        Objects.requireNonNull(monitor, "monitor is null");
        return WorkProcessor.create(() -> {
            WorkProcessor.ProcessState state = WorkProcessorUtils.getNextState(processor);
            monitor.accept(state);
            return state;
        });
    }

    static <T> WorkProcessor<T> finishWhen(WorkProcessor<T> processor, BooleanSupplier finishSignal) {
        Objects.requireNonNull(processor, "processor is null");
        Objects.requireNonNull(finishSignal, "finishSignal is null");
        return WorkProcessor.create(() -> {
            if (finishSignal.getAsBoolean()) {
                return WorkProcessor.ProcessState.finished();
            }
            return WorkProcessorUtils.getNextState(processor);
        });
    }

    private static <T> WorkProcessor.ProcessState<T> getNextState(WorkProcessor<T> processor) {
        if (processor.process()) {
            if (processor.isFinished()) {
                return WorkProcessor.ProcessState.finished();
            }
            return WorkProcessor.ProcessState.ofResult(processor.getResult());
        }
        if (processor.isBlocked()) {
            return WorkProcessor.ProcessState.blocked(processor.getBlockedFuture());
        }
        return WorkProcessor.ProcessState.yielded();
    }

    static <T, R> WorkProcessor<R> flatMap(WorkProcessor<T> processor, Function<T, WorkProcessor<R>> mapper) {
        Objects.requireNonNull(processor, "processor is null");
        Objects.requireNonNull(mapper, "mapper is null");
        return processor.flatTransform(element -> {
            if (element == null) {
                return WorkProcessor.TransformationState.finished();
            }
            return WorkProcessor.TransformationState.ofResult((WorkProcessor)mapper.apply(element));
        });
    }

    static <T, R> WorkProcessor<R> map(WorkProcessor<T> processor, Function<T, R> mapper) {
        Objects.requireNonNull(processor, "processor is null");
        Objects.requireNonNull(mapper, "mapper is null");
        return processor.transform(element -> {
            if (element == null) {
                return WorkProcessor.TransformationState.finished();
            }
            return WorkProcessor.TransformationState.ofResult(mapper.apply(element));
        });
    }

    static <T, R> WorkProcessor<R> flatTransform(WorkProcessor<T> processor, WorkProcessor.Transformation<T, WorkProcessor<R>> transformation) {
        Objects.requireNonNull(processor, "processor is null");
        Objects.requireNonNull(transformation, "transformation is null");
        return processor.transform(transformation).transformProcessor(WorkProcessorUtils::flatten);
    }

    static <T> WorkProcessor<T> flatten(WorkProcessor<WorkProcessor<T>> processor) {
        Objects.requireNonNull(processor, "processor is null");
        return processor.transform(nestedProcessor -> {
            if (nestedProcessor == null) {
                return WorkProcessor.TransformationState.finished();
            }
            if (nestedProcessor.process()) {
                if (nestedProcessor.isFinished()) {
                    return WorkProcessor.TransformationState.needsMoreData();
                }
                return WorkProcessor.TransformationState.ofResult(nestedProcessor.getResult(), false);
            }
            if (nestedProcessor.isBlocked()) {
                return WorkProcessor.TransformationState.blocked(nestedProcessor.getBlockedFuture());
            }
            return WorkProcessor.TransformationState.yielded();
        });
    }

    static <T, R> WorkProcessor<R> transform(final WorkProcessor<T> processor, final WorkProcessor.Transformation<T, R> transformation) {
        Objects.requireNonNull(processor, "processor is null");
        Objects.requireNonNull(transformation, "transformation is null");
        return WorkProcessorUtils.create(new WorkProcessor.Process<R>(){
            T element;

            @Override
            public WorkProcessor.ProcessState<R> process() {
                while (true) {
                    WorkProcessor.TransformationState state;
                    if (this.element == null && !processor.isFinished()) {
                        if (processor.process()) {
                            if (!processor.isFinished()) {
                                this.element = Objects.requireNonNull(processor.getResult(), "result is null");
                            }
                        } else {
                            if (processor.isBlocked()) {
                                return WorkProcessor.ProcessState.blocked(processor.getBlockedFuture());
                            }
                            return WorkProcessor.ProcessState.yielded();
                        }
                    }
                    if ((state = Objects.requireNonNull(transformation.process(this.element), "state is null")).isNeedsMoreData()) {
                        Preconditions.checkState((!processor.isFinished() ? 1 : 0) != 0, (Object)"Cannot request more data when base processor is finished");
                        this.element = null;
                    }
                    switch (state.getType()) {
                        case NEEDS_MORE_DATA: {
                            break;
                        }
                        case BLOCKED: {
                            return WorkProcessor.ProcessState.blocked(state.getBlocked());
                        }
                        case YIELD: {
                            return WorkProcessor.ProcessState.yielded();
                        }
                        case RESULT: {
                            return WorkProcessor.ProcessState.ofResult(state.getResult());
                        }
                        case FINISHED: {
                            return WorkProcessor.ProcessState.finished();
                        }
                    }
                }
            }
        });
    }

    static <T> WorkProcessor<T> create(WorkProcessor.Process<T> process) {
        return new ProcessWorkProcessor<T>(process);
    }

    private static class YieldingIterator<T>
    extends AbstractIterator<Optional<T>> {
        @Nullable
        WorkProcessor<T> processor;

        YieldingIterator(WorkProcessor<T> processor) {
            this.processor = Objects.requireNonNull(processor, "processorParameter is null");
        }

        protected Optional<T> computeNext() {
            if (this.processor.process()) {
                if (this.processor.isFinished()) {
                    this.processor = null;
                    return (Optional)this.endOfData();
                }
                return Optional.of(this.processor.getResult());
            }
            if (this.processor.isBlocked()) {
                throw new IllegalStateException("Cannot iterate over blocking WorkProcessor");
            }
            return Optional.empty();
        }
    }

    private static class YieldingProcess<T>
    implements WorkProcessor.Process<T> {
        final WorkProcessor<T> processor;
        final BooleanSupplier yieldSignal;
        boolean lastProcessYielded;

        YieldingProcess(WorkProcessor<T> processor, BooleanSupplier yieldSignal) {
            this.processor = Objects.requireNonNull(processor, "processor is null");
            this.yieldSignal = Objects.requireNonNull(yieldSignal, "yieldSignal is null");
        }

        @Override
        public WorkProcessor.ProcessState<T> process() {
            if (!this.lastProcessYielded && this.yieldSignal.getAsBoolean()) {
                this.lastProcessYielded = true;
                return WorkProcessor.ProcessState.yielded();
            }
            this.lastProcessYielded = false;
            return WorkProcessorUtils.getNextState(this.processor);
        }
    }

    private static class BlockingProcess<T>
    implements WorkProcessor.Process<T> {
        final WorkProcessor<T> processor;
        final Supplier<ListenableFuture<Void>> futureSupplier;
        WorkProcessor.ProcessState<T> state;

        BlockingProcess(WorkProcessor<T> processor, Supplier<ListenableFuture<Void>> futureSupplier) {
            this.processor = Objects.requireNonNull(processor, "processor is null");
            this.futureSupplier = Objects.requireNonNull(futureSupplier, "futureSupplier is null");
        }

        @Override
        public WorkProcessor.ProcessState<T> process() {
            ListenableFuture<Void> future;
            if (this.state == null) {
                this.state = WorkProcessorUtils.getNextState(this.processor);
            }
            if (!(future = this.futureSupplier.get()).isDone()) {
                if (this.state.getType() == WorkProcessor.ProcessState.Type.YIELD) {
                    this.state = null;
                }
                return WorkProcessor.ProcessState.blocked(future);
            }
            WorkProcessor.ProcessState<T> result = this.state;
            this.state = null;
            return result;
        }
    }

    private static class ProcessWorkProcessor<T>
    implements WorkProcessor<T> {
        @Nullable
        WorkProcessor.Process<T> process;
        WorkProcessor.ProcessState<T> state = WorkProcessor.ProcessState.yielded();

        ProcessWorkProcessor(WorkProcessor.Process<T> process) {
            this.process = Objects.requireNonNull(process, "process is null");
        }

        @Override
        public boolean process() {
            if (this.isBlocked()) {
                return false;
            }
            if (this.isFinished()) {
                return true;
            }
            this.state = Objects.requireNonNull(this.process.process());
            if (this.state.getType() == WorkProcessor.ProcessState.Type.FINISHED) {
                this.process = null;
                return true;
            }
            return this.state.getType() == WorkProcessor.ProcessState.Type.RESULT;
        }

        @Override
        public boolean isBlocked() {
            return this.state.getType() == WorkProcessor.ProcessState.Type.BLOCKED && !this.state.getBlocked().isDone();
        }

        @Override
        public ListenableFuture<Void> getBlockedFuture() {
            Preconditions.checkState((this.state.getType() == WorkProcessor.ProcessState.Type.BLOCKED ? 1 : 0) != 0, (Object)"Must be blocked to get blocked future");
            return this.state.getBlocked();
        }

        @Override
        public boolean isFinished() {
            return this.state.getType() == WorkProcessor.ProcessState.Type.FINISHED;
        }

        @Override
        public T getResult() {
            Preconditions.checkState((this.state.getType() == WorkProcessor.ProcessState.Type.RESULT ? 1 : 0) != 0, (Object)"process() must return true and must not be finished");
            return this.state.getResult();
        }
    }

    private static class ElementAndProcessor<T> {
        @Nullable
        final T element;
        final WorkProcessor<T> processor;

        ElementAndProcessor(T element, WorkProcessor<T> processor) {
            this.element = element;
            this.processor = Objects.requireNonNull(processor, "processor is null");
        }

        T getElement() {
            return this.element;
        }

        WorkProcessor<T> getProcessor() {
            return this.processor;
        }
    }
}

