/*
 * Decompiled with CFR 0.152.
 */
package io.trino.operator;

import com.google.common.annotations.VisibleForTesting;
import com.google.common.base.Preconditions;
import com.google.common.base.Verify;
import io.airlift.slice.SizeOf;
import io.trino.array.LongBigArray;
import io.trino.operator.PageWithPositionComparator;
import io.trino.operator.RowIdComparisonStrategy;
import io.trino.operator.RowReference;
import io.trino.operator.RowReferencePageManager;
import io.trino.spi.Page;
import io.trino.util.HeapTraversal;
import io.trino.util.LongBigArrayFIFOQueue;
import java.util.Objects;
import java.util.function.LongConsumer;
import javax.annotation.Nullable;

public class GroupedTopNRowNumberAccumulator {
    private static final long INSTANCE_SIZE = SizeOf.instanceSize(GroupedTopNRowNumberAccumulator.class);
    private static final long UNKNOWN_INDEX = -1L;
    private final GroupIdToHeapBuffer groupIdToHeapBuffer = new GroupIdToHeapBuffer();
    private final HeapNodeBuffer heapNodeBuffer = new HeapNodeBuffer();
    private final HeapTraversal heapTraversal = new HeapTraversal();
    private final RowIdComparisonStrategy strategy;
    private final int topN;
    private final LongConsumer rowIdEvictionListener;

    public GroupedTopNRowNumberAccumulator(RowIdComparisonStrategy strategy, int topN, LongConsumer rowIdEvictionListener) {
        this.strategy = Objects.requireNonNull(strategy, "strategy is null");
        Preconditions.checkArgument((topN > 0 ? 1 : 0) != 0, (Object)"topN must be greater than zero");
        this.topN = topN;
        this.rowIdEvictionListener = Objects.requireNonNull(rowIdEvictionListener, "rowIdEvictionListener is null");
    }

    public long sizeOf() {
        return INSTANCE_SIZE + this.groupIdToHeapBuffer.sizeOf() + this.heapNodeBuffer.sizeOf() + this.heapTraversal.sizeOf();
    }

    public int findFirstPositionToAdd(Page newPage, int groupCount, int[] groupIds, PageWithPositionComparator comparator, RowReferencePageManager pageManager) {
        int currentTotalGroups = this.groupIdToHeapBuffer.getTotalGroups();
        this.groupIdToHeapBuffer.allocateGroupIfNeeded(groupCount);
        for (int position = 0; position < newPage.getPositionCount(); ++position) {
            int rightPosition;
            int groupId = groupIds[position];
            if (groupId >= currentTotalGroups || this.calculateRootRowNumber(groupId) < (long)this.topN) {
                return position;
            }
            long heapRootNodeIndex = this.groupIdToHeapBuffer.getHeapRootNodeIndex(groupId);
            if (heapRootNodeIndex == -1L) {
                return position;
            }
            long rowId = this.heapNodeBuffer.getRowId(heapRootNodeIndex);
            Page rightPage = pageManager.getPage(rowId);
            if (comparator.compareTo(newPage, position, rightPage, rightPosition = pageManager.getPosition(rowId)) >= 0) continue;
            return position;
        }
        return -1;
    }

    public boolean add(int groupId, RowReference rowReference) {
        this.groupIdToHeapBuffer.allocateGroupIfNeeded(groupId);
        long heapRootNodeIndex = this.groupIdToHeapBuffer.getHeapRootNodeIndex(groupId);
        if (heapRootNodeIndex == -1L || this.calculateRootRowNumber(groupId) < (long)this.topN) {
            this.heapInsert(groupId, rowReference.allocateRowId());
            return true;
        }
        if (rowReference.compareTo(this.strategy, this.heapNodeBuffer.getRowId(heapRootNodeIndex)) < 0) {
            this.heapPopAndInsert(groupId, rowReference.allocateRowId(), this.rowIdEvictionListener);
            return true;
        }
        return false;
    }

    public long drainTo(int groupId, LongBigArray rowIdOutput) {
        long heapSize = this.groupIdToHeapBuffer.getHeapSize(groupId);
        rowIdOutput.ensureCapacity(heapSize);
        for (long i = heapSize - 1L; i >= 0L; --i) {
            rowIdOutput.set(i, this.peekRootRowId(groupId));
            this.heapPop(groupId, null);
        }
        return heapSize;
    }

    private long calculateRootRowNumber(int groupId) {
        return this.groupIdToHeapBuffer.getHeapSize(groupId);
    }

    private long peekRootRowId(int groupId) {
        long heapRootNodeIndex = this.groupIdToHeapBuffer.getHeapRootNodeIndex(groupId);
        Preconditions.checkArgument((heapRootNodeIndex != -1L ? 1 : 0) != 0, (Object)"No root to peek");
        return this.heapNodeBuffer.getRowId(heapRootNodeIndex);
    }

    private long getChildIndex(long heapNodeIndex, HeapTraversal.Child child) {
        return child == HeapTraversal.Child.LEFT ? this.heapNodeBuffer.getLeftChildHeapIndex(heapNodeIndex) : this.heapNodeBuffer.getRightChildHeapIndex(heapNodeIndex);
    }

    private void setChildIndex(long heapNodeIndex, HeapTraversal.Child child, long newChildIndex) {
        if (child == HeapTraversal.Child.LEFT) {
            this.heapNodeBuffer.setLeftChildHeapIndex(heapNodeIndex, newChildIndex);
        } else {
            this.heapNodeBuffer.setRightChildHeapIndex(heapNodeIndex, newChildIndex);
        }
    }

    private void heapPop(int groupId, @Nullable LongConsumer contextEvictionListener) {
        long heapRootNodeIndex = this.groupIdToHeapBuffer.getHeapRootNodeIndex(groupId);
        Preconditions.checkArgument((heapRootNodeIndex != -1L ? 1 : 0) != 0, (Object)"Group ID has an empty heap");
        long lastNodeIndex = this.heapDetachLastInsertionLeaf(groupId);
        long lastRowId = this.heapNodeBuffer.getRowId(lastNodeIndex);
        this.heapNodeBuffer.deallocate(lastNodeIndex);
        if (lastNodeIndex == heapRootNodeIndex) {
            if (contextEvictionListener != null) {
                contextEvictionListener.accept(lastRowId);
            }
        } else {
            this.heapPopAndInsert(groupId, lastRowId, contextEvictionListener);
        }
    }

    private long heapDetachLastInsertionLeaf(int groupId) {
        long heapRootNodeIndex = this.groupIdToHeapBuffer.getHeapRootNodeIndex(groupId);
        long heapSize = this.groupIdToHeapBuffer.getHeapSize(groupId);
        long previousNodeIndex = -1L;
        HeapTraversal.Child childPosition = null;
        long currentNodeIndex = heapRootNodeIndex;
        this.heapTraversal.resetWithPathTo(heapSize);
        while (!this.heapTraversal.isTarget()) {
            previousNodeIndex = currentNodeIndex;
            childPosition = this.heapTraversal.nextChild();
            Verify.verify(((currentNodeIndex = this.getChildIndex(currentNodeIndex, childPosition)) != -1L ? 1 : 0) != 0, (String)"Target node must exist", (Object[])new Object[0]);
        }
        if (previousNodeIndex == -1L) {
            this.groupIdToHeapBuffer.setHeapRootNodeIndex(groupId, -1L);
            this.groupIdToHeapBuffer.setHeapSize(groupId, 0L);
        } else {
            this.setChildIndex(previousNodeIndex, childPosition, -1L);
            this.groupIdToHeapBuffer.addHeapSize(groupId, -1L);
        }
        return currentNodeIndex;
    }

    private void heapInsert(int groupId, long newRowId) {
        long heapRootNodeIndex = this.groupIdToHeapBuffer.getHeapRootNodeIndex(groupId);
        if (heapRootNodeIndex == -1L) {
            heapRootNodeIndex = this.heapNodeBuffer.allocateNewNode(newRowId);
            this.groupIdToHeapBuffer.setHeapRootNodeIndex(groupId, heapRootNodeIndex);
            this.groupIdToHeapBuffer.setHeapSize(groupId, 1L);
            return;
        }
        long previousHeapNodeIndex = -1L;
        HeapTraversal.Child childPosition = null;
        long currentHeapNodeIndex = heapRootNodeIndex;
        this.heapTraversal.resetWithPathTo(this.groupIdToHeapBuffer.getHeapSize(groupId) + 1L);
        while (!this.heapTraversal.isTarget()) {
            long currentRowId = this.heapNodeBuffer.getRowId(currentHeapNodeIndex);
            if (this.strategy.compare(newRowId, currentRowId) > 0) {
                this.heapNodeBuffer.setRowId(currentHeapNodeIndex, newRowId);
                newRowId = currentRowId;
            }
            previousHeapNodeIndex = currentHeapNodeIndex;
            childPosition = this.heapTraversal.nextChild();
            currentHeapNodeIndex = this.getChildIndex(currentHeapNodeIndex, childPosition);
        }
        Verify.verify((previousHeapNodeIndex != -1L && childPosition != null ? 1 : 0) != 0, (String)"heap must have at least one node before starting traversal", (Object[])new Object[0]);
        Verify.verify((currentHeapNodeIndex == -1L ? 1 : 0) != 0, (String)"New child shouldn't exist yet", (Object[])new Object[0]);
        long newHeapNodeIndex = this.heapNodeBuffer.allocateNewNode(newRowId);
        this.setChildIndex(previousHeapNodeIndex, childPosition, newHeapNodeIndex);
        this.groupIdToHeapBuffer.incrementHeapSize(groupId);
    }

    private void heapPopAndInsert(int groupId, long newRowId, @Nullable LongConsumer contextEvictionListener) {
        long maxChildNodeIndex;
        long heapRootNodeIndex = this.groupIdToHeapBuffer.getHeapRootNodeIndex(groupId);
        Preconditions.checkState((heapRootNodeIndex != -1L ? 1 : 0) != 0, (Object)"popAndInsert() requires at least a root node");
        long poppedRowId = this.heapNodeBuffer.getRowId(heapRootNodeIndex);
        long currentNodeIndex = heapRootNodeIndex;
        while ((maxChildNodeIndex = this.heapNodeBuffer.getLeftChildHeapIndex(currentNodeIndex)) != -1L) {
            long rightRowId;
            long maxChildRowId = this.heapNodeBuffer.getRowId(maxChildNodeIndex);
            long rightChildNodeIndex = this.heapNodeBuffer.getRightChildHeapIndex(currentNodeIndex);
            if (rightChildNodeIndex != -1L && this.strategy.compare(rightRowId = this.heapNodeBuffer.getRowId(rightChildNodeIndex), maxChildRowId) > 0) {
                maxChildNodeIndex = rightChildNodeIndex;
                maxChildRowId = rightRowId;
            }
            if (this.strategy.compare(newRowId, maxChildRowId) >= 0) break;
            this.heapNodeBuffer.setRowId(currentNodeIndex, maxChildRowId);
            currentNodeIndex = maxChildNodeIndex;
        }
        this.heapNodeBuffer.setRowId(currentNodeIndex, newRowId);
        if (contextEvictionListener != null) {
            contextEvictionListener.accept(poppedRowId);
        }
    }

    @VisibleForTesting
    void verifyIntegrity() {
        long totalHeapNodes = 0L;
        for (int groupId = 0; groupId < this.groupIdToHeapBuffer.getTotalGroups(); ++groupId) {
            long heapSize = this.groupIdToHeapBuffer.getHeapSize(groupId);
            long rootNodeIndex = this.groupIdToHeapBuffer.getHeapRootNodeIndex(groupId);
            Verify.verify((rootNodeIndex == -1L || this.calculateRootRowNumber(groupId) <= (long)this.topN ? 1 : 0) != 0, (String)"Max heap has more values than needed", (Object[])new Object[0]);
            IntegrityStats integrityStats = this.verifyHeapIntegrity(rootNodeIndex);
            Verify.verify((integrityStats.getNodeCount() == heapSize ? 1 : 0) != 0, (String)"Recorded heap size does not match actual heap size", (Object[])new Object[0]);
            totalHeapNodes += integrityStats.getNodeCount();
        }
        Verify.verify((totalHeapNodes == this.heapNodeBuffer.getActiveNodeCount() ? 1 : 0) != 0, (String)"Failed to deallocate some unused nodes", (Object[])new Object[0]);
    }

    private IntegrityStats verifyHeapIntegrity(long heapNodeIndex) {
        if (heapNodeIndex == -1L) {
            return new IntegrityStats(0L, 0L);
        }
        long rowId = this.heapNodeBuffer.getRowId(heapNodeIndex);
        long leftChildHeapIndex = this.heapNodeBuffer.getLeftChildHeapIndex(heapNodeIndex);
        long rightChildHeapIndex = this.heapNodeBuffer.getRightChildHeapIndex(heapNodeIndex);
        if (leftChildHeapIndex != -1L) {
            Verify.verify((this.strategy.compare(rowId, this.heapNodeBuffer.getRowId(leftChildHeapIndex)) >= 0 ? 1 : 0) != 0, (String)"Max heap invariant violated", (Object[])new Object[0]);
        }
        if (rightChildHeapIndex != -1L) {
            Verify.verify((leftChildHeapIndex != -1L ? 1 : 0) != 0, (String)"Left should always be inserted before right", (Object[])new Object[0]);
            Verify.verify((this.strategy.compare(rowId, this.heapNodeBuffer.getRowId(rightChildHeapIndex)) >= 0 ? 1 : 0) != 0, (String)"Max heap invariant violated", (Object[])new Object[0]);
        }
        IntegrityStats leftIntegrityStats = this.verifyHeapIntegrity(leftChildHeapIndex);
        IntegrityStats rightIntegrityStats = this.verifyHeapIntegrity(rightChildHeapIndex);
        Verify.verify((Math.abs(leftIntegrityStats.getMaxDepth() - rightIntegrityStats.getMaxDepth()) <= 1L ? 1 : 0) != 0, (String)"Heap not balanced", (Object[])new Object[0]);
        return new IntegrityStats(Math.max(leftIntegrityStats.getMaxDepth(), rightIntegrityStats.getMaxDepth()) + 1L, leftIntegrityStats.getNodeCount() + rightIntegrityStats.getNodeCount() + 1L);
    }

    private static class GroupIdToHeapBuffer {
        private static final long INSTANCE_SIZE = SizeOf.instanceSize(GroupIdToHeapBuffer.class);
        private final LongBigArray heapIndexBuffer = new LongBigArray(-1L);
        private final LongBigArray sizeBuffer = new LongBigArray(0L);
        private int totalGroups;

        private GroupIdToHeapBuffer() {
        }

        public void allocateGroupIfNeeded(int groupId) {
            if (this.totalGroups > groupId) {
                return;
            }
            this.totalGroups = groupId + 1;
            this.heapIndexBuffer.ensureCapacity((long)this.totalGroups);
            this.sizeBuffer.ensureCapacity((long)this.totalGroups);
        }

        public int getTotalGroups() {
            return this.totalGroups;
        }

        public long getHeapRootNodeIndex(int groupId) {
            return this.heapIndexBuffer.get((long)groupId);
        }

        public void setHeapRootNodeIndex(int groupId, long heapNodeIndex) {
            this.heapIndexBuffer.set((long)groupId, heapNodeIndex);
        }

        public long getHeapSize(int groupId) {
            return this.sizeBuffer.get((long)groupId);
        }

        public void setHeapSize(int groupId, long count) {
            this.sizeBuffer.set((long)groupId, count);
        }

        public void addHeapSize(int groupId, long delta) {
            this.sizeBuffer.add((long)groupId, delta);
        }

        public void incrementHeapSize(int groupId) {
            this.sizeBuffer.increment((long)groupId);
        }

        public long sizeOf() {
            return INSTANCE_SIZE + this.heapIndexBuffer.sizeOf() + this.sizeBuffer.sizeOf();
        }
    }

    private static class HeapNodeBuffer {
        private static final long INSTANCE_SIZE = SizeOf.instanceSize(HeapNodeBuffer.class);
        private static final int POSITIONS_PER_ENTRY = 3;
        private static final int LEFT_CHILD_HEAP_INDEX_OFFSET = 1;
        private static final int RIGHT_CHILD_HEAP_INDEX_OFFSET = 2;
        private final LongBigArray buffer = new LongBigArray();
        private final LongBigArrayFIFOQueue emptySlots = new LongBigArrayFIFOQueue();
        private long capacity;

        private HeapNodeBuffer() {
        }

        public long allocateNewNode(long rowId) {
            long newHeapIndex;
            if (!this.emptySlots.isEmpty()) {
                newHeapIndex = this.emptySlots.dequeueLong();
            } else {
                newHeapIndex = this.capacity++;
                this.buffer.ensureCapacity(this.capacity * 3L);
            }
            this.setRowId(newHeapIndex, rowId);
            this.setLeftChildHeapIndex(newHeapIndex, -1L);
            this.setRightChildHeapIndex(newHeapIndex, -1L);
            return newHeapIndex;
        }

        public void deallocate(long index) {
            this.emptySlots.enqueue(index);
        }

        public long getActiveNodeCount() {
            return this.capacity - this.emptySlots.longSize();
        }

        public long getRowId(long index) {
            return this.buffer.get(index * 3L);
        }

        public void setRowId(long index, long rowId) {
            this.buffer.set(index * 3L, rowId);
        }

        public long getLeftChildHeapIndex(long index) {
            return this.buffer.get(index * 3L + 1L);
        }

        public void setLeftChildHeapIndex(long index, long childHeapIndex) {
            this.buffer.set(index * 3L + 1L, childHeapIndex);
        }

        public long getRightChildHeapIndex(long index) {
            return this.buffer.get(index * 3L + 2L);
        }

        public void setRightChildHeapIndex(long index, long childHeapIndex) {
            this.buffer.set(index * 3L + 2L, childHeapIndex);
        }

        public long sizeOf() {
            return INSTANCE_SIZE + this.buffer.sizeOf() + this.emptySlots.sizeOf();
        }
    }

    private static class IntegrityStats {
        private final long maxDepth;
        private final long nodeCount;

        public IntegrityStats(long maxDepth, long nodeCount) {
            this.maxDepth = maxDepth;
            this.nodeCount = nodeCount;
        }

        public long getMaxDepth() {
            return this.maxDepth;
        }

        public long getNodeCount() {
            return this.nodeCount;
        }
    }
}

