/*
 * Decompiled with CFR 0.152.
 */
package io.trino.execution.buffer;

import com.google.common.base.Preconditions;
import com.google.common.collect.ImmutableList;
import com.google.common.util.concurrent.ListenableFuture;
import com.google.errorprone.annotations.ThreadSafe;
import com.google.errorprone.annotations.concurrent.GuardedBy;
import io.airlift.units.DataSize;
import io.airlift.units.Duration;
import io.trino.execution.buffer.BufferResult;
import io.trino.execution.buffer.BufferTestUtils;
import io.trino.execution.buffer.ClientBuffer;
import io.trino.execution.buffer.PipelinedBufferInfo;
import io.trino.execution.buffer.PipelinedOutputBuffers;
import io.trino.execution.buffer.SerializedPageReference;
import io.trino.spi.Page;
import io.trino.spi.type.BigintType;
import java.util.ArrayDeque;
import java.util.ArrayList;
import java.util.Collection;
import java.util.Deque;
import java.util.List;
import java.util.Objects;
import java.util.Optional;
import java.util.concurrent.atomic.AtomicInteger;
import org.assertj.core.api.AbstractThrowableAssert;
import org.assertj.core.api.Assertions;
import org.testng.Assert;
import org.testng.annotations.Test;

public class TestClientBuffer {
    private static final String TASK_INSTANCE_ID = "task-instance-id";
    private static final ImmutableList<BigintType> TYPES = ImmutableList.of((Object)BigintType.BIGINT);
    private static final PipelinedOutputBuffers.OutputBufferId BUFFER_ID = new PipelinedOutputBuffers.OutputBufferId(33);
    private static final String INVALID_SEQUENCE_ID = "Invalid sequence id";
    private static final SerializedPageReference.PagesReleasedListener NOOP_RELEASE_LISTENER = (releasedPagesCount, releasedMemorySizeInBytes) -> {};

    @Test
    public void testSimplePushBuffer() {
        int i;
        ClientBuffer buffer = new ClientBuffer(TASK_INSTANCE_ID, BUFFER_ID, NOOP_RELEASE_LISTENER);
        for (i = 0; i < 3; ++i) {
            TestClientBuffer.addPage(buffer, BufferTestUtils.createPage(i));
        }
        TestClientBuffer.assertBufferInfo(buffer, 3, 0);
        BufferTestUtils.assertBufferResultEquals(TYPES, TestClientBuffer.getBufferResult(buffer, 0L, BufferTestUtils.sizeOfPages(10), BufferTestUtils.NO_WAIT), TestClientBuffer.bufferResult(0L, BufferTestUtils.createPage(0), BufferTestUtils.createPage(1), BufferTestUtils.createPage(2)));
        TestClientBuffer.assertBufferInfo(buffer, 3, 0);
        buffer.getPages(3L, BufferTestUtils.sizeOfPages(10)).cancel(true);
        TestClientBuffer.assertBufferInfo(buffer, 0, 3);
        for (i = 3; i < 6; ++i) {
            TestClientBuffer.addPage(buffer, BufferTestUtils.createPage(i));
        }
        TestClientBuffer.assertBufferInfo(buffer, 3, 3);
        BufferTestUtils.assertBufferResultEquals(TYPES, TestClientBuffer.getBufferResult(buffer, 3L, BufferTestUtils.sizeOfPages(1), BufferTestUtils.NO_WAIT), TestClientBuffer.bufferResult(3L, BufferTestUtils.createPage(3), new Page[0]));
        TestClientBuffer.assertBufferInfo(buffer, 3, 3);
        buffer.setNoMorePages();
        TestClientBuffer.assertBufferInfo(buffer, 3, 3);
        BufferTestUtils.assertBufferResultEquals(TYPES, TestClientBuffer.getBufferResult(buffer, 4L, BufferTestUtils.sizeOfPages(1), BufferTestUtils.NO_WAIT), TestClientBuffer.bufferResult(4L, BufferTestUtils.createPage(4), new Page[0]));
        TestClientBuffer.assertBufferInfo(buffer, 2, 4);
        BufferTestUtils.assertBufferResultEquals(TYPES, TestClientBuffer.getBufferResult(buffer, 5L, BufferTestUtils.sizeOfPages(30), BufferTestUtils.NO_WAIT), TestClientBuffer.bufferResult(5L, BufferTestUtils.createPage(5), new Page[0]));
        TestClientBuffer.assertBufferInfo(buffer, 1, 5);
        BufferTestUtils.assertBufferResultEquals(TYPES, TestClientBuffer.getBufferResult(buffer, 6L, BufferTestUtils.sizeOfPages(10), BufferTestUtils.NO_WAIT), BufferResult.emptyResults((String)TASK_INSTANCE_ID, (long)6L, (boolean)true));
        TestClientBuffer.assertBufferInfo(buffer, 0, 6);
        buffer.destroy();
        TestClientBuffer.assertBufferDestroyed(buffer, 6);
    }

    @Test
    public void testSimplePullBuffer() {
        ClientBuffer buffer = new ClientBuffer(TASK_INSTANCE_ID, BUFFER_ID, NOOP_RELEASE_LISTENER);
        TestingPagesSupplier supplier = new TestingPagesSupplier();
        for (int i = 0; i < 3; ++i) {
            supplier.addPage(BufferTestUtils.createPage(i));
        }
        Assert.assertEquals((int)supplier.getBufferedPages(), (int)3);
        BufferTestUtils.assertBufferResultEquals(TYPES, TestClientBuffer.getBufferResult(buffer, supplier, 0L, BufferTestUtils.sizeOfPages(10), BufferTestUtils.NO_WAIT), TestClientBuffer.bufferResult(0L, BufferTestUtils.createPage(0), BufferTestUtils.createPage(1), BufferTestUtils.createPage(2)));
        Assert.assertEquals((int)supplier.getBufferedPages(), (int)0);
        TestClientBuffer.assertBufferInfo(buffer, 3, 0);
        ListenableFuture pendingRead = buffer.getPages(3L, BufferTestUtils.sizeOfPages(1));
        Assert.assertEquals((int)supplier.getBufferedPages(), (int)0);
        TestClientBuffer.assertBufferInfo(buffer, 0, 3);
        Assert.assertFalse((boolean)pendingRead.isDone());
        for (int i = 3; i < 6; ++i) {
            supplier.addPage(BufferTestUtils.createPage(i));
        }
        Assert.assertEquals((int)supplier.getBufferedPages(), (int)3);
        buffer.loadPagesIfNecessary((ClientBuffer.PagesSupplier)supplier);
        BufferTestUtils.assertBufferResultEquals(TYPES, BufferTestUtils.getFuture((ListenableFuture<BufferResult>)pendingRead, BufferTestUtils.NO_WAIT), TestClientBuffer.bufferResult(3L, BufferTestUtils.createPage(3), new Page[0]));
        Assert.assertEquals((int)supplier.getBufferedPages(), (int)2);
        TestClientBuffer.assertBufferInfo(buffer, 1, 3);
        supplier.setNoMorePages();
        Assert.assertEquals((int)supplier.getBufferedPages(), (int)2);
        TestClientBuffer.assertBufferInfo(buffer, 1, 3);
        BufferTestUtils.assertBufferResultEquals(TYPES, TestClientBuffer.getBufferResult(buffer, supplier, 4L, BufferTestUtils.sizeOfPages(1), BufferTestUtils.NO_WAIT), TestClientBuffer.bufferResult(4L, BufferTestUtils.createPage(4), new Page[0]));
        TestClientBuffer.assertBufferInfo(buffer, 1, 4);
        Assert.assertEquals((int)supplier.getBufferedPages(), (int)1);
        BufferTestUtils.assertBufferResultEquals(TYPES, TestClientBuffer.getBufferResult(buffer, supplier, 5L, BufferTestUtils.sizeOfPages(30), BufferTestUtils.NO_WAIT), TestClientBuffer.bufferResult(5L, BufferTestUtils.createPage(5), new Page[0]));
        TestClientBuffer.assertBufferInfo(buffer, 1, 5);
        Assert.assertEquals((int)supplier.getBufferedPages(), (int)0);
        BufferTestUtils.assertBufferResultEquals(TYPES, TestClientBuffer.getBufferResult(buffer, supplier, 6L, BufferTestUtils.sizeOfPages(10), BufferTestUtils.NO_WAIT), BufferResult.emptyResults((String)TASK_INSTANCE_ID, (long)6L, (boolean)true));
        TestClientBuffer.assertBufferInfo(buffer, 0, 6);
        Assert.assertEquals((int)supplier.getBufferedPages(), (int)0);
        buffer.destroy();
        TestClientBuffer.assertBufferDestroyed(buffer, 6);
    }

    @Test
    public void testDuplicateRequests() {
        ClientBuffer buffer = new ClientBuffer(TASK_INSTANCE_ID, BUFFER_ID, NOOP_RELEASE_LISTENER);
        for (int i = 0; i < 3; ++i) {
            TestClientBuffer.addPage(buffer, BufferTestUtils.createPage(i));
        }
        TestClientBuffer.assertBufferInfo(buffer, 3, 0);
        BufferTestUtils.assertBufferResultEquals(TYPES, TestClientBuffer.getBufferResult(buffer, 0L, BufferTestUtils.sizeOfPages(10), BufferTestUtils.NO_WAIT), TestClientBuffer.bufferResult(0L, BufferTestUtils.createPage(0), BufferTestUtils.createPage(1), BufferTestUtils.createPage(2)));
        TestClientBuffer.assertBufferInfo(buffer, 3, 0);
        BufferTestUtils.assertBufferResultEquals(TYPES, TestClientBuffer.getBufferResult(buffer, 0L, BufferTestUtils.sizeOfPages(10), BufferTestUtils.NO_WAIT), TestClientBuffer.bufferResult(0L, BufferTestUtils.createPage(0), BufferTestUtils.createPage(1), BufferTestUtils.createPage(2)));
        TestClientBuffer.assertBufferInfo(buffer, 3, 0);
        buffer.getPages(3L, BufferTestUtils.sizeOfPages(10)).cancel(true);
        BufferTestUtils.assertBufferResultEquals(TYPES, TestClientBuffer.getBufferResult(buffer, 0L, BufferTestUtils.sizeOfPages(10), BufferTestUtils.NO_WAIT), BufferResult.emptyResults((String)TASK_INSTANCE_ID, (long)0L, (boolean)false));
        TestClientBuffer.assertBufferInfo(buffer, 0, 3);
    }

    @Test
    public void testAddAfterNoMorePages() {
        ClientBuffer buffer = new ClientBuffer(TASK_INSTANCE_ID, BUFFER_ID, NOOP_RELEASE_LISTENER);
        buffer.setNoMorePages();
        TestClientBuffer.addPage(buffer, BufferTestUtils.createPage(0));
        TestClientBuffer.addPage(buffer, BufferTestUtils.createPage(0));
        TestClientBuffer.assertBufferInfo(buffer, 0, 0);
    }

    @Test
    public void testAddAfterDestroy() {
        ClientBuffer buffer = new ClientBuffer(TASK_INSTANCE_ID, BUFFER_ID, NOOP_RELEASE_LISTENER);
        buffer.destroy();
        TestClientBuffer.addPage(buffer, BufferTestUtils.createPage(0));
        TestClientBuffer.addPage(buffer, BufferTestUtils.createPage(0));
        TestClientBuffer.assertBufferDestroyed(buffer, 0);
    }

    @Test
    public void testDestroy() {
        ClientBuffer buffer = new ClientBuffer(TASK_INSTANCE_ID, BUFFER_ID, NOOP_RELEASE_LISTENER);
        for (int i = 0; i < 5; ++i) {
            TestClientBuffer.addPage(buffer, BufferTestUtils.createPage(i));
        }
        buffer.setNoMorePages();
        BufferTestUtils.assertBufferResultEquals(TYPES, TestClientBuffer.getBufferResult(buffer, 0L, BufferTestUtils.sizeOfPages(1), BufferTestUtils.NO_WAIT), TestClientBuffer.bufferResult(0L, BufferTestUtils.createPage(0), new Page[0]));
        buffer.destroy();
        TestClientBuffer.assertBufferDestroyed(buffer, 0);
        BufferTestUtils.assertBufferResultEquals(TYPES, TestClientBuffer.getBufferResult(buffer, 1L, BufferTestUtils.sizeOfPages(1), BufferTestUtils.NO_WAIT), BufferResult.emptyResults((String)TASK_INSTANCE_ID, (long)0L, (boolean)true));
    }

    @Test
    public void testNoMorePagesFreesReader() {
        ClientBuffer buffer = new ClientBuffer(TASK_INSTANCE_ID, BUFFER_ID, NOOP_RELEASE_LISTENER);
        ListenableFuture future = buffer.getPages(0L, BufferTestUtils.sizeOfPages(10));
        Assert.assertFalse((boolean)future.isDone());
        TestClientBuffer.addPage(buffer, BufferTestUtils.createPage(0));
        BufferTestUtils.assertBufferResultEquals(TYPES, BufferTestUtils.getFuture((ListenableFuture<BufferResult>)future, BufferTestUtils.NO_WAIT), TestClientBuffer.bufferResult(0L, BufferTestUtils.createPage(0), new Page[0]));
        future = buffer.getPages(1L, BufferTestUtils.sizeOfPages(10));
        Assert.assertFalse((boolean)future.isDone());
        buffer.setNoMorePages();
        TestClientBuffer.assertBufferInfo(buffer, 0, 1);
        BufferTestUtils.assertBufferResultEquals(TYPES, BufferTestUtils.getFuture((ListenableFuture<BufferResult>)future, BufferTestUtils.NO_WAIT), BufferResult.emptyResults((String)TASK_INSTANCE_ID, (long)1L, (boolean)true));
    }

    @Test
    public void testDestroyFreesReader() {
        ClientBuffer buffer = new ClientBuffer(TASK_INSTANCE_ID, BUFFER_ID, NOOP_RELEASE_LISTENER);
        ListenableFuture future = buffer.getPages(0L, BufferTestUtils.sizeOfPages(10));
        Assert.assertFalse((boolean)future.isDone());
        TestClientBuffer.addPage(buffer, BufferTestUtils.createPage(0));
        Assert.assertTrue((boolean)future.isDone());
        BufferTestUtils.assertBufferResultEquals(TYPES, BufferTestUtils.getFuture((ListenableFuture<BufferResult>)future, BufferTestUtils.NO_WAIT), TestClientBuffer.bufferResult(0L, BufferTestUtils.createPage(0), new Page[0]));
        future = buffer.getPages(1L, BufferTestUtils.sizeOfPages(10));
        Assert.assertFalse((boolean)future.isDone());
        buffer.destroy();
        BufferTestUtils.assertBufferResultEquals(TYPES, BufferTestUtils.getFuture((ListenableFuture<BufferResult>)future, BufferTestUtils.NO_WAIT), BufferResult.emptyResults((String)TASK_INSTANCE_ID, (long)1L, (boolean)false));
        TestClientBuffer.assertBufferDestroyed(buffer, 1);
    }

    @Test
    public void testInvalidTokenFails() {
        ClientBuffer buffer = new ClientBuffer(TASK_INSTANCE_ID, BUFFER_ID, NOOP_RELEASE_LISTENER);
        TestClientBuffer.addPage(buffer, BufferTestUtils.createPage(0));
        TestClientBuffer.addPage(buffer, BufferTestUtils.createPage(1));
        buffer.getPages(1L, BufferTestUtils.sizeOfPages(10)).cancel(true);
        TestClientBuffer.assertBufferInfo(buffer, 1, 1);
        TestClientBuffer.assertInvalidSequenceId(buffer, -1);
        TestClientBuffer.assertBufferInfo(buffer, 1, 1);
        TestClientBuffer.assertInvalidSequenceId(buffer, 10);
        TestClientBuffer.assertBufferInfo(buffer, 1, 1);
    }

    @Test
    public void testReferenceCount() {
        AtomicInteger releasedPages = new AtomicInteger();
        SerializedPageReference.PagesReleasedListener onPagesReleased = (releasedPageCount, releasedMemorySizeInBytes) -> releasedPages.addAndGet(releasedPageCount);
        ClientBuffer buffer = new ClientBuffer(TASK_INSTANCE_ID, BUFFER_ID, onPagesReleased);
        TestClientBuffer.addPage(buffer, BufferTestUtils.createPage(0), onPagesReleased);
        TestClientBuffer.addPage(buffer, BufferTestUtils.createPage(1), onPagesReleased);
        Assert.assertEquals((int)releasedPages.get(), (int)0);
        TestClientBuffer.assertBufferInfo(buffer, 2, 0);
        BufferTestUtils.assertBufferResultEquals(TYPES, TestClientBuffer.getBufferResult(buffer, 0L, BufferTestUtils.sizeOfPages(0), BufferTestUtils.NO_WAIT), TestClientBuffer.bufferResult(0L, BufferTestUtils.createPage(0), new Page[0]));
        Assert.assertEquals((int)releasedPages.get(), (int)0);
        TestClientBuffer.assertBufferInfo(buffer, 2, 0);
        BufferTestUtils.assertBufferResultEquals(TYPES, TestClientBuffer.getBufferResult(buffer, 1L, BufferTestUtils.sizeOfPages(1), BufferTestUtils.NO_WAIT), TestClientBuffer.bufferResult(1L, BufferTestUtils.createPage(1), new Page[0]));
        Assert.assertEquals((int)releasedPages.get(), (int)1);
        TestClientBuffer.assertBufferInfo(buffer, 1, 1);
        buffer.destroy();
        Assert.assertEquals((int)releasedPages.get(), (int)2);
        TestClientBuffer.assertBufferDestroyed(buffer, 1);
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    @Test
    public void testProcessReadLockHolderAssertionsFireInTest() {
        ClientBuffer buffer = new ClientBuffer(TASK_INSTANCE_ID, BUFFER_ID, NOOP_RELEASE_LISTENER);
        try {
            ListenableFuture pendingRead = buffer.getPages(0L, DataSize.succinctBytes((long)1L));
            ClientBuffer clientBuffer = buffer;
            synchronized (clientBuffer) {
                TestClientBuffer.addPage(buffer, BufferTestUtils.createPage(0));
            }
            Assert.fail((String)"Expected AssertionError to be thrown, are assertions enabled in your testing environment?");
            Assert.assertTrue((boolean)BufferTestUtils.getFuture((ListenableFuture<BufferResult>)pendingRead, BufferTestUtils.NO_WAIT).isEmpty(), (String)"Code should not reach here");
        }
        catch (AssertionError ae) {
            Assert.assertEquals((String)((Throwable)((Object)ae)).getMessage(), (String)"Cannot process pending read while holding a lock on this");
        }
        finally {
            buffer.destroy();
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    @Test
    public void testGetPagesWithSupplierLockHolderAssertionsFireInTest() {
        ClientBuffer buffer = new ClientBuffer(TASK_INSTANCE_ID, BUFFER_ID, NOOP_RELEASE_LISTENER);
        TestingPagesSupplier supplier = new TestingPagesSupplier();
        supplier.addPage(BufferTestUtils.createPage(0));
        try {
            ListenableFuture result;
            ClientBuffer clientBuffer = buffer;
            synchronized (clientBuffer) {
                result = buffer.getPages(0L, BufferTestUtils.sizeOfPages(1), Optional.of(supplier));
            }
            Assert.fail((String)"Expected AssertionError to be thrown, are assertions enabled in your testing environment?");
            Assert.assertTrue((boolean)BufferTestUtils.getFuture((ListenableFuture<BufferResult>)result, BufferTestUtils.NO_WAIT).isEmpty(), (String)"Code should not reach here");
        }
        catch (AssertionError ae) {
            Assert.assertEquals((String)((Throwable)((Object)ae)).getMessage(), (String)"Cannot load pages while holding a lock on this");
        }
        finally {
            buffer.destroy();
        }
    }

    private static void assertInvalidSequenceId(ClientBuffer buffer, int sequenceId) {
        ((AbstractThrowableAssert)Assertions.assertThatThrownBy(() -> buffer.getPages((long)sequenceId, BufferTestUtils.sizeOfPages(10))).isInstanceOf(IllegalArgumentException.class)).hasMessage(INVALID_SEQUENCE_ID);
    }

    private static BufferResult getBufferResult(ClientBuffer buffer, long sequenceId, DataSize maxSize, Duration maxWait) {
        ListenableFuture future = buffer.getPages(sequenceId, maxSize);
        return BufferTestUtils.getFuture((ListenableFuture<BufferResult>)future, maxWait);
    }

    private static BufferResult getBufferResult(ClientBuffer buffer, ClientBuffer.PagesSupplier supplier, long sequenceId, DataSize maxSize, Duration maxWait) {
        ListenableFuture future = buffer.getPages(sequenceId, maxSize, Optional.of(supplier));
        return BufferTestUtils.getFuture((ListenableFuture<BufferResult>)future, maxWait);
    }

    private static void addPage(ClientBuffer buffer, Page page) {
        TestClientBuffer.addPage(buffer, page, NOOP_RELEASE_LISTENER);
    }

    private static void addPage(ClientBuffer buffer, Page page, SerializedPageReference.PagesReleasedListener onPagesReleased) {
        SerializedPageReference serializedPageReference = new SerializedPageReference(BufferTestUtils.serializePage(page), page.getPositionCount(), 1);
        buffer.enqueuePages((Collection)ImmutableList.of((Object)serializedPageReference));
        SerializedPageReference.dereferencePages((List)ImmutableList.of((Object)serializedPageReference), (SerializedPageReference.PagesReleasedListener)onPagesReleased);
    }

    private static void assertBufferInfo(ClientBuffer buffer, int bufferedPages, int pagesSent) {
        Assert.assertEquals((Object)buffer.getInfo(), (Object)new PipelinedBufferInfo(BUFFER_ID, (long)(bufferedPages + pagesSent), (long)(bufferedPages + pagesSent), bufferedPages, BufferTestUtils.sizeOfPages(bufferedPages).toBytes(), (long)pagesSent, false));
        Assert.assertFalse((boolean)buffer.isDestroyed());
    }

    private static BufferResult bufferResult(long token, Page firstPage, Page ... otherPages) {
        ImmutableList pages = ImmutableList.builder().add((Object)firstPage).add((Object[])otherPages).build();
        return BufferTestUtils.createBufferResult(TASK_INSTANCE_ID, token, (List<Page>)pages);
    }

    private static void assertBufferDestroyed(ClientBuffer buffer, int pagesSent) {
        PipelinedBufferInfo bufferInfo = buffer.getInfo();
        Assert.assertEquals((int)bufferInfo.getBufferedPages(), (int)0);
        Assert.assertEquals((long)bufferInfo.getPagesSent(), (long)pagesSent);
        Assert.assertTrue((boolean)bufferInfo.isFinished());
        Assert.assertTrue((boolean)buffer.isDestroyed());
    }

    @ThreadSafe
    private static class TestingPagesSupplier
    implements ClientBuffer.PagesSupplier {
        @GuardedBy(value="this")
        private final Deque<SerializedPageReference> buffer = new ArrayDeque<SerializedPageReference>();
        @GuardedBy(value="this")
        private boolean noMorePages;

        private TestingPagesSupplier() {
        }

        public synchronized boolean mayHaveMorePages() {
            return !this.noMorePages || !this.buffer.isEmpty();
        }

        synchronized void setNoMorePages() {
            this.noMorePages = true;
        }

        synchronized int getBufferedPages() {
            return this.buffer.size();
        }

        public synchronized void addPage(Page page) {
            Objects.requireNonNull(page, "page is null");
            Preconditions.checkState((!this.noMorePages ? 1 : 0) != 0);
            this.buffer.add(new SerializedPageReference(BufferTestUtils.serializePage(page), page.getPositionCount(), 1));
        }

        public synchronized List<SerializedPageReference> getPages(DataSize maxSize) {
            SerializedPageReference page;
            long maxBytes = maxSize.toBytes();
            ArrayList<SerializedPageReference> pages = new ArrayList<SerializedPageReference>();
            long bytesRemoved = 0L;
            while ((page = this.buffer.peek()) != null && (pages.isEmpty() || (bytesRemoved += page.getRetainedSizeInBytes()) <= maxBytes)) {
                Preconditions.checkState((this.buffer.poll() == page ? 1 : 0) != 0, (Object)"Buffer corrupted");
                pages.add(page);
            }
            return ImmutableList.copyOf(pages);
        }
    }
}

