/*
 * Decompiled with CFR 0.152.
 */
package io.trino.connector;

import com.google.common.base.MoreObjects;
import com.google.common.base.Preconditions;
import com.google.common.collect.ImmutableList;
import com.google.common.collect.ImmutableMap;
import com.google.common.collect.ImmutableSortedMap;
import com.google.common.hash.Hasher;
import com.google.common.hash.Hashing;
import com.google.inject.Inject;
import io.airlift.configuration.ConfigurationLoader;
import io.airlift.log.Logger;
import io.trino.connector.CatalogProperties;
import io.trino.connector.CatalogStore;
import io.trino.connector.ConnectorName;
import io.trino.connector.FileCatalogStoreConfig;
import io.trino.spi.ErrorCodeSupplier;
import io.trino.spi.StandardErrorCode;
import io.trino.spi.TrinoException;
import io.trino.spi.connector.CatalogHandle;
import java.io.File;
import java.io.FileOutputStream;
import java.io.IOException;
import java.io.UncheckedIOException;
import java.nio.file.Files;
import java.nio.file.StandardCopyOption;
import java.nio.file.attribute.FileAttribute;
import java.util.Arrays;
import java.util.Collection;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Properties;
import java.util.concurrent.ConcurrentHashMap;
import java.util.concurrent.ConcurrentMap;

public final class FileCatalogStore
implements CatalogStore {
    private static final Logger log = Logger.get(FileCatalogStore.class);
    private final boolean readOnly;
    private final File catalogsDirectory;
    private final ConcurrentMap<String, CatalogStore.StoredCatalog> catalogs = new ConcurrentHashMap<String, CatalogStore.StoredCatalog>();

    @Inject
    public FileCatalogStore(FileCatalogStoreConfig config) {
        Objects.requireNonNull(config, "config is null");
        this.readOnly = config.isReadOnly();
        this.catalogsDirectory = config.getCatalogConfigurationDir().getAbsoluteFile();
        List disabledCatalogs = (List)MoreObjects.firstNonNull(config.getDisabledCatalogs(), (Object)ImmutableList.of());
        for (File file : FileCatalogStore.listCatalogFiles(this.catalogsDirectory)) {
            String catalogName = com.google.common.io.Files.getNameWithoutExtension((String)file.getName());
            Preconditions.checkArgument((!catalogName.equals("system") ? 1 : 0) != 0, (Object)"Catalog name SYSTEM is reserved for internal usage");
            if (disabledCatalogs.contains(catalogName)) {
                log.info("Skipping disabled catalog %s", new Object[]{catalogName});
                continue;
            }
            this.catalogs.put(catalogName, new FileStoredCatalog(catalogName, file));
        }
    }

    @Override
    public Collection<CatalogStore.StoredCatalog> getCatalogs() {
        return ImmutableList.copyOf(this.catalogs.values());
    }

    @Override
    public CatalogProperties createCatalogProperties(String catalogName, ConnectorName connectorName, Map<String, String> properties) {
        this.checkModifiable();
        return new CatalogProperties(CatalogHandle.createRootCatalogHandle((String)catalogName, (CatalogHandle.CatalogVersion)FileCatalogStore.computeCatalogVersion(catalogName, connectorName, properties)), connectorName, (Map<String, String>)ImmutableMap.copyOf(properties));
    }

    @Override
    public void addOrReplaceCatalog(CatalogProperties catalogProperties) {
        this.checkModifiable();
        String catalogName = catalogProperties.getCatalogHandle().getCatalogName();
        File file = this.toFile(catalogName);
        Properties properties = new Properties();
        properties.setProperty("connector.name", catalogProperties.getConnectorName().toString());
        properties.putAll(catalogProperties.getProperties());
        try {
            File temporary = new File(file.getPath() + ".tmp");
            Files.createDirectories(temporary.getParentFile().toPath(), new FileAttribute[0]);
            try (FileOutputStream out = new FileOutputStream(temporary);){
                properties.store(out, null);
                out.flush();
                out.getFD().sync();
            }
            Files.move(temporary.toPath(), file.toPath(), StandardCopyOption.REPLACE_EXISTING);
        }
        catch (IOException e) {
            log.error((Throwable)e, "Could not store catalog properties for %s", new Object[]{catalogName});
            throw new TrinoException((ErrorCodeSupplier)StandardErrorCode.CATALOG_STORE_ERROR, "Could not store catalog properties");
        }
        this.catalogs.put(catalogName, new FileStoredCatalog(catalogName, file));
    }

    @Override
    public void removeCatalog(String catalogName) {
        this.checkModifiable();
        this.catalogs.remove(catalogName);
        try {
            Files.deleteIfExists(this.toFile(catalogName).toPath());
        }
        catch (IOException e) {
            log.warn((Throwable)e, "Could not remove catalog properties for %s", new Object[]{catalogName});
        }
    }

    private void checkModifiable() {
        if (this.readOnly) {
            throw new TrinoException((ErrorCodeSupplier)StandardErrorCode.NOT_SUPPORTED, "Catalog store is read only");
        }
    }

    private File toFile(String catalogName) {
        return new File(this.catalogsDirectory, catalogName + ".properties");
    }

    private static List<File> listCatalogFiles(File catalogsDirectory) {
        if (catalogsDirectory == null || !catalogsDirectory.isDirectory()) {
            return ImmutableList.of();
        }
        File[] files = catalogsDirectory.listFiles();
        if (files == null) {
            return ImmutableList.of();
        }
        return (List)Arrays.stream(files).filter(File::isFile).filter(file -> file.getName().endsWith(".properties")).collect(ImmutableList.toImmutableList());
    }

    static CatalogHandle.CatalogVersion computeCatalogVersion(String catalogName, ConnectorName connectorName, Map<String, String> properties) {
        Hasher hasher = Hashing.sha256().newHasher();
        hasher.putUnencodedChars((CharSequence)"catalog-hash");
        FileCatalogStore.hashLengthPrefixedString(hasher, catalogName);
        FileCatalogStore.hashLengthPrefixedString(hasher, connectorName.toString());
        hasher.putInt(properties.size());
        ImmutableSortedMap.copyOf(properties).forEach((key, value) -> {
            FileCatalogStore.hashLengthPrefixedString(hasher, key);
            FileCatalogStore.hashLengthPrefixedString(hasher, value);
        });
        return new CatalogHandle.CatalogVersion(hasher.hash().toString());
    }

    private static void hashLengthPrefixedString(Hasher hasher, String value) {
        hasher.putInt(value.length());
        hasher.putUnencodedChars((CharSequence)value);
    }

    private static class FileStoredCatalog
    implements CatalogStore.StoredCatalog {
        private final String name;
        private final File file;

        public FileStoredCatalog(String name, File file) {
            this.name = Objects.requireNonNull(name, "name is null");
            this.file = Objects.requireNonNull(file, "file is null");
        }

        @Override
        public String getName() {
            return this.name;
        }

        @Override
        public CatalogProperties loadProperties() {
            HashMap<String, String> properties;
            try {
                properties = new HashMap<String, String>(ConfigurationLoader.loadPropertiesFrom((String)this.file.getPath()));
            }
            catch (IOException e) {
                throw new UncheckedIOException("Error reading catalog property file " + this.file, e);
            }
            String connectorNameValue = (String)properties.remove("connector.name");
            Preconditions.checkState((connectorNameValue != null ? 1 : 0) != 0, (String)"Catalog configuration %s does not contain 'connector.name'", (Object)this.file.getAbsoluteFile());
            if (connectorNameValue.indexOf(45) >= 0) {
                String deprecatedConnectorName = connectorNameValue;
                connectorNameValue = connectorNameValue.replace('-', '_');
                log.warn("Catalog '%s' is using the deprecated connector name '%s'. The correct connector name is '%s'", new Object[]{this.name, deprecatedConnectorName, connectorNameValue});
            }
            ConnectorName connectorName = new ConnectorName(connectorNameValue);
            CatalogHandle catalogHandle = CatalogHandle.createRootCatalogHandle((String)this.name, (CatalogHandle.CatalogVersion)FileCatalogStore.computeCatalogVersion(this.name, connectorName, properties));
            return new CatalogProperties(catalogHandle, connectorName, (Map<String, String>)ImmutableMap.copyOf(properties));
        }
    }
}

