/*
 * Decompiled with CFR 0.152.
 */
package io.trino.operator.aggregation.histogram;

import com.google.common.base.Preconditions;
import com.google.common.base.Throwables;
import com.google.common.primitives.Ints;
import io.airlift.slice.SizeOf;
import io.trino.operator.VariableWidthData;
import io.trino.spi.ErrorCodeSupplier;
import io.trino.spi.StandardErrorCode;
import io.trino.spi.TrinoException;
import io.trino.spi.block.Block;
import io.trino.spi.block.BlockBuilder;
import io.trino.spi.block.MapBlockBuilder;
import io.trino.spi.type.BigintType;
import io.trino.spi.type.Type;
import jakarta.annotation.Nullable;
import java.lang.invoke.MethodHandle;
import java.lang.invoke.MethodHandles;
import java.lang.invoke.VarHandle;
import java.nio.ByteOrder;
import java.util.Arrays;
import java.util.Objects;

public final class TypedHistogram {
    private static final int INSTANCE_SIZE = SizeOf.instanceSize(TypedHistogram.class);
    private static final int MAX_ARRAY_SIZE = 0x7FFFFFF7;
    private static final int INITIAL_CAPACITY = 16;
    private static final long HASH_COMBINE_PRIME = 4999L;
    private static final int RECORDS_PER_GROUP_SHIFT = 10;
    private static final int RECORDS_PER_GROUP = 1024;
    private static final int RECORDS_PER_GROUP_MASK = 1023;
    private static final int VECTOR_LENGTH = 8;
    private static final VarHandle LONG_HANDLE = MethodHandles.byteArrayViewVarHandle(long[].class, ByteOrder.LITTLE_ENDIAN);
    private static final VarHandle INT_HANDLE = MethodHandles.byteArrayViewVarHandle(int[].class, ByteOrder.LITTLE_ENDIAN);
    private final Type type;
    private final MethodHandle readFlat;
    private final MethodHandle writeFlat;
    private final MethodHandle hashFlat;
    private final MethodHandle distinctFlatBlock;
    private final MethodHandle hashBlock;
    private final int recordSize;
    private final int recordGroupIdOffset;
    private final int recordNextIndexOffset;
    private final int recordCountOffset;
    private final int recordValueOffset;
    private int capacity;
    private int mask;
    private byte[] control;
    private byte[][] recordGroups;
    private final VariableWidthData variableWidthData;
    @Nullable
    private int[] groupRecordIndex;
    private int size;
    private int maxFill;

    private static int calculateMaxFill(int capacity) {
        return capacity / 16 * 15;
    }

    public TypedHistogram(Type type, MethodHandle readFlat, MethodHandle writeFlat, MethodHandle hashFlat, MethodHandle distinctFlatBlock, MethodHandle hashBlock, boolean grouped) {
        this.type = Objects.requireNonNull(type, "type is null");
        this.readFlat = Objects.requireNonNull(readFlat, "readFlat is null");
        this.writeFlat = Objects.requireNonNull(writeFlat, "writeFlat is null");
        this.hashFlat = Objects.requireNonNull(hashFlat, "hashFlat is null");
        this.distinctFlatBlock = Objects.requireNonNull(distinctFlatBlock, "distinctFlatBlock is null");
        this.hashBlock = Objects.requireNonNull(hashBlock, "hashBlock is null");
        this.capacity = 16;
        this.maxFill = TypedHistogram.calculateMaxFill(this.capacity);
        this.mask = this.capacity - 1;
        this.control = new byte[this.capacity + 8];
        this.groupRecordIndex = grouped ? new int[]{} : null;
        boolean variableWidth = type.isFlatVariableWidth();
        VariableWidthData variableWidthData = this.variableWidthData = variableWidth ? new VariableWidthData() : null;
        if (grouped) {
            this.recordGroupIdOffset = variableWidth ? 12 : 0;
            this.recordNextIndexOffset = this.recordGroupIdOffset + 4;
            this.recordCountOffset = this.recordNextIndexOffset + 4;
        } else {
            this.recordGroupIdOffset = Integer.MIN_VALUE;
            this.recordNextIndexOffset = Integer.MIN_VALUE;
            this.recordCountOffset = variableWidth ? 12 : 0;
        }
        this.recordValueOffset = this.recordCountOffset + 8;
        this.recordSize = this.recordValueOffset + type.getFlatFixedSize();
        this.recordGroups = TypedHistogram.createRecordGroups(this.capacity, this.recordSize);
    }

    private static byte[][] createRecordGroups(int capacity, int recordSize) {
        if (capacity < 1024) {
            return new byte[][]{new byte[Math.multiplyExact(capacity, recordSize)]};
        }
        byte[][] groups = new byte[capacity + 1 >> 10][];
        for (int i = 0; i < groups.length; ++i) {
            groups[i] = new byte[Math.multiplyExact(1024, recordSize)];
        }
        return groups;
    }

    public long getEstimatedSize() {
        return (long)INSTANCE_SIZE + SizeOf.sizeOf((byte[])this.control) + SizeOf.sizeOf((byte[])this.recordGroups[0]) * (long)this.recordGroups.length + (this.variableWidthData == null ? 0L : this.variableWidthData.getRetainedSizeBytes()) + (this.groupRecordIndex == null ? 0L : SizeOf.sizeOf((int[])this.groupRecordIndex));
    }

    public void setMaxGroupId(int maxGroupId) {
        Preconditions.checkState((this.groupRecordIndex != null ? 1 : 0) != 0, (Object)"grouping is not enabled");
        int requiredSize = maxGroupId + 1;
        Objects.checkIndex(requiredSize, 0x7FFFFFF7);
        int currentSize = this.groupRecordIndex.length;
        if (requiredSize > currentSize) {
            this.groupRecordIndex = Arrays.copyOf(this.groupRecordIndex, Ints.constrainToRange((int)(requiredSize * 2), (int)1024, (int)0x7FFFFFF7));
            Arrays.fill(this.groupRecordIndex, currentSize, this.groupRecordIndex.length, -1);
        }
    }

    public int size() {
        return this.size;
    }

    public void serialize(int groupId, MapBlockBuilder out) {
        if (this.size == 0) {
            out.appendNull();
            return;
        }
        if (this.groupRecordIndex == null) {
            Preconditions.checkArgument((groupId == 0 ? 1 : 0) != 0, (Object)"groupId must be zero when grouping is not enabled");
            out.buildEntry((keyBuilder, valueBuilder) -> {
                for (int i = 0; i < this.capacity; ++i) {
                    if (this.control[i] == 0) continue;
                    byte[] records = this.getRecords(i);
                    int recordOffset = this.getRecordOffset(i);
                    this.serializeEntry(keyBuilder, valueBuilder, records, recordOffset);
                }
            });
            return;
        }
        int index = this.groupRecordIndex[groupId];
        if (index == -1) {
            out.appendNull();
            return;
        }
        out.buildEntry((keyBuilder, valueBuilder) -> {
            int nextIndex = index;
            while (nextIndex >= 0) {
                byte[] records = this.getRecords(nextIndex);
                int recordOffset = this.getRecordOffset(nextIndex);
                this.serializeEntry(keyBuilder, valueBuilder, records, recordOffset);
                nextIndex = INT_HANDLE.get(records, recordOffset + this.recordNextIndexOffset);
            }
        });
    }

    private void serializeEntry(BlockBuilder keyBuilder, BlockBuilder valueBuilder, byte[] records, int recordOffset) {
        byte[] variableWidthChunk = VariableWidthData.EMPTY_CHUNK;
        if (this.variableWidthData != null) {
            variableWidthChunk = this.variableWidthData.getChunk(records, recordOffset);
        }
        try {
            this.readFlat.invokeExact(records, recordOffset + this.recordValueOffset, variableWidthChunk, keyBuilder);
        }
        catch (Throwable throwable) {
            Throwables.throwIfUnchecked((Throwable)throwable);
            throw new RuntimeException(throwable);
        }
        BigintType.BIGINT.writeLong(valueBuilder, LONG_HANDLE.get(records, recordOffset + this.recordCountOffset));
    }

    public void add(int groupId, Block block, int position, long count) {
        Preconditions.checkArgument((!block.isNull(position) ? 1 : 0) != 0, (Object)"value must not be null");
        Preconditions.checkArgument((groupId == 0 || this.groupRecordIndex != null ? 1 : 0) != 0, (Object)"groupId must be zero when grouping is not enabled");
        long hash = this.valueHashCode(groupId, block, position);
        byte hashPrefix = (byte)(hash & 0x7FL | 0x80L);
        int bucket = this.bucket((int)(hash >> 7));
        int step = 1;
        long repeated = TypedHistogram.repeat(hashPrefix);
        while (true) {
            long controlVector;
            int matchBucket;
            if ((matchBucket = this.matchInVector(groupId, block, position, bucket, repeated, controlVector = LONG_HANDLE.get(this.control, bucket))) >= 0) {
                this.addCount(matchBucket, count);
                return;
            }
            int emptyIndex = this.findEmptyInVector(controlVector, bucket);
            if (emptyIndex >= 0) {
                this.insert(emptyIndex, groupId, block, position, count, hashPrefix);
                ++this.size;
                if (this.size >= this.maxFill) {
                    this.rehash();
                }
                return;
            }
            bucket = this.bucket(bucket + step);
            step += 8;
        }
    }

    private int matchInVector(int groupId, Block block, int position, int vectorStartBucket, long repeated, long controlVector) {
        for (long controlMatches = TypedHistogram.match(controlVector, repeated); controlMatches != 0L; controlMatches &= controlMatches - 1L) {
            int bucket = this.bucket(vectorStartBucket + (Long.numberOfTrailingZeros(controlMatches) >>> 3));
            if (!this.valueNotDistinctFrom(bucket, block, position, groupId)) continue;
            return bucket;
        }
        return -1;
    }

    private int findEmptyInVector(long vector, int vectorStartBucket) {
        long controlMatches = TypedHistogram.match(vector, 0L);
        if (controlMatches == 0L) {
            return -1;
        }
        int slot = Long.numberOfTrailingZeros(controlMatches) >>> 3;
        return this.bucket(vectorStartBucket + slot);
    }

    private void addCount(int index, long increment) {
        byte[] records = this.getRecords(index);
        int countOffset = this.getRecordOffset(index) + this.recordCountOffset;
        LONG_HANDLE.set(records, countOffset, LONG_HANDLE.get(records, countOffset) + increment);
    }

    private void insert(int index, int groupId, Block block, int position, long count, byte hashPrefix) {
        this.setControl(index, hashPrefix);
        byte[] records = this.getRecords(index);
        int recordOffset = this.getRecordOffset(index);
        if (this.groupRecordIndex != null) {
            INT_HANDLE.set(records, recordOffset + this.recordGroupIdOffset, groupId);
            int nextRecordIndex = this.groupRecordIndex[groupId];
            this.groupRecordIndex[groupId] = index;
            INT_HANDLE.set(records, recordOffset + this.recordNextIndexOffset, nextRecordIndex);
        }
        LONG_HANDLE.set(records, recordOffset + this.recordCountOffset, count);
        byte[] variableWidthChunk = VariableWidthData.EMPTY_CHUNK;
        int variableWidthChunkOffset = 0;
        if (this.variableWidthData != null) {
            int variableWidthLength = this.type.getFlatVariableWidthSize(block, position);
            variableWidthChunk = this.variableWidthData.allocate(records, recordOffset, variableWidthLength);
            variableWidthChunkOffset = VariableWidthData.getChunkOffset(records, recordOffset);
        }
        try {
            this.writeFlat.invokeExact(block, position, records, recordOffset + this.recordValueOffset, variableWidthChunk, variableWidthChunkOffset);
        }
        catch (Throwable throwable) {
            Throwables.throwIfUnchecked((Throwable)throwable);
            throw new RuntimeException(throwable);
        }
    }

    private void setControl(int index, byte hashPrefix) {
        this.control[index] = hashPrefix;
        if (index < 8) {
            this.control[index + this.capacity] = hashPrefix;
        }
    }

    private void rehash() {
        int oldCapacity = this.capacity;
        byte[] oldControl = this.control;
        byte[][] oldRecordGroups = this.recordGroups;
        long newCapacityLong = (long)this.capacity * 2L;
        if (newCapacityLong > 0x7FFFFFF7L) {
            throw new TrinoException((ErrorCodeSupplier)StandardErrorCode.GENERIC_INSUFFICIENT_RESOURCES, "Size of hash table cannot exceed 1 billion entries");
        }
        this.capacity = (int)newCapacityLong;
        this.maxFill = TypedHistogram.calculateMaxFill(this.capacity);
        this.mask = this.capacity - 1;
        this.control = new byte[this.capacity + 8];
        this.recordGroups = TypedHistogram.createRecordGroups(this.capacity, this.recordSize);
        if (this.groupRecordIndex != null) {
            Arrays.fill(this.groupRecordIndex, -1);
        }
        block0: for (int oldIndex = 0; oldIndex < oldCapacity; ++oldIndex) {
            if (oldControl[oldIndex] == 0) continue;
            byte[] oldRecords = oldRecordGroups[oldIndex >> 10];
            int oldRecordOffset = this.getRecordOffset(oldIndex);
            int groupId = 0;
            if (this.groupRecordIndex != null) {
                groupId = INT_HANDLE.get(oldRecords, oldRecordOffset + this.recordGroupIdOffset);
            }
            long hash = this.valueHashCode(groupId, oldRecords, oldIndex);
            byte hashPrefix = (byte)(hash & 0x7FL | 0x80L);
            int bucket = this.bucket((int)(hash >> 7));
            int step = 1;
            while (true) {
                long controlVector;
                int emptyIndex;
                if ((emptyIndex = this.findEmptyInVector(controlVector = LONG_HANDLE.get(this.control, bucket), bucket)) >= 0) {
                    this.setControl(emptyIndex, hashPrefix);
                    byte[] records = this.getRecords(emptyIndex);
                    int recordOffset = this.getRecordOffset(emptyIndex);
                    System.arraycopy(oldRecords, oldRecordOffset, records, recordOffset, this.recordSize);
                    if (this.groupRecordIndex == null) continue block0;
                    INT_HANDLE.set(records, recordOffset + this.recordNextIndexOffset, this.groupRecordIndex[groupId]);
                    this.groupRecordIndex[groupId] = emptyIndex;
                    continue block0;
                }
                bucket = this.bucket(bucket + step);
                step += 8;
            }
        }
    }

    private int bucket(int hash) {
        return hash & this.mask;
    }

    private byte[] getRecords(int index) {
        return this.recordGroups[index >> 10];
    }

    private int getRecordOffset(int index) {
        return (index & 0x3FF) * this.recordSize;
    }

    private long valueHashCode(int groupId, byte[] records, int index) {
        int recordOffset = this.getRecordOffset(index);
        try {
            byte[] variableWidthChunk = VariableWidthData.EMPTY_CHUNK;
            if (this.variableWidthData != null) {
                variableWidthChunk = this.variableWidthData.getChunk(records, recordOffset);
            }
            long valueHash = this.hashFlat.invokeExact(records, recordOffset + this.recordValueOffset, variableWidthChunk);
            return (long)groupId * 4999L + valueHash;
        }
        catch (Throwable throwable) {
            Throwables.throwIfUnchecked((Throwable)throwable);
            throw new RuntimeException(throwable);
        }
    }

    private long valueHashCode(int groupId, Block right, int rightPosition) {
        try {
            long valueHash = this.hashBlock.invokeExact(right, rightPosition);
            return (long)groupId * 4999L + valueHash;
        }
        catch (Throwable throwable) {
            Throwables.throwIfUnchecked((Throwable)throwable);
            throw new RuntimeException(throwable);
        }
    }

    private boolean valueNotDistinctFrom(int leftPosition, Block right, int rightPosition, int rightGroupId) {
        long leftGroupId;
        byte[] leftRecords = this.getRecords(leftPosition);
        int leftRecordOffset = this.getRecordOffset(leftPosition);
        if (this.groupRecordIndex != null && (leftGroupId = (long)INT_HANDLE.get(leftRecords, leftRecordOffset + this.recordGroupIdOffset)) != (long)rightGroupId) {
            return false;
        }
        byte[] leftVariableWidthChunk = VariableWidthData.EMPTY_CHUNK;
        if (this.variableWidthData != null) {
            leftVariableWidthChunk = this.variableWidthData.getChunk(leftRecords, leftRecordOffset);
        }
        try {
            return !this.distinctFlatBlock.invokeExact(leftRecords, leftRecordOffset + this.recordValueOffset, leftVariableWidthChunk, right, rightPosition);
        }
        catch (Throwable throwable) {
            Throwables.throwIfUnchecked((Throwable)throwable);
            throw new RuntimeException(throwable);
        }
    }

    private static long repeat(byte value) {
        return (long)(value & 0xFF) * 0x101010101010101L;
    }

    private static long match(long vector, long repeatedValue) {
        long comparison = vector ^ repeatedValue;
        return comparison - 0x101010101010101L & (comparison ^ 0xFFFFFFFFFFFFFFFFL) & 0x8080808080808080L;
    }
}

