/*
 * Decompiled with CFR 0.152.
 */
package io.trino.server.protocol;

import com.google.common.base.MoreObjects;
import com.google.common.collect.ImmutableList;
import com.google.common.collect.Sets;
import io.airlift.log.Logger;
import io.trino.client.ClientTypeSignature;
import io.trino.client.ClientTypeSignatureParameter;
import io.trino.client.Column;
import io.trino.client.FailureInfo;
import io.trino.client.NamedClientTypeSignature;
import io.trino.client.QueryError;
import io.trino.client.RowFieldName;
import io.trino.client.StageStats;
import io.trino.client.StatementStats;
import io.trino.client.Warning;
import io.trino.execution.ExecutionFailureInfo;
import io.trino.execution.QueryInfo;
import io.trino.execution.QueryState;
import io.trino.execution.QueryStats;
import io.trino.execution.StageInfo;
import io.trino.execution.TaskInfo;
import io.trino.spi.ErrorCode;
import io.trino.spi.StandardErrorCode;
import io.trino.spi.TrinoWarning;
import io.trino.spi.WarningCode;
import io.trino.spi.type.Type;
import io.trino.spi.type.TypeSignature;
import io.trino.spi.type.TypeSignatureParameter;
import io.trino.sql.ExpressionFormatter;
import io.trino.sql.analyzer.TypeSignatureTranslator;
import io.trino.sql.tree.DataType;
import io.trino.sql.tree.DateTimeDataType;
import io.trino.sql.tree.Expression;
import io.trino.sql.tree.GenericDataType;
import io.trino.sql.tree.IntervalDayTimeDataType;
import io.trino.sql.tree.NumericParameter;
import io.trino.sql.tree.RowDataType;
import io.trino.sql.tree.TypeParameter;
import io.trino.util.Failures;
import java.util.HashSet;
import java.util.List;
import java.util.Set;
import java.util.stream.Collectors;

public final class ProtocolUtil {
    private static final Logger log = Logger.get(ProtocolUtil.class);

    private ProtocolUtil() {
    }

    public static Column createColumn(String name, Type type, boolean supportsParametricDateTime) {
        String formatted = ProtocolUtil.formatType(TypeSignatureTranslator.toSqlType(type), supportsParametricDateTime);
        return new Column(name, formatted, ProtocolUtil.toClientTypeSignature(type.getTypeSignature(), supportsParametricDateTime));
    }

    private static String formatType(DataType type, boolean supportsParametricDateTime) {
        if (type instanceof DateTimeDataType) {
            DateTimeDataType dataTimeType = (DateTimeDataType)type;
            if (!supportsParametricDateTime) {
                if (dataTimeType.getType() == DateTimeDataType.Type.TIMESTAMP && dataTimeType.isWithTimeZone()) {
                    return "timestamp with time zone";
                }
                if (dataTimeType.getType() == DateTimeDataType.Type.TIMESTAMP && !dataTimeType.isWithTimeZone()) {
                    return "timestamp";
                }
                if (dataTimeType.getType() == DateTimeDataType.Type.TIME && !dataTimeType.isWithTimeZone()) {
                    return "time";
                }
                if (dataTimeType.getType() == DateTimeDataType.Type.TIME && dataTimeType.isWithTimeZone()) {
                    return "time with time zone";
                }
            }
            return ExpressionFormatter.formatExpression((Expression)type);
        }
        if (type instanceof RowDataType) {
            RowDataType rowDataType = (RowDataType)type;
            return rowDataType.getFields().stream().map(field -> field.getName().map(name -> name + " ").orElse("") + ProtocolUtil.formatType(field.getType(), supportsParametricDateTime)).collect(Collectors.joining(", ", "row(", ")"));
        }
        if (type instanceof GenericDataType) {
            GenericDataType dataType = (GenericDataType)type;
            if (dataType.getArguments().isEmpty()) {
                return dataType.getName().getValue();
            }
            return dataType.getArguments().stream().map(parameter -> {
                if (parameter instanceof NumericParameter) {
                    return ((NumericParameter)parameter).getValue();
                }
                if (parameter instanceof TypeParameter) {
                    return ProtocolUtil.formatType(((TypeParameter)parameter).getValue(), supportsParametricDateTime);
                }
                throw new IllegalArgumentException("Unsupported parameter type: " + parameter.getClass().getName());
            }).collect(Collectors.joining(", ", dataType.getName().getValue() + "(", ")"));
        }
        if (type instanceof IntervalDayTimeDataType) {
            return ExpressionFormatter.formatExpression((Expression)type);
        }
        throw new IllegalArgumentException("Unsupported data type: " + type.getClass().getName());
    }

    private static ClientTypeSignature toClientTypeSignature(TypeSignature signature, boolean supportsParametricDateTime) {
        if (!supportsParametricDateTime) {
            if (signature.getBase().equalsIgnoreCase("timestamp")) {
                return new ClientTypeSignature("timestamp");
            }
            if (signature.getBase().equalsIgnoreCase("timestamp with time zone")) {
                return new ClientTypeSignature("timestamp with time zone");
            }
            if (signature.getBase().equalsIgnoreCase("time")) {
                return new ClientTypeSignature("time");
            }
            if (signature.getBase().equalsIgnoreCase("time with time zone")) {
                return new ClientTypeSignature("time with time zone");
            }
        }
        return new ClientTypeSignature(signature.getBase(), (List)signature.getParameters().stream().map(parameter -> ProtocolUtil.toClientTypeSignatureParameter(parameter, supportsParametricDateTime)).collect(ImmutableList.toImmutableList()));
    }

    private static ClientTypeSignatureParameter toClientTypeSignatureParameter(TypeSignatureParameter parameter, boolean supportsParametricDateTime) {
        switch (parameter.getKind()) {
            case TYPE: {
                return ClientTypeSignatureParameter.ofType((ClientTypeSignature)ProtocolUtil.toClientTypeSignature(parameter.getTypeSignature(), supportsParametricDateTime));
            }
            case NAMED_TYPE: {
                return ClientTypeSignatureParameter.ofNamedType((NamedClientTypeSignature)new NamedClientTypeSignature(parameter.getNamedTypeSignature().getFieldName().map(value -> new RowFieldName(value.getName())), ProtocolUtil.toClientTypeSignature(parameter.getNamedTypeSignature().getTypeSignature(), supportsParametricDateTime)));
            }
            case LONG: {
                return ClientTypeSignatureParameter.ofLong((long)parameter.getLongLiteral());
            }
        }
        throw new IllegalArgumentException("Unsupported kind: " + parameter.getKind());
    }

    public static StatementStats toStatementStats(QueryInfo queryInfo) {
        QueryStats queryStats = queryInfo.getQueryStats();
        StageInfo outputStage = queryInfo.getOutputStage().orElse(null);
        HashSet<String> globalUniqueNodes = new HashSet<String>();
        StageStats rootStageStats = ProtocolUtil.toStageStats(outputStage, globalUniqueNodes);
        return StatementStats.builder().setState(queryInfo.getState().toString()).setQueued(queryInfo.getState() == QueryState.QUEUED).setScheduled(queryInfo.isScheduled()).setProgressPercentage(queryInfo.getProgressPercentage()).setRunningPercentage(queryInfo.getRunningPercentage()).setNodes(globalUniqueNodes.size()).setTotalSplits(queryStats.getTotalDrivers()).setQueuedSplits(queryStats.getQueuedDrivers()).setRunningSplits(queryStats.getRunningDrivers() + queryStats.getBlockedDrivers()).setCompletedSplits(queryStats.getCompletedDrivers()).setCpuTimeMillis(queryStats.getTotalCpuTime().toMillis()).setWallTimeMillis(queryStats.getTotalScheduledTime().toMillis()).setQueuedTimeMillis(queryStats.getQueuedTime().toMillis()).setElapsedTimeMillis(queryStats.getElapsedTime().toMillis()).setProcessedRows(queryStats.getRawInputPositions()).setProcessedBytes(queryStats.getRawInputDataSize().toBytes()).setPhysicalInputBytes(queryStats.getPhysicalInputDataSize().toBytes()).setPhysicalWrittenBytes(queryStats.getPhysicalWrittenDataSize().toBytes()).setPeakMemoryBytes(queryStats.getPeakUserMemoryReservation().toBytes()).setSpilledBytes(queryStats.getSpilledDataSize().toBytes()).setRootStage(rootStageStats).build();
    }

    private static StageStats toStageStats(StageInfo stageInfo, Set<String> globalUniqueNodes) {
        if (stageInfo == null) {
            return null;
        }
        io.trino.execution.StageStats stageStats = stageInfo.getStageStats();
        StageStats.Builder builder = StageStats.builder().setStageId(String.valueOf(stageInfo.getStageId().getId())).setState(stageInfo.getState().toString()).setDone(stageInfo.getState().isDone()).setTotalSplits(stageStats.getTotalDrivers()).setQueuedSplits(stageStats.getQueuedDrivers()).setRunningSplits(stageStats.getRunningDrivers() + stageStats.getBlockedDrivers()).setCompletedSplits(stageStats.getCompletedDrivers()).setCpuTimeMillis(stageStats.getTotalCpuTime().toMillis()).setWallTimeMillis(stageStats.getTotalScheduledTime().toMillis()).setProcessedRows(stageStats.getRawInputPositions()).setProcessedBytes(stageStats.getRawInputDataSize().toBytes()).setPhysicalInputBytes(stageStats.getPhysicalInputDataSize().toBytes()).setFailedTasks(stageStats.getFailedTasks()).setCoordinatorOnly(stageInfo.isCoordinatorOnly()).setNodes(ProtocolUtil.countStageAndAddGlobalUniqueNodes(stageInfo, globalUniqueNodes));
        List<StageInfo> subStages = stageInfo.getSubStages();
        if (subStages.isEmpty()) {
            builder.setSubStages((List)ImmutableList.of());
        } else {
            ImmutableList.Builder subStagesBuilder = ImmutableList.builderWithExpectedSize((int)subStages.size());
            for (StageInfo subStage : subStages) {
                subStagesBuilder.add((Object)ProtocolUtil.toStageStats(subStage, globalUniqueNodes));
            }
            builder.setSubStages((List)subStagesBuilder.build());
        }
        return builder.build();
    }

    private static int countStageAndAddGlobalUniqueNodes(StageInfo stageInfo, Set<String> globalUniqueNodes) {
        List<TaskInfo> tasks = stageInfo.getTasks();
        HashSet stageUniqueNodes = Sets.newHashSetWithExpectedSize((int)tasks.size());
        for (TaskInfo task : tasks) {
            String nodeId = task.getTaskStatus().getNodeId();
            stageUniqueNodes.add(nodeId);
            globalUniqueNodes.add(nodeId);
        }
        return stageUniqueNodes.size();
    }

    public static Warning toClientWarning(TrinoWarning warning) {
        WarningCode code = warning.getWarningCode();
        return new Warning(new Warning.Code(code.getCode(), code.getName()), warning.getMessage());
    }

    public static QueryError toQueryError(QueryInfo queryInfo) {
        ErrorCode errorCode;
        ExecutionFailureInfo executionFailure;
        QueryState state = queryInfo.getState();
        if (state != QueryState.FAILED) {
            return null;
        }
        if (queryInfo.getFailureInfo() != null) {
            executionFailure = queryInfo.getFailureInfo();
        } else {
            log.warn("Query %s in state %s has no failure info", new Object[]{queryInfo.getQueryId(), state});
            executionFailure = Failures.toFailure(new RuntimeException(String.format("Query is %s (reason unknown)", new Object[]{state})));
        }
        FailureInfo failure = executionFailure.toFailureInfo();
        if (queryInfo.getErrorCode() != null) {
            errorCode = queryInfo.getErrorCode();
        } else {
            errorCode = StandardErrorCode.GENERIC_INTERNAL_ERROR.toErrorCode();
            log.warn("Failed query %s has no error code", new Object[]{queryInfo.getQueryId()});
        }
        return new QueryError((String)MoreObjects.firstNonNull((Object)failure.getMessage(), (Object)"Internal error"), null, errorCode.getCode(), errorCode.getName(), errorCode.getType().toString(), failure.getErrorLocation(), failure);
    }
}

