/*
 * Decompiled with CFR 0.152.
 */
package io.trino.server.security.jwt;

import com.google.common.base.CharMatcher;
import com.google.common.io.Files;
import com.google.inject.Inject;
import io.airlift.security.pem.PemReader;
import io.jsonwebtoken.Claims;
import io.jsonwebtoken.JwsHeader;
import io.jsonwebtoken.SignatureAlgorithm;
import io.jsonwebtoken.SigningKeyResolver;
import io.jsonwebtoken.UnsupportedJwtException;
import io.jsonwebtoken.security.SecurityException;
import io.trino.server.security.jwt.JwtAuthenticatorConfig;
import java.io.File;
import java.io.IOException;
import java.nio.charset.Charset;
import java.nio.charset.StandardCharsets;
import java.security.GeneralSecurityException;
import java.security.Key;
import java.util.Base64;
import java.util.Objects;
import java.util.concurrent.ConcurrentHashMap;
import java.util.concurrent.ConcurrentMap;
import javax.crypto.spec.SecretKeySpec;

public class FileSigningKeyResolver
implements SigningKeyResolver {
    private static final String DEFAULT_KEY = "default-key";
    private static final CharMatcher INVALID_KID_CHARS = CharMatcher.inRange((char)'a', (char)'z').or(CharMatcher.inRange((char)'A', (char)'Z')).or(CharMatcher.inRange((char)'0', (char)'9')).or(CharMatcher.anyOf((CharSequence)"_-")).negate();
    private static final String KEY_ID_VARIABLE = "${KID}";
    private final String keyFile;
    private final LoadedKey staticKey;
    private final ConcurrentMap<String, LoadedKey> keys = new ConcurrentHashMap<String, LoadedKey>();

    @Inject
    public FileSigningKeyResolver(JwtAuthenticatorConfig config) {
        this(config.getKeyFile());
    }

    public FileSigningKeyResolver(String keyFile) {
        this.keyFile = Objects.requireNonNull(keyFile, "keyFile is null");
        this.staticKey = keyFile.contains(KEY_ID_VARIABLE) ? null : FileSigningKeyResolver.loadKeyFile(new File(keyFile));
    }

    public Key resolveSigningKey(JwsHeader header, Claims claims) {
        return this.getKey(header);
    }

    public Key resolveSigningKey(JwsHeader header, String plaintext) {
        return this.getKey(header);
    }

    private Key getKey(JwsHeader<?> header) {
        SignatureAlgorithm algorithm = SignatureAlgorithm.forName((String)header.getAlgorithm());
        if (this.staticKey != null) {
            return this.staticKey.getKey(algorithm);
        }
        String keyId = FileSigningKeyResolver.getKeyId(header);
        LoadedKey key = this.keys.computeIfAbsent(keyId, this::loadKey);
        return key.getKey(algorithm);
    }

    private static String getKeyId(JwsHeader<?> header) {
        String keyId = header.getKeyId();
        if (keyId == null) {
            return DEFAULT_KEY;
        }
        keyId = INVALID_KID_CHARS.replaceFrom((CharSequence)keyId, '_');
        return keyId;
    }

    private LoadedKey loadKey(String keyId) {
        return FileSigningKeyResolver.loadKeyFile(new File(this.keyFile.replace(KEY_ID_VARIABLE, keyId)));
    }

    public static LoadedKey loadKeyFile(File file) {
        String data;
        if (!file.canRead()) {
            throw new SecurityException("Unknown signing key ID");
        }
        try {
            data = Files.asCharSource((File)file, (Charset)StandardCharsets.US_ASCII).read();
        }
        catch (IOException e) {
            throw new SecurityException("Unable to read signing key", (Throwable)e);
        }
        if (PemReader.isPem((String)data)) {
            try {
                return new LoadedKey(PemReader.loadPublicKey((String)data));
            }
            catch (RuntimeException | GeneralSecurityException e) {
                throw new SecurityException("Unable to decode PEM signing key id", (Throwable)e);
            }
        }
        try {
            byte[] rawKey = Base64.getMimeDecoder().decode(data.getBytes(StandardCharsets.US_ASCII));
            return new LoadedKey(rawKey);
        }
        catch (RuntimeException e) {
            throw new SecurityException("Unable to decode HMAC signing key", (Throwable)e);
        }
    }

    private static class LoadedKey {
        private final Key publicKey;
        private final byte[] hmacKey;

        public LoadedKey(Key publicKey) {
            this.publicKey = Objects.requireNonNull(publicKey, "publicKey is null");
            this.hmacKey = null;
        }

        public LoadedKey(byte[] hmacKey) {
            this.hmacKey = Objects.requireNonNull(hmacKey, "hmacKey is null");
            this.publicKey = null;
        }

        public Key getKey(SignatureAlgorithm algorithm) {
            if (algorithm.isHmac()) {
                if (this.hmacKey == null) {
                    throw new UnsupportedJwtException(String.format("JWT is signed with %s, but no HMAC key is configured", algorithm));
                }
                return new SecretKeySpec(this.hmacKey, algorithm.getJcaName());
            }
            if (this.publicKey == null) {
                throw new UnsupportedJwtException(String.format("JWT is signed with %s, but no key is configured", algorithm));
            }
            return this.publicKey;
        }
    }
}

