/*
 * Decompiled with CFR 0.152.
 */
package io.trino.metadata;

import com.google.common.base.Preconditions;
import com.google.common.collect.ImmutableList;
import com.google.common.collect.ImmutableMap;
import com.google.common.collect.Lists;
import io.trino.Session;
import io.trino.SystemSessionProperties;
import io.trino.metadata.Metadata;
import io.trino.metadata.QualifiedObjectName;
import io.trino.spi.ErrorCodeSupplier;
import io.trino.spi.StandardErrorCode;
import io.trino.spi.TrinoException;
import io.trino.spi.connector.CatalogHandle;
import io.trino.spi.connector.CatalogSchemaName;
import io.trino.spi.connector.ColumnMetadata;
import io.trino.spi.connector.ConnectorTableMetadata;
import io.trino.spi.connector.SchemaTableName;
import io.trino.spi.security.PrincipalType;
import io.trino.spi.security.TrinoPrincipal;
import io.trino.spi.type.Type;
import io.trino.sql.analyzer.SemanticExceptions;
import io.trino.sql.tree.GrantorSpecification;
import io.trino.sql.tree.Identifier;
import io.trino.sql.tree.Node;
import io.trino.sql.tree.PrincipalSpecification;
import io.trino.sql.tree.QualifiedName;
import java.util.List;
import java.util.Locale;
import java.util.Map;
import java.util.Objects;
import java.util.Optional;

public final class MetadataUtil {
    private MetadataUtil() {
    }

    public static void checkTableName(String catalogName, Optional<String> schemaName, Optional<String> tableName) {
        MetadataUtil.checkCatalogName(catalogName);
        schemaName.ifPresent(name -> MetadataUtil.checkLowerCase(name, "schemaName"));
        tableName.ifPresent(name -> MetadataUtil.checkLowerCase(name, "tableName"));
        Preconditions.checkArgument((schemaName.isPresent() || tableName.isEmpty() ? 1 : 0) != 0, (Object)"tableName specified but schemaName is missing");
    }

    public static String checkCatalogName(String catalogName) {
        return MetadataUtil.checkLowerCase(catalogName, "catalogName");
    }

    public static String checkSchemaName(String schemaName) {
        return MetadataUtil.checkLowerCase(schemaName, "schemaName");
    }

    public static String checkTableName(String tableName) {
        return MetadataUtil.checkLowerCase(tableName, "tableName");
    }

    public static void checkObjectName(String catalogName, String schemaName, String objectName) {
        MetadataUtil.checkLowerCase(catalogName, "catalogName");
        MetadataUtil.checkLowerCase(schemaName, "schemaName");
        MetadataUtil.checkLowerCase(objectName, "objectName");
    }

    public static String checkLowerCase(String value, String name) {
        if (value == null) {
            throw new NullPointerException(String.format("%s is null", name));
        }
        Preconditions.checkArgument((boolean)value.equals(value.toLowerCase(Locale.ENGLISH)), (String)"%s is not lowercase: %s", (Object)name, (Object)value);
        return value;
    }

    public static ColumnMetadata findColumnMetadata(ConnectorTableMetadata tableMetadata, String columnName) {
        for (ColumnMetadata columnMetadata : tableMetadata.getColumns()) {
            if (!columnName.equals(columnMetadata.getName())) continue;
            return columnMetadata;
        }
        return null;
    }

    public static CatalogHandle getRequiredCatalogHandle(Metadata metadata, Session session, Node node, String catalogName) {
        return metadata.getCatalogHandle(session, catalogName).orElseThrow(() -> SemanticExceptions.semanticException((ErrorCodeSupplier)StandardErrorCode.CATALOG_NOT_FOUND, node, "Catalog '%s' does not exist", catalogName));
    }

    public static CatalogSchemaName createCatalogSchemaName(Session session, Node node, Optional<QualifiedName> schema) {
        String catalogName = session.getCatalog().orElse(null);
        String schemaName = session.getSchema().orElse(null);
        if (schema.isPresent()) {
            List parts = schema.get().getParts();
            if (parts.size() > 2) {
                throw SemanticExceptions.semanticException((ErrorCodeSupplier)StandardErrorCode.SYNTAX_ERROR, node, "Too many parts in schema name: %s", schema.get());
            }
            if (parts.size() == 2) {
                catalogName = (String)parts.get(0);
            }
            schemaName = schema.get().getSuffix();
        }
        if (catalogName == null) {
            throw SemanticExceptions.semanticException((ErrorCodeSupplier)StandardErrorCode.MISSING_CATALOG_NAME, node, "Catalog must be specified when session catalog is not set", new Object[0]);
        }
        if (schemaName == null) {
            throw SemanticExceptions.semanticException((ErrorCodeSupplier)StandardErrorCode.MISSING_SCHEMA_NAME, node, "Schema must be specified when session schema is not set", new Object[0]);
        }
        return new CatalogSchemaName(catalogName, schemaName);
    }

    public static QualifiedObjectName createQualifiedObjectName(Session session, Node node, QualifiedName name) {
        Objects.requireNonNull(session, "session is null");
        Objects.requireNonNull(name, "name is null");
        if (name.getParts().size() > 3) {
            throw new TrinoException((ErrorCodeSupplier)StandardErrorCode.SYNTAX_ERROR, String.format("Too many dots in table name: %s", name));
        }
        List parts = Lists.reverse((List)name.getParts());
        String objectName = (String)parts.get(0);
        String schemaName = parts.size() > 1 ? (String)parts.get(1) : session.getSchema().orElseThrow(() -> SemanticExceptions.semanticException((ErrorCodeSupplier)StandardErrorCode.MISSING_SCHEMA_NAME, node, "Schema must be specified when session schema is not set", new Object[0]));
        String catalogName = parts.size() > 2 ? (String)parts.get(2) : session.getCatalog().orElseThrow(() -> SemanticExceptions.semanticException((ErrorCodeSupplier)StandardErrorCode.MISSING_CATALOG_NAME, node, "Catalog must be specified when session catalog is not set", new Object[0]));
        return new QualifiedObjectName(catalogName, schemaName, objectName);
    }

    public static TrinoPrincipal createPrincipal(Session session, GrantorSpecification specification) {
        GrantorSpecification.Type type = specification.getType();
        switch (type) {
            case PRINCIPAL: {
                return MetadataUtil.createPrincipal((PrincipalSpecification)specification.getPrincipal().get());
            }
            case CURRENT_USER: {
                return new TrinoPrincipal(PrincipalType.USER, session.getIdentity().getUser());
            }
            case CURRENT_ROLE: {
                throw new UnsupportedOperationException("CURRENT_ROLE is not yet supported");
            }
        }
        throw new IllegalArgumentException("Unsupported type: " + type);
    }

    public static TrinoPrincipal createPrincipal(PrincipalSpecification specification) {
        PrincipalSpecification.Type type = specification.getType();
        switch (type) {
            case UNSPECIFIED: 
            case USER: {
                return new TrinoPrincipal(PrincipalType.USER, specification.getName().getValue());
            }
            case ROLE: {
                return new TrinoPrincipal(PrincipalType.ROLE, specification.getName().getValue());
            }
        }
        throw new IllegalArgumentException("Unsupported type: " + type);
    }

    public static PrincipalSpecification createPrincipal(TrinoPrincipal principal) {
        PrincipalType type = principal.getType();
        switch (type) {
            case USER: {
                return new PrincipalSpecification(PrincipalSpecification.Type.USER, new Identifier(principal.getName()));
            }
            case ROLE: {
                return new PrincipalSpecification(PrincipalSpecification.Type.ROLE, new Identifier(principal.getName()));
            }
        }
        throw new IllegalArgumentException("Unsupported type: " + type);
    }

    public static boolean tableExists(Metadata metadata, Session session, String table) {
        if (session.getCatalog().isEmpty() || session.getSchema().isEmpty()) {
            return false;
        }
        QualifiedObjectName name = new QualifiedObjectName(session.getCatalog().get(), session.getSchema().get(), table);
        return metadata.getTableHandle(session, name).isPresent();
    }

    public static void checkRoleExists(Session session, Node node, Metadata metadata, TrinoPrincipal principal, Optional<String> catalog) {
        if (principal.getType() == PrincipalType.ROLE) {
            MetadataUtil.checkRoleExists(session, node, metadata, principal.getName(), catalog);
        }
    }

    public static void checkRoleExists(Session session, Node node, Metadata metadata, String role, Optional<String> catalog) {
        if (!metadata.roleExists(session, role, catalog)) {
            throw SemanticExceptions.semanticException((ErrorCodeSupplier)StandardErrorCode.ROLE_NOT_FOUND, node, "Role '%s' does not exist%s", role, catalog.map(c -> String.format(" in catalog '%s'", c)).orElse(""));
        }
    }

    public static Optional<String> processRoleCommandCatalog(Metadata metadata, Session session, Node node, Optional<String> catalog) {
        boolean legacyCatalogRoles = SystemSessionProperties.isLegacyCatalogRoles(session);
        if (catalog.isEmpty() && legacyCatalogRoles && (catalog = session.getCatalog()).isEmpty()) {
            throw SemanticExceptions.semanticException((ErrorCodeSupplier)StandardErrorCode.MISSING_CATALOG_NAME, node, "Session catalog must be set", new Object[0]);
        }
        catalog.ifPresent(catalogName -> MetadataUtil.getRequiredCatalogHandle(metadata, session, node, catalogName));
        if (catalog.isPresent() && !metadata.isCatalogManagedSecurity(session, catalog.get())) {
            throw SemanticExceptions.semanticException((ErrorCodeSupplier)StandardErrorCode.NOT_SUPPORTED, node, "Catalog '%s' does not support role management", catalog.get());
        }
        return catalog;
    }

    public static class TableMetadataBuilder {
        private final SchemaTableName tableName;
        private final ImmutableList.Builder<ColumnMetadata> columns = ImmutableList.builder();
        private final ImmutableMap.Builder<String, Object> properties = ImmutableMap.builder();
        private final Optional<String> comment;

        public static TableMetadataBuilder tableMetadataBuilder(SchemaTableName tableName) {
            return new TableMetadataBuilder(tableName);
        }

        private TableMetadataBuilder(SchemaTableName tableName) {
            this(tableName, Optional.empty());
        }

        private TableMetadataBuilder(SchemaTableName tableName, Optional<String> comment) {
            this.tableName = tableName;
            this.comment = comment;
        }

        public TableMetadataBuilder column(String columnName, Type type) {
            this.columns.add((Object)new ColumnMetadata(columnName, type));
            return this;
        }

        public TableMetadataBuilder hiddenColumn(String columnName, Type type) {
            this.columns.add((Object)ColumnMetadata.builder().setName(columnName).setType(type).setHidden(true).build());
            return this;
        }

        public TableMetadataBuilder property(String name, Object value) {
            this.properties.put((Object)name, value);
            return this;
        }

        public ConnectorTableMetadata build() {
            return new ConnectorTableMetadata(this.tableName, (List)this.columns.build(), (Map)this.properties.buildOrThrow(), this.comment);
        }
    }
}

