/*
 * Decompiled with CFR 0.152.
 */
package io.trino.sql;

import com.google.common.base.Joiner;
import com.google.common.collect.ImmutableMap;
import io.airlift.slice.Slice;
import io.airlift.slice.Slices;
import io.trino.Session;
import io.trino.SessionTestUtils;
import io.trino.metadata.Metadata;
import io.trino.security.AccessControl;
import io.trino.security.AllowAllAccessControl;
import io.trino.spi.ErrorCodeSupplier;
import io.trino.spi.StandardErrorCode;
import io.trino.spi.type.BigintType;
import io.trino.spi.type.BooleanType;
import io.trino.spi.type.CharType;
import io.trino.spi.type.DateType;
import io.trino.spi.type.DecimalType;
import io.trino.spi.type.DoubleType;
import io.trino.spi.type.Int128;
import io.trino.spi.type.IntegerType;
import io.trino.spi.type.SqlTimestampWithTimeZone;
import io.trino.spi.type.TimeType;
import io.trino.spi.type.TimeZoneKey;
import io.trino.spi.type.TimestampType;
import io.trino.spi.type.Type;
import io.trino.spi.type.VarbinaryType;
import io.trino.spi.type.VarcharType;
import io.trino.sql.ExpressionFormatter;
import io.trino.sql.ExpressionTestUtils;
import io.trino.sql.ExpressionUtils;
import io.trino.sql.PlannerContext;
import io.trino.sql.parser.SqlParser;
import io.trino.sql.planner.ExpressionInterpreter;
import io.trino.sql.planner.Symbol;
import io.trino.sql.planner.SymbolResolver;
import io.trino.sql.planner.TestingPlannerContext;
import io.trino.sql.planner.TypeAnalyzer;
import io.trino.sql.planner.TypeProvider;
import io.trino.sql.planner.assertions.SymbolAliases;
import io.trino.sql.planner.iterative.rule.CanonicalizeExpressionRewriter;
import io.trino.sql.tree.Expression;
import io.trino.sql.tree.LikePredicate;
import io.trino.sql.tree.NodeRef;
import io.trino.sql.tree.StringLiteral;
import io.trino.sql.tree.SymbolReference;
import io.trino.testing.assertions.TrinoExceptionAssert;
import io.trino.transaction.TestingTransactionManager;
import io.trino.transaction.TransactionBuilder;
import io.trino.transaction.TransactionManager;
import io.trino.type.DateTimes;
import io.trino.type.IntervalDayTimeType;
import java.util.Locale;
import java.util.Map;
import java.util.Optional;
import java.util.function.Function;
import java.util.stream.IntStream;
import org.intellij.lang.annotations.Language;
import org.joda.time.DateTime;
import org.joda.time.DateTimeZone;
import org.joda.time.LocalDate;
import org.joda.time.LocalTime;
import org.joda.time.ReadableInstant;
import org.junit.jupiter.api.Test;
import org.junit.jupiter.api.Timeout;
import org.testng.Assert;

public class TestExpressionInterpreter {
    private static final int TEST_CHAR_TYPE_LENGTH = 17;
    private static final int TEST_VARCHAR_TYPE_LENGTH = 17;
    private static final TypeProvider SYMBOL_TYPES = TypeProvider.copyOf((Map)ImmutableMap.builder().put((Object)new Symbol("bound_integer"), (Object)IntegerType.INTEGER).put((Object)new Symbol("bound_long"), (Object)BigintType.BIGINT).put((Object)new Symbol("bound_string"), (Object)VarcharType.createVarcharType((int)17)).put((Object)new Symbol("bound_varbinary"), (Object)VarbinaryType.VARBINARY).put((Object)new Symbol("bound_double"), (Object)DoubleType.DOUBLE).put((Object)new Symbol("bound_boolean"), (Object)BooleanType.BOOLEAN).put((Object)new Symbol("bound_date"), (Object)DateType.DATE).put((Object)new Symbol("bound_time"), (Object)TimeType.TIME_MILLIS).put((Object)new Symbol("bound_timestamp"), (Object)TimestampType.TIMESTAMP_MILLIS).put((Object)new Symbol("bound_pattern"), (Object)VarcharType.VARCHAR).put((Object)new Symbol("bound_null_string"), (Object)VarcharType.VARCHAR).put((Object)new Symbol("bound_decimal_short"), (Object)DecimalType.createDecimalType((int)5, (int)2)).put((Object)new Symbol("bound_decimal_long"), (Object)DecimalType.createDecimalType((int)23, (int)3)).put((Object)new Symbol("time"), (Object)BigintType.BIGINT).put((Object)new Symbol("unbound_integer"), (Object)IntegerType.INTEGER).put((Object)new Symbol("unbound_long"), (Object)BigintType.BIGINT).put((Object)new Symbol("unbound_long2"), (Object)BigintType.BIGINT).put((Object)new Symbol("unbound_long3"), (Object)BigintType.BIGINT).put((Object)new Symbol("unbound_char"), (Object)CharType.createCharType((int)17)).put((Object)new Symbol("unbound_string"), (Object)VarcharType.VARCHAR).put((Object)new Symbol("unbound_double"), (Object)DoubleType.DOUBLE).put((Object)new Symbol("unbound_boolean"), (Object)BooleanType.BOOLEAN).put((Object)new Symbol("unbound_date"), (Object)DateType.DATE).put((Object)new Symbol("unbound_time"), (Object)TimeType.TIME_MILLIS).put((Object)new Symbol("unbound_timestamp"), (Object)TimestampType.TIMESTAMP_MILLIS).put((Object)new Symbol("unbound_interval"), (Object)IntervalDayTimeType.INTERVAL_DAY_TIME).put((Object)new Symbol("unbound_pattern"), (Object)VarcharType.VARCHAR).put((Object)new Symbol("unbound_null_string"), (Object)VarcharType.VARCHAR).buildOrThrow());
    private static final SymbolResolver INPUTS = symbol -> {
        switch (symbol.getName().toLowerCase(Locale.ENGLISH)) {
            case "bound_integer": {
                return 1234L;
            }
            case "bound_long": {
                return 1234L;
            }
            case "bound_string": {
                return Slices.utf8Slice((String)"hello");
            }
            case "bound_double": {
                return 12.34;
            }
            case "bound_date": {
                return new LocalDate(2001, 8, 22).toDateMidnight(DateTimeZone.UTC).getMillis();
            }
            case "bound_time": {
                return new LocalTime(3, 4, 5, 321).toDateTime((ReadableInstant)new DateTime(0L, DateTimeZone.UTC)).getMillis();
            }
            case "bound_timestamp": {
                return DateTimes.scaleEpochMillisToMicros((long)new DateTime(2001, 8, 22, 3, 4, 5, 321, DateTimeZone.UTC).getMillis());
            }
            case "bound_pattern": {
                return Slices.utf8Slice((String)"%el%");
            }
            case "bound_timestamp_with_timezone": {
                return SqlTimestampWithTimeZone.newInstance((int)3, (long)new DateTime(1970, 1, 1, 1, 0, 0, 999, DateTimeZone.UTC).getMillis(), (int)0, (TimeZoneKey)TimeZoneKey.getTimeZoneKey((String)"Z"));
            }
            case "bound_varbinary": {
                return Slices.wrappedBuffer((byte[])new byte[]{-85});
            }
            case "bound_decimal_short": {
                return 12345L;
            }
            case "bound_decimal_long": {
                return Int128.valueOf((String)"12345678901234567890123");
            }
        }
        return symbol.toSymbolReference();
    };
    private static final SqlParser SQL_PARSER = new SqlParser();
    private static final TestingTransactionManager TRANSACTION_MANAGER = new TestingTransactionManager();
    private static final PlannerContext PLANNER_CONTEXT = TestingPlannerContext.plannerContextBuilder().withTransactionManager(TRANSACTION_MANAGER).build();

    @Test
    public void testAnd() {
        TestExpressionInterpreter.assertOptimizedEquals("true AND false", "false");
        TestExpressionInterpreter.assertOptimizedEquals("false AND true", "false");
        TestExpressionInterpreter.assertOptimizedEquals("false AND false", "false");
        TestExpressionInterpreter.assertOptimizedEquals("true AND NULL", "NULL");
        TestExpressionInterpreter.assertOptimizedEquals("false AND NULL", "false");
        TestExpressionInterpreter.assertOptimizedEquals("NULL AND true", "NULL");
        TestExpressionInterpreter.assertOptimizedEquals("NULL AND false", "false");
        TestExpressionInterpreter.assertOptimizedEquals("NULL AND NULL", "NULL");
        TestExpressionInterpreter.assertOptimizedEquals("unbound_string='z' AND true", "unbound_string='z'");
        TestExpressionInterpreter.assertOptimizedEquals("unbound_string='z' AND false", "false");
        TestExpressionInterpreter.assertOptimizedEquals("true AND unbound_string='z'", "unbound_string='z'");
        TestExpressionInterpreter.assertOptimizedEquals("false AND unbound_string='z'", "false");
        TestExpressionInterpreter.assertOptimizedEquals("bound_string='z' AND bound_long=1+1", "bound_string='z' AND bound_long=2");
    }

    @Test
    public void testOr() {
        TestExpressionInterpreter.assertOptimizedEquals("true OR true", "true");
        TestExpressionInterpreter.assertOptimizedEquals("true OR false", "true");
        TestExpressionInterpreter.assertOptimizedEquals("false OR true", "true");
        TestExpressionInterpreter.assertOptimizedEquals("false OR false", "false");
        TestExpressionInterpreter.assertOptimizedEquals("true OR NULL", "true");
        TestExpressionInterpreter.assertOptimizedEquals("NULL OR true", "true");
        TestExpressionInterpreter.assertOptimizedEquals("NULL OR NULL", "NULL");
        TestExpressionInterpreter.assertOptimizedEquals("false OR NULL", "NULL");
        TestExpressionInterpreter.assertOptimizedEquals("NULL OR false", "NULL");
        TestExpressionInterpreter.assertOptimizedEquals("bound_string='z' OR true", "true");
        TestExpressionInterpreter.assertOptimizedEquals("bound_string='z' OR false", "bound_string='z'");
        TestExpressionInterpreter.assertOptimizedEquals("true OR bound_string='z'", "true");
        TestExpressionInterpreter.assertOptimizedEquals("false OR bound_string='z'", "bound_string='z'");
        TestExpressionInterpreter.assertOptimizedEquals("bound_string='z' OR bound_long=1+1", "bound_string='z' OR bound_long=2");
    }

    @Test
    public void testComparison() {
        TestExpressionInterpreter.assertOptimizedEquals("NULL = NULL", "NULL");
        TestExpressionInterpreter.assertOptimizedEquals("'a' = 'b'", "false");
        TestExpressionInterpreter.assertOptimizedEquals("'a' = 'a'", "true");
        TestExpressionInterpreter.assertOptimizedEquals("'a' = NULL", "NULL");
        TestExpressionInterpreter.assertOptimizedEquals("NULL = 'a'", "NULL");
        TestExpressionInterpreter.assertOptimizedEquals("bound_integer = 1234", "true");
        TestExpressionInterpreter.assertOptimizedEquals("bound_integer = 12340000000", "false");
        TestExpressionInterpreter.assertOptimizedEquals("bound_long = BIGINT '1234'", "true");
        TestExpressionInterpreter.assertOptimizedEquals("bound_long = 1234", "true");
        TestExpressionInterpreter.assertOptimizedEquals("bound_double = 12.34", "true");
        TestExpressionInterpreter.assertOptimizedEquals("bound_string = 'hello'", "true");
        TestExpressionInterpreter.assertOptimizedEquals("unbound_long = bound_long", "unbound_long = 1234");
        TestExpressionInterpreter.assertOptimizedEquals("10151082135029368 = 10151082135029369", "false");
        TestExpressionInterpreter.assertOptimizedEquals("bound_varbinary = X'a b'", "true");
        TestExpressionInterpreter.assertOptimizedEquals("bound_varbinary = X'a d'", "false");
        TestExpressionInterpreter.assertOptimizedEquals("1.1 = 1.1", "true");
        TestExpressionInterpreter.assertOptimizedEquals("9876543210.9874561203 = 9876543210.9874561203", "true");
        TestExpressionInterpreter.assertOptimizedEquals("bound_decimal_short = 123.45", "true");
        TestExpressionInterpreter.assertOptimizedEquals("bound_decimal_long = 12345678901234567890.123", "true");
    }

    @Test
    public void testIsDistinctFrom() {
        TestExpressionInterpreter.assertOptimizedEquals("NULL IS DISTINCT FROM NULL", "false");
        TestExpressionInterpreter.assertOptimizedEquals("3 IS DISTINCT FROM 4", "true");
        TestExpressionInterpreter.assertOptimizedEquals("3 IS DISTINCT FROM BIGINT '4'", "true");
        TestExpressionInterpreter.assertOptimizedEquals("3 IS DISTINCT FROM 4000000000", "true");
        TestExpressionInterpreter.assertOptimizedEquals("3 IS DISTINCT FROM 3", "false");
        TestExpressionInterpreter.assertOptimizedEquals("3 IS DISTINCT FROM NULL", "true");
        TestExpressionInterpreter.assertOptimizedEquals("NULL IS DISTINCT FROM 3", "true");
        TestExpressionInterpreter.assertOptimizedEquals("10151082135029368 IS DISTINCT FROM 10151082135029369", "true");
        TestExpressionInterpreter.assertOptimizedEquals("1.1 IS DISTINCT FROM 1.1", "false");
        TestExpressionInterpreter.assertOptimizedEquals("9876543210.9874561203 IS DISTINCT FROM NULL", "true");
        TestExpressionInterpreter.assertOptimizedEquals("bound_decimal_short IS DISTINCT FROM NULL", "true");
        TestExpressionInterpreter.assertOptimizedEquals("bound_decimal_long IS DISTINCT FROM 12345678901234567890.123", "false");
        TestExpressionInterpreter.assertOptimizedMatches("unbound_integer IS DISTINCT FROM 1", "unbound_integer IS DISTINCT FROM 1");
        TestExpressionInterpreter.assertOptimizedMatches("unbound_integer IS DISTINCT FROM NULL", "unbound_integer IS NOT NULL");
        TestExpressionInterpreter.assertOptimizedMatches("NULL IS DISTINCT FROM unbound_integer", "unbound_integer IS NOT NULL");
    }

    @Test
    public void testIsNull() {
        TestExpressionInterpreter.assertOptimizedEquals("NULL IS NULL", "true");
        TestExpressionInterpreter.assertOptimizedEquals("1 IS NULL", "false");
        TestExpressionInterpreter.assertOptimizedEquals("10000000000 IS NULL", "false");
        TestExpressionInterpreter.assertOptimizedEquals("BIGINT '1' IS NULL", "false");
        TestExpressionInterpreter.assertOptimizedEquals("1.0 IS NULL", "false");
        TestExpressionInterpreter.assertOptimizedEquals("'a' IS NULL", "false");
        TestExpressionInterpreter.assertOptimizedEquals("true IS NULL", "false");
        TestExpressionInterpreter.assertOptimizedEquals("NULL+1 IS NULL", "true");
        TestExpressionInterpreter.assertOptimizedEquals("unbound_string IS NULL", "unbound_string IS NULL");
        TestExpressionInterpreter.assertOptimizedEquals("unbound_long+(1+1) IS NULL", "unbound_long+2 IS NULL");
        TestExpressionInterpreter.assertOptimizedEquals("1.1 IS NULL", "false");
        TestExpressionInterpreter.assertOptimizedEquals("9876543210.9874561203 IS NULL", "false");
        TestExpressionInterpreter.assertOptimizedEquals("bound_decimal_short IS NULL", "false");
        TestExpressionInterpreter.assertOptimizedEquals("bound_decimal_long IS NULL", "false");
    }

    @Test
    public void testIsNotNull() {
        TestExpressionInterpreter.assertOptimizedEquals("NULL IS NOT NULL", "false");
        TestExpressionInterpreter.assertOptimizedEquals("1 IS NOT NULL", "true");
        TestExpressionInterpreter.assertOptimizedEquals("10000000000 IS NOT NULL", "true");
        TestExpressionInterpreter.assertOptimizedEquals("BIGINT '1' IS NOT NULL", "true");
        TestExpressionInterpreter.assertOptimizedEquals("1.0 IS NOT NULL", "true");
        TestExpressionInterpreter.assertOptimizedEquals("'a' IS NOT NULL", "true");
        TestExpressionInterpreter.assertOptimizedEquals("true IS NOT NULL", "true");
        TestExpressionInterpreter.assertOptimizedEquals("NULL+1 IS NOT NULL", "false");
        TestExpressionInterpreter.assertOptimizedEquals("unbound_string IS NOT NULL", "unbound_string IS NOT NULL");
        TestExpressionInterpreter.assertOptimizedEquals("unbound_long+(1+1) IS NOT NULL", "unbound_long+2 IS NOT NULL");
        TestExpressionInterpreter.assertOptimizedEquals("1.1 IS NOT NULL", "true");
        TestExpressionInterpreter.assertOptimizedEquals("9876543210.9874561203 IS NOT NULL", "true");
        TestExpressionInterpreter.assertOptimizedEquals("bound_decimal_short IS NOT NULL", "true");
        TestExpressionInterpreter.assertOptimizedEquals("bound_decimal_long IS NOT NULL", "true");
    }

    @Test
    public void testLambdaBody() {
        TestExpressionInterpreter.assertOptimizedEquals("transform(ARRAY[bound_long], n -> CAST(n as BIGINT))", "transform(ARRAY[bound_long], n -> n)");
        TestExpressionInterpreter.assertOptimizedEquals("transform(ARRAY[bound_long], n -> CAST(n as VARCHAR(5)))", "transform(ARRAY[bound_long], n -> CAST(n as VARCHAR(5)))");
        TestExpressionInterpreter.assertOptimizedEquals("transform(ARRAY[bound_long], n -> IF(false, 1, 0 / 0))", "transform(ARRAY[bound_long], n -> 0 / 0)");
        TestExpressionInterpreter.assertOptimizedEquals("transform(ARRAY[bound_long], n -> 5 / 0)", "transform(ARRAY[bound_long], n -> 5 / 0)");
        TestExpressionInterpreter.assertOptimizedEquals("transform(ARRAY[bound_long], n -> nullif(true, true))", "transform(ARRAY[bound_long], n -> CAST(null AS Boolean))");
        TestExpressionInterpreter.assertOptimizedEquals("transform(ARRAY[bound_long], n -> n + 10 * 10)", "transform(ARRAY[bound_long], n -> n + 100)");
        TestExpressionInterpreter.assertOptimizedEquals("reduce_agg(bound_long, 0, (a, b) -> IF(false, a, b), (a, b) -> IF(true, a, b))", "reduce_agg(bound_long, 0, (a, b) -> b, (a, b) -> a)");
    }

    @Test
    public void testNullIf() {
        TestExpressionInterpreter.assertOptimizedEquals("nullif(true, true)", "NULL");
        TestExpressionInterpreter.assertOptimizedEquals("nullif(true, false)", "true");
        TestExpressionInterpreter.assertOptimizedEquals("nullif(NULL, false)", "NULL");
        TestExpressionInterpreter.assertOptimizedEquals("nullif(true, NULL)", "true");
        TestExpressionInterpreter.assertOptimizedEquals("nullif('a', 'a')", "NULL");
        TestExpressionInterpreter.assertOptimizedEquals("nullif('a', 'b')", "'a'");
        TestExpressionInterpreter.assertOptimizedEquals("nullif(NULL, 'b')", "NULL");
        TestExpressionInterpreter.assertOptimizedEquals("nullif('a', NULL)", "'a'");
        TestExpressionInterpreter.assertOptimizedEquals("nullif(1, 1)", "NULL");
        TestExpressionInterpreter.assertOptimizedEquals("nullif(1, 2)", "1");
        TestExpressionInterpreter.assertOptimizedEquals("nullif(1, BIGINT '2')", "1");
        TestExpressionInterpreter.assertOptimizedEquals("nullif(1, 20000000000)", "1");
        TestExpressionInterpreter.assertOptimizedEquals("nullif(1.0E0, 1)", "NULL");
        TestExpressionInterpreter.assertOptimizedEquals("nullif(10000000000.0E0, 10000000000)", "NULL");
        TestExpressionInterpreter.assertOptimizedEquals("nullif(1.1E0, 1)", "1.1E0");
        TestExpressionInterpreter.assertOptimizedEquals("nullif(1.1E0, 1.1E0)", "NULL");
        TestExpressionInterpreter.assertOptimizedEquals("nullif(1, 2-1)", "NULL");
        TestExpressionInterpreter.assertOptimizedEquals("nullif(NULL, NULL)", "NULL");
        TestExpressionInterpreter.assertOptimizedEquals("nullif(1, NULL)", "1");
        TestExpressionInterpreter.assertOptimizedEquals("nullif(unbound_long, 1)", "nullif(unbound_long, 1)");
        TestExpressionInterpreter.assertOptimizedEquals("nullif(unbound_long, unbound_long2)", "nullif(unbound_long, unbound_long2)");
        TestExpressionInterpreter.assertOptimizedEquals("nullif(unbound_long, unbound_long2+(1+1))", "nullif(unbound_long, unbound_long2+2)");
        TestExpressionInterpreter.assertOptimizedEquals("nullif(1.1, 1.2)", "1.1");
        TestExpressionInterpreter.assertOptimizedEquals("nullif(9876543210.9874561203, 9876543210.9874561203)", "NULL");
        TestExpressionInterpreter.assertOptimizedEquals("nullif(bound_decimal_short, 123.45)", "NULL");
        TestExpressionInterpreter.assertOptimizedEquals("nullif(bound_decimal_long, 12345678901234567890.123)", "NULL");
        TestExpressionInterpreter.assertOptimizedEquals("nullif(ARRAY[CAST(1 AS bigint)], ARRAY[CAST(1 AS bigint)]) IS NULL", "true");
        TestExpressionInterpreter.assertOptimizedEquals("nullif(ARRAY[CAST(1 AS bigint)], ARRAY[CAST(NULL AS bigint)]) IS NULL", "false");
        TestExpressionInterpreter.assertOptimizedEquals("nullif(ARRAY[CAST(NULL AS bigint)], ARRAY[CAST(NULL AS bigint)]) IS NULL", "false");
    }

    @Test
    public void testNegative() {
        TestExpressionInterpreter.assertOptimizedEquals("-(1)", "-1");
        TestExpressionInterpreter.assertOptimizedEquals("-(BIGINT '1')", "BIGINT '-1'");
        TestExpressionInterpreter.assertOptimizedEquals("-(unbound_long+1)", "-(unbound_long+1)");
        TestExpressionInterpreter.assertOptimizedEquals("-(1+1)", "-2");
        TestExpressionInterpreter.assertOptimizedEquals("-(1+ BIGINT '1')", "BIGINT '-2'");
        TestExpressionInterpreter.assertOptimizedEquals("-(CAST(NULL AS bigint))", "NULL");
        TestExpressionInterpreter.assertOptimizedEquals("-(unbound_long+(1+1))", "-(unbound_long+2)");
        TestExpressionInterpreter.assertOptimizedEquals("-(1.1+1.2)", "-2.3");
        TestExpressionInterpreter.assertOptimizedEquals("-(9876543210.9874561203-9876543210.9874561203)", "CAST(0 AS decimal(20,10))");
        TestExpressionInterpreter.assertOptimizedEquals("-(bound_decimal_short+123.45)", "-246.90");
        TestExpressionInterpreter.assertOptimizedEquals("-(bound_decimal_long-12345678901234567890.123)", "CAST(0 AS decimal(20,10))");
    }

    @Test
    public void testArithmeticUnary() {
        TestExpressionInterpreter.assertOptimizedEquals("-rand()", "-rand()");
        TestExpressionInterpreter.assertOptimizedEquals("-(0 / 0)", "-(0 / 0)");
        TestExpressionInterpreter.assertOptimizedEquals("-(-(0 / 0))", "0 / 0");
        TestExpressionInterpreter.assertOptimizedEquals("-(-(-(0 / 0)))", "-(0 / 0)");
        TestExpressionInterpreter.assertOptimizedEquals("+rand()", "rand()");
        TestExpressionInterpreter.assertOptimizedEquals("+(0 / 0)", "0 / 0");
        TestExpressionInterpreter.assertOptimizedEquals("+++(0 / 0)", "0 / 0");
    }

    @Test
    public void testNot() {
        TestExpressionInterpreter.assertOptimizedEquals("not true", "false");
        TestExpressionInterpreter.assertOptimizedEquals("not false", "true");
        TestExpressionInterpreter.assertOptimizedEquals("not NULL", "NULL");
        TestExpressionInterpreter.assertOptimizedEquals("not 1=1", "false");
        TestExpressionInterpreter.assertOptimizedEquals("not 1=BIGINT '1'", "false");
        TestExpressionInterpreter.assertOptimizedEquals("not 1!=1", "true");
        TestExpressionInterpreter.assertOptimizedEquals("not unbound_long=1", "not unbound_long=1");
        TestExpressionInterpreter.assertOptimizedEquals("not unbound_long=(1+1)", "not unbound_long=2");
    }

    @Test
    public void testFunctionCall() {
        TestExpressionInterpreter.assertOptimizedEquals("abs(-5)", "5");
        TestExpressionInterpreter.assertOptimizedEquals("abs(-10-5)", "15");
        TestExpressionInterpreter.assertOptimizedEquals("abs(-bound_integer + 1)", "1233");
        TestExpressionInterpreter.assertOptimizedEquals("abs(-bound_long + 1)", "1233");
        TestExpressionInterpreter.assertOptimizedEquals("abs(-bound_long + BIGINT '1')", "1233");
        TestExpressionInterpreter.assertOptimizedEquals("abs(-bound_long)", "1234");
        TestExpressionInterpreter.assertOptimizedEquals("abs(unbound_long)", "abs(unbound_long)");
        TestExpressionInterpreter.assertOptimizedEquals("abs(unbound_long + 1)", "abs(unbound_long + 1)");
    }

    @Test
    public void testNonDeterministicFunctionCall() {
        TestExpressionInterpreter.assertOptimizedEquals("random()", "random()");
        Object value = TestExpressionInterpreter.evaluate("random()");
        Assert.assertTrue((boolean)(value instanceof Double));
        double randomValue = (Double)value;
        Assert.assertTrue((0.0 <= randomValue && randomValue < 1.0 ? 1 : 0) != 0);
    }

    @Test
    public void testBetween() {
        TestExpressionInterpreter.assertOptimizedEquals("3 BETWEEN 2 AND 4", "true");
        TestExpressionInterpreter.assertOptimizedEquals("2 BETWEEN 3 AND 4", "false");
        TestExpressionInterpreter.assertOptimizedEquals("NULL BETWEEN 2 AND 4", "NULL");
        TestExpressionInterpreter.assertOptimizedEquals("3 BETWEEN NULL AND 4", "NULL");
        TestExpressionInterpreter.assertOptimizedEquals("3 BETWEEN 2 AND NULL", "NULL");
        TestExpressionInterpreter.assertOptimizedEquals("2 BETWEEN 3 AND NULL", "false");
        TestExpressionInterpreter.assertOptimizedEquals("8 BETWEEN NULL AND 6", "false");
        TestExpressionInterpreter.assertOptimizedEquals("'cc' BETWEEN 'b' AND 'd'", "true");
        TestExpressionInterpreter.assertOptimizedEquals("'b' BETWEEN 'cc' AND 'd'", "false");
        TestExpressionInterpreter.assertOptimizedEquals("NULL BETWEEN 'b' AND 'd'", "NULL");
        TestExpressionInterpreter.assertOptimizedEquals("'cc' BETWEEN NULL AND 'd'", "NULL");
        TestExpressionInterpreter.assertOptimizedEquals("'cc' BETWEEN 'b' AND NULL", "NULL");
        TestExpressionInterpreter.assertOptimizedEquals("bound_integer BETWEEN 1000 AND 2000", "true");
        TestExpressionInterpreter.assertOptimizedEquals("bound_integer BETWEEN 3 AND 4", "false");
        TestExpressionInterpreter.assertOptimizedEquals("bound_long BETWEEN 1000 AND 2000", "true");
        TestExpressionInterpreter.assertOptimizedEquals("bound_long BETWEEN 3 AND 4", "false");
        TestExpressionInterpreter.assertOptimizedEquals("bound_long BETWEEN bound_integer AND (bound_long + 1)", "true");
        TestExpressionInterpreter.assertOptimizedEquals("bound_string BETWEEN 'e' AND 'i'", "true");
        TestExpressionInterpreter.assertOptimizedEquals("bound_string BETWEEN 'a' AND 'b'", "false");
        TestExpressionInterpreter.assertOptimizedEquals("bound_long BETWEEN unbound_long AND 2000 + 1", "1234 BETWEEN unbound_long AND 2001");
        TestExpressionInterpreter.assertOptimizedEquals("bound_string BETWEEN unbound_string AND 'bar'", String.format("CAST('hello' AS varchar(%s)) BETWEEN unbound_string AND 'bar'", 17));
        TestExpressionInterpreter.assertOptimizedEquals("1.15 BETWEEN 1.1 AND 1.2", "true");
        TestExpressionInterpreter.assertOptimizedEquals("9876543210.98745612035 BETWEEN 9876543210.9874561203 AND 9876543210.9874561204", "true");
        TestExpressionInterpreter.assertOptimizedEquals("123.455 BETWEEN bound_decimal_short AND 123.46", "true");
        TestExpressionInterpreter.assertOptimizedEquals("12345678901234567890.1235 BETWEEN bound_decimal_long AND 12345678901234567890.123", "false");
    }

    @Test
    public void testExtract() {
        DateTime dateTime = new DateTime(2001, 8, 22, 3, 4, 5, 321, DateTimeZone.UTC);
        double seconds = (double)dateTime.getMillis() / 1000.0;
        TestExpressionInterpreter.assertOptimizedEquals("extract(YEAR FROM from_unixtime(" + seconds + ",'UTC'))", "2001");
        TestExpressionInterpreter.assertOptimizedEquals("extract(QUARTER FROM from_unixtime(" + seconds + ",'UTC'))", "3");
        TestExpressionInterpreter.assertOptimizedEquals("extract(MONTH FROM from_unixtime(" + seconds + ",'UTC'))", "8");
        TestExpressionInterpreter.assertOptimizedEquals("extract(WEEK FROM from_unixtime(" + seconds + ",'UTC'))", "34");
        TestExpressionInterpreter.assertOptimizedEquals("extract(DOW FROM from_unixtime(" + seconds + ",'UTC'))", "3");
        TestExpressionInterpreter.assertOptimizedEquals("extract(DOY FROM from_unixtime(" + seconds + ",'UTC'))", "234");
        TestExpressionInterpreter.assertOptimizedEquals("extract(DAY FROM from_unixtime(" + seconds + ",'UTC'))", "22");
        TestExpressionInterpreter.assertOptimizedEquals("extract(HOUR FROM from_unixtime(" + seconds + ",'UTC'))", "3");
        TestExpressionInterpreter.assertOptimizedEquals("extract(MINUTE FROM from_unixtime(" + seconds + ",'UTC'))", "4");
        TestExpressionInterpreter.assertOptimizedEquals("extract(SECOND FROM from_unixtime(" + seconds + ",'UTC'))", "5");
        TestExpressionInterpreter.assertOptimizedEquals("extract(TIMEZONE_HOUR FROM from_unixtime(" + seconds + ", 7, 9))", "7");
        TestExpressionInterpreter.assertOptimizedEquals("extract(TIMEZONE_MINUTE FROM from_unixtime(" + seconds + ", 7, 9))", "9");
        TestExpressionInterpreter.assertOptimizedEquals("extract(YEAR FROM bound_timestamp)", "2001");
        TestExpressionInterpreter.assertOptimizedEquals("extract(QUARTER FROM bound_timestamp)", "3");
        TestExpressionInterpreter.assertOptimizedEquals("extract(MONTH FROM bound_timestamp)", "8");
        TestExpressionInterpreter.assertOptimizedEquals("extract(WEEK FROM bound_timestamp)", "34");
        TestExpressionInterpreter.assertOptimizedEquals("extract(DOW FROM bound_timestamp)", "3");
        TestExpressionInterpreter.assertOptimizedEquals("extract(DOY FROM bound_timestamp)", "234");
        TestExpressionInterpreter.assertOptimizedEquals("extract(DAY FROM bound_timestamp)", "22");
        TestExpressionInterpreter.assertOptimizedEquals("extract(HOUR FROM bound_timestamp)", "3");
        TestExpressionInterpreter.assertOptimizedEquals("extract(MINUTE FROM bound_timestamp)", "4");
        TestExpressionInterpreter.assertOptimizedEquals("extract(SECOND FROM bound_timestamp)", "5");
        TestExpressionInterpreter.assertOptimizedEquals("extract(YEAR FROM unbound_timestamp)", "extract(YEAR FROM unbound_timestamp)");
        TestExpressionInterpreter.assertOptimizedEquals("extract(SECOND FROM bound_timestamp + INTERVAL '3' SECOND)", "8");
    }

    @Test
    public void testIn() {
        TestExpressionInterpreter.assertOptimizedEquals("3 IN (2, 4, 3, 5)", "true");
        TestExpressionInterpreter.assertOptimizedEquals("3 IN (2, 4, 9, 5)", "false");
        TestExpressionInterpreter.assertOptimizedEquals("3 IN (2, NULL, 3, 5)", "true");
        TestExpressionInterpreter.assertOptimizedEquals("'foo' IN ('bar', 'baz', 'foo', 'blah')", "true");
        TestExpressionInterpreter.assertOptimizedEquals("'foo' IN ('bar', 'baz', 'buz', 'blah')", "false");
        TestExpressionInterpreter.assertOptimizedEquals("'foo' IN ('bar', NULL, 'foo', 'blah')", "true");
        TestExpressionInterpreter.assertOptimizedEquals("NULL IN (2, NULL, 3, 5)", "NULL");
        TestExpressionInterpreter.assertOptimizedEquals("3 IN (2, NULL)", "NULL");
        TestExpressionInterpreter.assertOptimizedEquals("bound_integer IN (2, 1234, 3, 5)", "true");
        TestExpressionInterpreter.assertOptimizedEquals("bound_integer IN (2, 4, 3, 5)", "false");
        TestExpressionInterpreter.assertOptimizedEquals("1234 IN (2, bound_integer, 3, 5)", "true");
        TestExpressionInterpreter.assertOptimizedEquals("99 IN (2, bound_integer, 3, 5)", "false");
        TestExpressionInterpreter.assertOptimizedEquals("bound_integer IN (2, bound_integer, 3, 5)", "true");
        TestExpressionInterpreter.assertOptimizedEquals("bound_long IN (2, 1234, 3, 5)", "true");
        TestExpressionInterpreter.assertOptimizedEquals("bound_long IN (2, 4, 3, 5)", "false");
        TestExpressionInterpreter.assertOptimizedEquals("1234 IN (2, bound_long, 3, 5)", "true");
        TestExpressionInterpreter.assertOptimizedEquals("99 IN (2, bound_long, 3, 5)", "false");
        TestExpressionInterpreter.assertOptimizedEquals("bound_long IN (2, bound_long, 3, 5)", "true");
        TestExpressionInterpreter.assertOptimizedEquals("bound_string IN ('bar', 'hello', 'foo', 'blah')", "true");
        TestExpressionInterpreter.assertOptimizedEquals("bound_string IN ('bar', 'baz', 'foo', 'blah')", "false");
        TestExpressionInterpreter.assertOptimizedEquals("'hello' IN ('bar', bound_string, 'foo', 'blah')", "true");
        TestExpressionInterpreter.assertOptimizedEquals("'baz' IN ('bar', bound_string, 'foo', 'blah')", "false");
        TestExpressionInterpreter.assertOptimizedEquals("bound_long IN (2, 1234, unbound_long, 5)", "true");
        TestExpressionInterpreter.assertOptimizedEquals("bound_string IN ('bar', 'hello', unbound_string, 'blah')", "true");
        TestExpressionInterpreter.assertOptimizedEquals("bound_long IN (2, 4, unbound_long, unbound_long2, 9)", "1234 IN (unbound_long, unbound_long2)");
        TestExpressionInterpreter.assertOptimizedEquals("unbound_long IN (2, 4, bound_long, unbound_long2, 5)", "unbound_long IN (2, 4, 1234, unbound_long2, 5)");
        TestExpressionInterpreter.assertOptimizedEquals("1.15 IN (1.1, 1.2, 1.3, 1.15)", "true");
        TestExpressionInterpreter.assertOptimizedEquals("9876543210.98745612035 IN (9876543210.9874561203, 9876543210.9874561204, 9876543210.98745612035)", "true");
        TestExpressionInterpreter.assertOptimizedEquals("bound_decimal_short IN (123.455, 123.46, 123.45)", "true");
        TestExpressionInterpreter.assertOptimizedEquals("bound_decimal_long IN (12345678901234567890.123, 9876543210.9874561204, 9876543210.98745612035)", "true");
        TestExpressionInterpreter.assertOptimizedEquals("bound_decimal_long IN (9876543210.9874561204, NULL, 9876543210.98745612035)", "NULL");
        TestExpressionInterpreter.assertOptimizedEquals("unbound_integer IN (1)", "unbound_integer = 1");
        TestExpressionInterpreter.assertOptimizedEquals("unbound_long IN (unbound_long2)", "unbound_long = unbound_long2");
        TestExpressionInterpreter.assertOptimizedEquals("3 in (2, 4, 3, 5 / 0)", "3 in (2, 4, 3, 5 / 0)");
        TestExpressionInterpreter.assertOptimizedEquals("null in (2, 4, 3, 5 / 0)", "null in (2, 4, 3, 5 / 0)");
        TestExpressionInterpreter.assertOptimizedEquals("3 in (2, 4, 3, null, 5 / 0)", "3 in (2, 4, 3, null, 5 / 0)");
        TestExpressionInterpreter.assertOptimizedEquals("null in (2, 4, null, 5 / 0)", "null in (2, 4, null, 5 / 0)");
        TestExpressionInterpreter.assertOptimizedEquals("3 in (5 / 0, 5 / 0)", "3 in (5 / 0, 5 / 0)");
        TrinoExceptionAssert.assertTrinoExceptionThrownBy(() -> TestExpressionInterpreter.evaluate("3 in (2, 4, 3, 5 / 0)")).hasErrorCode(new ErrorCodeSupplier[]{StandardErrorCode.DIVISION_BY_ZERO});
        TestExpressionInterpreter.assertOptimizedEquals("0 / 0 in (2, 4, 3, 5)", "0 / 0 in (2, 4, 3, 5)");
        TestExpressionInterpreter.assertOptimizedEquals("0 / 0 in (2, 4, 2, 4)", "0 / 0 in (2, 4)");
        TestExpressionInterpreter.assertOptimizedEquals("0 / 0 in (rand(), 2, 4)", "0 / 0 in (2, 4, rand())");
        TestExpressionInterpreter.assertOptimizedEquals("0 / 0 in (2, 2)", "0 / 0 = 2");
    }

    @Test
    public void testInComplexTypes() {
        TestExpressionInterpreter.assertEvaluatedEquals("ARRAY[1] IN (ARRAY[1])", "true");
        TestExpressionInterpreter.assertEvaluatedEquals("ARRAY[1] IN (ARRAY[2])", "false");
        TestExpressionInterpreter.assertEvaluatedEquals("ARRAY[1] IN (ARRAY[2], ARRAY[1])", "true");
        TestExpressionInterpreter.assertEvaluatedEquals("ARRAY[1] IN (NULL)", "NULL");
        TestExpressionInterpreter.assertEvaluatedEquals("ARRAY[1] IN (NULL, ARRAY[1])", "true");
        TestExpressionInterpreter.assertEvaluatedEquals("ARRAY[1, 2, NULL] IN (ARRAY[2, NULL], ARRAY[1, NULL])", "false");
        TestExpressionInterpreter.assertEvaluatedEquals("ARRAY[1, NULL] IN (ARRAY[2, NULL], NULL)", "NULL");
        TestExpressionInterpreter.assertEvaluatedEquals("ARRAY[NULL] IN (ARRAY[NULL])", "NULL");
        TestExpressionInterpreter.assertEvaluatedEquals("ARRAY[1] IN (ARRAY[NULL])", "NULL");
        TestExpressionInterpreter.assertEvaluatedEquals("ARRAY[NULL] IN (ARRAY[1])", "NULL");
        TestExpressionInterpreter.assertEvaluatedEquals("ARRAY[1, NULL] IN (ARRAY[1, NULL])", "NULL");
        TestExpressionInterpreter.assertEvaluatedEquals("ARRAY[1, NULL] IN (ARRAY[2, NULL])", "false");
        TestExpressionInterpreter.assertEvaluatedEquals("ARRAY[1, NULL] IN (ARRAY[1, NULL], ARRAY[2, NULL])", "NULL");
        TestExpressionInterpreter.assertEvaluatedEquals("ARRAY[1, NULL] IN (ARRAY[1, NULL], ARRAY[2, NULL], ARRAY[1, NULL])", "NULL");
        TestExpressionInterpreter.assertEvaluatedEquals("ARRAY[ARRAY[1, 2], ARRAY[3, 4]] in (ARRAY[ARRAY[1, 2], ARRAY[3, NULL]])", "NULL");
        TestExpressionInterpreter.assertEvaluatedEquals("ROW(1) IN (ROW(1))", "true");
        TestExpressionInterpreter.assertEvaluatedEquals("ROW(1) IN (ROW(2))", "false");
        TestExpressionInterpreter.assertEvaluatedEquals("ROW(1) IN (ROW(2), ROW(1), ROW(2))", "true");
        TestExpressionInterpreter.assertEvaluatedEquals("ROW(1) IN (NULL)", "NULL");
        TestExpressionInterpreter.assertEvaluatedEquals("ROW(1) IN (NULL, ROW(1))", "true");
        TestExpressionInterpreter.assertEvaluatedEquals("ROW(1, NULL) IN (ROW(2, NULL), NULL)", "NULL");
        TestExpressionInterpreter.assertEvaluatedEquals("ROW(NULL) IN (ROW(NULL))", "NULL");
        TestExpressionInterpreter.assertEvaluatedEquals("ROW(1) IN (ROW(NULL))", "NULL");
        TestExpressionInterpreter.assertEvaluatedEquals("ROW(NULL) IN (ROW(1))", "NULL");
        TestExpressionInterpreter.assertEvaluatedEquals("ROW(1, NULL) IN (ROW(1, NULL))", "NULL");
        TestExpressionInterpreter.assertEvaluatedEquals("ROW(1, NULL) IN (ROW(2, NULL))", "false");
        TestExpressionInterpreter.assertEvaluatedEquals("ROW(1, NULL) IN (ROW(1, NULL), ROW(2, NULL))", "NULL");
        TestExpressionInterpreter.assertEvaluatedEquals("ROW(1, NULL) IN (ROW(1, NULL), ROW(2, NULL), ROW(1, NULL))", "NULL");
        TestExpressionInterpreter.assertEvaluatedEquals("map(ARRAY[1], ARRAY[1]) IN (map(ARRAY[1], ARRAY[1]))", "true");
        TestExpressionInterpreter.assertEvaluatedEquals("map(ARRAY[1], ARRAY[1]) IN (NULL)", "NULL");
        TestExpressionInterpreter.assertEvaluatedEquals("map(ARRAY[1], ARRAY[1]) IN (NULL, map(ARRAY[1], ARRAY[1]))", "true");
        TestExpressionInterpreter.assertEvaluatedEquals("map(ARRAY[1], ARRAY[1]) IN (map(ARRAY[1, 2], ARRAY[1, NULL]))", "false");
        TestExpressionInterpreter.assertEvaluatedEquals("map(ARRAY[1, 2], ARRAY[1, NULL]) IN (map(ARRAY[1, 2], ARRAY[2, NULL]), NULL)", "NULL");
        TestExpressionInterpreter.assertEvaluatedEquals("map(ARRAY[1, 2], ARRAY[1, NULL]) IN (map(ARRAY[1, 2], ARRAY[1, NULL]))", "NULL");
        TestExpressionInterpreter.assertEvaluatedEquals("map(ARRAY[1, 2], ARRAY[1, NULL]) IN (map(ARRAY[1, 3], ARRAY[1, NULL]))", "false");
        TestExpressionInterpreter.assertEvaluatedEquals("map(ARRAY[1], ARRAY[NULL]) IN (map(ARRAY[1], ARRAY[NULL]))", "NULL");
        TestExpressionInterpreter.assertEvaluatedEquals("map(ARRAY[1], ARRAY[1]) IN (map(ARRAY[1], ARRAY[NULL]))", "NULL");
        TestExpressionInterpreter.assertEvaluatedEquals("map(ARRAY[1], ARRAY[NULL]) IN (map(ARRAY[1], ARRAY[1]))", "NULL");
        TestExpressionInterpreter.assertEvaluatedEquals("map(ARRAY[1, 2], ARRAY[1, NULL]) IN (map(ARRAY[1, 2], ARRAY[1, NULL]))", "NULL");
        TestExpressionInterpreter.assertEvaluatedEquals("map(ARRAY[1, 2], ARRAY[1, NULL]) IN (map(ARRAY[1, 3], ARRAY[1, NULL]))", "false");
        TestExpressionInterpreter.assertEvaluatedEquals("map(ARRAY[1, 2], ARRAY[1, NULL]) IN (map(ARRAY[1, 2], ARRAY[2, NULL]))", "false");
        TestExpressionInterpreter.assertEvaluatedEquals("map(ARRAY[1, 2], ARRAY[1, NULL]) IN (map(ARRAY[1, 2], ARRAY[1, NULL]), map(ARRAY[1, 2], ARRAY[2, NULL]))", "NULL");
        TestExpressionInterpreter.assertEvaluatedEquals("map(ARRAY[1, 2], ARRAY[1, NULL]) IN (map(ARRAY[1, 2], ARRAY[1, NULL]), map(ARRAY[1, 2], ARRAY[2, NULL]), map(ARRAY[1, 2], ARRAY[1, NULL]))", "NULL");
    }

    @Test
    public void testDereference() {
        TestExpressionInterpreter.assertOptimizedEquals("CAST(ROW(1, true) AS ROW(id BIGINT, value BOOLEAN)).value", "true");
        TestExpressionInterpreter.assertOptimizedEquals("CAST(ROW(1, null) AS ROW(id BIGINT, value BOOLEAN)).value", "null");
        TestExpressionInterpreter.assertOptimizedEquals("CAST(ROW(0 / 0, true) AS ROW(id DOUBLE, value BOOLEAN)).value", "CAST(ROW(0 / 0, true) AS ROW(id DOUBLE, value BOOLEAN)).value");
        TrinoExceptionAssert.assertTrinoExceptionThrownBy(() -> TestExpressionInterpreter.evaluate("CAST(ROW(0 / 0, true) AS ROW(id DOUBLE, value BOOLEAN)).value")).hasErrorCode(new ErrorCodeSupplier[]{StandardErrorCode.DIVISION_BY_ZERO});
    }

    @Test
    public void testCurrentUser() {
        TestExpressionInterpreter.assertOptimizedEquals("current_user", "'" + SessionTestUtils.TEST_SESSION.getUser() + "'");
    }

    @Test
    public void testCastToString() {
        TestExpressionInterpreter.assertOptimizedEquals("CAST(123 AS varchar(20))", "'123'");
        TestExpressionInterpreter.assertOptimizedEquals("CAST(-123 AS varchar(20))", "'-123'");
        TestExpressionInterpreter.assertOptimizedEquals("CAST(BIGINT '123' AS varchar)", "'123'");
        TestExpressionInterpreter.assertOptimizedEquals("CAST(12300000000 AS varchar)", "'12300000000'");
        TestExpressionInterpreter.assertOptimizedEquals("CAST(-12300000000 AS varchar)", "'-12300000000'");
        TestExpressionInterpreter.assertOptimizedEquals("CAST(123.0E0 AS varchar)", "'1.23E2'");
        TestExpressionInterpreter.assertOptimizedEquals("CAST(-123.0E0 AS varchar)", "'-1.23E2'");
        TestExpressionInterpreter.assertOptimizedEquals("CAST(123.456E0 AS varchar)", "'1.23456E2'");
        TestExpressionInterpreter.assertOptimizedEquals("CAST(-123.456E0 AS varchar)", "'-1.23456E2'");
        TestExpressionInterpreter.assertOptimizedEquals("CAST(true AS varchar)", "'true'");
        TestExpressionInterpreter.assertOptimizedEquals("CAST(false AS varchar)", "'false'");
        TestExpressionInterpreter.assertOptimizedEquals("VARCHAR 'xyz'", "'xyz'");
        TestExpressionInterpreter.assertOptimizedEquals("CAST(NULL AS varchar)", "NULL");
        TestExpressionInterpreter.assertOptimizedEquals("CAST(1.1 AS varchar)", "'1.1'");
    }

    @Test
    public void testCastBigintToBoundedVarchar() {
        TestExpressionInterpreter.assertEvaluatedEquals("CAST(12300000000 AS varchar(11))", "'12300000000'");
        TestExpressionInterpreter.assertEvaluatedEquals("CAST(12300000000 AS varchar(50))", "'12300000000'");
        TrinoExceptionAssert.assertTrinoExceptionThrownBy(() -> TestExpressionInterpreter.evaluate("CAST(12300000000 AS varchar(3))")).hasErrorCode(new ErrorCodeSupplier[]{StandardErrorCode.INVALID_CAST_ARGUMENT}).hasMessage("Value 12300000000 cannot be represented as varchar(3)");
        TrinoExceptionAssert.assertTrinoExceptionThrownBy(() -> TestExpressionInterpreter.evaluate("CAST(-12300000000 AS varchar(3))")).hasErrorCode(new ErrorCodeSupplier[]{StandardErrorCode.INVALID_CAST_ARGUMENT}).hasMessage("Value -12300000000 cannot be represented as varchar(3)");
    }

    @Test
    public void testCastIntegerToBoundedVarchar() {
        TestExpressionInterpreter.assertEvaluatedEquals("CAST(1234 AS varchar(4))", "'1234'");
        TestExpressionInterpreter.assertEvaluatedEquals("CAST(1234 AS varchar(50))", "'1234'");
        TrinoExceptionAssert.assertTrinoExceptionThrownBy(() -> TestExpressionInterpreter.evaluate("CAST(1234 AS varchar(3))")).hasErrorCode(new ErrorCodeSupplier[]{StandardErrorCode.INVALID_CAST_ARGUMENT}).hasMessage("Value 1234 cannot be represented as varchar(3)");
        TrinoExceptionAssert.assertTrinoExceptionThrownBy(() -> TestExpressionInterpreter.evaluate("CAST(-1234 AS varchar(3))")).hasErrorCode(new ErrorCodeSupplier[]{StandardErrorCode.INVALID_CAST_ARGUMENT}).hasMessage("Value -1234 cannot be represented as varchar(3)");
    }

    @Test
    public void testCastSmallintToBoundedVarchar() {
        TestExpressionInterpreter.assertEvaluatedEquals("CAST(SMALLINT '1234' AS varchar(4))", "'1234'");
        TestExpressionInterpreter.assertEvaluatedEquals("CAST(SMALLINT '1234' AS varchar(50))", "'1234'");
        TrinoExceptionAssert.assertTrinoExceptionThrownBy(() -> TestExpressionInterpreter.evaluate("CAST(SMALLINT '1234' AS varchar(3))")).hasErrorCode(new ErrorCodeSupplier[]{StandardErrorCode.INVALID_CAST_ARGUMENT}).hasMessage("Value 1234 cannot be represented as varchar(3)");
        TrinoExceptionAssert.assertTrinoExceptionThrownBy(() -> TestExpressionInterpreter.evaluate("CAST(SMALLINT '-1234' AS varchar(3))")).hasErrorCode(new ErrorCodeSupplier[]{StandardErrorCode.INVALID_CAST_ARGUMENT}).hasMessage("Value -1234 cannot be represented as varchar(3)");
    }

    @Test
    public void testCastTinyintToBoundedVarchar() {
        TestExpressionInterpreter.assertEvaluatedEquals("CAST(TINYINT '123' AS varchar(3))", "'123'");
        TestExpressionInterpreter.assertEvaluatedEquals("CAST(TINYINT '123' AS varchar(50))", "'123'");
        TrinoExceptionAssert.assertTrinoExceptionThrownBy(() -> TestExpressionInterpreter.evaluate("CAST(TINYINT '123' AS varchar(2))")).hasErrorCode(new ErrorCodeSupplier[]{StandardErrorCode.INVALID_CAST_ARGUMENT}).hasMessage("Value 123 cannot be represented as varchar(2)");
        TrinoExceptionAssert.assertTrinoExceptionThrownBy(() -> TestExpressionInterpreter.evaluate("CAST(TINYINT '-123' AS varchar(2))")).hasErrorCode(new ErrorCodeSupplier[]{StandardErrorCode.INVALID_CAST_ARGUMENT}).hasMessage("Value -123 cannot be represented as varchar(2)");
    }

    @Test
    public void testCastDecimalToBoundedVarchar() {
        TestExpressionInterpreter.assertEvaluatedEquals("CAST(DECIMAL '12.4' AS varchar(4))", "'12.4'");
        TestExpressionInterpreter.assertEvaluatedEquals("CAST(DECIMAL '12.4' AS varchar(50))", "'12.4'");
        TrinoExceptionAssert.assertTrinoExceptionThrownBy(() -> TestExpressionInterpreter.evaluate("CAST(DECIMAL '12.4' AS varchar(3))")).hasErrorCode(new ErrorCodeSupplier[]{StandardErrorCode.INVALID_CAST_ARGUMENT}).hasMessage("Value 12.4 cannot be represented as varchar(3)");
        TrinoExceptionAssert.assertTrinoExceptionThrownBy(() -> TestExpressionInterpreter.evaluate("CAST(DECIMAL '-12.4' AS varchar(3))")).hasErrorCode(new ErrorCodeSupplier[]{StandardErrorCode.INVALID_CAST_ARGUMENT}).hasMessage("Value -12.4 cannot be represented as varchar(3)");
        TrinoExceptionAssert.assertTrinoExceptionThrownBy(() -> TestExpressionInterpreter.evaluate("CAST(DECIMAL '12.40' AS varchar(4))")).hasErrorCode(new ErrorCodeSupplier[]{StandardErrorCode.INVALID_CAST_ARGUMENT}).hasMessage("Value 12.40 cannot be represented as varchar(4)");
        TrinoExceptionAssert.assertTrinoExceptionThrownBy(() -> TestExpressionInterpreter.evaluate("CAST(DECIMAL '-12.40' AS varchar(5))")).hasErrorCode(new ErrorCodeSupplier[]{StandardErrorCode.INVALID_CAST_ARGUMENT}).hasMessage("Value -12.40 cannot be represented as varchar(5)");
        TestExpressionInterpreter.assertEvaluatedEquals("CAST(DECIMAL '100000000000000000.1' AS varchar(20))", "'100000000000000000.1'");
        TestExpressionInterpreter.assertEvaluatedEquals("CAST(DECIMAL '100000000000000000.1' AS varchar(50))", "'100000000000000000.1'");
        TrinoExceptionAssert.assertTrinoExceptionThrownBy(() -> TestExpressionInterpreter.evaluate("CAST(DECIMAL '100000000000000000.1' AS varchar(3))")).hasErrorCode(new ErrorCodeSupplier[]{StandardErrorCode.INVALID_CAST_ARGUMENT}).hasMessage("Value 100000000000000000.1 cannot be represented as varchar(3)");
        TrinoExceptionAssert.assertTrinoExceptionThrownBy(() -> TestExpressionInterpreter.evaluate("CAST(DECIMAL '-100000000000000000.1' AS varchar(3))")).hasErrorCode(new ErrorCodeSupplier[]{StandardErrorCode.INVALID_CAST_ARGUMENT}).hasMessage("Value -100000000000000000.1 cannot be represented as varchar(3)");
        TrinoExceptionAssert.assertTrinoExceptionThrownBy(() -> TestExpressionInterpreter.evaluate("CAST(DECIMAL '100000000000000000.10' AS varchar(20))")).hasErrorCode(new ErrorCodeSupplier[]{StandardErrorCode.INVALID_CAST_ARGUMENT}).hasMessage("Value 100000000000000000.10 cannot be represented as varchar(20)");
        TrinoExceptionAssert.assertTrinoExceptionThrownBy(() -> TestExpressionInterpreter.evaluate("CAST(DECIMAL '-100000000000000000.10' AS varchar(21))")).hasErrorCode(new ErrorCodeSupplier[]{StandardErrorCode.INVALID_CAST_ARGUMENT}).hasMessage("Value -100000000000000000.10 cannot be represented as varchar(21)");
    }

    @Test
    public void testCastDoubleToBoundedVarchar() {
        TestExpressionInterpreter.assertEvaluatedEquals("CAST(0e0 / 0e0 AS varchar(3))", "'NaN'");
        TestExpressionInterpreter.assertEvaluatedEquals("CAST(0e0 / 0e0 AS varchar(50))", "'NaN'");
        TestExpressionInterpreter.assertEvaluatedEquals("CAST(DOUBLE 'Infinity' AS varchar(8))", "'Infinity'");
        TestExpressionInterpreter.assertEvaluatedEquals("CAST(DOUBLE 'Infinity' AS varchar(50))", "'Infinity'");
        TestExpressionInterpreter.assertEvaluatedEquals("CAST(0e0 AS varchar(3))", "'0E0'");
        TestExpressionInterpreter.assertEvaluatedEquals("CAST(DOUBLE '0' AS varchar(3))", "'0E0'");
        TestExpressionInterpreter.assertEvaluatedEquals("CAST(DOUBLE '-0' AS varchar(4))", "'-0E0'");
        TestExpressionInterpreter.assertEvaluatedEquals("CAST(DOUBLE '0' AS varchar(50))", "'0E0'");
        TestExpressionInterpreter.assertEvaluatedEquals("CAST(12e0 AS varchar(5))", "'1.2E1'");
        TestExpressionInterpreter.assertEvaluatedEquals("CAST(12e2 AS varchar(6))", "'1.2E3'");
        TestExpressionInterpreter.assertEvaluatedEquals("CAST(12e-2 AS varchar(6))", "'1.2E-1'");
        TestExpressionInterpreter.assertEvaluatedEquals("CAST(12e0 AS varchar(50))", "'1.2E1'");
        TestExpressionInterpreter.assertEvaluatedEquals("CAST(12e2 AS varchar(50))", "'1.2E3'");
        TestExpressionInterpreter.assertEvaluatedEquals("CAST(12e-2 AS varchar(50))", "'1.2E-1'");
        TestExpressionInterpreter.assertEvaluatedEquals("CAST(-12e0 AS varchar(6))", "'-1.2E1'");
        TestExpressionInterpreter.assertEvaluatedEquals("CAST(-12e2 AS varchar(6))", "'-1.2E3'");
        TestExpressionInterpreter.assertEvaluatedEquals("CAST(-12e-2 AS varchar(7))", "'-1.2E-1'");
        TestExpressionInterpreter.assertEvaluatedEquals("CAST(-12e0 AS varchar(50))", "'-1.2E1'");
        TestExpressionInterpreter.assertEvaluatedEquals("CAST(-12e2 AS varchar(50))", "'-1.2E3'");
        TestExpressionInterpreter.assertEvaluatedEquals("CAST(-12e-2 AS varchar(50))", "'-1.2E-1'");
        TestExpressionInterpreter.assertEvaluatedEquals("CAST(12345678.9e0 AS varchar(12))", "'1.23456789E7'");
        TestExpressionInterpreter.assertEvaluatedEquals("CAST(0.00001e0 AS varchar(6))", "'1.0E-5'");
        TrinoExceptionAssert.assertTrinoExceptionThrownBy(() -> TestExpressionInterpreter.evaluate("CAST(12e0 AS varchar(1))")).hasErrorCode(new ErrorCodeSupplier[]{StandardErrorCode.INVALID_CAST_ARGUMENT}).hasMessage("Value 12.0 (1.2E1) cannot be represented as varchar(1)");
        TrinoExceptionAssert.assertTrinoExceptionThrownBy(() -> TestExpressionInterpreter.evaluate("CAST(-12e2 AS varchar(1))")).hasErrorCode(new ErrorCodeSupplier[]{StandardErrorCode.INVALID_CAST_ARGUMENT}).hasMessage("Value -1200.0 (-1.2E3) cannot be represented as varchar(1)");
        TrinoExceptionAssert.assertTrinoExceptionThrownBy(() -> TestExpressionInterpreter.evaluate("CAST(0e0 AS varchar(1))")).hasErrorCode(new ErrorCodeSupplier[]{StandardErrorCode.INVALID_CAST_ARGUMENT}).hasMessage("Value 0.0 (0E0) cannot be represented as varchar(1)");
        TrinoExceptionAssert.assertTrinoExceptionThrownBy(() -> TestExpressionInterpreter.evaluate("CAST(0e0 / 0e0 AS varchar(1))")).hasErrorCode(new ErrorCodeSupplier[]{StandardErrorCode.INVALID_CAST_ARGUMENT}).hasMessage("Value NaN (NaN) cannot be represented as varchar(1)");
        TrinoExceptionAssert.assertTrinoExceptionThrownBy(() -> TestExpressionInterpreter.evaluate("CAST(DOUBLE 'Infinity' AS varchar(1))")).hasErrorCode(new ErrorCodeSupplier[]{StandardErrorCode.INVALID_CAST_ARGUMENT}).hasMessage("Value Infinity (Infinity) cannot be represented as varchar(1)");
        TestExpressionInterpreter.assertEvaluatedEquals("CAST(1200000e0 AS varchar(5))", "'1.2E6'");
    }

    @Test
    public void testCastRealToBoundedVarchar() {
        TestExpressionInterpreter.assertEvaluatedEquals("CAST(REAL '0e0' / REAL '0e0' AS varchar(3))", "'NaN'");
        TestExpressionInterpreter.assertEvaluatedEquals("CAST(REAL '0e0' / REAL '0e0' AS varchar(50))", "'NaN'");
        TestExpressionInterpreter.assertEvaluatedEquals("CAST(REAL 'Infinity' AS varchar(8))", "'Infinity'");
        TestExpressionInterpreter.assertEvaluatedEquals("CAST(REAL 'Infinity' AS varchar(50))", "'Infinity'");
        TestExpressionInterpreter.assertEvaluatedEquals("CAST(REAL '0' AS varchar(3))", "'0E0'");
        TestExpressionInterpreter.assertEvaluatedEquals("CAST(REAL '-0' AS varchar(4))", "'-0E0'");
        TestExpressionInterpreter.assertEvaluatedEquals("CAST(REAL '0' AS varchar(50))", "'0E0'");
        TestExpressionInterpreter.assertEvaluatedEquals("CAST(REAL '12' AS varchar(5))", "'1.2E1'");
        TestExpressionInterpreter.assertEvaluatedEquals("CAST(REAL '12e2' AS varchar(5))", "'1.2E3'");
        TestExpressionInterpreter.assertEvaluatedEquals("CAST(REAL '12e-2' AS varchar(6))", "'1.2E-1'");
        TestExpressionInterpreter.assertEvaluatedEquals("CAST(REAL '12' AS varchar(50))", "'1.2E1'");
        TestExpressionInterpreter.assertEvaluatedEquals("CAST(REAL '12e2' AS varchar(50))", "'1.2E3'");
        TestExpressionInterpreter.assertEvaluatedEquals("CAST(REAL '12e-2' AS varchar(50))", "'1.2E-1'");
        TestExpressionInterpreter.assertEvaluatedEquals("CAST(REAL '-12' AS varchar(6))", "'-1.2E1'");
        TestExpressionInterpreter.assertEvaluatedEquals("CAST(REAL '-12e2' AS varchar(6))", "'-1.2E3'");
        TestExpressionInterpreter.assertEvaluatedEquals("CAST(REAL '-12e-2' AS varchar(7))", "'-1.2E-1'");
        TestExpressionInterpreter.assertEvaluatedEquals("CAST(REAL '-12' AS varchar(50))", "'-1.2E1'");
        TestExpressionInterpreter.assertEvaluatedEquals("CAST(REAL '-12e2' AS varchar(50))", "'-1.2E3'");
        TestExpressionInterpreter.assertEvaluatedEquals("CAST(REAL '-12e-2' AS varchar(50))", "'-1.2E-1'");
        TestExpressionInterpreter.assertEvaluatedEquals("CAST(REAL '12345678.9e0' AS varchar(12))", "'1.234568E7'");
        TestExpressionInterpreter.assertEvaluatedEquals("CAST(REAL '0.00001e0' AS varchar(12))", "'1.0E-5'");
        TrinoExceptionAssert.assertTrinoExceptionThrownBy(() -> TestExpressionInterpreter.evaluate("CAST(REAL '12' AS varchar(1))")).hasErrorCode(new ErrorCodeSupplier[]{StandardErrorCode.INVALID_CAST_ARGUMENT}).hasMessage("Value 12.0 (1.2E1) cannot be represented as varchar(1)");
        TrinoExceptionAssert.assertTrinoExceptionThrownBy(() -> TestExpressionInterpreter.evaluate("CAST(REAL '-12e2' AS varchar(1))")).hasErrorCode(new ErrorCodeSupplier[]{StandardErrorCode.INVALID_CAST_ARGUMENT}).hasMessage("Value -1200.0 (-1.2E3) cannot be represented as varchar(1)");
        TrinoExceptionAssert.assertTrinoExceptionThrownBy(() -> TestExpressionInterpreter.evaluate("CAST(REAL '0' AS varchar(1))")).hasErrorCode(new ErrorCodeSupplier[]{StandardErrorCode.INVALID_CAST_ARGUMENT}).hasMessage("Value 0.0 (0E0) cannot be represented as varchar(1)");
        TrinoExceptionAssert.assertTrinoExceptionThrownBy(() -> TestExpressionInterpreter.evaluate("CAST(REAL '0e0' / REAL '0e0' AS varchar(1))")).hasErrorCode(new ErrorCodeSupplier[]{StandardErrorCode.INVALID_CAST_ARGUMENT}).hasMessage("Value NaN (NaN) cannot be represented as varchar(1)");
        TrinoExceptionAssert.assertTrinoExceptionThrownBy(() -> TestExpressionInterpreter.evaluate("CAST(REAL 'Infinity' AS varchar(1))")).hasErrorCode(new ErrorCodeSupplier[]{StandardErrorCode.INVALID_CAST_ARGUMENT}).hasMessage("Value Infinity (Infinity) cannot be represented as varchar(1)");
        TestExpressionInterpreter.assertEvaluatedEquals("CAST(REAL '1200000' AS varchar(5))", "'1.2E6'");
    }

    @Test
    public void testCastDateToBoundedVarchar() {
        TestExpressionInterpreter.assertEvaluatedEquals("CAST(DATE '2013-02-02' AS varchar(10))", "'2013-02-02'");
        TestExpressionInterpreter.assertEvaluatedEquals("CAST(DATE '-2013-02-02' AS varchar(50))", "'-2013-02-02'");
        TrinoExceptionAssert.assertTrinoExceptionThrownBy(() -> TestExpressionInterpreter.evaluate("CAST(DATE '2013-02-02' AS varchar(9))")).hasErrorCode(new ErrorCodeSupplier[]{StandardErrorCode.INVALID_CAST_ARGUMENT}).hasMessage("Value 2013-02-02 cannot be represented as varchar(9)");
        TrinoExceptionAssert.assertTrinoExceptionThrownBy(() -> TestExpressionInterpreter.evaluate("CAST(DATE '-2013-02-02' AS varchar(9))")).hasErrorCode(new ErrorCodeSupplier[]{StandardErrorCode.INVALID_CAST_ARGUMENT}).hasMessage("Value -2013-02-02 cannot be represented as varchar(9)");
    }

    @Test
    public void testCastToBoolean() {
        TestExpressionInterpreter.assertOptimizedEquals("CAST(123 AS boolean)", "true");
        TestExpressionInterpreter.assertOptimizedEquals("CAST(-123 AS boolean)", "true");
        TestExpressionInterpreter.assertOptimizedEquals("CAST(0 AS boolean)", "false");
        TestExpressionInterpreter.assertOptimizedEquals("CAST(12300000000 AS boolean)", "true");
        TestExpressionInterpreter.assertOptimizedEquals("CAST(-12300000000 AS boolean)", "true");
        TestExpressionInterpreter.assertOptimizedEquals("CAST(BIGINT '0' AS boolean)", "false");
        TestExpressionInterpreter.assertOptimizedEquals("CAST(true AS boolean)", "true");
        TestExpressionInterpreter.assertOptimizedEquals("CAST(false AS boolean)", "false");
        TestExpressionInterpreter.assertOptimizedEquals("CAST('true' AS boolean)", "true");
        TestExpressionInterpreter.assertOptimizedEquals("CAST('false' AS boolean)", "false");
        TestExpressionInterpreter.assertOptimizedEquals("CAST('t' AS boolean)", "true");
        TestExpressionInterpreter.assertOptimizedEquals("CAST('f' AS boolean)", "false");
        TestExpressionInterpreter.assertOptimizedEquals("CAST('1' AS boolean)", "true");
        TestExpressionInterpreter.assertOptimizedEquals("CAST('0' AS boolean)", "false");
        TestExpressionInterpreter.assertOptimizedEquals("CAST(NULL AS boolean)", "NULL");
        TestExpressionInterpreter.assertOptimizedEquals("CAST(123.45E0 AS boolean)", "true");
        TestExpressionInterpreter.assertOptimizedEquals("CAST(-123.45E0 AS boolean)", "true");
        TestExpressionInterpreter.assertOptimizedEquals("CAST(0.0E0 AS boolean)", "false");
        TestExpressionInterpreter.assertOptimizedEquals("CAST(0.00 AS boolean)", "false");
        TestExpressionInterpreter.assertOptimizedEquals("CAST(7.8 AS boolean)", "true");
        TestExpressionInterpreter.assertOptimizedEquals("CAST(12345678901234567890.123 AS boolean)", "true");
        TestExpressionInterpreter.assertOptimizedEquals("CAST(00000000000000000000.000 AS boolean)", "false");
    }

    @Test
    public void testCastToBigint() {
        TestExpressionInterpreter.assertOptimizedEquals("CAST(0 AS bigint)", "0");
        TestExpressionInterpreter.assertOptimizedEquals("CAST(123 AS bigint)", "123");
        TestExpressionInterpreter.assertOptimizedEquals("CAST(-123 AS bigint)", "-123");
        TestExpressionInterpreter.assertOptimizedEquals("CAST(BIGINT '0' AS bigint)", "0");
        TestExpressionInterpreter.assertOptimizedEquals("CAST(BIGINT '123' AS bigint)", "123");
        TestExpressionInterpreter.assertOptimizedEquals("CAST(BIGINT '-123' AS bigint)", "-123");
        TestExpressionInterpreter.assertOptimizedEquals("CAST(123.0E0 AS bigint)", "123");
        TestExpressionInterpreter.assertOptimizedEquals("CAST(-123.0E0 AS bigint)", "-123");
        TestExpressionInterpreter.assertOptimizedEquals("CAST(123.456E0 AS bigint)", "123");
        TestExpressionInterpreter.assertOptimizedEquals("CAST(-123.456E0 AS bigint)", "-123");
        TestExpressionInterpreter.assertOptimizedEquals("CAST(true AS bigint)", "1");
        TestExpressionInterpreter.assertOptimizedEquals("CAST(false AS bigint)", "0");
        TestExpressionInterpreter.assertOptimizedEquals("CAST('123' AS bigint)", "123");
        TestExpressionInterpreter.assertOptimizedEquals("CAST('-123' AS bigint)", "-123");
        TestExpressionInterpreter.assertOptimizedEquals("CAST(NULL AS bigint)", "NULL");
        TestExpressionInterpreter.assertOptimizedEquals("CAST(DECIMAL '1.01' AS bigint)", "1");
        TestExpressionInterpreter.assertOptimizedEquals("CAST(DECIMAL '7.8' AS bigint)", "8");
        TestExpressionInterpreter.assertOptimizedEquals("CAST(DECIMAL '1234567890.123' AS bigint)", "1234567890");
        TestExpressionInterpreter.assertOptimizedEquals("CAST(DECIMAL '00000000000000000000.000' AS bigint)", "0");
    }

    @Test
    public void testCastToInteger() {
        TestExpressionInterpreter.assertOptimizedEquals("CAST(0 AS integer)", "0");
        TestExpressionInterpreter.assertOptimizedEquals("CAST(123 AS integer)", "123");
        TestExpressionInterpreter.assertOptimizedEquals("CAST(-123 AS integer)", "-123");
        TestExpressionInterpreter.assertOptimizedEquals("CAST(BIGINT '0' AS integer)", "0");
        TestExpressionInterpreter.assertOptimizedEquals("CAST(BIGINT '123' AS integer)", "123");
        TestExpressionInterpreter.assertOptimizedEquals("CAST(BIGINT '-123' AS integer)", "-123");
        TestExpressionInterpreter.assertOptimizedEquals("CAST(123.0E0 AS integer)", "123");
        TestExpressionInterpreter.assertOptimizedEquals("CAST(-123.0E0 AS integer)", "-123");
        TestExpressionInterpreter.assertOptimizedEquals("CAST(123.456E0 AS integer)", "123");
        TestExpressionInterpreter.assertOptimizedEquals("CAST(-123.456E0 AS integer)", "-123");
        TestExpressionInterpreter.assertOptimizedEquals("CAST(true AS integer)", "1");
        TestExpressionInterpreter.assertOptimizedEquals("CAST(false AS integer)", "0");
        TestExpressionInterpreter.assertOptimizedEquals("CAST('123' AS integer)", "123");
        TestExpressionInterpreter.assertOptimizedEquals("CAST('-123' AS integer)", "-123");
        TestExpressionInterpreter.assertOptimizedEquals("CAST(NULL AS integer)", "NULL");
    }

    @Test
    public void testCastToDouble() {
        TestExpressionInterpreter.assertOptimizedEquals("CAST(0 AS double)", "0.0E0");
        TestExpressionInterpreter.assertOptimizedEquals("CAST(123 AS double)", "123.0E0");
        TestExpressionInterpreter.assertOptimizedEquals("CAST(-123 AS double)", "-123.0E0");
        TestExpressionInterpreter.assertOptimizedEquals("CAST(BIGINT '0' AS double)", "0.0E0");
        TestExpressionInterpreter.assertOptimizedEquals("CAST(12300000000 AS double)", "12300000000.0E0");
        TestExpressionInterpreter.assertOptimizedEquals("CAST(-12300000000 AS double)", "-12300000000.0E0");
        TestExpressionInterpreter.assertOptimizedEquals("CAST(123.0E0 AS double)", "123.0E0");
        TestExpressionInterpreter.assertOptimizedEquals("CAST(-123.0E0 AS double)", "-123.0E0");
        TestExpressionInterpreter.assertOptimizedEquals("CAST(123.456E0 AS double)", "123.456E0");
        TestExpressionInterpreter.assertOptimizedEquals("CAST(-123.456E0 AS double)", "-123.456E0");
        TestExpressionInterpreter.assertOptimizedEquals("CAST('0' AS double)", "0.0E0");
        TestExpressionInterpreter.assertOptimizedEquals("CAST('123' AS double)", "123.0E0");
        TestExpressionInterpreter.assertOptimizedEquals("CAST('-123' AS double)", "-123.0E0");
        TestExpressionInterpreter.assertOptimizedEquals("CAST('123.0E0' AS double)", "123.0E0");
        TestExpressionInterpreter.assertOptimizedEquals("CAST('-123.0E0' AS double)", "-123.0E0");
        TestExpressionInterpreter.assertOptimizedEquals("CAST('123.456E0' AS double)", "123.456E0");
        TestExpressionInterpreter.assertOptimizedEquals("CAST('-123.456E0' AS double)", "-123.456E0");
        TestExpressionInterpreter.assertOptimizedEquals("CAST(NULL AS double)", "NULL");
        TestExpressionInterpreter.assertOptimizedEquals("CAST(true AS double)", "1.0E0");
        TestExpressionInterpreter.assertOptimizedEquals("CAST(false AS double)", "0.0E0");
        TestExpressionInterpreter.assertOptimizedEquals("CAST(1.01 AS double)", "DOUBLE '1.01'");
        TestExpressionInterpreter.assertOptimizedEquals("CAST(7.8 AS double)", "DOUBLE '7.8'");
        TestExpressionInterpreter.assertOptimizedEquals("CAST(1234567890.123 AS double)", "DOUBLE '1234567890.123'");
        TestExpressionInterpreter.assertOptimizedEquals("CAST(00000000000000000000.000 AS double)", "DOUBLE '0.0'");
    }

    @Test
    public void testCastToDecimal() {
        TestExpressionInterpreter.assertOptimizedEquals("CAST(0 AS decimal(1,0))", "DECIMAL '0'");
        TestExpressionInterpreter.assertOptimizedEquals("CAST(123 AS decimal(3,0))", "DECIMAL '123'");
        TestExpressionInterpreter.assertOptimizedEquals("CAST(-123 AS decimal(3,0))", "DECIMAL '-123'");
        TestExpressionInterpreter.assertOptimizedEquals("CAST(-123 AS decimal(20,10))", "CAST(-123 AS decimal(20,10))");
        TestExpressionInterpreter.assertOptimizedEquals("CAST(0E0 AS decimal(1,0))", "DECIMAL '0'");
        TestExpressionInterpreter.assertOptimizedEquals("CAST(123.2E0 AS decimal(4,1))", "DECIMAL '123.2'");
        TestExpressionInterpreter.assertOptimizedEquals("CAST(-123.0E0 AS decimal(3,0))", "DECIMAL '-123'");
        TestExpressionInterpreter.assertOptimizedEquals("CAST(-123.55E0 AS decimal(20,10))", "CAST(-123.55 AS decimal(20,10))");
        TestExpressionInterpreter.assertOptimizedEquals("CAST('0' AS decimal(1,0))", "DECIMAL '0'");
        TestExpressionInterpreter.assertOptimizedEquals("CAST('123.2' AS decimal(4,1))", "DECIMAL '123.2'");
        TestExpressionInterpreter.assertOptimizedEquals("CAST('-123.0' AS decimal(3,0))", "DECIMAL '-123'");
        TestExpressionInterpreter.assertOptimizedEquals("CAST('-123.55' AS decimal(20,10))", "CAST(-123.55 AS decimal(20,10))");
        TestExpressionInterpreter.assertOptimizedEquals("CAST(NULL AS decimal(1,0))", "NULL");
        TestExpressionInterpreter.assertOptimizedEquals("CAST(NULL AS decimal(20,10))", "NULL");
        TestExpressionInterpreter.assertOptimizedEquals("CAST(true AS decimal(1,0))", "DECIMAL '1'");
        TestExpressionInterpreter.assertOptimizedEquals("CAST(false AS decimal(4,1))", "DECIMAL '000.0'");
        TestExpressionInterpreter.assertOptimizedEquals("CAST(true AS decimal(3,0))", "DECIMAL '001'");
        TestExpressionInterpreter.assertOptimizedEquals("CAST(false AS decimal(20,10))", "CAST(0 AS decimal(20,10))");
        TestExpressionInterpreter.assertOptimizedEquals("CAST(0.0 AS decimal(1,0))", "DECIMAL '0'");
        TestExpressionInterpreter.assertOptimizedEquals("CAST(123.2 AS decimal(4,1))", "DECIMAL '123.2'");
        TestExpressionInterpreter.assertOptimizedEquals("CAST(-123.0 AS decimal(3,0))", "DECIMAL '-123'");
        TestExpressionInterpreter.assertOptimizedEquals("CAST(-123.55 AS decimal(20,10))", "CAST(-123.55 AS decimal(20,10))");
    }

    @Test
    public void testCastOptimization() {
        TestExpressionInterpreter.assertOptimizedEquals("CAST(bound_integer AS varchar)", "'1234'");
        TestExpressionInterpreter.assertOptimizedEquals("CAST(bound_long AS varchar)", "'1234'");
        TestExpressionInterpreter.assertOptimizedEquals("CAST(bound_integer + 1 AS varchar)", "'1235'");
        TestExpressionInterpreter.assertOptimizedEquals("CAST(bound_long + 1 AS varchar)", "'1235'");
        TestExpressionInterpreter.assertOptimizedEquals("CAST(unbound_string AS varchar)", "CAST(unbound_string AS varchar)");
        TestExpressionInterpreter.assertOptimizedMatches("CAST(unbound_string AS varchar)", "unbound_string");
        TestExpressionInterpreter.assertOptimizedMatches("CAST(unbound_integer AS integer)", "unbound_integer");
        TestExpressionInterpreter.assertOptimizedMatches("CAST(unbound_string AS varchar(10))", "CAST(unbound_string AS varchar(10))");
    }

    @Test
    public void testTryCast() {
        TestExpressionInterpreter.assertOptimizedEquals("TRY_CAST(NULL AS bigint)", "NULL");
        TestExpressionInterpreter.assertOptimizedEquals("TRY_CAST(123 AS bigint)", "123");
        TestExpressionInterpreter.assertOptimizedEquals("TRY_CAST(NULL AS integer)", "NULL");
        TestExpressionInterpreter.assertOptimizedEquals("TRY_CAST(123 AS integer)", "123");
        TestExpressionInterpreter.assertOptimizedEquals("TRY_CAST('foo' AS varchar)", "'foo'");
        TestExpressionInterpreter.assertOptimizedEquals("TRY_CAST('foo' AS bigint)", "NULL");
        TestExpressionInterpreter.assertOptimizedEquals("TRY_CAST(unbound_string AS bigint)", "TRY_CAST(unbound_string AS bigint)");
        TestExpressionInterpreter.assertOptimizedEquals("TRY_CAST('foo' AS decimal(2,1))", "NULL");
    }

    @Test
    public void testReservedWithDoubleQuotes() {
        TestExpressionInterpreter.assertOptimizedEquals("\"time\"", "\"time\"");
    }

    @Test
    public void testSearchCase() {
        TestExpressionInterpreter.assertOptimizedEquals("CASE WHEN true THEN 33 END", "33");
        TestExpressionInterpreter.assertOptimizedEquals("CASE WHEN false THEN 1 ELSE 33 END", "33");
        TestExpressionInterpreter.assertOptimizedEquals("CASE WHEN false THEN 10000000000 ELSE 33 END", "33");
        TestExpressionInterpreter.assertOptimizedEquals("CASE WHEN bound_long = 1234 THEN 33 END", "33");
        TestExpressionInterpreter.assertOptimizedEquals("CASE WHEN true THEN bound_long END", "1234");
        TestExpressionInterpreter.assertOptimizedEquals("CASE WHEN false THEN 1 ELSE bound_long END", "1234");
        TestExpressionInterpreter.assertOptimizedEquals("CASE WHEN bound_integer = 1234 THEN 33 END", "33");
        TestExpressionInterpreter.assertOptimizedEquals("CASE WHEN true THEN bound_integer END", "1234");
        TestExpressionInterpreter.assertOptimizedEquals("CASE WHEN false THEN 1 ELSE bound_integer END", "1234");
        TestExpressionInterpreter.assertOptimizedEquals("CASE WHEN bound_long = 1234 THEN 33 ELSE unbound_long END", "33");
        TestExpressionInterpreter.assertOptimizedEquals("CASE WHEN true THEN bound_long ELSE unbound_long END", "1234");
        TestExpressionInterpreter.assertOptimizedEquals("CASE WHEN false THEN unbound_long ELSE bound_long END", "1234");
        TestExpressionInterpreter.assertOptimizedEquals("CASE WHEN bound_integer = 1234 THEN 33 ELSE unbound_integer END", "33");
        TestExpressionInterpreter.assertOptimizedEquals("CASE WHEN true THEN bound_integer ELSE unbound_integer END", "1234");
        TestExpressionInterpreter.assertOptimizedEquals("CASE WHEN false THEN unbound_integer ELSE bound_integer END", "1234");
        TestExpressionInterpreter.assertOptimizedEquals("CASE WHEN unbound_long = 1234 THEN 33 ELSE 1 END", "CASE WHEN unbound_long = 1234 THEN 33 ELSE 1 END");
        TestExpressionInterpreter.assertOptimizedMatches("CASE WHEN 0 / 0 = 0 THEN 1 END", "CASE WHEN ((0 / 0) = 0) THEN 1 END");
        TestExpressionInterpreter.assertOptimizedMatches("IF(false, 1, 0 / 0)", "0 / 0");
        TestExpressionInterpreter.assertOptimizedEquals("CASE WHEN false THEN 2.2 WHEN true THEN 2.2 END", "2.2");
        TestExpressionInterpreter.assertOptimizedEquals("CASE WHEN false THEN 1234567890.0987654321 WHEN true THEN 3.3 END", "CAST(3.3 AS decimal(20,10))");
        TestExpressionInterpreter.assertOptimizedEquals("CASE WHEN false THEN 1 WHEN true THEN 2.2 END", "2.2");
        TestExpressionInterpreter.assertOptimizedEquals("CASE WHEN ARRAY[CAST(1 AS bigint)] = ARRAY[CAST(1 AS bigint)] THEN 'matched' ELSE 'not_matched' END", "'matched'");
        TestExpressionInterpreter.assertOptimizedEquals("CASE WHEN ARRAY[CAST(2 AS bigint)] = ARRAY[CAST(1 AS bigint)] THEN 'matched' ELSE 'not_matched' END", "'not_matched'");
        TestExpressionInterpreter.assertOptimizedEquals("CASE WHEN ARRAY[CAST(NULL AS bigint)] = ARRAY[CAST(1 AS bigint)] THEN 'matched' ELSE 'not_matched' END", "'not_matched'");
        TestExpressionInterpreter.assertOptimizedEquals("CASE WHEN 0 / 0 = 0 THEN 'a' ELSE 'b' END", "CASE WHEN 0 / 0 = 0 THEN 'a' ELSE 'b' END");
        TestExpressionInterpreter.assertOptimizedEquals("CASE WHEN true THEN 'a' WHEN 0 / 0 = 0 THEN 'b' ELSE 'c' END", "'a'");
        TestExpressionInterpreter.assertOptimizedEquals("CASE WHEN 0 / 0 = 0 THEN 'a' WHEN true THEN 'b' ELSE 'c' END", "CASE WHEN 0 / 0 = 0 THEN 'a' ELSE 'b' END");
        TestExpressionInterpreter.assertOptimizedEquals("CASE WHEN 0 / 0 = 0 THEN 'a' WHEN false THEN 'b' ELSE 'c' END", "CASE WHEN 0 / 0 = 0 THEN 'a' ELSE 'c' END");
        TestExpressionInterpreter.assertOptimizedEquals("CASE WHEN 0 / 0 = 0 THEN 'a' WHEN false THEN 'b' END", "CASE WHEN 0 / 0 = 0 THEN 'a' END");
        TestExpressionInterpreter.assertOptimizedEquals("CASE WHEN false THEN 'a' WHEN false THEN 'b' ELSE 'c' END", "'c'");
        TestExpressionInterpreter.assertOptimizedEquals("CASE WHEN false THEN 'a' WHEN false THEN 'b' END", "null");
        TestExpressionInterpreter.assertOptimizedEquals("CASE WHEN 0 > 1 THEN 'a' WHEN 1 > 2 THEN 'b' END", "null");
        TestExpressionInterpreter.assertOptimizedEquals("CASE WHEN true THEN 0 / 0 WHEN false THEN 1 END", "0 / 0");
        TestExpressionInterpreter.assertOptimizedEquals("CASE WHEN false THEN 1 WHEN false THEN 2 ELSE 0 / 0 END", "0 / 0");
        TestExpressionInterpreter.assertEvaluatedEquals("CASE WHEN false THEN 0 / 0 WHEN true THEN 1 ELSE 0 / 0 END", "1");
        TestExpressionInterpreter.assertEvaluatedEquals("CASE WHEN true THEN 1 WHEN 0 / 0 = 0 THEN 2 ELSE 0 / 0 END", "1");
    }

    @Test
    public void testSimpleCase() {
        TestExpressionInterpreter.assertOptimizedEquals("CASE 1 WHEN 1 THEN 32 + 1 WHEN 1 THEN 34 END", "33");
        TestExpressionInterpreter.assertOptimizedEquals("CASE NULL WHEN true THEN 33 END", "NULL");
        TestExpressionInterpreter.assertOptimizedEquals("CASE NULL WHEN true THEN 33 ELSE 33 END", "33");
        TestExpressionInterpreter.assertOptimizedEquals("CASE 33 WHEN NULL THEN 1 ELSE 33 END", "33");
        TestExpressionInterpreter.assertOptimizedEquals("CASE NULL WHEN true THEN 3300000000 END", "NULL");
        TestExpressionInterpreter.assertOptimizedEquals("CASE NULL WHEN true THEN 3300000000 ELSE 3300000000 END", "3300000000");
        TestExpressionInterpreter.assertOptimizedEquals("CASE 33 WHEN NULL THEN 3300000000 ELSE 33 END", "33");
        TestExpressionInterpreter.assertOptimizedEquals("CASE true WHEN true THEN 33 END", "33");
        TestExpressionInterpreter.assertOptimizedEquals("CASE true WHEN false THEN 1 ELSE 33 END", "33");
        TestExpressionInterpreter.assertOptimizedEquals("CASE bound_long WHEN 1234 THEN 33 END", "33");
        TestExpressionInterpreter.assertOptimizedEquals("CASE 1234 WHEN bound_long THEN 33 END", "33");
        TestExpressionInterpreter.assertOptimizedEquals("CASE true WHEN true THEN bound_long END", "1234");
        TestExpressionInterpreter.assertOptimizedEquals("CASE true WHEN false THEN 1 ELSE bound_long END", "1234");
        TestExpressionInterpreter.assertOptimizedEquals("CASE bound_integer WHEN 1234 THEN 33 END", "33");
        TestExpressionInterpreter.assertOptimizedEquals("CASE 1234 WHEN bound_integer THEN 33 END", "33");
        TestExpressionInterpreter.assertOptimizedEquals("CASE true WHEN true THEN bound_integer END", "1234");
        TestExpressionInterpreter.assertOptimizedEquals("CASE true WHEN false THEN 1 ELSE bound_integer END", "1234");
        TestExpressionInterpreter.assertOptimizedEquals("CASE bound_long WHEN 1234 THEN 33 ELSE unbound_long END", "33");
        TestExpressionInterpreter.assertOptimizedEquals("CASE true WHEN true THEN bound_long ELSE unbound_long END", "1234");
        TestExpressionInterpreter.assertOptimizedEquals("CASE true WHEN false THEN unbound_long ELSE bound_long END", "1234");
        TestExpressionInterpreter.assertOptimizedEquals("CASE unbound_long WHEN 1234 THEN 33 ELSE 1 END", "CASE unbound_long WHEN 1234 THEN 33 ELSE 1 END");
        TestExpressionInterpreter.assertOptimizedEquals("CASE 33 WHEN 0 THEN 0 WHEN 33 THEN unbound_long ELSE 1 END", "unbound_long");
        TestExpressionInterpreter.assertOptimizedEquals("CASE 33 WHEN 0 THEN 0 WHEN 33 THEN 1 WHEN unbound_long THEN 2 ELSE 1 END", "1");
        TestExpressionInterpreter.assertOptimizedEquals("CASE 33 WHEN unbound_long THEN 0 WHEN 1 THEN 1 WHEN 33 THEN 2 ELSE 0 END", "CASE 33 WHEN unbound_long THEN 0 ELSE 2 END");
        TestExpressionInterpreter.assertOptimizedEquals("CASE 33 WHEN 0 THEN 0 WHEN 1 THEN 1 ELSE unbound_long END", "unbound_long");
        TestExpressionInterpreter.assertOptimizedEquals("CASE 33 WHEN unbound_long THEN 0 WHEN 1 THEN 1 WHEN unbound_long2 THEN 2 ELSE 3 END", "CASE 33 WHEN unbound_long THEN 0 WHEN unbound_long2 THEN 2 ELSE 3 END");
        TestExpressionInterpreter.assertOptimizedEquals("CASE true WHEN unbound_long = 1 THEN 1 WHEN 0 / 0 = 0 THEN 2 ELSE 33 END", "CASE true WHEN unbound_long = 1 THEN 1 WHEN 0 / 0 = 0 THEN 2 ELSE 33 END");
        TestExpressionInterpreter.assertOptimizedEquals("CASE bound_long WHEN unbound_long + 123 * 10  THEN 1 = 1 ELSE 1 = 2 END", "CASE bound_long WHEN unbound_long + 1230 THEN true ELSE false END");
        TestExpressionInterpreter.assertOptimizedEquals("CASE bound_long WHEN unbound_long THEN 2 + 2 END", "CASE bound_long WHEN unbound_long THEN 4 END");
        TestExpressionInterpreter.assertOptimizedEquals("CASE bound_long WHEN unbound_long THEN 2 + 2 WHEN 1 THEN NULL WHEN 2 THEN NULL END", "CASE bound_long WHEN unbound_long THEN 4 END");
        TestExpressionInterpreter.assertOptimizedMatches("CASE 1 WHEN unbound_long THEN 1 WHEN 0 / 0 THEN 2 ELSE 1 END", "CASE BIGINT '1' WHEN unbound_long THEN 1 WHEN CAST((0 / 0) AS bigint) THEN 2 ELSE 1 END");
        TestExpressionInterpreter.assertOptimizedMatches("CASE 1 WHEN 0 / 0 THEN 1 WHEN 0 / 0 THEN 2 ELSE 1 END", "CASE 1 WHEN 0 / 0 THEN 1 WHEN 0 / 0 THEN 2 ELSE 1 END");
        TestExpressionInterpreter.assertOptimizedEquals("CASE true WHEN false THEN 2.2 WHEN true THEN 2.2 END", "2.2");
        TestExpressionInterpreter.assertOptimizedEquals("CASE true WHEN false THEN 1 WHEN true THEN 2.2 END", "2.2");
        TestExpressionInterpreter.assertOptimizedEquals("CASE ARRAY[CAST(1 AS bigint)] WHEN ARRAY[CAST(1 AS bigint)] THEN 'matched' ELSE 'not_matched' END", "'matched'");
        TestExpressionInterpreter.assertOptimizedEquals("CASE ARRAY[CAST(2 AS bigint)] WHEN ARRAY[CAST(1 AS bigint)] THEN 'matched' ELSE 'not_matched' END", "'not_matched'");
        TestExpressionInterpreter.assertOptimizedEquals("CASE ARRAY[CAST(NULL AS bigint)] WHEN ARRAY[CAST(1 AS bigint)] THEN 'matched' ELSE 'not_matched' END", "'not_matched'");
        TestExpressionInterpreter.assertOptimizedEquals("CASE null WHEN 0 / 0 THEN 0 / 0 ELSE 1 END", "1");
        TestExpressionInterpreter.assertOptimizedEquals("CASE null WHEN 1 THEN 2 ELSE 0 / 0 END", "0 / 0");
        TestExpressionInterpreter.assertOptimizedEquals("CASE 0 / 0 WHEN 1 THEN 2 ELSE 3 END", "CASE 0 / 0 WHEN 1 THEN 2 ELSE 3 END");
        TestExpressionInterpreter.assertOptimizedEquals("CASE 1 WHEN 0 / 0 THEN 2 ELSE 3 END", "CASE 1 WHEN 0 / 0 THEN 2 ELSE 3 END");
        TestExpressionInterpreter.assertOptimizedEquals("CASE 1 WHEN 2 THEN 2 WHEN 0 / 0 THEN 3 ELSE 4 END", "CASE 1 WHEN 0 / 0 THEN 3 ELSE 4 END");
        TestExpressionInterpreter.assertOptimizedEquals("CASE 1 WHEN 0 / 0 THEN 2 END", "CASE 1 WHEN 0 / 0 THEN 2 END");
        TestExpressionInterpreter.assertOptimizedEquals("CASE 1 WHEN 2 THEN 2 WHEN 3 THEN 3 END", "null");
        TestExpressionInterpreter.assertEvaluatedEquals("CASE null WHEN 0 / 0 THEN 0 / 0 ELSE 1 END", "1");
        TestExpressionInterpreter.assertEvaluatedEquals("CASE 1 WHEN 2 THEN 0 / 0 ELSE 3 END", "3");
        TestExpressionInterpreter.assertEvaluatedEquals("CASE 1 WHEN 1 THEN 2 WHEN 1 THEN 0 / 0 END", "2");
        TestExpressionInterpreter.assertEvaluatedEquals("CASE 1 WHEN 1 THEN 2 ELSE 0 / 0 END", "2");
    }

    @Test
    public void testCoalesce() {
        TestExpressionInterpreter.assertOptimizedEquals("coalesce(unbound_long * (2 * 3), 1 - 1, NULL)", "coalesce(unbound_long * 6, 0)");
        TestExpressionInterpreter.assertOptimizedEquals("coalesce(unbound_long * (2 * 3), 1.0E0/2.0E0, NULL)", "coalesce(unbound_long * 6, 0.5E0)");
        TestExpressionInterpreter.assertOptimizedEquals("coalesce(unbound_long, 2, 1.0E0/2.0E0, 12.34E0, NULL)", "coalesce(unbound_long, 2.0E0, 0.5E0, 12.34E0)");
        TestExpressionInterpreter.assertOptimizedEquals("coalesce(unbound_integer * (2 * 3), 1 - 1, NULL)", "coalesce(6 * unbound_integer, 0)");
        TestExpressionInterpreter.assertOptimizedEquals("coalesce(unbound_integer * (2 * 3), 1.0E0/2.0E0, NULL)", "coalesce(6 * unbound_integer, 0.5E0)");
        TestExpressionInterpreter.assertOptimizedEquals("coalesce(unbound_integer, 2, 1.0E0/2.0E0, 12.34E0, NULL)", "coalesce(unbound_integer, 2.0E0, 0.5E0, 12.34E0)");
        TestExpressionInterpreter.assertOptimizedMatches("coalesce(0 / 0 < 1, unbound_boolean, 0 / 0 = 0)", "COALESCE(((0 / 0) < 1), unbound_boolean, ((0 / 0) = 0))");
        TestExpressionInterpreter.assertOptimizedMatches("coalesce(unbound_long, unbound_long)", "unbound_long");
        TestExpressionInterpreter.assertOptimizedMatches("coalesce(2 * unbound_long, 2 * unbound_long)", "unbound_long * BIGINT '2'");
        TestExpressionInterpreter.assertOptimizedMatches("coalesce(unbound_long, unbound_long2, unbound_long)", "coalesce(unbound_long, unbound_long2)");
        TestExpressionInterpreter.assertOptimizedMatches("coalesce(unbound_long, unbound_long2, unbound_long, unbound_long3)", "coalesce(unbound_long, unbound_long2, unbound_long3)");
        TestExpressionInterpreter.assertOptimizedEquals("coalesce(6, unbound_long2, unbound_long, unbound_long3)", "6");
        TestExpressionInterpreter.assertOptimizedEquals("coalesce(2 * 3, unbound_long2, unbound_long, unbound_long3)", "6");
        TestExpressionInterpreter.assertOptimizedMatches("coalesce(random(), random(), 5)", "coalesce(random(), random(), 5E0)");
        TestExpressionInterpreter.assertOptimizedMatches("coalesce(unbound_long, coalesce(unbound_long, 1))", "coalesce(unbound_long, BIGINT '1')");
        TestExpressionInterpreter.assertOptimizedMatches("coalesce(coalesce(unbound_long, coalesce(unbound_long, 1)), unbound_long2)", "coalesce(unbound_long, BIGINT '1')");
        TestExpressionInterpreter.assertOptimizedMatches("coalesce(unbound_long, 2, coalesce(unbound_long, 1))", "coalesce(unbound_long, BIGINT '2')");
        TestExpressionInterpreter.assertOptimizedMatches("coalesce(coalesce(unbound_long, coalesce(unbound_long2, unbound_long3)), 1)", "coalesce(unbound_long, unbound_long2, unbound_long3, BIGINT '1')");
        TestExpressionInterpreter.assertOptimizedMatches("coalesce(unbound_double, coalesce(random(), unbound_double))", "coalesce(unbound_double, random())");
        TestExpressionInterpreter.assertOptimizedEquals("coalesce(null, coalesce(null, null))", "null");
        TestExpressionInterpreter.assertOptimizedEquals("coalesce(null, coalesce(null, coalesce(null, null, 1)))", "1");
        TestExpressionInterpreter.assertOptimizedEquals("coalesce(1, 0 / 0)", "1");
        TestExpressionInterpreter.assertOptimizedEquals("coalesce(0 / 0, 1)", "coalesce(0 / 0, 1)");
        TestExpressionInterpreter.assertOptimizedEquals("coalesce(0 / 0, 1, null)", "coalesce(0 / 0, 1)");
        TestExpressionInterpreter.assertOptimizedEquals("coalesce(1, coalesce(0 / 0, 2))", "1");
        TestExpressionInterpreter.assertOptimizedEquals("coalesce(0 / 0, null, 1 / 0, null, 0 / 0)", "coalesce(0 / 0, 1 / 0)");
        TestExpressionInterpreter.assertOptimizedEquals("coalesce(0 / 0, null, 0 / 0, null)", "0 / 0");
        TestExpressionInterpreter.assertOptimizedEquals("coalesce(0 / 0, null, coalesce(0 / 0, null))", "0 / 0");
        TestExpressionInterpreter.assertOptimizedEquals("coalesce(rand(), rand(), 1, rand())", "coalesce(rand(), rand(), 1)");
        TestExpressionInterpreter.assertEvaluatedEquals("coalesce(1, 0 / 0)", "1");
        TrinoExceptionAssert.assertTrinoExceptionThrownBy(() -> TestExpressionInterpreter.evaluate("coalesce(0 / 0, 1)")).hasErrorCode(new ErrorCodeSupplier[]{StandardErrorCode.DIVISION_BY_ZERO});
    }

    @Test
    public void testIf() {
        TestExpressionInterpreter.assertOptimizedEquals("IF(2 = 2, 3, 4)", "3");
        TestExpressionInterpreter.assertOptimizedEquals("IF(1 = 2, 3, 4)", "4");
        TestExpressionInterpreter.assertOptimizedEquals("IF(1 = 2, BIGINT '3', 4)", "4");
        TestExpressionInterpreter.assertOptimizedEquals("IF(1 = 2, 3000000000, 4)", "4");
        TestExpressionInterpreter.assertOptimizedEquals("IF(true, 3, 4)", "3");
        TestExpressionInterpreter.assertOptimizedEquals("IF(false, 3, 4)", "4");
        TestExpressionInterpreter.assertOptimizedEquals("IF(NULL, 3, 4)", "4");
        TestExpressionInterpreter.assertOptimizedEquals("IF(true, 3, NULL)", "3");
        TestExpressionInterpreter.assertOptimizedEquals("IF(false, 3, NULL)", "NULL");
        TestExpressionInterpreter.assertOptimizedEquals("IF(true, NULL, 4)", "NULL");
        TestExpressionInterpreter.assertOptimizedEquals("IF(false, NULL, 4)", "4");
        TestExpressionInterpreter.assertOptimizedEquals("IF(true, NULL, NULL)", "NULL");
        TestExpressionInterpreter.assertOptimizedEquals("IF(false, NULL, NULL)", "NULL");
        TestExpressionInterpreter.assertOptimizedEquals("IF(true, 3.5E0, 4.2E0)", "3.5E0");
        TestExpressionInterpreter.assertOptimizedEquals("IF(false, 3.5E0, 4.2E0)", "4.2E0");
        TestExpressionInterpreter.assertOptimizedEquals("IF(true, 'foo', 'bar')", "'foo'");
        TestExpressionInterpreter.assertOptimizedEquals("IF(false, 'foo', 'bar')", "'bar'");
        TestExpressionInterpreter.assertOptimizedEquals("IF(true, 1.01, 1.02)", "1.01");
        TestExpressionInterpreter.assertOptimizedEquals("IF(false, 1.01, 1.02)", "1.02");
        TestExpressionInterpreter.assertOptimizedEquals("IF(true, 1234567890.123, 1.02)", "1234567890.123");
        TestExpressionInterpreter.assertOptimizedEquals("IF(false, 1.01, 1234567890.123)", "1234567890.123");
        TestExpressionInterpreter.assertOptimizedEquals("IF(unbound_boolean, 1 + 2, 3 + 4)", "CASE WHEN unbound_boolean THEN (1 + 2) ELSE (3 + 4) END");
        TestExpressionInterpreter.assertOptimizedEquals("IF(unbound_boolean, BIGINT '1' + 2, 3 + 4)", "CASE WHEN unbound_boolean THEN (BIGINT '1' + 2) ELSE (3 + 4) END");
        TestExpressionInterpreter.assertOptimizedEquals("IF(true, 0 / 0)", "0 / 0");
        TestExpressionInterpreter.assertOptimizedEquals("IF(true, 1, 0 / 0)", "1");
        TestExpressionInterpreter.assertOptimizedEquals("IF(false, 1, 0 / 0)", "0 / 0");
        TestExpressionInterpreter.assertOptimizedEquals("IF(false, 0 / 0, 1)", "1");
        TestExpressionInterpreter.assertOptimizedEquals("IF(0 / 0 = 0, 1, 2)", "IF(0 / 0 = 0, 1, 2)");
        TestExpressionInterpreter.assertEvaluatedEquals("IF(true, 1, 0 / 0)", "1");
        TestExpressionInterpreter.assertEvaluatedEquals("IF(false, 0 / 0, 1)", "1");
        TrinoExceptionAssert.assertTrinoExceptionThrownBy(() -> TestExpressionInterpreter.evaluate("IF(0 / 0 = 0, 1, 2)")).hasErrorCode(new ErrorCodeSupplier[]{StandardErrorCode.DIVISION_BY_ZERO});
    }

    @Test
    public void testLike() {
        TestExpressionInterpreter.assertOptimizedEquals("'a' LIKE 'a'", "true");
        TestExpressionInterpreter.assertOptimizedEquals("'' LIKE 'a'", "false");
        TestExpressionInterpreter.assertOptimizedEquals("'abc' LIKE 'a'", "false");
        TestExpressionInterpreter.assertOptimizedEquals("'a' LIKE '_'", "true");
        TestExpressionInterpreter.assertOptimizedEquals("'' LIKE '_'", "false");
        TestExpressionInterpreter.assertOptimizedEquals("'abc' LIKE '_'", "false");
        TestExpressionInterpreter.assertOptimizedEquals("'a' LIKE '%'", "true");
        TestExpressionInterpreter.assertOptimizedEquals("'' LIKE '%'", "true");
        TestExpressionInterpreter.assertOptimizedEquals("'abc' LIKE '%'", "true");
        TestExpressionInterpreter.assertOptimizedEquals("'abc' LIKE '___'", "true");
        TestExpressionInterpreter.assertOptimizedEquals("'ab' LIKE '___'", "false");
        TestExpressionInterpreter.assertOptimizedEquals("'abcd' LIKE '___'", "false");
        TestExpressionInterpreter.assertOptimizedEquals("'abc' LIKE 'abc'", "true");
        TestExpressionInterpreter.assertOptimizedEquals("'xyz' LIKE 'abc'", "false");
        TestExpressionInterpreter.assertOptimizedEquals("'abc0' LIKE 'abc'", "false");
        TestExpressionInterpreter.assertOptimizedEquals("'0abc' LIKE 'abc'", "false");
        TestExpressionInterpreter.assertOptimizedEquals("'abc' LIKE 'abc%'", "true");
        TestExpressionInterpreter.assertOptimizedEquals("'abc0' LIKE 'abc%'", "true");
        TestExpressionInterpreter.assertOptimizedEquals("'0abc' LIKE 'abc%'", "false");
        TestExpressionInterpreter.assertOptimizedEquals("'abc' LIKE '%abc'", "true");
        TestExpressionInterpreter.assertOptimizedEquals("'0abc' LIKE '%abc'", "true");
        TestExpressionInterpreter.assertOptimizedEquals("'abc0' LIKE '%abc'", "false");
        TestExpressionInterpreter.assertOptimizedEquals("'abc' LIKE '%abc%'", "true");
        TestExpressionInterpreter.assertOptimizedEquals("'0abc' LIKE '%abc%'", "true");
        TestExpressionInterpreter.assertOptimizedEquals("'abc0' LIKE '%abc%'", "true");
        TestExpressionInterpreter.assertOptimizedEquals("'0abc0' LIKE '%abc%'", "true");
        TestExpressionInterpreter.assertOptimizedEquals("'xyzw' LIKE '%abc%'", "false");
        TestExpressionInterpreter.assertOptimizedEquals("'abc' LIKE '%ab%c%'", "true");
        TestExpressionInterpreter.assertOptimizedEquals("'0abc' LIKE '%ab%c%'", "true");
        TestExpressionInterpreter.assertOptimizedEquals("'abc0' LIKE '%ab%c%'", "true");
        TestExpressionInterpreter.assertOptimizedEquals("'0abc0' LIKE '%ab%c%'", "true");
        TestExpressionInterpreter.assertOptimizedEquals("'ab01c' LIKE '%ab%c%'", "true");
        TestExpressionInterpreter.assertOptimizedEquals("'0ab01c' LIKE '%ab%c%'", "true");
        TestExpressionInterpreter.assertOptimizedEquals("'ab01c0' LIKE '%ab%c%'", "true");
        TestExpressionInterpreter.assertOptimizedEquals("'0ab01c0' LIKE '%ab%c%'", "true");
        TestExpressionInterpreter.assertOptimizedEquals("'xyzw' LIKE '%ab%c%'", "false");
        TestExpressionInterpreter.assertOptimizedEquals("'' LIKE ''", "true");
        TestExpressionInterpreter.assertOptimizedEquals("'.*' LIKE '.*'", "true");
        TestExpressionInterpreter.assertOptimizedEquals("'[' LIKE '['", "true");
        TestExpressionInterpreter.assertOptimizedEquals("']' LIKE ']'", "true");
        TestExpressionInterpreter.assertOptimizedEquals("'{' LIKE '{'", "true");
        TestExpressionInterpreter.assertOptimizedEquals("'}' LIKE '}'", "true");
        TestExpressionInterpreter.assertOptimizedEquals("'?' LIKE '?'", "true");
        TestExpressionInterpreter.assertOptimizedEquals("'+' LIKE '+'", "true");
        TestExpressionInterpreter.assertOptimizedEquals("'(' LIKE '('", "true");
        TestExpressionInterpreter.assertOptimizedEquals("')' LIKE ')'", "true");
        TestExpressionInterpreter.assertOptimizedEquals("'|' LIKE '|'", "true");
        TestExpressionInterpreter.assertOptimizedEquals("'^' LIKE '^'", "true");
        TestExpressionInterpreter.assertOptimizedEquals("'$' LIKE '$'", "true");
        TestExpressionInterpreter.assertOptimizedEquals("NULL LIKE '%'", "NULL");
        TestExpressionInterpreter.assertOptimizedEquals("'a' LIKE NULL", "NULL");
        TestExpressionInterpreter.assertOptimizedEquals("'a' LIKE '%' ESCAPE NULL", "NULL");
        TestExpressionInterpreter.assertOptimizedEquals("'%' LIKE 'z%' ESCAPE 'z'", "true");
    }

    @Test
    public void testLikeChar() {
        TestExpressionInterpreter.assertOptimizedEquals("CAST('abc' AS char(3)) LIKE 'abc'", "true");
        TestExpressionInterpreter.assertOptimizedEquals("CAST('abc' AS char(4)) LIKE 'abc'", "false");
        TestExpressionInterpreter.assertOptimizedEquals("CAST('abc' AS char(4)) LIKE 'abc '", "true");
        TestExpressionInterpreter.assertOptimizedEquals("CAST('abc' AS char(3)) LIKE '%abc'", "true");
        TestExpressionInterpreter.assertOptimizedEquals("CAST('abc' AS char(4)) LIKE '%abc'", "false");
        TestExpressionInterpreter.assertOptimizedEquals("CAST('abc' AS char(4)) LIKE '%abc '", "true");
        TestExpressionInterpreter.assertOptimizedEquals("CAST('abc' AS char(4)) LIKE '%c'", "false");
        TestExpressionInterpreter.assertOptimizedEquals("CAST('abc' AS char(4)) LIKE '%c '", "true");
        TestExpressionInterpreter.assertOptimizedEquals("CAST('abc' AS char(3)) LIKE '%a%b%c'", "true");
        TestExpressionInterpreter.assertOptimizedEquals("CAST('abc' AS char(4)) LIKE '%a%b%c'", "false");
        TestExpressionInterpreter.assertOptimizedEquals("CAST('abc' AS char(4)) LIKE '%a%b%c '", "true");
        TestExpressionInterpreter.assertOptimizedEquals("CAST('abc' AS char(4)) LIKE '%a%b%c_'", "true");
        TestExpressionInterpreter.assertOptimizedEquals("CAST('abc' AS char(4)) LIKE '%a%b%c%'", "true");
    }

    @Test
    public void testLikeOptimization() {
        TestExpressionInterpreter.assertOptimizedEquals("unbound_string LIKE 'abc'", "unbound_string = VARCHAR 'abc'");
        TestExpressionInterpreter.assertOptimizedEquals("unbound_string LIKE '' ESCAPE '#'", "unbound_string LIKE '' ESCAPE '#'");
        TestExpressionInterpreter.assertOptimizedEquals("unbound_string LIKE 'abc' ESCAPE '#'", "unbound_string = VARCHAR 'abc'");
        TestExpressionInterpreter.assertOptimizedEquals("unbound_string LIKE 'a#_b' ESCAPE '#'", "unbound_string = VARCHAR 'a_b'");
        TestExpressionInterpreter.assertOptimizedEquals("unbound_string LIKE 'a#%b' ESCAPE '#'", "unbound_string = VARCHAR 'a%b'");
        TestExpressionInterpreter.assertOptimizedEquals("unbound_string LIKE 'a#_##b' ESCAPE '#'", "unbound_string = VARCHAR 'a_#b'");
        TestExpressionInterpreter.assertOptimizedEquals("unbound_string LIKE 'a#__b' ESCAPE '#'", "unbound_string LIKE 'a#__b' ESCAPE '#'");
        TestExpressionInterpreter.assertOptimizedEquals("unbound_string LIKE 'a##%b' ESCAPE '#'", "unbound_string LIKE 'a##%b' ESCAPE '#'");
        TestExpressionInterpreter.assertOptimizedEquals("bound_string LIKE bound_pattern", "true");
        TestExpressionInterpreter.assertOptimizedEquals("'abc' LIKE bound_pattern", "false");
        TestExpressionInterpreter.assertOptimizedEquals("unbound_string LIKE bound_pattern", "unbound_string LIKE bound_pattern");
        TestExpressionInterpreter.assertOptimizedEquals("unbound_string LIKE unbound_pattern ESCAPE unbound_string", "unbound_string LIKE unbound_pattern ESCAPE unbound_string");
    }

    @Test
    public void testLikeCharOptimization() {
        TestExpressionInterpreter.assertOptimizedEquals("unbound_char LIKE 'abc'", "false");
        TestExpressionInterpreter.assertOptimizedEquals("unbound_char LIKE 'abc' ESCAPE '#'", "false");
        TestExpressionInterpreter.assertOptimizedEquals("unbound_char LIKE 'ab#_' ESCAPE '#'", "false");
        TestExpressionInterpreter.assertOptimizedEquals("unbound_char LIKE 'ab#%' ESCAPE '#'", "false");
        TestExpressionInterpreter.assertOptimizedEquals("CAST(unbound_char AS char(4)) LIKE 'abc'", "false");
        TestExpressionInterpreter.assertOptimizedEquals("CAST(unbound_char AS char(4)) LIKE 'abc' ESCAPE '#'", "false");
        TestExpressionInterpreter.assertOptimizedEquals("unbound_char LIKE 'ab_'", "unbound_char LIKE 'ab_'");
        TestExpressionInterpreter.assertOptimizedEquals("unbound_char LIKE 'ab%'", "unbound_char LIKE 'ab%'");
        TestExpressionInterpreter.assertOptimizedEquals("unbound_char LIKE 'ab%' ESCAPE '#'", "unbound_char LIKE 'ab%' ESCAPE '#'");
        TestExpressionInterpreter.assertOptimizedEquals("CAST(unbound_char AS char(4)) LIKE 'abcd'", "CAST(unbound_char AS char(4)) = CAST('abcd' AS char(4))");
        TestExpressionInterpreter.assertOptimizedEquals("CAST(unbound_char AS char(4)) LIKE 'abcd' ESCAPE '#'", "CAST(unbound_char AS char(4)) = CAST('abcd' AS char(4))");
        TestExpressionInterpreter.assertOptimizedEquals("CAST(unbound_char AS char(4)) LIKE 'ja\u017a\u0144'", "CAST(unbound_char AS char(4)) = CAST('ja\u017a\u0144' AS char(4))");
        TestExpressionInterpreter.assertOptimizedEquals("CAST(unbound_char AS char(4)) LIKE 'ab#_' ESCAPE '#'", "false");
        TestExpressionInterpreter.assertOptimizedEquals("CAST(unbound_char AS char(4)) LIKE 'ab#%' ESCAPE '#'", "false");
        TestExpressionInterpreter.assertOptimizedEquals("CAST(unbound_char AS char(4)) LIKE 'ab#%' ESCAPE '\\'", "CAST(unbound_char AS char(4)) LIKE 'ab#%' ESCAPE '\\'");
        TestExpressionInterpreter.assertOptimizedEquals("CAST(unbound_char AS char(4)) LIKE 'abcde'", "false");
        TestExpressionInterpreter.assertOptimizedEquals("CAST(unbound_char AS char(4)) LIKE 'abcde' ESCAPE '#'", "false");
        TestExpressionInterpreter.assertOptimizedEquals("CAST(unbound_char AS char(4)) LIKE '#%a#%b#%c#%d#%' ESCAPE '#'", "false");
        TestExpressionInterpreter.assertOptimizedEquals("CAST(unbound_char AS char(4)) LIKE '%a%b%c%d%'", "CAST(unbound_char AS char(4)) LIKE '%a%b%c%d%'");
        TestExpressionInterpreter.assertOptimizedEquals("CAST(unbound_char AS char(4)) LIKE '%a%b%c%d%' ESCAPE '#'", "CAST(unbound_char AS char(4)) LIKE '%a%b%c%d%' ESCAPE '#'");
        TestExpressionInterpreter.assertOptimizedEquals("unbound_char LIKE CAST(CAST('abc' AS char( 17)) AS varchar(17))", "unbound_char = CAST('abc' AS char(17))");
        TestExpressionInterpreter.assertOptimizedEquals("unbound_char LIKE CAST(CAST('abc' AS char( 17)) AS varchar)", "unbound_char = CAST('abc' AS char(17))");
        TestExpressionInterpreter.assertOptimizedEquals("unbound_char LIKE CAST(CAST('' AS char(17)) AS varchar(17)) ESCAPE '#'", "unbound_char LIKE CAST('                 ' AS varchar(17)) ESCAPE '#'");
        TestExpressionInterpreter.assertOptimizedEquals("unbound_char LIKE CAST(CAST('' AS char(17)) AS varchar) ESCAPE '#'", "unbound_char LIKE CAST('                 ' AS varchar(17)) ESCAPE '#'");
        TestExpressionInterpreter.assertOptimizedEquals("unbound_char LIKE bound_pattern", "unbound_char LIKE VARCHAR '%el%'");
        TestExpressionInterpreter.assertOptimizedEquals("unbound_char LIKE unbound_pattern", "unbound_char LIKE unbound_pattern");
        TestExpressionInterpreter.assertOptimizedEquals("unbound_char LIKE unbound_pattern ESCAPE unbound_string", "unbound_char LIKE unbound_pattern ESCAPE unbound_string");
    }

    @Test
    public void testEvaluateInvalidLike() {
        TrinoExceptionAssert.assertTrinoExceptionThrownBy(() -> TestExpressionInterpreter.evaluate("unbound_string LIKE 'abc' ESCAPE 'bc'")).hasErrorCode(new ErrorCodeSupplier[]{StandardErrorCode.INVALID_FUNCTION_ARGUMENT}).hasMessage("Escape string must be a single character");
        TrinoExceptionAssert.assertTrinoExceptionThrownBy(() -> TestExpressionInterpreter.evaluate("unbound_string LIKE '#' ESCAPE '#'")).hasErrorCode(new ErrorCodeSupplier[]{StandardErrorCode.INVALID_FUNCTION_ARGUMENT}).hasMessage("Escape character must be followed by '%', '_' or the escape character itself");
        TrinoExceptionAssert.assertTrinoExceptionThrownBy(() -> TestExpressionInterpreter.evaluate("unbound_string LIKE '#abc' ESCAPE '#'")).hasErrorCode(new ErrorCodeSupplier[]{StandardErrorCode.INVALID_FUNCTION_ARGUMENT}).hasMessage("Escape character must be followed by '%', '_' or the escape character itself");
        TrinoExceptionAssert.assertTrinoExceptionThrownBy(() -> TestExpressionInterpreter.evaluate("unbound_string LIKE 'ab#' ESCAPE '#'")).hasErrorCode(new ErrorCodeSupplier[]{StandardErrorCode.INVALID_FUNCTION_ARGUMENT}).hasMessage("Escape character must be followed by '%', '_' or the escape character itself");
    }

    @Test
    public void testFailedExpressionOptimization() {
        TestExpressionInterpreter.assertOptimizedEquals("IF(unbound_boolean, 1, 0 / 0)", "CASE WHEN unbound_boolean THEN 1 ELSE 0 / 0 END");
        TestExpressionInterpreter.assertOptimizedEquals("IF(unbound_boolean, 0 / 0, 1)", "CASE WHEN unbound_boolean THEN 0 / 0 ELSE 1 END");
        TestExpressionInterpreter.assertOptimizedMatches("CASE unbound_long WHEN 1 THEN 1 WHEN 0 / 0 THEN 2 END", "CASE unbound_long WHEN BIGINT '1' THEN 1 WHEN CAST((0 / 0) AS bigint) THEN 2 END");
        TestExpressionInterpreter.assertOptimizedMatches("CASE unbound_boolean WHEN true THEN 1 ELSE 0 / 0 END", "CASE unbound_boolean WHEN true THEN 1 ELSE 0 / 0 END");
        TestExpressionInterpreter.assertOptimizedMatches("CASE bound_long WHEN unbound_long THEN 1 WHEN 0 / 0 THEN 2 ELSE 1 END", "CASE BIGINT '1234' WHEN unbound_long THEN 1 WHEN CAST((0 / 0) AS bigint) THEN 2 ELSE 1 END");
        TestExpressionInterpreter.assertOptimizedMatches("CASE WHEN unbound_boolean THEN 1 WHEN 0 / 0 = 0 THEN 2 END", "CASE WHEN unbound_boolean THEN 1 WHEN 0 / 0 = 0 THEN 2 END");
        TestExpressionInterpreter.assertOptimizedMatches("CASE WHEN unbound_boolean THEN 1 ELSE 0 / 0 END", "CASE WHEN unbound_boolean THEN 1 ELSE 0 / 0 END");
        TestExpressionInterpreter.assertOptimizedMatches("CASE WHEN unbound_boolean THEN 0 / 0 ELSE 1 END", "CASE WHEN unbound_boolean THEN 0 / 0 ELSE 1 END");
    }

    @Test
    public void testOptimizeDivideByZero() {
        TestExpressionInterpreter.assertOptimizedEquals("0 / 0", "0 / 0");
        TrinoExceptionAssert.assertTrinoExceptionThrownBy(() -> TestExpressionInterpreter.evaluate("0 / 0")).hasErrorCode(new ErrorCodeSupplier[]{StandardErrorCode.DIVISION_BY_ZERO});
    }

    @Test
    public void testMassiveArrayConstructor() {
        TestExpressionInterpreter.optimize(String.format("ARRAY[%s]", Joiner.on((String)", ").join(IntStream.range(0, 10000).mapToObj(i -> "(bound_long + " + i + ")").iterator())));
        TestExpressionInterpreter.optimize(String.format("ARRAY[%s]", Joiner.on((String)", ").join(IntStream.range(0, 10000).mapToObj(i -> "(bound_integer + " + i + ")").iterator())));
        TestExpressionInterpreter.optimize(String.format("ARRAY[%s]", Joiner.on((String)", ").join(IntStream.range(0, 10000).mapToObj(i -> "'" + i + "'").iterator())));
        TestExpressionInterpreter.optimize(String.format("ARRAY[%s]", Joiner.on((String)", ").join(IntStream.range(0, 10000).mapToObj(i -> "ARRAY['" + i + "']").iterator())));
    }

    @Test
    public void testArrayConstructor() {
        TestExpressionInterpreter.optimize("ARRAY[]");
        TestExpressionInterpreter.assertOptimizedEquals("ARRAY[(unbound_long + 0), (unbound_long + 1), (unbound_long + 2)]", "\"$array\"((unbound_long + 0), (unbound_long + 1), (unbound_long + 2))");
        TestExpressionInterpreter.assertOptimizedEquals("ARRAY[(bound_long + 0), (unbound_long + 1), (bound_long + 2)]", "\"$array\"((bound_long + 0), (unbound_long + 1), (bound_long + 2))");
        TestExpressionInterpreter.assertOptimizedEquals("ARRAY[(bound_long + 0), (unbound_long + 1), NULL]", "\"$array\"((bound_long + 0), (unbound_long + 1), NULL)");
    }

    @Test
    public void testRowConstructor() {
        TestExpressionInterpreter.optimize("ROW(NULL)");
        TestExpressionInterpreter.optimize("ROW(1)");
        TestExpressionInterpreter.optimize("ROW(unbound_long + 0)");
        TestExpressionInterpreter.optimize("ROW(unbound_long + unbound_long2, unbound_string, unbound_double)");
        TestExpressionInterpreter.optimize("ROW(unbound_boolean, FALSE, ARRAY[unbound_long, unbound_long2], unbound_null_string, unbound_interval)");
        TestExpressionInterpreter.optimize("ARRAY[ROW(unbound_string, unbound_double), ROW(unbound_string, 0.0E0)]");
        TestExpressionInterpreter.optimize("ARRAY[ROW('string', unbound_double), ROW('string', bound_double)]");
        TestExpressionInterpreter.optimize("ROW(ROW(NULL), ROW(ROW(ROW(ROW('rowception')))))");
        TestExpressionInterpreter.optimize("ROW(unbound_string, bound_string)");
        TestExpressionInterpreter.optimize("ARRAY[ROW(unbound_string, unbound_double), ROW(CAST(bound_string AS varchar), 0.0E0)]");
        TestExpressionInterpreter.optimize("ARRAY[ROW(CAST(bound_string AS varchar), 0.0E0), ROW(unbound_string, unbound_double)]");
        TestExpressionInterpreter.optimize("ARRAY[ROW(unbound_string, unbound_double), CAST(NULL AS row(varchar, double))]");
        TestExpressionInterpreter.optimize("ARRAY[CAST(NULL AS row(varchar, double)), ROW(unbound_string, unbound_double)]");
    }

    @Test
    public void testRowSubscript() {
        TestExpressionInterpreter.assertOptimizedEquals("ROW(1, 'a', true)[3]", "true");
        TestExpressionInterpreter.assertOptimizedEquals("ROW(1, 'a', ROW(2, 'b', ROW(3, 'c')))[3][3][2]", "'c'");
        TestExpressionInterpreter.assertOptimizedEquals("ROW(1, null)[2]", "null");
        TestExpressionInterpreter.assertOptimizedEquals("ROW(0 / 0, 1)[1]", "ROW(0 / 0, 1)[1]");
        TestExpressionInterpreter.assertOptimizedEquals("ROW(0 / 0, 1)[2]", "ROW(0 / 0, 1)[2]");
        TrinoExceptionAssert.assertTrinoExceptionThrownBy(() -> TestExpressionInterpreter.evaluate("ROW(0 / 0, 1)[1]")).hasErrorCode(new ErrorCodeSupplier[]{StandardErrorCode.DIVISION_BY_ZERO});
        TrinoExceptionAssert.assertTrinoExceptionThrownBy(() -> TestExpressionInterpreter.evaluate("ROW(0 / 0, 1)[2]")).hasErrorCode(new ErrorCodeSupplier[]{StandardErrorCode.DIVISION_BY_ZERO});
    }

    @Test
    public void testArraySubscriptConstantNegativeIndex() {
        TestExpressionInterpreter.assertOptimizedEquals("ARRAY[1, 2, 3][-1]", "ARRAY[1,2,3][CAST(-1 AS bigint)]");
        TrinoExceptionAssert.assertTrinoExceptionThrownBy(() -> TestExpressionInterpreter.evaluate("ARRAY[1, 2, 3][-1]")).hasErrorCode(new ErrorCodeSupplier[]{StandardErrorCode.INVALID_FUNCTION_ARGUMENT}).hasMessage("Array subscript is negative: -1");
    }

    @Test
    public void testArraySubscriptConstantZeroIndex() {
        TestExpressionInterpreter.assertOptimizedEquals("ARRAY[1, 2, 3][0]", "ARRAY[1,2,3][CAST(0 AS bigint)]");
        TrinoExceptionAssert.assertTrinoExceptionThrownBy(() -> TestExpressionInterpreter.evaluate("ARRAY[1, 2, 3][0]")).hasErrorCode(new ErrorCodeSupplier[]{StandardErrorCode.INVALID_FUNCTION_ARGUMENT}).hasMessage("SQL array indices start at 1");
    }

    @Test
    public void testMapSubscriptMissingKey() {
        TestExpressionInterpreter.assertOptimizedEquals("MAP(ARRAY[1, 2], ARRAY[3, 4])[-1]", "MAP(ARRAY[1, 2], ARRAY[3, 4])[-1]");
        TrinoExceptionAssert.assertTrinoExceptionThrownBy(() -> TestExpressionInterpreter.evaluate("MAP(ARRAY[1, 2], ARRAY[3, 4])[-1]")).hasErrorCode(new ErrorCodeSupplier[]{StandardErrorCode.INVALID_FUNCTION_ARGUMENT}).hasMessage("Key not present in map: -1");
    }

    @Test
    public void testMapSubscriptConstantIndexes() {
        TestExpressionInterpreter.optimize("MAP(ARRAY[1, 2], ARRAY[3, 4])[1]");
        TestExpressionInterpreter.optimize("MAP(ARRAY[BIGINT '1', 2], ARRAY[3, 4])[1]");
        TestExpressionInterpreter.optimize("MAP(ARRAY[1, 2], ARRAY[3, 4])[2]");
        TestExpressionInterpreter.optimize("MAP(ARRAY[ARRAY[1,1]], ARRAY['a'])[ARRAY[1,1]]");
    }

    @Test
    @Timeout(value=60L)
    public void testLikeInvalidUtf8() {
        TestExpressionInterpreter.assertLike(new byte[]{97, 98, 99}, "%b%", true);
        TestExpressionInterpreter.assertLike(new byte[]{97, 98, 99, -1, 120, 121}, "%b%", true);
    }

    @Test
    public void testLiterals() {
        TestExpressionInterpreter.optimize("DATE '2013-04-03' + unbound_interval");
        TestExpressionInterpreter.optimize("TIME '03:04:05.321' + unbound_interval");
        TestExpressionInterpreter.optimize("TIME '03:04:05.321+00:00' + unbound_interval");
        TestExpressionInterpreter.optimize("TIMESTAMP '2013-04-03 03:04:05.321' +     unbound_interval");
        TestExpressionInterpreter.optimize("TIMESTAMP '2013-04-03 03:04:05.321 UTC' + unbound_interval");
        TestExpressionInterpreter.optimize("INTERVAL '3' DAY * unbound_long");
        TestExpressionInterpreter.optimize("INTERVAL '3' YEAR * unbound_long");
        Assert.assertEquals((Object)TestExpressionInterpreter.optimize("X'1234'"), (Object)Slices.wrappedBuffer((byte[])new byte[]{18, 52}));
    }

    private static void assertLike(byte[] value, String pattern, boolean expected) {
        LikePredicate predicate = new LikePredicate((Expression)TestExpressionInterpreter.rawStringLiteral(Slices.wrappedBuffer((byte[])value)), (Expression)new StringLiteral(pattern), Optional.empty());
        Assert.assertEquals((Object)TestExpressionInterpreter.evaluate((Expression)predicate), (Object)expected);
    }

    private static StringLiteral rawStringLiteral(Slice slice) {
        return new StringLiteral(slice.toStringUtf8());
    }

    private static void assertOptimizedEquals(@Language(value="SQL") String actual, @Language(value="SQL") String expected) {
        Assert.assertEquals((Object)TestExpressionInterpreter.optimize(actual), (Object)TestExpressionInterpreter.optimize(expected));
    }

    private static void assertOptimizedMatches(@Language(value="SQL") String actual, @Language(value="SQL") String expected) {
        Expression actualOptimized = (Expression)TestExpressionInterpreter.optimize(actual);
        SymbolAliases.Builder aliases = SymbolAliases.builder().putAll((Map)SYMBOL_TYPES.allTypes().keySet().stream().map(Symbol::getName).collect(ImmutableMap.toImmutableMap(Function.identity(), SymbolReference::new)));
        Expression rewrittenExpected = ExpressionUtils.rewriteIdentifiersToSymbolReferences((Expression)SQL_PARSER.createExpression(expected));
        ExpressionTestUtils.assertExpressionEquals(actualOptimized, rewrittenExpected, aliases.build());
    }

    private static Object optimize(@Language(value="SQL") String expression) {
        TestExpressionInterpreter.assertRoundTrip(expression);
        return TestExpressionInterpreter.optimize(TestExpressionInterpreter.planExpression(expression));
    }

    static Object optimize(Expression parsedExpression) {
        Map<NodeRef<Expression>, Type> expressionTypes = ExpressionTestUtils.getTypes(SessionTestUtils.TEST_SESSION, PLANNER_CONTEXT, SYMBOL_TYPES, parsedExpression);
        ExpressionInterpreter interpreter = new ExpressionInterpreter(parsedExpression, PLANNER_CONTEXT, SessionTestUtils.TEST_SESSION, expressionTypes);
        return interpreter.optimize(INPUTS);
    }

    static Expression planExpression(@Language(value="SQL") String expression) {
        return (Expression)TransactionBuilder.transaction((TransactionManager)TRANSACTION_MANAGER, (Metadata)PLANNER_CONTEXT.getMetadata(), (AccessControl)new AllowAllAccessControl()).singleStatement().execute(SessionTestUtils.TEST_SESSION, transactionSession -> {
            Expression parsedExpression = SQL_PARSER.createExpression(expression);
            parsedExpression = ExpressionUtils.rewriteIdentifiersToSymbolReferences((Expression)parsedExpression);
            parsedExpression = ExpressionTestUtils.resolveFunctionCalls(PLANNER_CONTEXT, transactionSession, SYMBOL_TYPES, parsedExpression);
            parsedExpression = CanonicalizeExpressionRewriter.rewrite((Expression)parsedExpression, (Session)transactionSession, (PlannerContext)PLANNER_CONTEXT, (TypeAnalyzer)TypeAnalyzer.createTestingTypeAnalyzer((PlannerContext)PLANNER_CONTEXT), (TypeProvider)SYMBOL_TYPES);
            return parsedExpression;
        });
    }

    private static void assertEvaluatedEquals(@Language(value="SQL") String actual, @Language(value="SQL") String expected) {
        Assert.assertEquals((Object)TestExpressionInterpreter.evaluate(actual), (Object)TestExpressionInterpreter.evaluate(expected));
    }

    private static Object evaluate(String expression) {
        TestExpressionInterpreter.assertRoundTrip(expression);
        Expression parsedExpression = ExpressionTestUtils.createExpression(expression, TRANSACTION_MANAGER, PLANNER_CONTEXT, SYMBOL_TYPES);
        return TestExpressionInterpreter.evaluate(parsedExpression);
    }

    private static void assertRoundTrip(String expression) {
        Expression parsed = SQL_PARSER.createExpression(expression);
        String formatted = ExpressionFormatter.formatExpression((Expression)parsed);
        Assert.assertEquals((Object)parsed, (Object)SQL_PARSER.createExpression(formatted));
    }

    private static Object evaluate(Expression expression) {
        Map<NodeRef<Expression>, Type> expressionTypes = ExpressionTestUtils.getTypes(SessionTestUtils.TEST_SESSION, PLANNER_CONTEXT, SYMBOL_TYPES, expression);
        ExpressionInterpreter interpreter = new ExpressionInterpreter(expression, PLANNER_CONTEXT, SessionTestUtils.TEST_SESSION, expressionTypes);
        return interpreter.evaluate(INPUTS);
    }
}

