/*
 * Decompiled with CFR 0.152.
 */
package io.trino.memory;

import com.google.common.annotations.VisibleForTesting;
import com.google.common.base.MoreObjects;
import com.google.common.base.Preconditions;
import com.google.common.collect.ImmutableMap;
import com.google.common.util.concurrent.AbstractFuture;
import com.google.common.util.concurrent.ListenableFuture;
import com.google.errorprone.annotations.concurrent.GuardedBy;
import io.airlift.units.DataSize;
import io.trino.execution.TaskId;
import io.trino.memory.MemoryPoolListener;
import io.trino.operator.Operator;
import io.trino.spi.QueryId;
import io.trino.spi.memory.MemoryAllocation;
import io.trino.spi.memory.MemoryPoolInfo;
import jakarta.annotation.Nullable;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.concurrent.CopyOnWriteArrayList;
import org.weakref.jmx.Managed;

public class MemoryPool {
    private final long maxBytes;
    @GuardedBy(value="this")
    private long reservedBytes;
    @GuardedBy(value="this")
    private long reservedRevocableBytes;
    @Nullable
    @GuardedBy(value="this")
    private NonCancellableMemoryFuture<Void> future;
    @GuardedBy(value="this")
    private final Map<QueryId, Long> queryMemoryReservations = new HashMap<QueryId, Long>();
    @GuardedBy(value="this")
    private final Map<QueryId, Map<String, Long>> taggedMemoryAllocations = new HashMap<QueryId, Map<String, Long>>();
    @GuardedBy(value="this")
    private final Map<QueryId, Long> queryRevocableMemoryReservations = new HashMap<QueryId, Long>();
    @GuardedBy(value="this")
    private final Map<TaskId, Long> taskMemoryReservations = new HashMap<TaskId, Long>();
    @GuardedBy(value="this")
    private final Map<TaskId, Long> taskRevocableMemoryReservations = new HashMap<TaskId, Long>();
    private final List<MemoryPoolListener> listeners = new CopyOnWriteArrayList<MemoryPoolListener>();

    public MemoryPool(DataSize size) {
        Objects.requireNonNull(size, "size is null");
        this.maxBytes = size.toBytes();
    }

    public synchronized MemoryPoolInfo getInfo() {
        HashMap memoryAllocations = new HashMap();
        for (Map.Entry<QueryId, Map<String, Long>> entry2 : this.taggedMemoryAllocations.entrySet()) {
            ArrayList allocations = new ArrayList();
            if (entry2.getValue() != null) {
                entry2.getValue().forEach((tag, allocation) -> allocations.add(new MemoryAllocation(tag, allocation.longValue())));
            }
            memoryAllocations.put(entry2.getKey(), allocations);
        }
        Map stringKeyedTaskMemoryReservations = (Map)this.taskMemoryReservations.entrySet().stream().collect(ImmutableMap.toImmutableMap(entry -> ((TaskId)entry.getKey()).toString(), Map.Entry::getValue));
        Map stringKeyedTaskRevocableMemoryReservations = (Map)this.taskRevocableMemoryReservations.entrySet().stream().collect(ImmutableMap.toImmutableMap(entry -> ((TaskId)entry.getKey()).toString(), Map.Entry::getValue));
        return new MemoryPoolInfo(this.maxBytes, this.reservedBytes, this.reservedRevocableBytes, this.queryMemoryReservations, memoryAllocations, this.queryRevocableMemoryReservations, stringKeyedTaskMemoryReservations, stringKeyedTaskRevocableMemoryReservations);
    }

    public void addListener(MemoryPoolListener listener) {
        this.listeners.add(Objects.requireNonNull(listener, "listener cannot be null"));
    }

    public void removeListener(MemoryPoolListener listener) {
        this.listeners.remove(Objects.requireNonNull(listener, "listener cannot be null"));
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    public ListenableFuture<Void> reserve(TaskId taskId, String allocationTag, long bytes) {
        Object result;
        Preconditions.checkArgument((bytes >= 0L ? 1 : 0) != 0, (String)"'%s' is negative", (long)bytes);
        MemoryPool memoryPool = this;
        synchronized (memoryPool) {
            if (bytes != 0L) {
                QueryId queryId = taskId.getQueryId();
                this.queryMemoryReservations.merge(queryId, bytes, Long::sum);
                this.updateTaggedMemoryAllocations(queryId, allocationTag, bytes);
                this.taskMemoryReservations.merge(taskId, bytes, Long::sum);
            }
            this.reservedBytes += bytes;
            if (this.getFreeBytes() <= 0L) {
                if (this.future == null) {
                    this.future = NonCancellableMemoryFuture.create();
                }
                Preconditions.checkState((!this.future.isDone() ? 1 : 0) != 0, (Object)"future is already completed");
                result = this.future;
            } else {
                result = Operator.NOT_BLOCKED;
            }
        }
        this.onMemoryReserved();
        return result;
    }

    private void onMemoryReserved() {
        this.listeners.forEach(listener -> listener.onMemoryReserved(this));
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    public ListenableFuture<Void> reserveRevocable(TaskId taskId, long bytes) {
        Object result;
        Preconditions.checkArgument((bytes >= 0L ? 1 : 0) != 0, (String)"'%s' is negative", (long)bytes);
        MemoryPool memoryPool = this;
        synchronized (memoryPool) {
            if (bytes != 0L) {
                this.queryRevocableMemoryReservations.merge(taskId.getQueryId(), bytes, Long::sum);
                this.taskRevocableMemoryReservations.merge(taskId, bytes, Long::sum);
            }
            this.reservedRevocableBytes += bytes;
            if (this.getFreeBytes() <= 0L) {
                if (this.future == null) {
                    this.future = NonCancellableMemoryFuture.create();
                }
                Preconditions.checkState((!this.future.isDone() ? 1 : 0) != 0, (Object)"future is already completed");
                result = this.future;
            } else {
                result = Operator.NOT_BLOCKED;
            }
        }
        this.onMemoryReserved();
        return result;
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    public boolean tryReserve(TaskId taskId, String allocationTag, long bytes) {
        Preconditions.checkArgument((bytes >= 0L ? 1 : 0) != 0, (String)"'%s' is negative", (long)bytes);
        MemoryPool memoryPool = this;
        synchronized (memoryPool) {
            if (this.getFreeBytes() - bytes < 0L) {
                return false;
            }
            this.reservedBytes += bytes;
            if (bytes != 0L) {
                QueryId queryId = taskId.getQueryId();
                this.queryMemoryReservations.merge(queryId, bytes, Long::sum);
                this.updateTaggedMemoryAllocations(queryId, allocationTag, bytes);
                this.taskMemoryReservations.merge(taskId, bytes, Long::sum);
            }
        }
        this.onMemoryReserved();
        return true;
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    public boolean tryReserveRevocable(long bytes) {
        Preconditions.checkArgument((bytes >= 0L ? 1 : 0) != 0, (String)"'%s' is negative", (long)bytes);
        MemoryPool memoryPool = this;
        synchronized (memoryPool) {
            if (this.getFreeBytes() - bytes < 0L) {
                return false;
            }
            this.reservedRevocableBytes += bytes;
        }
        this.onMemoryReserved();
        return true;
    }

    public synchronized void free(TaskId taskId, String allocationTag, long bytes) {
        Preconditions.checkArgument((bytes >= 0L ? 1 : 0) != 0, (String)"'%s' is negative", (long)bytes);
        Preconditions.checkArgument((this.reservedBytes >= bytes ? 1 : 0) != 0, (Object)"tried to free more memory than is reserved");
        if (bytes == 0L) {
            return;
        }
        QueryId queryId = taskId.getQueryId();
        Long queryReservation = this.queryMemoryReservations.get(queryId);
        Objects.requireNonNull(queryReservation, "queryReservation is null");
        Preconditions.checkArgument((queryReservation >= bytes ? 1 : 0) != 0, (Object)"tried to free more memory than is reserved by query");
        Long taskReservation = this.taskMemoryReservations.get(taskId);
        Objects.requireNonNull(taskReservation, "taskReservation is null");
        Preconditions.checkArgument((taskReservation >= bytes ? 1 : 0) != 0, (Object)"tried to free more memory than is reserved by task");
        queryReservation = queryReservation - bytes;
        if (queryReservation == 0L) {
            this.queryMemoryReservations.remove(queryId);
            this.taggedMemoryAllocations.remove(queryId);
        } else {
            this.queryMemoryReservations.put(queryId, queryReservation);
            this.updateTaggedMemoryAllocations(queryId, allocationTag, -bytes);
        }
        taskReservation = taskReservation - bytes;
        if (taskReservation == 0L) {
            this.taskMemoryReservations.remove(taskId);
        } else {
            this.taskMemoryReservations.put(taskId, taskReservation);
        }
        this.reservedBytes -= bytes;
        if (this.getFreeBytes() > 0L && this.future != null) {
            this.future.set(null);
            this.future = null;
        }
    }

    public synchronized void freeRevocable(TaskId taskId, long bytes) {
        Preconditions.checkArgument((bytes >= 0L ? 1 : 0) != 0, (String)"'%s' is negative", (long)bytes);
        Preconditions.checkArgument((this.reservedRevocableBytes >= bytes ? 1 : 0) != 0, (Object)"tried to free more revocable memory than is reserved");
        if (bytes == 0L) {
            return;
        }
        QueryId queryId = taskId.getQueryId();
        Long queryReservation = this.queryRevocableMemoryReservations.get(queryId);
        Objects.requireNonNull(queryReservation, "queryReservation is null");
        Preconditions.checkArgument((queryReservation >= bytes ? 1 : 0) != 0, (Object)"tried to free more revocable memory than is reserved by query");
        Long taskReservation = this.taskRevocableMemoryReservations.get(taskId);
        Objects.requireNonNull(taskReservation, "taskReservation is null");
        Preconditions.checkArgument((taskReservation >= bytes ? 1 : 0) != 0, (Object)"tried to free more revocable memory than is reserved by task");
        queryReservation = queryReservation - bytes;
        if (queryReservation == 0L) {
            this.queryRevocableMemoryReservations.remove(queryId);
        } else {
            this.queryRevocableMemoryReservations.put(queryId, queryReservation);
        }
        taskReservation = taskReservation - bytes;
        if (taskReservation == 0L) {
            this.taskRevocableMemoryReservations.remove(taskId);
        } else {
            this.taskRevocableMemoryReservations.put(taskId, taskReservation);
        }
        this.reservedRevocableBytes -= bytes;
        if (this.getFreeBytes() > 0L && this.future != null) {
            this.future.set(null);
            this.future = null;
        }
    }

    public synchronized void freeRevocable(long bytes) {
        Preconditions.checkArgument((bytes >= 0L ? 1 : 0) != 0, (String)"'%s' is negative", (long)bytes);
        Preconditions.checkArgument((this.reservedRevocableBytes >= bytes ? 1 : 0) != 0, (Object)"tried to free more revocable memory than is reserved");
        if (bytes == 0L) {
            return;
        }
        this.reservedRevocableBytes -= bytes;
        if (this.getFreeBytes() > 0L && this.future != null) {
            this.future.set(null);
            this.future = null;
        }
    }

    @Managed
    public synchronized long getFreeBytes() {
        return this.maxBytes - this.reservedBytes - this.reservedRevocableBytes;
    }

    @Managed
    public long getMaxBytes() {
        return this.maxBytes;
    }

    @Managed
    public synchronized long getReservedBytes() {
        return this.reservedBytes;
    }

    @Managed
    public synchronized long getReservedRevocableBytes() {
        return this.reservedRevocableBytes;
    }

    synchronized long getQueryMemoryReservation(QueryId queryId) {
        return this.queryMemoryReservations.getOrDefault(queryId, 0L);
    }

    @VisibleForTesting
    synchronized long getQueryRevocableMemoryReservation(QueryId queryId) {
        return this.queryRevocableMemoryReservations.getOrDefault(queryId, 0L);
    }

    @VisibleForTesting
    synchronized long getTaskMemoryReservation(TaskId taskId) {
        return this.taskMemoryReservations.getOrDefault(taskId, 0L);
    }

    @VisibleForTesting
    synchronized long getTaskRevocableMemoryReservation(TaskId taskId) {
        return this.taskRevocableMemoryReservations.getOrDefault(taskId, 0L);
    }

    public synchronized String toString() {
        return MoreObjects.toStringHelper((Object)this).add("maxBytes", this.maxBytes).add("freeBytes", this.getFreeBytes()).add("reservedBytes", this.reservedBytes).add("reservedRevocableBytes", this.reservedRevocableBytes).add("future", this.future).toString();
    }

    private synchronized void updateTaggedMemoryAllocations(QueryId queryId, String allocationTag, long delta) {
        if (delta == 0L) {
            return;
        }
        Map allocations = this.taggedMemoryAllocations.computeIfAbsent(queryId, ignored -> new HashMap());
        allocations.compute(allocationTag, (ignored, oldValue) -> {
            if (oldValue == null) {
                return delta;
            }
            long newValue = oldValue + delta;
            if (newValue == 0L) {
                return null;
            }
            return newValue;
        });
    }

    @VisibleForTesting
    public synchronized Map<QueryId, Long> getQueryMemoryReservations() {
        return ImmutableMap.copyOf(this.queryMemoryReservations);
    }

    @VisibleForTesting
    public synchronized Map<QueryId, Map<String, Long>> getTaggedMemoryAllocations() {
        return ImmutableMap.copyOf(this.taggedMemoryAllocations);
    }

    @VisibleForTesting
    public synchronized Map<QueryId, Long> getQueryRevocableMemoryReservations() {
        return ImmutableMap.copyOf(this.queryRevocableMemoryReservations);
    }

    @VisibleForTesting
    public synchronized Map<TaskId, Long> getTaskMemoryReservations() {
        return ImmutableMap.copyOf(this.taskMemoryReservations);
    }

    @VisibleForTesting
    public synchronized Map<TaskId, Long> getTaskRevocableMemoryReservations() {
        return ImmutableMap.copyOf(this.taskRevocableMemoryReservations);
    }

    private static class NonCancellableMemoryFuture<V>
    extends AbstractFuture<V> {
        private NonCancellableMemoryFuture() {
        }

        public static <V> NonCancellableMemoryFuture<V> create() {
            return new NonCancellableMemoryFuture<V>();
        }

        public boolean set(@Nullable V value) {
            return super.set(value);
        }

        public boolean cancel(boolean mayInterruptIfRunning) {
            throw new UnsupportedOperationException("cancellation is not supported");
        }
    }
}

