/*
 * Decompiled with CFR 0.152.
 */
package io.trino.operator.scalar;

import com.google.common.net.InetAddresses;
import com.google.common.primitives.Ints;
import io.airlift.slice.Slice;
import io.trino.spi.ErrorCodeSupplier;
import io.trino.spi.StandardErrorCode;
import io.trino.spi.TrinoException;
import io.trino.spi.function.Description;
import io.trino.spi.function.ScalarFunction;
import io.trino.spi.function.SqlType;
import java.math.BigInteger;
import java.net.Inet4Address;
import java.net.InetAddress;
import java.util.regex.Pattern;

public final class IpAddressFunctions {
    private static final Pattern IPV4_PATTERN = Pattern.compile("^\\d{1,3}\\.\\d{1,3}\\.\\d{1,3}\\.\\d{1,3}$");

    private IpAddressFunctions() {
    }

    @Description(value="Determines whether given IP address exists in the CIDR")
    @ScalarFunction
    @SqlType(value="boolean")
    public static boolean contains(@SqlType(value="varchar") Slice network, @SqlType(value="ipaddress") Slice address) {
        InetAddress cidrAddress;
        String cidr = network.toStringUtf8();
        int separator = cidr.indexOf("/");
        if (separator == -1) {
            throw new TrinoException((ErrorCodeSupplier)StandardErrorCode.INVALID_FUNCTION_ARGUMENT, "Invalid CIDR");
        }
        String cidrBase = cidr.substring(0, separator);
        try {
            cidrAddress = InetAddresses.forString((String)cidrBase);
        }
        catch (IllegalArgumentException e) {
            throw new TrinoException((ErrorCodeSupplier)StandardErrorCode.INVALID_FUNCTION_ARGUMENT, "Invalid network IP address");
        }
        byte[] cidrBytes = IpAddressFunctions.toBytes(cidrAddress);
        int prefixLength = Integer.parseInt(cidr.substring(separator + 1));
        if (prefixLength < 0) {
            throw new TrinoException((ErrorCodeSupplier)StandardErrorCode.INVALID_FUNCTION_ARGUMENT, "Invalid prefix length");
        }
        if (IPV4_PATTERN.matcher(cidrBase).matches()) {
            if (!IpAddressFunctions.isValidIpV4Cidr(cidrBytes, 12, prefixLength)) {
                throw new TrinoException((ErrorCodeSupplier)StandardErrorCode.INVALID_FUNCTION_ARGUMENT, "Invalid CIDR");
            }
            prefixLength += 96;
        } else if (!IpAddressFunctions.isValidIpV6Cidr(prefixLength)) {
            throw new TrinoException((ErrorCodeSupplier)StandardErrorCode.INVALID_FUNCTION_ARGUMENT, "Invalid CIDR");
        }
        if (prefixLength == 0) {
            return true;
        }
        byte[] ipAddress = address.getBytes();
        BigInteger cidrPrefix = new BigInteger(cidrBytes).shiftRight(cidrBytes.length * 8 - prefixLength);
        BigInteger addressPrefix = new BigInteger(ipAddress).shiftRight(ipAddress.length * 8 - prefixLength);
        return cidrPrefix.equals(addressPrefix);
    }

    private static boolean isValidIpV6Cidr(int prefixLength) {
        return prefixLength >= 0 && prefixLength <= 128;
    }

    private static boolean isValidIpV4Cidr(byte[] address, int offset, int prefix) {
        if (prefix < 0 || prefix > 32) {
            return false;
        }
        long mask = 0xFFFFFFFFL >>> prefix;
        return ((long)Ints.fromBytes((byte)address[offset], (byte)address[offset + 1], (byte)address[offset + 2], (byte)address[offset + 3]) & mask) == 0L;
    }

    private static byte[] toBytes(InetAddress address) {
        byte[] bytes = address.getAddress();
        if (address instanceof Inet4Address) {
            byte[] temp = new byte[16];
            temp[10] = -1;
            temp[11] = -1;
            System.arraycopy(bytes, 0, temp, 12, 4);
            bytes = temp;
        }
        return bytes;
    }
}

