/*
 * Decompiled with CFR 0.152.
 */
package io.trino.server;

import com.google.common.collect.ImmutableList;
import com.google.common.util.concurrent.Uninterruptibles;
import com.google.errorprone.annotations.concurrent.GuardedBy;
import com.google.inject.Inject;
import io.airlift.bootstrap.LifeCycleManager;
import io.airlift.concurrent.Threads;
import io.airlift.log.Logger;
import io.airlift.units.Duration;
import io.trino.execution.SqlTaskManager;
import io.trino.execution.TaskInfo;
import io.trino.server.ServerConfig;
import io.trino.server.ShutdownAction;
import java.util.List;
import java.util.Objects;
import java.util.concurrent.CountDownLatch;
import java.util.concurrent.ExecutionException;
import java.util.concurrent.ExecutorService;
import java.util.concurrent.Executors;
import java.util.concurrent.Future;
import java.util.concurrent.ScheduledExecutorService;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.TimeoutException;

public class GracefulShutdownHandler {
    private static final Logger log = Logger.get(GracefulShutdownHandler.class);
    private static final Duration LIFECYCLE_STOP_TIMEOUT = new Duration(30.0, TimeUnit.SECONDS);
    private final ScheduledExecutorService shutdownHandler = Executors.newSingleThreadScheduledExecutor(Threads.threadsNamed((String)"shutdown-handler-%s"));
    private final ExecutorService lifeCycleStopper = Executors.newSingleThreadExecutor(Threads.threadsNamed((String)"lifecycle-stopper-%s"));
    private final LifeCycleManager lifeCycleManager;
    private final SqlTaskManager sqlTaskManager;
    private final boolean isCoordinator;
    private final ShutdownAction shutdownAction;
    private final Duration gracePeriod;
    @GuardedBy(value="this")
    private boolean shutdownRequested;

    @Inject
    public GracefulShutdownHandler(SqlTaskManager sqlTaskManager, ServerConfig serverConfig, ShutdownAction shutdownAction, LifeCycleManager lifeCycleManager) {
        this.sqlTaskManager = Objects.requireNonNull(sqlTaskManager, "sqlTaskManager is null");
        this.shutdownAction = Objects.requireNonNull(shutdownAction, "shutdownAction is null");
        this.lifeCycleManager = Objects.requireNonNull(lifeCycleManager, "lifeCycleManager is null");
        this.isCoordinator = serverConfig.isCoordinator();
        this.gracePeriod = serverConfig.getGracePeriod();
    }

    public synchronized void requestShutdown() {
        log.info("Shutdown requested");
        if (this.isCoordinator) {
            throw new UnsupportedOperationException("Cannot shutdown coordinator");
        }
        if (this.shutdownRequested) {
            return;
        }
        this.shutdownRequested = true;
        this.shutdownHandler.schedule(this::shutdown, this.gracePeriod.toMillis(), TimeUnit.MILLISECONDS);
    }

    private void shutdown() {
        List<TaskInfo> activeTasks = this.getActiveTasks();
        while (activeTasks.size() > 0) {
            CountDownLatch countDownLatch = new CountDownLatch(activeTasks.size());
            for (TaskInfo taskInfo : activeTasks) {
                this.sqlTaskManager.addStateChangeListener(taskInfo.getTaskStatus().getTaskId(), newState -> {
                    if (newState.isDone()) {
                        countDownLatch.countDown();
                    }
                });
            }
            log.info("Waiting for all tasks to finish");
            try {
                countDownLatch.await();
            }
            catch (InterruptedException e) {
                log.warn("Interrupted while waiting for all tasks to finish");
                Thread.currentThread().interrupt();
            }
            activeTasks = this.getActiveTasks();
        }
        Uninterruptibles.sleepUninterruptibly((long)this.gracePeriod.toMillis(), (TimeUnit)TimeUnit.MILLISECONDS);
        Future<Object> shutdownFuture = this.lifeCycleStopper.submit(() -> {
            this.lifeCycleManager.stop();
            return null;
        });
        try {
            shutdownFuture.get(LIFECYCLE_STOP_TIMEOUT.toMillis(), TimeUnit.MILLISECONDS);
        }
        catch (TimeoutException e) {
            log.warn((Throwable)e, "Timed out waiting for the life cycle to stop");
        }
        catch (InterruptedException e) {
            log.warn((Throwable)e, "Interrupted while waiting for the life cycle to stop");
            Thread.currentThread().interrupt();
        }
        catch (ExecutionException e) {
            log.warn((Throwable)e, "Problem stopping the life cycle");
        }
        this.shutdownAction.onShutdown();
    }

    private List<TaskInfo> getActiveTasks() {
        return (List)this.sqlTaskManager.getAllTaskInfo().stream().filter(taskInfo -> !taskInfo.getTaskStatus().getState().isDone()).collect(ImmutableList.toImmutableList());
    }

    public synchronized boolean isShutdownRequested() {
        return this.shutdownRequested;
    }
}

