/*
 * Decompiled with CFR 0.152.
 */
package io.trino.server.security.oauth2;

import com.nimbusds.jose.KeyLengthException;
import io.airlift.units.Duration;
import io.jsonwebtoken.ClaimsBuilder;
import io.jsonwebtoken.ExpiredJwtException;
import io.jsonwebtoken.Jwts;
import io.trino.server.security.oauth2.JweTokenSerializer;
import io.trino.server.security.oauth2.OAuth2Client;
import io.trino.server.security.oauth2.RefreshTokensConfig;
import io.trino.server.security.oauth2.TokenPairSerializer;
import java.net.URI;
import java.security.GeneralSecurityException;
import java.security.SecureRandom;
import java.time.Clock;
import java.time.Instant;
import java.time.ZoneId;
import java.time.ZonedDateTime;
import java.time.temporal.ChronoUnit;
import java.util.Base64;
import java.util.Calendar;
import java.util.Date;
import java.util.Map;
import java.util.Optional;
import java.util.Random;
import java.util.concurrent.TimeUnit;
import org.assertj.core.api.AbstractThrowableAssert;
import org.assertj.core.api.Assertions;
import org.testng.annotations.DataProvider;
import org.testng.annotations.Test;

public class TestJweTokenSerializer {
    @Test
    public void testSerialization() throws Exception {
        JweTokenSerializer serializer = this.tokenSerializer(Clock.systemUTC(), Duration.succinctDuration((double)5.0, (TimeUnit)TimeUnit.SECONDS), TestJweTokenSerializer.randomEncodedSecret());
        Date expiration = new Calendar.Builder().setDate(2022, 6, 22).build().getTime();
        String serializedTokenPair = serializer.serialize(TokenPairSerializer.TokenPair.withAccessAndRefreshTokens((String)"access_token", (Date)expiration, (String)"refresh_token"));
        TokenPairSerializer.TokenPair deserializedTokenPair = serializer.deserialize(serializedTokenPair);
        Assertions.assertThat((String)deserializedTokenPair.accessToken()).isEqualTo("access_token");
        Assertions.assertThat((Date)deserializedTokenPair.expiration()).isEqualTo((Object)expiration);
        Assertions.assertThat((Optional)deserializedTokenPair.refreshToken()).isEqualTo(Optional.of("refresh_token"));
    }

    @Test(dataProvider="wrongSecretsProvider")
    public void testDeserializationWithWrongSecret(String encryptionSecret, String decryptionSecret) {
        ((AbstractThrowableAssert)Assertions.assertThatThrownBy(() -> this.assertRoundTrip(Optional.ofNullable(encryptionSecret), Optional.ofNullable(decryptionSecret))).isInstanceOf(RuntimeException.class)).hasMessageContaining("decryption failed: Tag mismatch");
    }

    @DataProvider
    public Object[][] wrongSecretsProvider() {
        return new Object[][]{{TestJweTokenSerializer.randomEncodedSecret(), TestJweTokenSerializer.randomEncodedSecret()}, {TestJweTokenSerializer.randomEncodedSecret(16), TestJweTokenSerializer.randomEncodedSecret(24)}, {null, null}, {null, TestJweTokenSerializer.randomEncodedSecret()}, {TestJweTokenSerializer.randomEncodedSecret(), null}};
    }

    @Test
    public void testSerializationDeserializationRoundTripWithDifferentKeyLengths() throws Exception {
        for (int keySize : new int[]{16, 24, 32}) {
            String secret = TestJweTokenSerializer.randomEncodedSecret(keySize);
            this.assertRoundTrip(secret, secret);
        }
    }

    @Test
    public void testSerializationFailsWithWrongKeySize() {
        for (int wrongKeySize : new int[]{8, 64, 128}) {
            String tooShortSecret = TestJweTokenSerializer.randomEncodedSecret(wrongKeySize);
            Assertions.assertThatThrownBy(() -> this.assertRoundTrip(tooShortSecret, tooShortSecret)).hasStackTraceContaining("Secret key size must be either 16, 24 or 32 bytes but was " + wrongKeySize);
        }
    }

    private void assertRoundTrip(String serializerSecret, String deserializerSecret) throws Exception {
        this.assertRoundTrip(Optional.of(serializerSecret), Optional.of(deserializerSecret));
    }

    private void assertRoundTrip(Optional<String> serializerSecret, Optional<String> deserializerSecret) throws Exception {
        JweTokenSerializer serializer = this.tokenSerializer(Clock.systemUTC(), Duration.succinctDuration((double)5.0, (TimeUnit)TimeUnit.SECONDS), serializerSecret);
        JweTokenSerializer deserializer = this.tokenSerializer(Clock.systemUTC(), Duration.succinctDuration((double)5.0, (TimeUnit)TimeUnit.SECONDS), deserializerSecret);
        Date expiration = new Calendar.Builder().setDate(2023, 6, 22).build().getTime();
        TokenPairSerializer.TokenPair tokenPair = TokenPairSerializer.TokenPair.withAccessAndRefreshTokens((String)TestJweTokenSerializer.randomEncodedSecret(), (Date)expiration, (String)TestJweTokenSerializer.randomEncodedSecret());
        Assertions.assertThat((Object)deserializer.deserialize(serializer.serialize(tokenPair))).isEqualTo((Object)tokenPair);
    }

    @Test
    public void testTokenDeserializationAfterTimeoutButBeforeExpirationExtension() throws Exception {
        TestingClock clock = new TestingClock();
        JweTokenSerializer serializer = this.tokenSerializer((Clock)clock, Duration.succinctDuration((double)12.0, (TimeUnit)TimeUnit.MINUTES), TestJweTokenSerializer.randomEncodedSecret());
        Date expiration = new Calendar.Builder().setDate(2022, 6, 22).build().getTime();
        String serializedTokenPair = serializer.serialize(TokenPairSerializer.TokenPair.withAccessAndRefreshTokens((String)"access_token", (Date)expiration, (String)"refresh_token"));
        clock.advanceBy(Duration.succinctDuration((double)10.0, (TimeUnit)TimeUnit.MINUTES));
        TokenPairSerializer.TokenPair deserializedTokenPair = serializer.deserialize(serializedTokenPair);
        Assertions.assertThat((String)deserializedTokenPair.accessToken()).isEqualTo("access_token");
        Assertions.assertThat((Date)deserializedTokenPair.expiration()).isEqualTo((Object)expiration);
        Assertions.assertThat((Optional)deserializedTokenPair.refreshToken()).isEqualTo(Optional.of("refresh_token"));
    }

    @Test
    public void testTokenDeserializationAfterTimeoutAndExpirationExtension() throws Exception {
        TestingClock clock = new TestingClock();
        JweTokenSerializer serializer = this.tokenSerializer((Clock)clock, Duration.succinctDuration((double)12.0, (TimeUnit)TimeUnit.MINUTES), TestJweTokenSerializer.randomEncodedSecret());
        Date expiration = new Calendar.Builder().setDate(2022, 6, 22).build().getTime();
        String serializedTokenPair = serializer.serialize(TokenPairSerializer.TokenPair.withAccessAndRefreshTokens((String)"access_token", (Date)expiration, (String)"refresh_token"));
        clock.advanceBy(Duration.succinctDuration((double)20.0, (TimeUnit)TimeUnit.MINUTES));
        Assertions.assertThatThrownBy(() -> serializer.deserialize(serializedTokenPair)).isExactlyInstanceOf(ExpiredJwtException.class);
    }

    @Test
    public void testTokenDeserializationWhenNonJWETokenIsPassed() throws Exception {
        JweTokenSerializer serializer = this.tokenSerializer((Clock)new TestingClock(), Duration.succinctDuration((double)12.0, (TimeUnit)TimeUnit.MINUTES), TestJweTokenSerializer.randomEncodedSecret());
        String nonJWEToken = "non_jwe_token";
        TokenPairSerializer.TokenPair tokenPair = serializer.deserialize(nonJWEToken);
        Assertions.assertThat((String)tokenPair.accessToken()).isEqualTo(nonJWEToken);
        Assertions.assertThat((Optional)tokenPair.refreshToken()).isEmpty();
    }

    private JweTokenSerializer tokenSerializer(Clock clock, Duration tokenExpiration, String encodedSecretKey) throws GeneralSecurityException, KeyLengthException {
        return this.tokenSerializer(clock, tokenExpiration, Optional.of(encodedSecretKey));
    }

    private JweTokenSerializer tokenSerializer(Clock clock, Duration tokenExpiration, Optional<String> secretKey) throws GeneralSecurityException, KeyLengthException {
        RefreshTokensConfig refreshTokensConfig = new RefreshTokensConfig();
        secretKey.ifPresent(arg_0 -> ((RefreshTokensConfig)refreshTokensConfig).setSecretKey(arg_0));
        return new JweTokenSerializer(refreshTokensConfig, (OAuth2Client)new Oauth2ClientStub(), "trino_coordinator_test_version", "trino_coordinator", "sub", clock, tokenExpiration);
    }

    private static String randomEncodedSecret() {
        return TestJweTokenSerializer.randomEncodedSecret(24);
    }

    private static String randomEncodedSecret(int length) {
        SecureRandom random = new SecureRandom();
        byte[] buffer = new byte[length];
        ((Random)random).nextBytes(buffer);
        return Base64.getEncoder().encodeToString(buffer);
    }

    private static class TestingClock
    extends Clock {
        private Instant currentTime = ZonedDateTime.of(2022, 5, 6, 10, 15, 0, 0, ZoneId.systemDefault()).toInstant();

        private TestingClock() {
        }

        @Override
        public ZoneId getZone() {
            return ZoneId.systemDefault();
        }

        @Override
        public Clock withZone(ZoneId zone) {
            return this;
        }

        @Override
        public Instant instant() {
            return this.currentTime;
        }

        public void advanceBy(Duration currentTimeDelta) {
            this.currentTime = this.currentTime.plus(currentTimeDelta.toMillis(), ChronoUnit.MILLIS);
        }
    }

    static class Oauth2ClientStub
    implements OAuth2Client {
        private final Map<String, Object> claims = (Map)((ClaimsBuilder)Jwts.claims().subject("user")).build();

        Oauth2ClientStub() {
        }

        public void load() {
        }

        public OAuth2Client.Request createAuthorizationRequest(String state, URI callbackUri) {
            throw new UnsupportedOperationException("operation is not yet supported");
        }

        public OAuth2Client.Response getOAuth2Response(String code, URI callbackUri, Optional<String> nonce) {
            throw new UnsupportedOperationException("operation is not yet supported");
        }

        public Optional<Map<String, Object>> getClaims(String accessToken) {
            return Optional.of(this.claims);
        }

        public OAuth2Client.Response refreshTokens(String refreshToken) {
            throw new UnsupportedOperationException("operation is not yet supported");
        }

        public Optional<URI> getLogoutEndpoint(Optional<String> idToken, URI callbackUrl) {
            return Optional.empty();
        }
    }
}

