/*
 * Decompiled with CFR 0.152.
 */
package io.trino.operator;

import com.google.common.base.Preconditions;
import com.google.common.base.Throwables;
import com.google.common.base.Verify;
import io.airlift.slice.SizeOf;
import io.trino.operator.FlatHashStrategy;
import io.trino.operator.UpdateMemory;
import io.trino.operator.VariableWidthData;
import io.trino.spi.ErrorCodeSupplier;
import io.trino.spi.StandardErrorCode;
import io.trino.spi.TrinoException;
import io.trino.spi.block.Block;
import io.trino.spi.block.BlockBuilder;
import io.trino.spi.type.BigintType;
import java.lang.invoke.MethodHandles;
import java.lang.invoke.VarHandle;
import java.nio.ByteOrder;

public final class FlatHash {
    private static final int INSTANCE_SIZE = SizeOf.instanceSize(FlatHash.class);
    private static final double DEFAULT_LOAD_FACTOR = 0.9375;
    private static final int RECORDS_PER_GROUP_SHIFT = 10;
    private static final int RECORDS_PER_GROUP = 1024;
    private static final int RECORDS_PER_GROUP_MASK = 1023;
    private static final int VECTOR_LENGTH = 8;
    private static final VarHandle LONG_HANDLE = MethodHandles.byteArrayViewVarHandle(long[].class, ByteOrder.LITTLE_ENDIAN);
    private static final VarHandle INT_HANDLE = MethodHandles.byteArrayViewVarHandle(int[].class, ByteOrder.LITTLE_ENDIAN);
    private final FlatHashStrategy flatHashStrategy;
    private final boolean hasPrecomputedHash;
    private final int recordSize;
    private final int recordGroupIdOffset;
    private final int recordHashOffset;
    private final int recordValueOffset;
    private int capacity;
    private int mask;
    private byte[] control;
    private byte[][] recordGroups;
    private final VariableWidthData variableWidthData;
    private int[] groupRecordIndex;
    private final UpdateMemory checkMemoryReservation;
    private long fixedSizeEstimate;
    private long rehashMemoryReservation;
    private int nextGroupId;
    private int maxFill;

    private static int computeCapacity(int maxSize, double loadFactor) {
        int capacity = (int)((double)maxSize / loadFactor);
        return Math.max(Math.toIntExact(1L << 64 - Long.numberOfLeadingZeros(capacity - 1)), 16);
    }

    public FlatHash(FlatHashStrategy flatHashStrategy, boolean hasPrecomputedHash, int expectedSize, UpdateMemory checkMemoryReservation) {
        this.flatHashStrategy = flatHashStrategy;
        this.hasPrecomputedHash = hasPrecomputedHash;
        this.checkMemoryReservation = checkMemoryReservation;
        this.capacity = Math.max(8, FlatHash.computeCapacity(expectedSize, 0.9375));
        this.maxFill = FlatHash.calculateMaxFill(this.capacity);
        this.mask = this.capacity - 1;
        this.control = new byte[this.capacity + 8];
        this.groupRecordIndex = new int[this.maxFill];
        boolean variableWidth = flatHashStrategy.isAnyVariableWidth();
        this.variableWidthData = variableWidth ? new VariableWidthData() : null;
        this.recordGroupIdOffset = variableWidth ? 12 : 0;
        this.recordHashOffset = this.recordGroupIdOffset + 4;
        this.recordValueOffset = this.recordHashOffset + (hasPrecomputedHash ? 8 : 0);
        this.recordSize = this.recordValueOffset + flatHashStrategy.getTotalFlatFixedLength();
        this.recordGroups = FlatHash.createRecordGroups(this.capacity, this.recordSize);
        this.fixedSizeEstimate = FlatHash.computeFixedSizeEstimate(this.capacity, this.recordSize);
    }

    public long getEstimatedSize() {
        return FlatHash.sumExact(this.fixedSizeEstimate, this.variableWidthData == null ? 0L : this.variableWidthData.getRetainedSizeBytes(), this.rehashMemoryReservation);
    }

    public int size() {
        return this.nextGroupId;
    }

    public int getCapacity() {
        return this.capacity;
    }

    public long hashPosition(int groupId) {
        Preconditions.checkArgument((groupId < this.nextGroupId ? 1 : 0) != 0, (Object)"groupId out of range");
        int index = this.groupRecordIndex[groupId];
        byte[] records = this.getRecords(index);
        if (this.hasPrecomputedHash) {
            return LONG_HANDLE.get(records, this.getRecordOffset(index) + this.recordHashOffset);
        }
        return this.valueHashCode(records, index);
    }

    public void appendTo(int groupId, BlockBuilder[] blockBuilders) {
        Preconditions.checkArgument((groupId < this.nextGroupId ? 1 : 0) != 0, (Object)"groupId out of range");
        int index = this.groupRecordIndex[groupId];
        byte[] records = this.getRecords(index);
        int recordOffset = this.getRecordOffset(index);
        byte[] variableWidthChunk = VariableWidthData.EMPTY_CHUNK;
        if (this.variableWidthData != null) {
            variableWidthChunk = this.variableWidthData.getChunk(records, recordOffset);
        }
        this.flatHashStrategy.readFlat(records, recordOffset + this.recordValueOffset, variableWidthChunk, blockBuilders);
        if (this.hasPrecomputedHash) {
            BigintType.BIGINT.writeLong(blockBuilders[blockBuilders.length - 1], LONG_HANDLE.get(records, recordOffset + this.recordHashOffset));
        }
    }

    public boolean contains(Block[] blocks, int position) {
        return this.contains(blocks, position, this.flatHashStrategy.hash(blocks, position));
    }

    public boolean contains(Block[] blocks, int position, long hash) {
        return this.getIndex(blocks, position, hash) >= 0;
    }

    public void computeHashes(Block[] blocks, long[] hashes, int offset, int length) {
        if (this.hasPrecomputedHash) {
            Block hashBlock = blocks[blocks.length - 1];
            for (int i = 0; i < length; ++i) {
                hashes[i] = BigintType.BIGINT.getLong(hashBlock, offset + i);
            }
        } else {
            this.flatHashStrategy.hashBlocksBatched(blocks, hashes, offset, length);
        }
    }

    public int putIfAbsent(Block[] blocks, int position, long hash) {
        int index = this.getIndex(blocks, position, hash);
        if (index >= 0) {
            return INT_HANDLE.get(this.getRecords(index), this.getRecordOffset(index) + this.recordGroupIdOffset);
        }
        index = -index - 1;
        int groupId = this.addNewGroup(index, blocks, position, hash);
        if (this.nextGroupId >= this.maxFill) {
            this.rehash(0);
        }
        return groupId;
    }

    public int putIfAbsent(Block[] blocks, int position) {
        long hash = this.hasPrecomputedHash ? BigintType.BIGINT.getLong(blocks[blocks.length - 1], position) : this.flatHashStrategy.hash(blocks, position);
        return this.putIfAbsent(blocks, position, hash);
    }

    private int getIndex(Block[] blocks, int position, long hash) {
        byte hashPrefix = (byte)(hash & 0x7FL | 0x80L);
        int bucket = this.bucket((int)(hash >> 7));
        int step = 1;
        long repeated = FlatHash.repeat(hashPrefix);
        long controlVector;
        int matchIndex;
        while ((matchIndex = this.matchInVector(blocks, position, hash, bucket, repeated, controlVector = LONG_HANDLE.get(this.control, bucket))) < 0) {
            int emptyIndex = this.findEmptyInVector(controlVector, bucket);
            if (emptyIndex >= 0) {
                return -emptyIndex - 1;
            }
            bucket = this.bucket(bucket + step);
            step += 8;
        }
        return matchIndex;
    }

    private int matchInVector(Block[] blocks, int position, long hash, int vectorStartBucket, long repeated, long controlVector) {
        for (long controlMatches = FlatHash.match(controlVector, repeated); controlMatches != 0L; controlMatches &= controlMatches - 1L) {
            int index = this.bucket(vectorStartBucket + (Long.numberOfTrailingZeros(controlMatches) >>> 3));
            if (!this.valueNotDistinctFrom(index, blocks, position, hash)) continue;
            return index;
        }
        return -1;
    }

    private int findEmptyInVector(long vector, int vectorStartBucket) {
        long controlMatches = FlatHash.match(vector, 0L);
        if (controlMatches == 0L) {
            return -1;
        }
        int slot = Long.numberOfTrailingZeros(controlMatches) >>> 3;
        return this.bucket(vectorStartBucket + slot);
    }

    private int addNewGroup(int index, Block[] blocks, int position, long hash) {
        this.setControl(index, (byte)(hash & 0x7FL | 0x80L));
        byte[] records = this.getRecords(index);
        int recordOffset = this.getRecordOffset(index);
        int groupId = this.nextGroupId++;
        INT_HANDLE.set(records, recordOffset + this.recordGroupIdOffset, groupId);
        this.groupRecordIndex[groupId] = index;
        if (this.hasPrecomputedHash) {
            LONG_HANDLE.set(records, recordOffset + this.recordHashOffset, hash);
        }
        byte[] variableWidthChunk = VariableWidthData.EMPTY_CHUNK;
        int variableWidthChunkOffset = 0;
        if (this.variableWidthData != null) {
            int variableWidthSize = this.flatHashStrategy.getTotalVariableWidth(blocks, position);
            variableWidthChunk = this.variableWidthData.allocate(records, recordOffset, variableWidthSize);
            variableWidthChunkOffset = VariableWidthData.getChunkOffset(records, recordOffset);
        }
        this.flatHashStrategy.writeFlat(blocks, position, records, recordOffset + this.recordValueOffset, variableWidthChunk, variableWidthChunkOffset);
        return groupId;
    }

    private void setControl(int index, byte hashPrefix) {
        this.control[index] = hashPrefix;
        if (index < 8) {
            this.control[index + this.capacity] = hashPrefix;
        }
    }

    public boolean ensureAvailableCapacity(int batchSize) {
        long requiredMaxFill = this.nextGroupId + batchSize;
        if (requiredMaxFill >= (long)this.maxFill) {
            long minimumRequiredCapacity = (requiredMaxFill + 1L) * 16L / 15L;
            return this.tryRehash(Math.toIntExact(minimumRequiredCapacity));
        }
        return true;
    }

    private boolean tryRehash(int minimumRequiredCapacity) {
        int newCapacity = this.computeNewCapacity(minimumRequiredCapacity);
        this.fixedSizeEstimate = FlatHash.computeFixedSizeEstimate(newCapacity, this.recordSize);
        this.rehashMemoryReservation = FlatHash.sumExact(SizeOf.sizeOf((byte[])this.control), SizeOf.sizeOf((byte[])this.recordGroups[0]));
        Verify.verify((this.rehashMemoryReservation >= 0L ? 1 : 0) != 0, (String)"rehashMemoryReservation is negative", (Object[])new Object[0]);
        if (!this.checkMemoryReservation.update()) {
            return false;
        }
        this.rehash(minimumRequiredCapacity);
        return true;
    }

    private void rehash(int minimumRequiredCapacity) {
        int oldCapacity = this.capacity;
        byte[] oldControl = this.control;
        byte[][] oldRecordGroups = this.recordGroups;
        this.capacity = this.computeNewCapacity(minimumRequiredCapacity);
        this.maxFill = FlatHash.calculateMaxFill(this.capacity);
        this.mask = this.capacity - 1;
        this.control = new byte[this.capacity + 8];
        this.recordGroups = this.capacity <= 1024 ? (Object)new byte[][]{new byte[Math.multiplyExact(this.capacity, this.recordSize)]} : (byte[][])new byte[this.capacity + 1 >> 10][];
        this.groupRecordIndex = new int[this.maxFill];
        for (int oldRecordGroupIndex = 0; oldRecordGroupIndex < oldRecordGroups.length; ++oldRecordGroupIndex) {
            byte[] oldRecords = oldRecordGroups[oldRecordGroupIndex];
            oldRecordGroups[oldRecordGroupIndex] = null;
            block1: for (int indexInRecordGroup = 0; indexInRecordGroup < Math.min(1024, oldCapacity); ++indexInRecordGroup) {
                int oldIndex = (oldRecordGroupIndex << 10) + indexInRecordGroup;
                if (oldControl[oldIndex] == 0) continue;
                long hash = this.hasPrecomputedHash ? LONG_HANDLE.get(oldRecords, this.getRecordOffset(oldIndex) + this.recordHashOffset) : this.valueHashCode(oldRecords, oldIndex);
                byte hashPrefix = (byte)(hash & 0x7FL | 0x80L);
                int bucket = this.bucket((int)(hash >> 7));
                int step = 1;
                while (true) {
                    long controlVector;
                    int emptyIndex;
                    if ((emptyIndex = this.findEmptyInVector(controlVector = LONG_HANDLE.get(this.control, bucket), bucket)) >= 0) {
                        this.setControl(emptyIndex, hashPrefix);
                        int newRecordGroupIndex = emptyIndex >> 10;
                        byte[] records = this.recordGroups[newRecordGroupIndex];
                        if (records == null) {
                            records = new byte[Math.multiplyExact(1024, this.recordSize)];
                            this.recordGroups[newRecordGroupIndex] = records;
                        }
                        int recordOffset = this.getRecordOffset(emptyIndex);
                        int oldRecordOffset = this.getRecordOffset(oldIndex);
                        System.arraycopy(oldRecords, oldRecordOffset, records, recordOffset, this.recordSize);
                        int groupId = INT_HANDLE.get(records, recordOffset + this.recordGroupIdOffset);
                        this.groupRecordIndex[groupId] = emptyIndex;
                        continue block1;
                    }
                    bucket = this.bucket(bucket + step);
                    step += 8;
                }
            }
        }
        for (int i = 0; i < this.recordGroups.length; ++i) {
            if (this.recordGroups[i] != null) continue;
            this.recordGroups[i] = new byte[Math.multiplyExact(1024, this.recordSize)];
        }
        this.rehashMemoryReservation = 0L;
        this.fixedSizeEstimate = FlatHash.computeFixedSizeEstimate(this.capacity, this.recordSize);
        this.checkMemoryReservation.update();
    }

    private int computeNewCapacity(int minimumRequiredCapacity) {
        Preconditions.checkArgument((minimumRequiredCapacity >= 0 ? 1 : 0) != 0, (Object)"minimumRequiredCapacity must be positive");
        long newCapacityLong = (long)this.capacity * 2L;
        while (newCapacityLong < (long)minimumRequiredCapacity) {
            newCapacityLong = Math.multiplyExact(newCapacityLong, 2);
        }
        if (newCapacityLong > Integer.MAX_VALUE) {
            throw new TrinoException((ErrorCodeSupplier)StandardErrorCode.GENERIC_INSUFFICIENT_RESOURCES, "Size of hash table cannot exceed 1 billion entries");
        }
        return Math.toIntExact(newCapacityLong);
    }

    private int bucket(int hash) {
        return hash & this.mask;
    }

    private byte[] getRecords(int index) {
        return this.recordGroups[index >> 10];
    }

    private int getRecordOffset(int index) {
        return (index & 0x3FF) * this.recordSize;
    }

    private long valueHashCode(byte[] records, int index) {
        int recordOffset = this.getRecordOffset(index);
        try {
            byte[] variableWidthChunk = VariableWidthData.EMPTY_CHUNK;
            if (this.variableWidthData != null) {
                variableWidthChunk = this.variableWidthData.getChunk(records, recordOffset);
            }
            return this.flatHashStrategy.hash(records, recordOffset + this.recordValueOffset, variableWidthChunk);
        }
        catch (Throwable throwable) {
            Throwables.throwIfUnchecked((Throwable)throwable);
            throw new RuntimeException(throwable);
        }
    }

    private boolean valueNotDistinctFrom(int leftIndex, Block[] rightBlocks, int rightPosition, long rightHash) {
        long leftHash;
        byte[] leftRecords = this.getRecords(leftIndex);
        int leftRecordOffset = this.getRecordOffset(leftIndex);
        if (this.hasPrecomputedHash && (leftHash = LONG_HANDLE.get(leftRecords, leftRecordOffset + this.recordHashOffset)) != rightHash) {
            return false;
        }
        byte[] leftVariableWidthChunk = VariableWidthData.EMPTY_CHUNK;
        if (this.variableWidthData != null) {
            leftVariableWidthChunk = this.variableWidthData.getChunk(leftRecords, leftRecordOffset);
        }
        return this.flatHashStrategy.valueNotDistinctFrom(leftRecords, leftRecordOffset + this.recordValueOffset, leftVariableWidthChunk, rightBlocks, rightPosition);
    }

    private static long repeat(byte value) {
        return (long)(value & 0xFF) * 0x101010101010101L;
    }

    private static long match(long vector, long repeatedValue) {
        long comparison = vector ^ repeatedValue;
        return comparison - 0x101010101010101L & (comparison ^ 0xFFFFFFFFFFFFFFFFL) & 0x8080808080808080L;
    }

    public int getPhysicalPosition(int groupId) {
        return this.groupRecordIndex[groupId];
    }

    private static int calculateMaxFill(int capacity) {
        return Math.toIntExact((long)capacity * 15L / 16L);
    }

    private static byte[][] createRecordGroups(int capacity, int recordSize) {
        if (capacity <= 1024) {
            return new byte[][]{new byte[Math.multiplyExact(capacity, recordSize)]};
        }
        byte[][] groups = new byte[capacity + 1 >> 10][];
        for (int i = 0; i < groups.length; ++i) {
            groups[i] = new byte[Math.multiplyExact(1024, recordSize)];
        }
        return groups;
    }

    private static long computeRecordGroupsSize(int capacity, int recordSize) {
        if (capacity <= 1024) {
            return SizeOf.sizeOfObjectArray((int)1) + SizeOf.sizeOfByteArray((int)Math.multiplyExact(capacity, recordSize));
        }
        int groupCount = Math.addExact(capacity, 1) >> 10;
        return SizeOf.sizeOfObjectArray((int)groupCount) + Math.multiplyExact((long)groupCount, SizeOf.sizeOfByteArray((int)Math.multiplyExact(1024, recordSize)));
    }

    private static long computeFixedSizeEstimate(int capacity, int recordSize) {
        return FlatHash.sumExact(INSTANCE_SIZE, SizeOf.sizeOfByteArray((int)(capacity + 8)), FlatHash.computeRecordGroupsSize(capacity, recordSize), SizeOf.sizeOfIntArray((int)capacity));
    }

    public static long sumExact(long ... values) {
        long result = 0L;
        for (long value : values) {
            result = Math.addExact(result, value);
        }
        return result;
    }
}

