/*
 * Decompiled with CFR 0.152.
 */
package io.trino.server.remotetask;

import com.google.common.util.concurrent.Futures;
import com.google.common.util.concurrent.ListenableFuture;
import com.google.errorprone.annotations.ThreadSafe;
import io.airlift.event.client.ServiceUnavailableException;
import io.airlift.log.Logger;
import io.airlift.units.Duration;
import io.trino.execution.TaskId;
import io.trino.server.remotetask.Backoff;
import io.trino.spi.ErrorCodeSupplier;
import io.trino.spi.HostAddress;
import io.trino.spi.StandardErrorCode;
import io.trino.spi.TrinoException;
import io.trino.spi.TrinoTransportException;
import java.io.EOFException;
import java.net.SocketException;
import java.net.SocketTimeoutException;
import java.net.URI;
import java.util.Objects;
import java.util.Queue;
import java.util.concurrent.CancellationException;
import java.util.concurrent.ConcurrentLinkedQueue;
import java.util.concurrent.RejectedExecutionException;
import java.util.concurrent.ScheduledExecutorService;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.TimeoutException;

@ThreadSafe
class RequestErrorTracker {
    private static final Logger log = Logger.get(RequestErrorTracker.class);
    private final TaskId taskId;
    private final URI taskUri;
    private final ScheduledExecutorService scheduledExecutor;
    private final String jobDescription;
    private final Backoff backoff;
    private final Queue<Throwable> errorsSinceLastSuccess = new ConcurrentLinkedQueue<Throwable>();

    public RequestErrorTracker(TaskId taskId, URI taskUri, Duration maxErrorDuration, ScheduledExecutorService scheduledExecutor, String jobDescription) {
        this.taskId = Objects.requireNonNull(taskId, "taskId is null");
        this.taskUri = Objects.requireNonNull(taskUri, "taskUri is null");
        this.scheduledExecutor = Objects.requireNonNull(scheduledExecutor, "scheduledExecutor is null");
        this.backoff = new Backoff(Objects.requireNonNull(maxErrorDuration, "maxErrorDuration is null"));
        this.jobDescription = Objects.requireNonNull(jobDescription, "jobDescription is null");
    }

    public ListenableFuture<Void> acquireRequestPermit() {
        long delayNanos = this.backoff.getBackoffDelayNanos();
        if (delayNanos == 0L) {
            return Futures.immediateVoidFuture();
        }
        return Futures.scheduleAsync(Futures::immediateVoidFuture, (long)delayNanos, (TimeUnit)TimeUnit.NANOSECONDS, (ScheduledExecutorService)this.scheduledExecutor);
    }

    public void startRequest() {
        if (this.backoff.getFailureCount() == 0L) {
            this.requestSucceeded();
        }
        this.backoff.startRequest();
    }

    public void requestSucceeded() {
        this.backoff.success();
        this.errorsSinceLastSuccess.clear();
    }

    public void requestFailed(Throwable reason) throws TrinoException {
        if (reason instanceof CancellationException) {
            return;
        }
        if (reason instanceof RejectedExecutionException) {
            throw new TrinoException((ErrorCodeSupplier)StandardErrorCode.REMOTE_TASK_ERROR, reason);
        }
        if (RequestErrorTracker.isExpectedError(reason)) {
            log.warn("Error %s %s: %s: %s", new Object[]{this.jobDescription, this.taskId, reason.getMessage(), this.taskUri});
        } else {
            log.warn(reason, "Error %s %s: %s", new Object[]{this.jobDescription, this.taskId, this.taskUri});
        }
        if (this.errorsSinceLastSuccess.size() < 10) {
            this.errorsSinceLastSuccess.add(reason);
        }
        if (this.backoff.failure()) {
            TrinoTransportException exception = new TrinoTransportException((ErrorCodeSupplier)StandardErrorCode.TOO_MANY_REQUESTS_FAILED, HostAddress.fromUri((URI)this.taskUri), String.format("%s (%s %s - %s failures, failure duration %s, total failed request time %s)", "Encountered too many errors talking to a worker node. The node may have crashed or be under too much load. This is probably a transient issue, so please retry your query in a few minutes.", this.jobDescription, this.taskUri, this.backoff.getFailureCount(), this.backoff.getFailureDuration().convertTo(TimeUnit.SECONDS), this.backoff.getFailureRequestTimeTotal().convertTo(TimeUnit.SECONDS)));
            this.errorsSinceLastSuccess.forEach(arg_0 -> ((TrinoException)exception).addSuppressed(arg_0));
            throw exception;
        }
    }

    static void logError(Throwable t, String message) {
        if (RequestErrorTracker.isExpectedError(t)) {
            log.error("%s: %s", new Object[]{message, t});
        } else {
            log.error(t, message);
        }
    }

    private static boolean isExpectedError(Throwable t) {
        while (t != null) {
            if (t instanceof SocketException || t instanceof SocketTimeoutException || t instanceof EOFException || t instanceof TimeoutException || t instanceof ServiceUnavailableException) {
                return true;
            }
            t = t.getCause();
        }
        return false;
    }
}

