/*
 * Decompiled with CFR 0.152.
 */
package io.trino.execution.executor.scheduler;

import com.google.common.base.Preconditions;
import com.google.common.base.Verify;
import com.google.common.collect.ImmutableSet;
import io.trino.annotation.NotThreadSafe;
import io.trino.execution.executor.scheduler.PriorityQueue;
import io.trino.execution.executor.scheduler.SchedulingGroup;
import io.trino.execution.executor.scheduler.State;
import java.util.Collection;
import java.util.HashMap;
import java.util.Map;
import java.util.Set;
import java.util.stream.Collectors;

@NotThreadSafe
final class SchedulingQueue<G, T> {
    private final PriorityQueue<G> runnableQueue = new PriorityQueue();
    private final Map<G, SchedulingGroup<T>> groups = new HashMap<G, SchedulingGroup<T>>();
    private final PriorityQueue<G> baselineWeights = new PriorityQueue();

    SchedulingQueue() {
    }

    public void startGroup(G group) {
        Preconditions.checkArgument((!this.groups.containsKey(group) ? 1 : 0) != 0, (String)"Group already started: %s", group);
        SchedulingGroup info = new SchedulingGroup();
        this.groups.put(group, info);
    }

    public Set<T> finishGroup(G group) {
        SchedulingGroup<T> info = this.groups.remove(group);
        Preconditions.checkArgument((info != null ? 1 : 0) != 0, (String)"Unknown group: %s", group);
        this.runnableQueue.removeIfPresent(group);
        this.baselineWeights.removeIfPresent(group);
        return info.tasks();
    }

    public boolean containsGroup(G group) {
        return this.groups.containsKey(group);
    }

    public Set<T> getTasks(G group) {
        Preconditions.checkArgument((boolean)this.groups.containsKey(group), (String)"Unknown group: %s", group);
        return this.groups.get(group).tasks();
    }

    public Set<T> finishAll() {
        ImmutableSet groups = ImmutableSet.copyOf(this.groups.keySet());
        return groups.stream().map(this::finishGroup).flatMap(Collection::stream).collect(Collectors.toSet());
    }

    public void finish(G group, T task) {
        Preconditions.checkArgument((boolean)this.groups.containsKey(group), (String)"Unknown group: %s", group);
        SchedulingGroup<T> info = this.groups.get(group);
        State previousState = info.state();
        info.finish(task);
        State newState = info.state();
        if (newState == State.RUNNABLE) {
            this.runnableQueue.addOrReplace(group, info.weight());
            this.baselineWeights.addOrReplace(group, info.weight());
        } else if (newState == State.RUNNING) {
            this.runnableQueue.removeIfPresent(group);
            this.baselineWeights.addOrReplace(group, info.weight());
        } else if (newState == State.BLOCKED && previousState != State.BLOCKED) {
            info.addWeight(-this.baselineWeight());
            this.runnableQueue.removeIfPresent(group);
            this.baselineWeights.removeIfPresent(group);
        }
        this.verifyState(group);
    }

    public void enqueue(G group, T task, long deltaWeight) {
        Preconditions.checkArgument((boolean)this.groups.containsKey(group), (String)"Unknown group: %s", group);
        SchedulingGroup<T> info = this.groups.get(group);
        State previousState = info.state();
        info.enqueue(task, deltaWeight);
        Verify.verify((info.state() == State.RUNNABLE ? 1 : 0) != 0);
        if (previousState == State.BLOCKED) {
            info.addWeight(this.baselineWeight());
        }
        this.runnableQueue.addOrReplace(group, info.weight());
        this.baselineWeights.addOrReplace(group, info.weight());
        this.verifyState(group);
    }

    public void block(G group, T task, long deltaWeight) {
        SchedulingGroup<T> info = this.groups.get(group);
        Preconditions.checkArgument((info != null ? 1 : 0) != 0, (String)"Unknown group: %s", group);
        Preconditions.checkArgument((info.state() == State.RUNNABLE || info.state() == State.RUNNING ? 1 : 0) != 0, (String)"Group is already blocked: %s", group);
        State previousState = info.state();
        info.block(task, deltaWeight);
        this.doTransition(group, info, previousState, info.state());
    }

    public T dequeue(long expectedWeight) {
        G group = this.runnableQueue.poll();
        if (group == null) {
            return null;
        }
        SchedulingGroup<T> info = this.groups.get(group);
        Verify.verify((info.state() == State.RUNNABLE ? 1 : 0) != 0, (String)"Group is not runnable: %s", group);
        T task = info.dequeue(expectedWeight);
        Verify.verify((task != null ? 1 : 0) != 0);
        this.baselineWeights.addOrReplace(group, info.weight());
        if (info.state() == State.RUNNABLE) {
            this.runnableQueue.add(group, info.weight());
        }
        Preconditions.checkState((info.state() == State.RUNNABLE || info.state() == State.RUNNING ? 1 : 0) != 0);
        this.verifyState(group);
        return task;
    }

    public T peek() {
        G group = this.runnableQueue.peek();
        if (group == null) {
            return null;
        }
        SchedulingGroup<T> info = this.groups.get(group);
        Verify.verify((info.state() == State.RUNNABLE ? 1 : 0) != 0, (String)"Group is not runnable: %s", group);
        T task = info.peek();
        Preconditions.checkState((task != null ? 1 : 0) != 0);
        return task;
    }

    public int getRunnableCount() {
        return this.runnableQueue.values().stream().map(this.groups::get).mapToInt(SchedulingGroup::runnableCount).sum();
    }

    public State state(G group) {
        SchedulingGroup<T> info = this.groups.get(group);
        Preconditions.checkArgument((info != null ? 1 : 0) != 0, (String)"Unknown group: %s", group);
        return info.state();
    }

    private long baselineWeight() {
        if (this.baselineWeights.isEmpty()) {
            return 0L;
        }
        return this.baselineWeights.nextPriority();
    }

    private void doTransition(G group, SchedulingGroup<T> info, State previousState, State newState) {
        if (newState == State.RUNNABLE) {
            this.runnableQueue.addOrReplace(group, info.weight());
            this.baselineWeights.addOrReplace(group, info.weight());
        } else if (newState == State.RUNNING) {
            this.runnableQueue.removeIfPresent(group);
            this.baselineWeights.addOrReplace(group, info.weight());
        } else if (newState == State.BLOCKED && previousState != State.BLOCKED) {
            info.addWeight(-this.baselineWeight());
            this.runnableQueue.removeIfPresent(group);
            this.baselineWeights.removeIfPresent(group);
        }
        this.verifyState(group);
    }

    private void verifyState(G groupKey) {
        SchedulingGroup<T> group = this.groups.get(groupKey);
        Preconditions.checkArgument((group != null ? 1 : 0) != 0, (String)"Unknown group: %s", groupKey);
        switch (group.state()) {
            case BLOCKED: {
                Preconditions.checkState((!this.runnableQueue.contains(groupKey) ? 1 : 0) != 0, (String)"Group in BLOCKED state should not be in queue: %s", groupKey);
                Preconditions.checkState((!this.baselineWeights.contains(groupKey) ? 1 : 0) != 0);
                break;
            }
            case RUNNABLE: {
                Preconditions.checkState((boolean)this.runnableQueue.contains(groupKey), (String)"Group in RUNNABLE state should be in queue: %s", groupKey);
                Preconditions.checkState((boolean)this.baselineWeights.contains(groupKey));
                break;
            }
            case RUNNING: {
                Preconditions.checkState((!this.runnableQueue.contains(groupKey) ? 1 : 0) != 0, (String)"Group in RUNNING state should not be in queue: %s", groupKey);
                Preconditions.checkState((boolean)this.baselineWeights.contains(groupKey));
            }
        }
    }

    public String toString() {
        StringBuilder builder = new StringBuilder();
        builder.append("Baseline weight: %s\n".formatted(this.baselineWeight()));
        builder.append("\n");
        for (Map.Entry<G, SchedulingGroup<T>> entry : this.groups.entrySet()) {
            G group = entry.getKey();
            SchedulingGroup<T> info = entry.getValue();
            String prefix = "%s %s".formatted(group == this.runnableQueue.peek() ? "=>" : " -", group);
            String details = switch (entry.getValue().state()) {
                default -> throw new MatchException(null, null);
                case State.BLOCKED -> "[BLOCKED, saved delta = %s]".formatted(info.weight());
                case State.RUNNABLE, State.RUNNING -> "[%s, weight = %s, baseline = %s]".formatted(new Object[]{info.state(), info.weight(), info.baselineWeight()});
            };
            builder.append((prefix + " " + details).indent(4));
            builder.append(info.toString().indent(8));
        }
        return builder.toString();
    }
}

