/*
 * Decompiled with CFR 0.152.
 */
package io.trino.execution.executor.timesharing;

import com.google.common.annotations.VisibleForTesting;
import com.google.common.base.Preconditions;
import com.google.common.math.DoubleMath;
import com.google.errorprone.annotations.ThreadSafe;
import com.google.errorprone.annotations.concurrent.GuardedBy;
import com.google.inject.Inject;
import io.airlift.stats.CounterStat;
import io.trino.execution.TaskManagerConfig;
import io.trino.execution.executor.timesharing.PrioritizedSplitRunner;
import io.trino.execution.executor.timesharing.Priority;
import java.math.RoundingMode;
import java.util.Collection;
import java.util.PriorityQueue;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.atomic.AtomicLong;
import java.util.concurrent.locks.Condition;
import java.util.concurrent.locks.ReentrantLock;
import org.weakref.jmx.Managed;
import org.weakref.jmx.Nested;

@ThreadSafe
public class MultilevelSplitQueue {
    static final int[] LEVEL_THRESHOLD_SECONDS = new int[]{0, 1, 10, 60, 300};
    static final long LEVEL_CONTRIBUTION_CAP = TimeUnit.SECONDS.toNanos(30L);
    @GuardedBy(value="lock")
    private final PriorityQueue<PrioritizedSplitRunner>[] levelWaitingSplits;
    private final AtomicLong[] levelScheduledTime;
    private final AtomicLong[] levelMinPriority;
    private final CounterStat[] selectedLevelCounters;
    private final ReentrantLock lock = new ReentrantLock();
    private final Condition notEmpty = this.lock.newCondition();
    private final double levelTimeMultiplier;

    @Inject
    public MultilevelSplitQueue(TaskManagerConfig taskManagerConfig) {
        this(taskManagerConfig.getLevelTimeMultiplier().doubleValue());
    }

    public MultilevelSplitQueue(double levelTimeMultiplier) {
        this.levelScheduledTime = new AtomicLong[LEVEL_THRESHOLD_SECONDS.length];
        this.levelMinPriority = new AtomicLong[LEVEL_THRESHOLD_SECONDS.length];
        this.levelWaitingSplits = new PriorityQueue[LEVEL_THRESHOLD_SECONDS.length];
        this.selectedLevelCounters = new CounterStat[LEVEL_THRESHOLD_SECONDS.length];
        for (int level = 0; level < LEVEL_THRESHOLD_SECONDS.length; ++level) {
            this.levelScheduledTime[level] = new AtomicLong();
            this.levelMinPriority[level] = new AtomicLong(-1L);
            this.levelWaitingSplits[level] = new PriorityQueue();
            this.selectedLevelCounters[level] = new CounterStat();
        }
        this.levelTimeMultiplier = levelTimeMultiplier;
    }

    private void addLevelTime(int level, long nanos) {
        this.levelScheduledTime[level].addAndGet(nanos);
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    public void offer(PrioritizedSplitRunner split) {
        Preconditions.checkArgument((split != null ? 1 : 0) != 0, (Object)"split is null");
        split.setReady();
        int level = split.getPriority().getLevel();
        this.lock.lock();
        try {
            if (this.levelWaitingSplits[level].isEmpty()) {
                long level0Time = this.getLevel0TargetTime();
                long levelExpectedTime = (long)((double)level0Time / Math.pow(this.levelTimeMultiplier, level));
                long delta = levelExpectedTime - this.levelScheduledTime[level].get();
                this.levelScheduledTime[level].addAndGet(delta);
            }
            this.levelWaitingSplits[level].offer(split);
            this.notEmpty.signal();
        }
        finally {
            this.lock.unlock();
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    public PrioritizedSplitRunner take() throws InterruptedException {
        while (true) {
            this.lock.lockInterruptibly();
            try {
                PrioritizedSplitRunner result;
                while ((result = this.pollSplit()) == null) {
                    this.notEmpty.await();
                }
                if (result.updateLevelPriority()) {
                    this.offer(result);
                    continue;
                }
                int selectedLevel = result.getPriority().getLevel();
                this.levelMinPriority[selectedLevel].set(result.getPriority().getLevelPriority());
                this.selectedLevelCounters[selectedLevel].update(1L);
                PrioritizedSplitRunner prioritizedSplitRunner = result;
                return prioritizedSplitRunner;
            }
            finally {
                this.lock.unlock();
                continue;
            }
            break;
        }
    }

    @GuardedBy(value="lock")
    private PrioritizedSplitRunner pollSplit() {
        long targetScheduledTime = this.getLevel0TargetTime();
        double worstRatio = 1.0;
        int selectedLevel = -1;
        for (int level = 0; level < LEVEL_THRESHOLD_SECONDS.length; ++level) {
            if (!this.levelWaitingSplits[level].isEmpty()) {
                double ratio;
                long levelTime = this.levelScheduledTime[level].get();
                double d = ratio = levelTime == 0L ? 0.0 : (double)targetScheduledTime / (1.0 * (double)levelTime);
                if (selectedLevel == -1 || ratio > worstRatio) {
                    worstRatio = ratio;
                    selectedLevel = level;
                }
            }
            targetScheduledTime = DoubleMath.roundToLong((double)((double)targetScheduledTime / this.levelTimeMultiplier), (RoundingMode)RoundingMode.HALF_UP);
        }
        if (selectedLevel == -1) {
            return null;
        }
        PrioritizedSplitRunner result = this.levelWaitingSplits[selectedLevel].poll();
        Preconditions.checkState((result != null ? 1 : 0) != 0, (Object)"pollSplit cannot return null");
        return result;
    }

    @GuardedBy(value="lock")
    private long getLevel0TargetTime() {
        long level0TargetTime = this.levelScheduledTime[0].get();
        double currentMultiplier = this.levelTimeMultiplier;
        for (int level = 0; level < LEVEL_THRESHOLD_SECONDS.length; ++level) {
            long levelTime = this.levelScheduledTime[level].get();
            level0TargetTime = Math.max(level0TargetTime, (long)((double)levelTime / (currentMultiplier /= this.levelTimeMultiplier)));
        }
        return level0TargetTime;
    }

    public Priority updatePriority(Priority oldPriority, long quantaNanos, long scheduledNanos) {
        int oldLevel = oldPriority.getLevel();
        int newLevel = MultilevelSplitQueue.computeLevel(scheduledNanos);
        long levelContribution = Math.min(quantaNanos, LEVEL_CONTRIBUTION_CAP);
        if (oldLevel == newLevel) {
            this.addLevelTime(oldLevel, levelContribution);
            return new Priority(oldLevel, oldPriority.getLevelPriority() + quantaNanos);
        }
        long remainingLevelContribution = levelContribution;
        long remainingTaskTime = quantaNanos;
        for (int currentLevel = oldLevel; currentLevel < newLevel; ++currentLevel) {
            long timeAccruedToLevel = Math.min(TimeUnit.SECONDS.toNanos(LEVEL_THRESHOLD_SECONDS[currentLevel + 1] - LEVEL_THRESHOLD_SECONDS[currentLevel]), remainingLevelContribution);
            this.addLevelTime(currentLevel, timeAccruedToLevel);
            remainingLevelContribution -= timeAccruedToLevel;
            remainingTaskTime -= timeAccruedToLevel;
        }
        this.addLevelTime(newLevel, remainingLevelContribution);
        long newLevelMinPriority = this.getLevelMinPriority(newLevel, scheduledNanos);
        return new Priority(newLevel, newLevelMinPriority + remainingTaskTime);
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    public void remove(PrioritizedSplitRunner split) {
        Preconditions.checkArgument((split != null ? 1 : 0) != 0, (Object)"split is null");
        this.lock.lock();
        try {
            for (PriorityQueue<PrioritizedSplitRunner> level : this.levelWaitingSplits) {
                level.remove(split);
            }
        }
        finally {
            this.lock.unlock();
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    public void removeAll(Collection<PrioritizedSplitRunner> splits) {
        this.lock.lock();
        try {
            for (PriorityQueue<PrioritizedSplitRunner> level : this.levelWaitingSplits) {
                level.removeAll(splits);
            }
        }
        finally {
            this.lock.unlock();
        }
    }

    public long getLevelMinPriority(int level, long taskThreadUsageNanos) {
        this.levelMinPriority[level].compareAndSet(-1L, taskThreadUsageNanos);
        return this.levelMinPriority[level].get();
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    public int size() {
        this.lock.lock();
        try {
            int total = 0;
            for (PriorityQueue<PrioritizedSplitRunner> level : this.levelWaitingSplits) {
                total += level.size();
            }
            int n = total;
            return n;
        }
        finally {
            this.lock.unlock();
        }
    }

    public static int computeLevel(long threadUsageNanos) {
        long seconds = TimeUnit.NANOSECONDS.toSeconds(threadUsageNanos);
        for (int level = 0; level < LEVEL_THRESHOLD_SECONDS.length - 1; ++level) {
            if (seconds >= (long)LEVEL_THRESHOLD_SECONDS[level + 1]) continue;
            return level;
        }
        return LEVEL_THRESHOLD_SECONDS.length - 1;
    }

    @VisibleForTesting
    long getLevelScheduledTime(int level) {
        return this.levelScheduledTime[level].longValue();
    }

    @Managed
    public long getLevel0Time() {
        return this.getLevelScheduledTime(0);
    }

    @Managed
    public long getLevel1Time() {
        return this.getLevelScheduledTime(1);
    }

    @Managed
    public long getLevel2Time() {
        return this.getLevelScheduledTime(2);
    }

    @Managed
    public long getLevel3Time() {
        return this.getLevelScheduledTime(3);
    }

    @Managed
    public long getLevel4Time() {
        return this.getLevelScheduledTime(4);
    }

    @Managed
    @Nested
    public CounterStat getSelectedCountLevel0() {
        return this.selectedLevelCounters[0];
    }

    @Managed
    @Nested
    public CounterStat getSelectedCountLevel1() {
        return this.selectedLevelCounters[1];
    }

    @Managed
    @Nested
    public CounterStat getSelectedCountLevel2() {
        return this.selectedLevelCounters[2];
    }

    @Managed
    @Nested
    public CounterStat getSelectedCountLevel3() {
        return this.selectedLevelCounters[3];
    }

    @Managed
    @Nested
    public CounterStat getSelectedCountLevel4() {
        return this.selectedLevelCounters[4];
    }
}

