/*
 * Decompiled with CFR 0.152.
 */
package io.trino.server;

import com.google.common.base.StandardSystemProperty;
import com.google.common.collect.ImmutableList;
import com.google.common.collect.ImmutableSet;
import com.google.errorprone.annotations.FormatMethod;
import com.sun.management.UnixOperatingSystemMXBean;
import io.airlift.slice.Slice;
import io.airlift.slice.Slices;
import java.lang.management.ManagementFactory;
import java.lang.management.MemoryManagerMXBean;
import java.nio.ByteOrder;
import java.nio.charset.Charset;
import java.nio.charset.StandardCharsets;
import java.util.List;
import java.util.Locale;
import java.util.OptionalLong;
import java.util.stream.Stream;
import org.joda.time.DateTime;

final class TrinoSystemRequirements {
    private static final int MIN_FILE_DESCRIPTORS = 4096;
    private static final int RECOMMENDED_FILE_DESCRIPTORS = 8192;

    private TrinoSystemRequirements() {
    }

    public static void verifySystemRequirements() {
        TrinoSystemRequirements.verifyJvmRequirements();
        TrinoSystemRequirements.verifySystemTimeIsReasonable();
    }

    private static void verifyJvmRequirements() {
        TrinoSystemRequirements.verifyJavaVersion();
        TrinoSystemRequirements.verify64BitJvm();
        TrinoSystemRequirements.verifyOsArchitecture();
        TrinoSystemRequirements.verifyByteOrder();
        TrinoSystemRequirements.verifyUsingG1Gc();
        TrinoSystemRequirements.verifyFileDescriptor();
        TrinoSystemRequirements.verifySlice();
        TrinoSystemRequirements.verifyUtf8();
    }

    private static void verify64BitJvm() {
        String dataModel = System.getProperty("sun.arch.data.model");
        if (!"64".equals(dataModel)) {
            TrinoSystemRequirements.failRequirement("Trino requires a 64-bit JVM (found %s)", dataModel);
        }
    }

    private static void verifyByteOrder() {
        ByteOrder order = ByteOrder.nativeOrder();
        if (!order.equals(ByteOrder.LITTLE_ENDIAN)) {
            TrinoSystemRequirements.failRequirement("Trino requires a little endian platform (found %s)", order);
        }
    }

    private static void verifyOsArchitecture() {
        String osName = StandardSystemProperty.OS_NAME.value();
        String osArch = StandardSystemProperty.OS_ARCH.value();
        if ("Linux".equals(osName)) {
            if (!ImmutableSet.of((Object)"amd64", (Object)"aarch64", (Object)"ppc64le").contains((Object)osArch)) {
                TrinoSystemRequirements.failRequirement("Trino requires amd64, aarch64, or ppc64le on Linux (found %s)", osArch);
            } else if ("ppc64le".equals(osArch)) {
                TrinoSystemRequirements.warnRequirement("Support for the POWER architecture is experimental", new Object[0]);
            }
        } else if ("Mac OS X".equals(osName)) {
            if (!"x86_64".equals(osArch) && !"aarch64".equals(osArch)) {
                TrinoSystemRequirements.failRequirement("Trino requires x86_64 or aarch64 on Mac OS X (found %s)", osArch);
            }
        } else {
            TrinoSystemRequirements.failRequirement("Trino requires Linux or Mac OS X (found %s)", osName);
        }
    }

    private static void verifyJavaVersion() {
        Runtime.Version required = Runtime.Version.parse("23+37");
        if (Runtime.version().compareTo(required) < 0) {
            TrinoSystemRequirements.failRequirement("Trino requires Java %s at minimum (found %s)", required, Runtime.version());
        }
    }

    private static void verifyUsingG1Gc() {
        try {
            List garbageCollectors = (List)ManagementFactory.getGarbageCollectorMXBeans().stream().map(MemoryManagerMXBean::getName).collect(ImmutableList.toImmutableList());
            if (garbageCollectors.stream().noneMatch(name -> name.toUpperCase(Locale.US).startsWith("G1 "))) {
                TrinoSystemRequirements.warnRequirement("Current garbage collectors are %s. Trino recommends the G1 garbage collector.", garbageCollectors);
            }
        }
        catch (RuntimeException e) {
            TrinoSystemRequirements.failRequirement("Cannot read garbage collector information: %s", e);
        }
    }

    private static void verifyFileDescriptor() {
        OptionalLong maxFileDescriptorCount = TrinoSystemRequirements.getMaxFileDescriptorCount();
        if (maxFileDescriptorCount.isEmpty()) {
            TrinoSystemRequirements.failRequirement("Cannot read OS file descriptor limit", new Object[0]);
        }
        if (maxFileDescriptorCount.getAsLong() < 4096L) {
            TrinoSystemRequirements.failRequirement("Trino requires at least %s file descriptors (found %s)", 4096, maxFileDescriptorCount.getAsLong());
        }
        if (maxFileDescriptorCount.getAsLong() < 8192L) {
            TrinoSystemRequirements.warnRequirement("Current OS file descriptor limit is %s. Trino recommends at least %s", maxFileDescriptorCount.getAsLong(), 8192);
        }
    }

    private static OptionalLong getMaxFileDescriptorCount() {
        return Stream.of(ManagementFactory.getOperatingSystemMXBean()).filter(UnixOperatingSystemMXBean.class::isInstance).map(UnixOperatingSystemMXBean.class::cast).mapToLong(UnixOperatingSystemMXBean::getMaxFileDescriptorCount).findFirst();
    }

    private static void verifySlice() {
        Slice slice = Slices.wrappedBuffer((byte[])new byte[5]);
        slice.setByte(4, 222);
        slice.setByte(3, 173);
        slice.setByte(2, 190);
        slice.setByte(1, 239);
        if (slice.getInt(1) != -559038737) {
            TrinoSystemRequirements.failRequirement("Slice library produced an unexpected result", new Object[0]);
        }
    }

    private static void verifyUtf8() {
        Charset defaultCharset = Charset.defaultCharset();
        if (!defaultCharset.equals(StandardCharsets.UTF_8)) {
            TrinoSystemRequirements.failRequirement("Trino requires that the default charset is UTF-8 (found %s). This can be set with the JVM command line option -Dfile.encoding=UTF-8", defaultCharset.name());
        }
    }

    private static void verifySystemTimeIsReasonable() {
        int currentYear = DateTime.now().year().get();
        if (currentYear < 2024) {
            TrinoSystemRequirements.failRequirement("Trino requires the system time to be current (found year %s)", currentYear);
        }
    }

    @FormatMethod
    private static void failRequirement(String format, Object ... args) {
        System.err.println("ERROR: " + String.format(format, args));
        System.exit(100);
    }

    @FormatMethod
    private static void warnRequirement(String format, Object ... args) {
        System.err.println("WARNING: " + String.format(format, args));
    }
}

