/*
 * Decompiled with CFR 0.152.
 */
package io.trino.operator.output;

import com.google.common.annotations.VisibleForTesting;
import com.google.common.collect.ImmutableList;
import com.google.errorprone.annotations.ThreadSafe;
import com.google.errorprone.annotations.concurrent.GuardedBy;
import io.airlift.log.Logger;
import io.trino.Session;
import io.trino.SystemSessionProperties;
import io.trino.execution.resourcegroups.IndexedPriorityQueue;
import io.trino.operator.PartitionFunction;
import io.trino.operator.RetryPolicy;
import io.trino.spi.connector.ConnectorBucketNodeMap;
import io.trino.spi.type.Type;
import io.trino.sql.planner.NodePartitioningManager;
import io.trino.sql.planner.PartitioningHandle;
import io.trino.sql.planner.PartitioningScheme;
import io.trino.sql.planner.SystemPartitioningHandle;
import java.util.ArrayList;
import java.util.List;
import java.util.Objects;
import java.util.concurrent.CopyOnWriteArrayList;
import java.util.concurrent.atomic.AtomicLong;
import java.util.concurrent.atomic.AtomicLongArray;
import java.util.stream.IntStream;

@ThreadSafe
public class SkewedPartitionRebalancer {
    private static final Logger log = Logger.get(SkewedPartitionRebalancer.class);
    private static final int SCALE_WRITERS_PARTITION_COUNT = 4096;
    private static final double TASK_BUCKET_SKEWNESS_THRESHOLD = 0.7;
    private final int partitionCount;
    private final int taskCount;
    private final int taskBucketCount;
    private final long minPartitionDataProcessedRebalanceThreshold;
    private final long minDataProcessedRebalanceThreshold;
    private final AtomicLongArray partitionRowCount;
    private final AtomicLong dataProcessed;
    private final AtomicLong dataProcessedAtLastRebalance;
    @GuardedBy(value="this")
    private final long[] partitionDataSize;
    @GuardedBy(value="this")
    private final long[] partitionDataSizeAtLastRebalance;
    @GuardedBy(value="this")
    private final long[] partitionDataSizeSinceLastRebalancePerTask;
    @GuardedBy(value="this")
    private final long[] estimatedTaskBucketDataSizeSinceLastRebalance;
    private final List<List<TaskBucket>> partitionAssignments;

    public static boolean checkCanScalePartitionsRemotely(Session session, int taskCount, PartitioningHandle partitioningHandle, NodePartitioningManager nodePartitioningManager) {
        if (SystemSessionProperties.getRetryPolicy(session) == RetryPolicy.TASK) {
            return false;
        }
        boolean hasFixedNodeMapping = partitioningHandle.getCatalogHandle().map(catalogHandle -> nodePartitioningManager.getConnectorBucketNodeMap(session, partitioningHandle).map(ConnectorBucketNodeMap::hasFixedMapping).orElse(false)).orElse(false);
        return taskCount > 1 && !hasFixedNodeMapping && PartitioningHandle.isScaledWriterHashDistribution(partitioningHandle);
    }

    public static PartitionFunction createPartitionFunction(Session session, NodePartitioningManager nodePartitioningManager, PartitioningScheme scheme, List<Type> partitionChannelTypes) {
        PartitioningHandle handle = scheme.getPartitioning().getHandle();
        int bucketCount = handle.getConnectorHandle() instanceof SystemPartitioningHandle ? 4096 : nodePartitioningManager.getBucketCount(session, handle);
        return nodePartitioningManager.getPartitionFunction(session, scheme, partitionChannelTypes, IntStream.range(0, bucketCount).toArray());
    }

    public static int getMaxWritersBasedOnMemory(Session session) {
        return (int)Math.ceil((double)SystemSessionProperties.getQueryMaxMemoryPerNode(session).toBytes() / (double)SystemSessionProperties.getMaxMemoryPerPartitionWriter(session).toBytes());
    }

    public static int getTaskCount(PartitioningScheme partitioningScheme) {
        int[] bucketToPartition = partitioningScheme.getBucketToPartition().orElseThrow(() -> new IllegalArgumentException("Bucket to partition must be set before calculating taskCount"));
        return IntStream.of(bucketToPartition).max().getAsInt() + 1;
    }

    public SkewedPartitionRebalancer(int partitionCount, int taskCount, int taskBucketCount, long minPartitionDataProcessedRebalanceThreshold, long maxDataProcessedRebalanceThreshold) {
        this.partitionCount = partitionCount;
        this.taskCount = taskCount;
        this.taskBucketCount = taskBucketCount;
        this.minPartitionDataProcessedRebalanceThreshold = minPartitionDataProcessedRebalanceThreshold;
        this.minDataProcessedRebalanceThreshold = Math.max(minPartitionDataProcessedRebalanceThreshold, maxDataProcessedRebalanceThreshold);
        this.partitionRowCount = new AtomicLongArray(partitionCount);
        this.dataProcessed = new AtomicLong();
        this.dataProcessedAtLastRebalance = new AtomicLong();
        this.partitionDataSize = new long[partitionCount];
        this.partitionDataSizeAtLastRebalance = new long[partitionCount];
        this.partitionDataSizeSinceLastRebalancePerTask = new long[partitionCount];
        this.estimatedTaskBucketDataSizeSinceLastRebalance = new long[taskCount * taskBucketCount];
        int[] taskBucketIds = new int[taskCount];
        ImmutableList.Builder partitionAssignments = ImmutableList.builder();
        for (int partition = 0; partition < partitionCount; ++partition) {
            int taskId;
            int n = taskId = partition % taskCount;
            int n2 = taskBucketIds[n];
            taskBucketIds[n] = n2 + 1;
            int bucketId = n2 % taskBucketCount;
            partitionAssignments.add(new CopyOnWriteArrayList(ImmutableList.of((Object)new TaskBucket(this, taskId, bucketId))));
        }
        this.partitionAssignments = partitionAssignments.build();
    }

    @VisibleForTesting
    List<List<Integer>> getPartitionAssignments() {
        ImmutableList.Builder assignedTasks = ImmutableList.builder();
        for (List<TaskBucket> partitionAssignment : this.partitionAssignments) {
            List tasks = (List)partitionAssignment.stream().map(taskBucket -> taskBucket.taskId).collect(ImmutableList.toImmutableList());
            assignedTasks.add((Object)tasks);
        }
        return assignedTasks.build();
    }

    public int getTaskCount() {
        return this.taskCount;
    }

    public int getTaskId(int partitionId, long index) {
        List<TaskBucket> taskIds = this.partitionAssignments.get(partitionId);
        return taskIds.get((int)Math.floorMod((long)index, (int)taskIds.size())).taskId;
    }

    public void addDataProcessed(long dataSize) {
        this.dataProcessed.addAndGet(dataSize);
    }

    public void addPartitionRowCount(int partition, long rowCount) {
        this.partitionRowCount.addAndGet(partition, rowCount);
    }

    public void rebalance() {
        long currentDataProcessed = this.dataProcessed.get();
        if (this.shouldRebalance(currentDataProcessed)) {
            this.rebalancePartitions(currentDataProcessed);
        }
    }

    private boolean shouldRebalance(long dataProcessed) {
        return dataProcessed - this.dataProcessedAtLastRebalance.get() >= this.minDataProcessedRebalanceThreshold;
    }

    private synchronized void rebalancePartitions(long dataProcessed) {
        if (!this.shouldRebalance(dataProcessed)) {
            return;
        }
        this.calculatePartitionDataSize(dataProcessed);
        for (int partition = 0; partition < this.partitionCount; ++partition) {
            int totalAssignedTasks = this.partitionAssignments.get(partition).size();
            long dataSize = this.partitionDataSize[partition];
            this.partitionDataSizeSinceLastRebalancePerTask[partition] = (dataSize - this.partitionDataSizeAtLastRebalance[partition]) / (long)totalAssignedTasks;
            this.partitionDataSizeAtLastRebalance[partition] = dataSize;
        }
        ArrayList<IndexedPriorityQueue<Integer>> taskBucketMaxPartitions = new ArrayList<IndexedPriorityQueue<Integer>>(this.taskCount * this.taskBucketCount);
        for (int taskId = 0; taskId < this.taskCount; ++taskId) {
            for (int bucketId = 0; bucketId < this.taskBucketCount; ++bucketId) {
                taskBucketMaxPartitions.add(new IndexedPriorityQueue());
            }
        }
        for (int partition = 0; partition < this.partitionCount; ++partition) {
            List<TaskBucket> taskAssignments = this.partitionAssignments.get(partition);
            for (TaskBucket taskBucket : taskAssignments) {
                IndexedPriorityQueue queue = (IndexedPriorityQueue)taskBucketMaxPartitions.get(taskBucket.id);
                queue.addOrUpdate(partition, this.partitionDataSizeSinceLastRebalancePerTask[partition]);
            }
        }
        IndexedPriorityQueue<TaskBucket> maxTaskBuckets = new IndexedPriorityQueue<TaskBucket>();
        IndexedPriorityQueue<TaskBucket> minTaskBuckets = new IndexedPriorityQueue<TaskBucket>();
        for (int taskId = 0; taskId < this.taskCount; ++taskId) {
            for (int bucketId = 0; bucketId < this.taskBucketCount; ++bucketId) {
                TaskBucket taskBucket = new TaskBucket(this, taskId, bucketId);
                this.estimatedTaskBucketDataSizeSinceLastRebalance[taskBucket.id] = this.calculateTaskBucketDataSizeSinceLastRebalance((IndexedPriorityQueue)taskBucketMaxPartitions.get(taskBucket.id));
                maxTaskBuckets.addOrUpdate(taskBucket, this.estimatedTaskBucketDataSizeSinceLastRebalance[taskBucket.id]);
                minTaskBuckets.addOrUpdate(taskBucket, Long.MAX_VALUE - this.estimatedTaskBucketDataSizeSinceLastRebalance[taskBucket.id]);
            }
        }
        this.rebalanceBasedOnTaskBucketSkewness(maxTaskBuckets, minTaskBuckets, taskBucketMaxPartitions);
        this.dataProcessedAtLastRebalance.set(dataProcessed);
    }

    @GuardedBy(value="this")
    private void calculatePartitionDataSize(long dataProcessed) {
        int partition;
        long totalPartitionRowCount = 0L;
        for (partition = 0; partition < this.partitionCount; ++partition) {
            totalPartitionRowCount += this.partitionRowCount.get(partition);
        }
        for (partition = 0; partition < this.partitionCount; ++partition) {
            this.partitionDataSize[partition] = Math.max(this.partitionRowCount.get(partition) * dataProcessed / totalPartitionRowCount, this.partitionDataSize[partition]);
        }
    }

    @GuardedBy(value="this")
    private long calculateTaskBucketDataSizeSinceLastRebalance(IndexedPriorityQueue<Integer> maxPartitions) {
        long estimatedDataSizeSinceLastRebalance = 0L;
        for (int partition : maxPartitions) {
            estimatedDataSizeSinceLastRebalance += this.partitionDataSizeSinceLastRebalancePerTask[partition];
        }
        return estimatedDataSizeSinceLastRebalance;
    }

    @GuardedBy(value="this")
    private void rebalanceBasedOnTaskBucketSkewness(IndexedPriorityQueue<TaskBucket> maxTaskBuckets, IndexedPriorityQueue<TaskBucket> minTaskBuckets, List<IndexedPriorityQueue<Integer>> taskBucketMaxPartitions) {
        TaskBucket maxTaskBucket;
        ArrayList<Integer> scaledPartitions = new ArrayList<Integer>();
        block0: while ((maxTaskBucket = maxTaskBuckets.poll()) != null) {
            Integer maxPartition;
            IndexedPriorityQueue<Integer> maxPartitions = taskBucketMaxPartitions.get(maxTaskBucket.id);
            if (maxPartitions.isEmpty()) continue;
            List<TaskBucket> minSkewedTaskBuckets = this.findSkewedMinTaskBuckets(maxTaskBucket, minTaskBuckets);
            if (minSkewedTaskBuckets.isEmpty()) break;
            block1: while ((maxPartition = maxPartitions.poll()) != null) {
                if (scaledPartitions.contains(maxPartition)) continue;
                int totalAssignedTasks = this.partitionAssignments.get(maxPartition).size();
                if (this.partitionDataSize[maxPartition] < this.minPartitionDataProcessedRebalanceThreshold * (long)totalAssignedTasks) continue block0;
                for (TaskBucket minTaskBucket : minSkewedTaskBuckets) {
                    if (!this.rebalancePartition(maxPartition, minTaskBucket, maxTaskBuckets, minTaskBuckets)) continue;
                    scaledPartitions.add(maxPartition);
                    continue block1;
                }
            }
        }
    }

    @GuardedBy(value="this")
    private List<TaskBucket> findSkewedMinTaskBuckets(TaskBucket maxTaskBucket, IndexedPriorityQueue<TaskBucket> minTaskBuckets) {
        ImmutableList.Builder minSkewedTaskBuckets = ImmutableList.builder();
        for (TaskBucket minTaskBucket : minTaskBuckets) {
            double skewness = (double)(this.estimatedTaskBucketDataSizeSinceLastRebalance[maxTaskBucket.id] - this.estimatedTaskBucketDataSizeSinceLastRebalance[minTaskBucket.id]) / (double)this.estimatedTaskBucketDataSizeSinceLastRebalance[maxTaskBucket.id];
            if (skewness <= 0.7 || Double.isNaN(skewness)) break;
            if (maxTaskBucket.taskId == minTaskBucket.taskId) continue;
            minSkewedTaskBuckets.add((Object)minTaskBucket);
        }
        return minSkewedTaskBuckets.build();
    }

    @GuardedBy(value="this")
    private boolean rebalancePartition(int partitionId, TaskBucket toTaskBucket, IndexedPriorityQueue<TaskBucket> maxTasks, IndexedPriorityQueue<TaskBucket> minTasks) {
        List<TaskBucket> assignments = this.partitionAssignments.get(partitionId);
        if (assignments.stream().anyMatch(taskBucket -> taskBucket.taskId == toTaskBucket.taskId)) {
            return false;
        }
        assignments.add(toTaskBucket);
        int newTaskCount = assignments.size();
        int oldTaskCount = newTaskCount - 1;
        for (TaskBucket taskBucket2 : assignments) {
            if (taskBucket2.equals(toTaskBucket)) {
                int n = taskBucket2.id;
                this.estimatedTaskBucketDataSizeSinceLastRebalance[n] = this.estimatedTaskBucketDataSizeSinceLastRebalance[n] + this.partitionDataSizeSinceLastRebalancePerTask[partitionId] * (long)oldTaskCount / (long)newTaskCount;
            } else {
                int n = taskBucket2.id;
                this.estimatedTaskBucketDataSizeSinceLastRebalance[n] = this.estimatedTaskBucketDataSizeSinceLastRebalance[n] - this.partitionDataSizeSinceLastRebalancePerTask[partitionId] / (long)newTaskCount;
            }
            maxTasks.addOrUpdate(taskBucket2, this.estimatedTaskBucketDataSizeSinceLastRebalance[taskBucket2.id]);
            minTasks.addOrUpdate(taskBucket2, Long.MAX_VALUE - this.estimatedTaskBucketDataSizeSinceLastRebalance[taskBucket2.id]);
        }
        log.debug("Rebalanced partition %s to task %s with taskCount %s", new Object[]{partitionId, toTaskBucket.taskId, assignments.size()});
        return true;
    }

    private final class TaskBucket {
        private final int taskId;
        private final int id;

        private TaskBucket(SkewedPartitionRebalancer skewedPartitionRebalancer, int taskId, int bucketId) {
            this.taskId = taskId;
            this.id = taskId * skewedPartitionRebalancer.taskBucketCount + bucketId;
        }

        public int hashCode() {
            return Objects.hash(this.taskId, this.id);
        }

        public boolean equals(Object o) {
            if (this == o) {
                return true;
            }
            if (o == null || this.getClass() != o.getClass()) {
                return false;
            }
            TaskBucket that = (TaskBucket)o;
            return that.id == this.id;
        }
    }
}

