/*
 * Decompiled with CFR 0.152.
 */
package io.trino.server;

import com.google.common.collect.ImmutableList;
import com.google.common.util.concurrent.Uninterruptibles;
import com.google.inject.Inject;
import io.airlift.bootstrap.LifeCycleManager;
import io.airlift.concurrent.Threads;
import io.airlift.log.Logger;
import io.airlift.units.Duration;
import io.trino.execution.SqlTaskManager;
import io.trino.execution.TaskInfo;
import io.trino.metadata.NodeState;
import io.trino.server.ServerConfig;
import io.trino.server.ShutdownAction;
import java.util.List;
import java.util.Objects;
import java.util.concurrent.CountDownLatch;
import java.util.concurrent.ExecutionException;
import java.util.concurrent.ExecutorService;
import java.util.concurrent.Executors;
import java.util.concurrent.Future;
import java.util.concurrent.ScheduledExecutorService;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.TimeoutException;
import java.util.concurrent.atomic.AtomicReference;

public class NodeStateManager {
    private static final Logger log = Logger.get(NodeStateManager.class);
    private static final Duration LIFECYCLE_STOP_TIMEOUT = new Duration(30.0, TimeUnit.SECONDS);
    private final ScheduledExecutorService shutdownHandler = Executors.newSingleThreadScheduledExecutor(Threads.threadsNamed((String)"shutdown-handler-%s"));
    private final ExecutorService lifeCycleStopper = Executors.newSingleThreadExecutor(Threads.threadsNamed((String)"lifecycle-stopper-%s"));
    private final LifeCycleManager lifeCycleManager;
    private final SqlTaskManager sqlTaskManager;
    private final boolean isCoordinator;
    private final ShutdownAction shutdownAction;
    private final Duration gracePeriod;
    private final ScheduledExecutorService executor = Executors.newSingleThreadScheduledExecutor(Threads.threadsNamed((String)"drain-handler-%s"));
    private final AtomicReference<NodeState> nodeState = new AtomicReference<NodeState>(NodeState.ACTIVE);

    @Inject
    public NodeStateManager(SqlTaskManager sqlTaskManager, ServerConfig serverConfig, ShutdownAction shutdownAction, LifeCycleManager lifeCycleManager) {
        this.sqlTaskManager = Objects.requireNonNull(sqlTaskManager, "sqlTaskManager is null");
        this.shutdownAction = Objects.requireNonNull(shutdownAction, "shutdownAction is null");
        this.lifeCycleManager = Objects.requireNonNull(lifeCycleManager, "lifeCycleManager is null");
        this.isCoordinator = serverConfig.isCoordinator();
        this.gracePeriod = serverConfig.getGracePeriod();
    }

    public NodeState getServerState() {
        return this.nodeState.get();
    }

    public synchronized void transitionState(NodeState state) {
        NodeState currState = this.nodeState.get();
        if (currState == state) {
            return;
        }
        switch (state) {
            case ACTIVE: {
                if (currState == NodeState.DRAINING && this.nodeState.compareAndSet(NodeState.DRAINING, NodeState.ACTIVE)) {
                    return;
                }
                if (currState != NodeState.DRAINED || !this.nodeState.compareAndSet(NodeState.DRAINED, NodeState.ACTIVE)) break;
                return;
            }
            case SHUTTING_DOWN: {
                if (currState == NodeState.DRAINED && this.nodeState.compareAndSet(NodeState.DRAINED, NodeState.SHUTTING_DOWN)) {
                    this.requestTerminate();
                    return;
                }
                this.requestGracefulShutdown();
                this.nodeState.set(NodeState.SHUTTING_DOWN);
                return;
            }
            case DRAINING: {
                if (currState != NodeState.ACTIVE || !this.nodeState.compareAndSet(NodeState.ACTIVE, NodeState.DRAINING)) break;
                this.requestDrain();
                return;
            }
            case DRAINED: {
                throw new IllegalStateException(String.format("Invalid state transition from %s to %s, transition to DRAINED is internal only", new Object[]{currState, state}));
            }
            case INACTIVE: {
                throw new IllegalStateException(String.format("Invalid state transition from %s to %s, INACTIVE is not a valid internal state", new Object[]{currState, state}));
            }
        }
        throw new IllegalStateException(String.format("Invalid state transition from %s to %s", new Object[]{currState, state}));
    }

    private synchronized void requestDrain() {
        log.debug("Drain requested, NodeState: " + String.valueOf((Object)this.getServerState()));
        if (this.isCoordinator) {
            throw new UnsupportedOperationException("Cannot drain coordinator");
        }
        this.executor.schedule(this::drain, this.gracePeriod.toMillis(), TimeUnit.MILLISECONDS);
    }

    private void requestTerminate() {
        log.info("Immediate Shutdown requested");
        if (this.isCoordinator) {
            throw new UnsupportedOperationException("Cannot shutdown coordinator");
        }
        this.shutdownHandler.schedule(this::terminate, 0L, TimeUnit.MILLISECONDS);
    }

    private void requestGracefulShutdown() {
        log.info("Shutdown requested");
        if (this.isCoordinator) {
            throw new UnsupportedOperationException("Cannot shutdown coordinator");
        }
        this.shutdownHandler.schedule(this::shutdown, this.gracePeriod.toMillis(), TimeUnit.MILLISECONDS);
    }

    private void shutdown() {
        this.waitActiveTasksToFinish();
        this.terminate();
    }

    private void terminate() {
        Future<Object> shutdownFuture = this.lifeCycleStopper.submit(() -> {
            this.lifeCycleManager.stop();
            return null;
        });
        try {
            shutdownFuture.get(LIFECYCLE_STOP_TIMEOUT.toMillis(), TimeUnit.MILLISECONDS);
        }
        catch (TimeoutException e) {
            log.warn((Throwable)e, "Timed out waiting for the life cycle to stop");
        }
        catch (InterruptedException e) {
            log.warn((Throwable)e, "Interrupted while waiting for the life cycle to stop");
            Thread.currentThread().interrupt();
        }
        catch (ExecutionException e) {
            log.warn((Throwable)e, "Problem stopping the life cycle");
        }
        this.shutdownAction.onShutdown();
    }

    private void drain() {
        if (this.nodeState.get() == NodeState.DRAINING) {
            this.waitActiveTasksToFinish();
        }
        this.drainingComplete();
    }

    private void drainingComplete() {
        boolean success = this.nodeState.compareAndSet(NodeState.DRAINING, NodeState.DRAINED);
        if (success) {
            log.info("NodeState: DRAINED, server can be safely SHUT DOWN.");
        } else {
            log.info("NodeState: " + String.valueOf((Object)this.nodeState.get()) + ", will not transition to DRAINED");
        }
    }

    private void waitActiveTasksToFinish() {
        while (this.isShuttingDownOrDraining()) {
            List<TaskInfo> activeTasks = this.getActiveTasks();
            log.info("Waiting for " + activeTasks.size() + " active tasks to finish");
            if (activeTasks.isEmpty()) break;
            this.waitTasksToFinish(activeTasks);
        }
        if (this.isShuttingDownOrDraining()) {
            Uninterruptibles.sleepUninterruptibly((long)this.gracePeriod.toMillis(), (TimeUnit)TimeUnit.MILLISECONDS);
        }
    }

    private void waitTasksToFinish(List<TaskInfo> activeTasks) {
        CountDownLatch countDownLatch = new CountDownLatch(activeTasks.size());
        for (TaskInfo taskInfo : activeTasks) {
            this.sqlTaskManager.addStateChangeListener(taskInfo.taskStatus().getTaskId(), newState -> {
                if (newState.isDone()) {
                    countDownLatch.countDown();
                }
            });
        }
        try {
            while (!countDownLatch.await(1L, TimeUnit.SECONDS)) {
                if (this.isShuttingDownOrDraining()) continue;
                log.info("Wait for tasks interrupted, worker is no longer draining.");
                break;
            }
        }
        catch (InterruptedException e) {
            log.warn("Interrupted while waiting for all tasks to finish");
            Thread.currentThread().interrupt();
        }
    }

    private boolean isShuttingDownOrDraining() {
        NodeState state = this.nodeState.get();
        return state == NodeState.SHUTTING_DOWN || state == NodeState.DRAINING;
    }

    private List<TaskInfo> getActiveTasks() {
        return (List)this.sqlTaskManager.getAllTaskInfo().stream().filter(taskInfo -> !taskInfo.taskStatus().getState().isDone()).collect(ImmutableList.toImmutableList());
    }
}

