/*
 * Decompiled with CFR 0.152.
 */
package io.trino.eventlistener;

import com.google.common.base.Function;
import com.google.common.base.Preconditions;
import com.google.common.base.Strings;
import com.google.common.collect.ImmutableList;
import com.google.inject.Inject;
import io.airlift.configuration.ConfigurationLoader;
import io.airlift.configuration.secrets.SecretsResolver;
import io.airlift.log.Logger;
import io.airlift.stats.TimeStat;
import io.opentelemetry.api.OpenTelemetry;
import io.opentelemetry.api.trace.Tracer;
import io.trino.client.NodeVersion;
import io.trino.eventlistener.EventListenerConfig;
import io.trino.eventlistener.EventListenerContextInstance;
import io.trino.spi.classloader.ThreadContextClassLoader;
import io.trino.spi.eventlistener.EventListener;
import io.trino.spi.eventlistener.EventListenerFactory;
import io.trino.spi.eventlistener.QueryCompletedEvent;
import io.trino.spi.eventlistener.QueryCreatedEvent;
import io.trino.spi.eventlistener.SplitCompletedEvent;
import jakarta.annotation.PreDestroy;
import java.io.File;
import java.io.IOException;
import java.io.UncheckedIOException;
import java.util.ArrayList;
import java.util.Collections;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.concurrent.ConcurrentHashMap;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.atomic.AtomicBoolean;
import java.util.concurrent.atomic.AtomicInteger;
import java.util.concurrent.atomic.AtomicReference;
import org.weakref.jmx.Managed;
import org.weakref.jmx.Nested;

public class EventListenerManager {
    private static final Logger log = Logger.get(EventListenerManager.class);
    private static final File CONFIG_FILE = new File("etc/event-listener.properties");
    private static final String EVENT_LISTENER_NAME_PROPERTY = "event-listener.name";
    private final List<File> configFiles;
    private final Map<String, EventListenerFactory> eventListenerFactories = new ConcurrentHashMap<String, EventListenerFactory>();
    private final List<EventListener> providedEventListeners = Collections.synchronizedList(new ArrayList());
    private final AtomicReference<List<EventListener>> configuredEventListeners = new AtomicReference<ImmutableList>(ImmutableList.of());
    private final AtomicBoolean loading = new AtomicBoolean(false);
    private final AtomicInteger concurrentQueryCompletedEvents = new AtomicInteger();
    private final TimeStat queryCreatedTime = new TimeStat(TimeUnit.MILLISECONDS);
    private final TimeStat queryCompletedTime = new TimeStat(TimeUnit.MILLISECONDS);
    private final TimeStat splitCompletedTime = new TimeStat(TimeUnit.MILLISECONDS);
    private final SecretsResolver secretsResolver;
    private final EventListenerContextInstance context;

    @Inject
    public EventListenerManager(EventListenerConfig config, SecretsResolver secretsResolver, OpenTelemetry openTelemetry, Tracer tracer, NodeVersion version) {
        this.configFiles = ImmutableList.copyOf(config.getEventListenerFiles());
        this.secretsResolver = Objects.requireNonNull(secretsResolver, "secretsResolver is null");
        this.context = new EventListenerContextInstance(version.toString(), openTelemetry, tracer);
    }

    public void addEventListenerFactory(EventListenerFactory eventListenerFactory) {
        Objects.requireNonNull(eventListenerFactory, "eventListenerFactory is null");
        if (this.eventListenerFactories.putIfAbsent(eventListenerFactory.getName(), eventListenerFactory) != null) {
            throw new IllegalArgumentException(String.format("Event listener factory '%s' is already registered", eventListenerFactory.getName()));
        }
    }

    public void addEventListener(EventListener eventListener) {
        Objects.requireNonNull(eventListener, "EventListener is null");
        this.providedEventListeners.add(eventListener);
    }

    public void loadEventListeners() {
        Preconditions.checkState((boolean)this.loading.compareAndSet(false, true), (Object)"Event listeners already loaded");
        this.configuredEventListeners.set((List<EventListener>)ImmutableList.builder().addAll(this.providedEventListeners).addAll(this.configuredEventListeners()).build());
    }

    private List<EventListener> configuredEventListeners() {
        ImmutableList configFiles = this.configFiles;
        if (configFiles.isEmpty()) {
            if (!CONFIG_FILE.exists()) {
                return ImmutableList.of();
            }
            configFiles = ImmutableList.of((Object)CONFIG_FILE);
        }
        return (List)configFiles.stream().map(this::createEventListener).collect(ImmutableList.toImmutableList());
    }

    private EventListener createEventListener(File configFile) {
        EventListener eventListener;
        log.info("-- Loading event listener %s --", new Object[]{configFile});
        configFile = configFile.getAbsoluteFile();
        Map<String, String> properties = EventListenerManager.loadEventListenerProperties(configFile);
        String name = properties.remove(EVENT_LISTENER_NAME_PROPERTY);
        Preconditions.checkArgument((!Strings.isNullOrEmpty((String)name) ? 1 : 0) != 0, (String)"EventListener plugin configuration for %s does not contain %s", (Object)configFile, (Object)EVENT_LISTENER_NAME_PROPERTY);
        EventListenerFactory factory = this.eventListenerFactories.get(name);
        Preconditions.checkArgument((factory != null ? 1 : 0) != 0, (String)"Event listener factory '%s' is not registered. Available factories: %s", (Object)name, this.eventListenerFactories.keySet());
        try (ThreadContextClassLoader threadContextClassLoader = new ThreadContextClassLoader(factory.getClass().getClassLoader());){
            eventListener = factory.create(this.secretsResolver.getResolvedConfiguration(properties), (EventListenerFactory.EventListenerContext)this.context);
        }
        log.info("-- Loaded event listener %s --", new Object[]{configFile});
        return eventListener;
    }

    private static Map<String, String> loadEventListenerProperties(File configFile) {
        try {
            return new HashMap<String, String>(ConfigurationLoader.loadPropertiesFrom((String)configFile.getPath()));
        }
        catch (IOException e) {
            throw new UncheckedIOException("Failed to read configuration file: " + String.valueOf(configFile), e);
        }
    }

    public void queryCompleted(Function<Boolean, QueryCompletedEvent> queryCompletedEventProvider) {
        try (TimeStat.BlockTimer blockTimer = this.queryCompletedTime.time();){
            this.concurrentQueryCompletedEvents.incrementAndGet();
            this.doQueryCompleted(queryCompletedEventProvider);
            this.concurrentQueryCompletedEvents.decrementAndGet();
        }
    }

    private void doQueryCompleted(Function<Boolean, QueryCompletedEvent> queryCompletedEventProvider) {
        for (EventListener listener : this.configuredEventListeners.get()) {
            QueryCompletedEvent event = (QueryCompletedEvent)queryCompletedEventProvider.apply((Object)listener.requiresAnonymizedPlan());
            try {
                listener.queryCompleted(event);
            }
            catch (Throwable e) {
                log.warn(e, "Failed to publish QueryCompletedEvent for query %s", new Object[]{event.getMetadata().getQueryId()});
            }
        }
    }

    public void queryCreated(QueryCreatedEvent queryCreatedEvent) {
        try (TimeStat.BlockTimer blockTimer = this.queryCreatedTime.time();){
            this.doQueryCreated(queryCreatedEvent);
        }
    }

    private void doQueryCreated(QueryCreatedEvent queryCreatedEvent) {
        for (EventListener listener : this.configuredEventListeners.get()) {
            try {
                listener.queryCreated(queryCreatedEvent);
            }
            catch (Throwable e) {
                log.warn(e, "Failed to publish QueryCreatedEvent for query %s", new Object[]{queryCreatedEvent.getMetadata().getQueryId()});
            }
        }
    }

    public void splitCompleted(SplitCompletedEvent splitCompletedEvent) {
        try (TimeStat.BlockTimer blockTimer = this.splitCompletedTime.time();){
            this.doSplitCompleted(splitCompletedEvent);
        }
    }

    private void doSplitCompleted(SplitCompletedEvent splitCompletedEvent) {
        for (EventListener listener : this.configuredEventListeners.get()) {
            try {
                listener.splitCompleted(splitCompletedEvent);
            }
            catch (Throwable e) {
                log.warn(e, "Failed to publish SplitCompletedEvent for query %s", new Object[]{splitCompletedEvent.getQueryId()});
            }
        }
    }

    @Managed
    @Nested
    public TimeStat getQueryCreatedTime() {
        return this.queryCreatedTime;
    }

    @Managed
    @Nested
    public TimeStat getQueryCompletedTime() {
        return this.queryCompletedTime;
    }

    @Managed
    @Nested
    public TimeStat getSplitCompletedTime() {
        return this.splitCompletedTime;
    }

    @Managed
    public int getConcurrentQueryCompletedEvents() {
        return this.concurrentQueryCompletedEvents.get();
    }

    @PreDestroy
    public void shutdown() {
        for (EventListener listener : this.configuredEventListeners.getAndSet(List.of())) {
            try {
                listener.shutdown();
            }
            catch (Throwable e) {
                log.warn(e, "Failed to shutdown event listener: %s", new Object[]{listener.getClass().getCanonicalName()});
            }
        }
    }
}

