/*
 * Decompiled with CFR 0.152.
 */
package io.trino.exchange;

import com.google.common.base.Preconditions;
import com.google.common.base.Strings;
import com.google.inject.Inject;
import io.airlift.configuration.ConfigurationLoader;
import io.airlift.configuration.secrets.SecretsResolver;
import io.airlift.log.Logger;
import io.opentelemetry.api.OpenTelemetry;
import io.opentelemetry.api.trace.Tracer;
import io.trino.exchange.ExchangeManagerContextInstance;
import io.trino.spi.ErrorCodeSupplier;
import io.trino.spi.StandardErrorCode;
import io.trino.spi.TrinoException;
import io.trino.spi.classloader.ThreadContextClassLoader;
import io.trino.spi.exchange.ExchangeManager;
import io.trino.spi.exchange.ExchangeManagerContext;
import io.trino.spi.exchange.ExchangeManagerFactory;
import jakarta.annotation.PreDestroy;
import java.io.File;
import java.io.IOException;
import java.io.UncheckedIOException;
import java.util.HashMap;
import java.util.Map;
import java.util.Objects;
import java.util.concurrent.ConcurrentHashMap;

public class ExchangeManagerRegistry {
    private static final Logger log = Logger.get(ExchangeManagerRegistry.class);
    private static final File CONFIG_FILE = new File("etc/exchange-manager.properties");
    private static final String EXCHANGE_MANAGER_NAME_PROPERTY = "exchange-manager.name";
    private final OpenTelemetry openTelemetry;
    private final Tracer tracer;
    private final Map<String, ExchangeManagerFactory> exchangeManagerFactories = new ConcurrentHashMap<String, ExchangeManagerFactory>();
    private volatile ExchangeManager exchangeManager;
    private final SecretsResolver secretsResolver;

    @Inject
    public ExchangeManagerRegistry(OpenTelemetry openTelemetry, Tracer tracer, SecretsResolver secretsResolver) {
        this.openTelemetry = Objects.requireNonNull(openTelemetry, "openTelemetry is null");
        this.tracer = Objects.requireNonNull(tracer, "tracer is null");
        this.secretsResolver = Objects.requireNonNull(secretsResolver, "secretsResolver is null");
    }

    public void addExchangeManagerFactory(ExchangeManagerFactory factory) {
        Objects.requireNonNull(factory, "factory is null");
        if (this.exchangeManagerFactories.putIfAbsent(factory.getName(), factory) != null) {
            throw new IllegalArgumentException(String.format("Exchange manager factory '%s' is already registered", factory.getName()));
        }
    }

    public void loadExchangeManager() {
        if (!CONFIG_FILE.exists()) {
            return;
        }
        Map<String, String> properties = ExchangeManagerRegistry.loadProperties(CONFIG_FILE);
        String name = properties.remove(EXCHANGE_MANAGER_NAME_PROPERTY);
        Preconditions.checkArgument((!Strings.isNullOrEmpty((String)name) ? 1 : 0) != 0, (String)"Exchange manager configuration %s does not contain %s", (Object)CONFIG_FILE, (Object)EXCHANGE_MANAGER_NAME_PROPERTY);
        this.loadExchangeManager(name, properties);
    }

    public synchronized void loadExchangeManager(String name, Map<String, String> properties) {
        ExchangeManager exchangeManager;
        log.info("-- Loading exchange manager %s --", new Object[]{name});
        Preconditions.checkState((this.exchangeManager == null ? 1 : 0) != 0, (Object)"exchangeManager is already loaded");
        ExchangeManagerFactory factory = this.exchangeManagerFactories.get(name);
        Preconditions.checkArgument((factory != null ? 1 : 0) != 0, (String)"Exchange manager factory '%s' is not registered. Available factories: %s", (Object)name, this.exchangeManagerFactories.keySet());
        try (ThreadContextClassLoader threadContextClassLoader = new ThreadContextClassLoader(factory.getClass().getClassLoader());){
            exchangeManager = factory.create(this.secretsResolver.getResolvedConfiguration(properties), (ExchangeManagerContext)new ExchangeManagerContextInstance(this.openTelemetry, this.tracer));
        }
        log.info("-- Loaded exchange manager %s --", new Object[]{name});
        this.exchangeManager = exchangeManager;
    }

    public ExchangeManager getExchangeManager() {
        ExchangeManager exchangeManager = this.exchangeManager;
        if (exchangeManager == null) {
            throw new TrinoException((ErrorCodeSupplier)StandardErrorCode.EXCHANGE_MANAGER_NOT_CONFIGURED, "Exchange manager must be configured for the failure recovery capabilities to be fully functional");
        }
        return exchangeManager;
    }

    @PreDestroy
    public void shutdown() {
        try {
            if (this.exchangeManager != null) {
                this.exchangeManager.shutdown();
            }
        }
        catch (Throwable t) {
            log.error(t, "Error shutting down exchange manager: %s", new Object[]{this.exchangeManager});
        }
    }

    private static Map<String, String> loadProperties(File configFile) {
        try {
            return new HashMap<String, String>(ConfigurationLoader.loadPropertiesFrom((String)configFile.getPath()));
        }
        catch (IOException e) {
            throw new UncheckedIOException("Failed to read configuration file: " + String.valueOf(configFile), e);
        }
    }
}

