/*
 * Decompiled with CFR 0.152.
 */
package io.trino.plugin.mongodb;

import com.google.common.annotations.VisibleForTesting;
import com.google.common.base.Preconditions;
import com.google.common.base.Throwables;
import com.google.common.base.Verify;
import com.google.common.cache.CacheBuilder;
import com.google.common.cache.CacheLoader;
import com.google.common.cache.LoadingCache;
import com.google.common.collect.ImmutableList;
import com.google.common.collect.ImmutableMap;
import com.google.common.collect.ImmutableSet;
import com.google.common.primitives.Primitives;
import com.google.common.primitives.Shorts;
import com.google.common.primitives.SignedBytes;
import com.google.common.util.concurrent.UncheckedExecutionException;
import com.mongodb.MongoClient;
import com.mongodb.client.FindIterable;
import com.mongodb.client.ListIndexesIterable;
import com.mongodb.client.MongoCollection;
import com.mongodb.client.MongoCursor;
import com.mongodb.client.MongoDatabase;
import com.mongodb.client.model.IndexOptions;
import com.mongodb.client.result.DeleteResult;
import io.airlift.log.Logger;
import io.airlift.slice.Slice;
import io.trino.plugin.mongodb.MongoClientConfig;
import io.trino.plugin.mongodb.MongoColumnHandle;
import io.trino.plugin.mongodb.MongoIndex;
import io.trino.plugin.mongodb.MongoTable;
import io.trino.plugin.mongodb.MongoTableHandle;
import io.trino.plugin.mongodb.ObjectIdType;
import io.trino.spi.TrinoException;
import io.trino.spi.connector.ColumnHandle;
import io.trino.spi.connector.ColumnMetadata;
import io.trino.spi.connector.SchemaNotFoundException;
import io.trino.spi.connector.SchemaTableName;
import io.trino.spi.connector.TableNotFoundException;
import io.trino.spi.predicate.Domain;
import io.trino.spi.predicate.Range;
import io.trino.spi.predicate.TupleDomain;
import io.trino.spi.type.BigintType;
import io.trino.spi.type.BooleanType;
import io.trino.spi.type.DoubleType;
import io.trino.spi.type.IntegerType;
import io.trino.spi.type.NamedTypeSignature;
import io.trino.spi.type.RowFieldName;
import io.trino.spi.type.SmallintType;
import io.trino.spi.type.TimestampType;
import io.trino.spi.type.TinyintType;
import io.trino.spi.type.Type;
import io.trino.spi.type.TypeManager;
import io.trino.spi.type.TypeSignature;
import io.trino.spi.type.TypeSignatureParameter;
import io.trino.spi.type.VarcharType;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Date;
import java.util.HashSet;
import java.util.List;
import java.util.Locale;
import java.util.Map;
import java.util.Objects;
import java.util.Optional;
import java.util.Set;
import java.util.concurrent.TimeUnit;
import java.util.stream.Collectors;
import java.util.stream.IntStream;
import org.bson.Document;
import org.bson.conversions.Bson;
import org.bson.types.ObjectId;

public class MongoSession {
    private static final Logger log = Logger.get(MongoSession.class);
    private static final List<String> SYSTEM_TABLES = Arrays.asList("system.indexes", "system.users", "system.version");
    private static final String TABLE_NAME_KEY = "table";
    private static final String FIELDS_KEY = "fields";
    private static final String FIELDS_NAME_KEY = "name";
    private static final String FIELDS_TYPE_KEY = "type";
    private static final String FIELDS_HIDDEN_KEY = "hidden";
    private static final String OR_OP = "$or";
    private static final String EQ_OP = "$eq";
    private static final String NOT_EQ_OP = "$ne";
    private static final String GTE_OP = "$gte";
    private static final String GT_OP = "$gt";
    private static final String LT_OP = "$lt";
    private static final String LTE_OP = "$lte";
    private static final String IN_OP = "$in";
    private final TypeManager typeManager;
    private final MongoClient client;
    private final String schemaCollection;
    private final boolean caseInsensitiveNameMatching;
    private final int cursorBatchSize;
    private final LoadingCache<SchemaTableName, MongoTable> tableCache;
    private final String implicitPrefix;

    public MongoSession(TypeManager typeManager, MongoClient client, MongoClientConfig config) {
        this.typeManager = Objects.requireNonNull(typeManager, "typeManager is null");
        this.client = Objects.requireNonNull(client, "client is null");
        this.schemaCollection = Objects.requireNonNull(config.getSchemaCollection(), "config.getSchemaCollection() is null");
        this.caseInsensitiveNameMatching = config.isCaseInsensitiveNameMatching();
        this.cursorBatchSize = config.getCursorBatchSize();
        this.implicitPrefix = Objects.requireNonNull(config.getImplicitRowFieldPrefix(), "config.getImplicitRowFieldPrefix() is null");
        this.tableCache = CacheBuilder.newBuilder().expireAfterWrite(1L, TimeUnit.HOURS).refreshAfterWrite(1L, TimeUnit.MINUTES).build(CacheLoader.from(this::loadTableSchema));
    }

    public void shutdown() {
        this.client.close();
    }

    public List<String> getAllSchemas() {
        return (List)ImmutableList.copyOf((Iterable)this.client.listDatabaseNames()).stream().map(name -> name.toLowerCase(Locale.ENGLISH)).collect(ImmutableList.toImmutableList());
    }

    public Set<String> getAllTables(String schema) throws SchemaNotFoundException {
        String schemaName = this.toRemoteSchemaName(schema);
        ImmutableSet.Builder builder = ImmutableSet.builder();
        builder.addAll((Iterable)ImmutableList.copyOf((Iterable)this.client.getDatabase(schemaName).listCollectionNames()).stream().filter(name -> !name.equals(this.schemaCollection)).filter(name -> !SYSTEM_TABLES.contains(name)).collect(Collectors.toSet()));
        builder.addAll(this.getTableMetadataNames(schema));
        return builder.build();
    }

    public MongoTable getTable(SchemaTableName tableName) throws TableNotFoundException {
        try {
            return (MongoTable)this.tableCache.getUnchecked((Object)tableName);
        }
        catch (UncheckedExecutionException e) {
            Throwables.throwIfInstanceOf((Throwable)e.getCause(), TrinoException.class);
            throw e;
        }
    }

    public void createTable(SchemaTableName name, List<MongoColumnHandle> columns) {
        this.createTableMetadata(name, columns);
    }

    public void dropTable(SchemaTableName tableName) {
        this.deleteTableMetadata(tableName);
        this.getCollection(tableName).drop();
        this.tableCache.invalidate((Object)tableName);
    }

    public void addColumn(SchemaTableName schemaTableName, ColumnMetadata columnMetadata) {
        String schemaName = this.toRemoteSchemaName(schemaTableName.getSchemaName());
        String tableName = this.toRemoteTableName(schemaName, schemaTableName.getTableName());
        Document metadata = this.getTableMetadata(schemaName, tableName);
        ArrayList<Document> columns = new ArrayList<Document>(this.getColumnMetadata(metadata));
        Document newColumn = new Document();
        newColumn.append(FIELDS_NAME_KEY, (Object)columnMetadata.getName());
        newColumn.append(FIELDS_TYPE_KEY, (Object)columnMetadata.getType().getTypeSignature().toString());
        newColumn.append(FIELDS_HIDDEN_KEY, (Object)false);
        columns.add(newColumn);
        metadata.append(FIELDS_KEY, columns);
        MongoDatabase db = this.client.getDatabase(schemaName);
        MongoCollection schema = db.getCollection(this.schemaCollection);
        schema.findOneAndReplace((Bson)new Document(TABLE_NAME_KEY, (Object)tableName), (Object)metadata);
        this.tableCache.invalidate((Object)schemaTableName);
    }

    public void dropColumn(SchemaTableName schemaTableName, String columnName) {
        String remoteSchemaName = this.toRemoteSchemaName(schemaTableName.getSchemaName());
        String remoteTableName = this.toRemoteTableName(remoteSchemaName, schemaTableName.getTableName());
        Document metadata = this.getTableMetadata(remoteSchemaName, remoteTableName);
        List columns = (List)this.getColumnMetadata(metadata).stream().filter(document -> !document.getString((Object)FIELDS_NAME_KEY).equals(columnName)).collect(ImmutableList.toImmutableList());
        metadata.append(FIELDS_KEY, (Object)columns);
        MongoDatabase database = this.client.getDatabase(remoteSchemaName);
        MongoCollection schema = database.getCollection(this.schemaCollection);
        schema.findOneAndReplace((Bson)new Document(TABLE_NAME_KEY, (Object)remoteTableName), (Object)metadata);
        this.tableCache.invalidate((Object)schemaTableName);
    }

    private MongoTable loadTableSchema(SchemaTableName schemaTableName) throws TableNotFoundException {
        String schemaName = this.toRemoteSchemaName(schemaTableName.getSchemaName());
        String tableName = this.toRemoteTableName(schemaName, schemaTableName.getTableName());
        Document tableMeta = this.getTableMetadata(schemaName, tableName);
        ImmutableList.Builder columnHandles = ImmutableList.builder();
        for (Document columnMetadata : this.getColumnMetadata(tableMeta)) {
            MongoColumnHandle columnHandle = this.buildColumnHandle(columnMetadata);
            columnHandles.add((Object)columnHandle);
        }
        MongoTableHandle tableHandle = new MongoTableHandle(schemaTableName);
        return new MongoTable(tableHandle, (List<MongoColumnHandle>)columnHandles.build(), this.getIndexes(schemaName, tableName));
    }

    private MongoColumnHandle buildColumnHandle(Document columnMeta) {
        String name = columnMeta.getString((Object)FIELDS_NAME_KEY);
        String typeString = columnMeta.getString((Object)FIELDS_TYPE_KEY);
        boolean hidden = columnMeta.getBoolean((Object)FIELDS_HIDDEN_KEY, false);
        Type type = this.typeManager.fromSqlType(typeString);
        return new MongoColumnHandle(name, type, hidden);
    }

    private List<Document> getColumnMetadata(Document doc) {
        if (!doc.containsKey((Object)FIELDS_KEY)) {
            return ImmutableList.of();
        }
        return (List)doc.get((Object)FIELDS_KEY);
    }

    public MongoCollection<Document> getCollection(SchemaTableName tableName) {
        return this.getCollection(tableName.getSchemaName(), tableName.getTableName());
    }

    private MongoCollection<Document> getCollection(String schema, String table) {
        String schemaName = this.toRemoteSchemaName(schema);
        String tableName = this.toRemoteTableName(schemaName, table);
        return this.client.getDatabase(schemaName).getCollection(tableName);
    }

    public List<MongoIndex> getIndexes(String schemaName, String tableName) {
        if (this.isView(schemaName, tableName)) {
            return ImmutableList.of();
        }
        MongoCollection collection = this.client.getDatabase(schemaName).getCollection(tableName);
        return MongoIndex.parse((ListIndexesIterable<Document>)collection.listIndexes());
    }

    public MongoCursor<Document> execute(MongoTableHandle tableHandle, List<MongoColumnHandle> columns) {
        Document output = new Document();
        for (MongoColumnHandle column : columns) {
            output.append(column.getName(), (Object)1);
        }
        MongoCollection<Document> collection = this.getCollection(tableHandle.getSchemaTableName());
        Document query = MongoSession.buildQuery(tableHandle.getConstraint());
        FindIterable iterable = collection.find((Bson)query).projection((Bson)output);
        tableHandle.getLimit().ifPresent(arg_0 -> ((FindIterable)iterable).limit(arg_0));
        log.debug("Find documents: collection: %s, filter: %s, projection: %s", new Object[]{tableHandle.getSchemaTableName(), query.toJson(), output.toJson()});
        if (this.cursorBatchSize != 0) {
            iterable.batchSize(this.cursorBatchSize);
        }
        return iterable.iterator();
    }

    @VisibleForTesting
    static Document buildQuery(TupleDomain<ColumnHandle> tupleDomain) {
        Document query = new Document();
        if (tupleDomain.getDomains().isPresent()) {
            for (Map.Entry entry : ((Map)tupleDomain.getDomains().get()).entrySet()) {
                MongoColumnHandle column = (MongoColumnHandle)entry.getKey();
                Optional<Document> predicate = MongoSession.buildPredicate(column, (Domain)entry.getValue());
                predicate.ifPresent(arg_0 -> ((Document)query).putAll(arg_0));
            }
        }
        return query;
    }

    private static Optional<Document> buildPredicate(MongoColumnHandle column, Domain domain) {
        String name = column.getName();
        Type type = column.getType();
        if (domain.getValues().isNone() && domain.isNullAllowed()) {
            return Optional.of(MongoSession.documentOf(name, MongoSession.isNullPredicate()));
        }
        if (domain.getValues().isAll() && !domain.isNullAllowed()) {
            return Optional.of(MongoSession.documentOf(name, MongoSession.isNotNullPredicate()));
        }
        ArrayList<Object> singleValues = new ArrayList<Object>();
        ArrayList<Document> disjuncts = new ArrayList<Document>();
        for (Range range : domain.getValues().getRanges().getOrderedRanges()) {
            Optional<Object> translated;
            if (range.isSingleValue()) {
                Optional<Object> translated2 = MongoSession.translateValue(range.getSingleValue(), type);
                if (translated2.isEmpty()) {
                    return Optional.empty();
                }
                singleValues.add(translated2.get());
                continue;
            }
            Document rangeConjuncts = new Document();
            if (!range.isLowUnbounded()) {
                translated = MongoSession.translateValue(range.getLowBoundedValue(), type);
                if (translated.isEmpty()) {
                    return Optional.empty();
                }
                rangeConjuncts.put(range.isLowInclusive() ? GTE_OP : GT_OP, translated.get());
            }
            if (!range.isHighUnbounded()) {
                translated = MongoSession.translateValue(range.getHighBoundedValue(), type);
                if (translated.isEmpty()) {
                    return Optional.empty();
                }
                rangeConjuncts.put(range.isHighInclusive() ? LTE_OP : LT_OP, translated.get());
            }
            Verify.verify((!rangeConjuncts.isEmpty() ? 1 : 0) != 0);
            disjuncts.add(rangeConjuncts);
        }
        if (singleValues.size() == 1) {
            disjuncts.add(MongoSession.documentOf(EQ_OP, singleValues.get(0)));
        } else if (singleValues.size() > 1) {
            disjuncts.add(MongoSession.documentOf(IN_OP, singleValues));
        }
        if (domain.isNullAllowed()) {
            disjuncts.add(MongoSession.isNullPredicate());
        }
        return Optional.of(MongoSession.orPredicate((List)disjuncts.stream().map(disjunct -> new Document(name, disjunct)).collect(ImmutableList.toImmutableList())));
    }

    private static Optional<Object> translateValue(Object trinoNativeValue, Type type) {
        Objects.requireNonNull(trinoNativeValue, "trinoNativeValue is null");
        Objects.requireNonNull(type, "type is null");
        Preconditions.checkArgument((boolean)Primitives.wrap((Class)type.getJavaType()).isInstance(trinoNativeValue), (String)"%s (%s) is not a valid representation for %s", (Object)trinoNativeValue, trinoNativeValue.getClass(), (Object)type);
        if (type == TinyintType.TINYINT) {
            return Optional.of(Long.valueOf(SignedBytes.checkedCast((long)((Long)trinoNativeValue))));
        }
        if (type == SmallintType.SMALLINT) {
            return Optional.of(Long.valueOf(Shorts.checkedCast((long)((Long)trinoNativeValue))));
        }
        if (type == IntegerType.INTEGER) {
            return Optional.of(Long.valueOf(Math.toIntExact((Long)trinoNativeValue)));
        }
        if (type == BigintType.BIGINT) {
            return Optional.of(trinoNativeValue);
        }
        if (type instanceof ObjectIdType) {
            return Optional.of(new ObjectId(((Slice)trinoNativeValue).getBytes()));
        }
        if (type instanceof VarcharType) {
            return Optional.of(((Slice)trinoNativeValue).toStringUtf8());
        }
        return Optional.empty();
    }

    private static Document documentOf(String key, Object value) {
        return new Document(key, value);
    }

    private static Document orPredicate(List<Document> values) {
        Preconditions.checkState((!values.isEmpty() ? 1 : 0) != 0);
        if (values.size() == 1) {
            return values.get(0);
        }
        return new Document(OR_OP, values);
    }

    private static Document isNullPredicate() {
        return MongoSession.documentOf(EQ_OP, null);
    }

    private static Document isNotNullPredicate() {
        return MongoSession.documentOf(NOT_EQ_OP, null);
    }

    private Document getTableMetadata(String schemaName, String tableName) throws TableNotFoundException {
        MongoDatabase db = this.client.getDatabase(schemaName);
        MongoCollection schema = db.getCollection(this.schemaCollection);
        Document doc = (Document)schema.find((Bson)new Document(TABLE_NAME_KEY, (Object)tableName)).first();
        if (doc == null) {
            if (!this.collectionExists(db, tableName)) {
                throw new TableNotFoundException(new SchemaTableName(schemaName, tableName), String.format("Table '%s.%s' not found", schemaName, tableName), null);
            }
            Document metadata = new Document(TABLE_NAME_KEY, (Object)tableName);
            metadata.append(FIELDS_KEY, this.guessTableFields(schemaName, tableName));
            if (!this.indexExists((MongoCollection<Document>)schema)) {
                schema.createIndex((Bson)new Document(TABLE_NAME_KEY, (Object)1), new IndexOptions().unique(true));
            }
            schema.insertOne((Object)metadata);
            return metadata;
        }
        return doc;
    }

    public boolean collectionExists(MongoDatabase db, String collectionName) {
        for (String name : db.listCollectionNames()) {
            if (!name.equalsIgnoreCase(collectionName)) continue;
            return true;
        }
        return false;
    }

    private boolean indexExists(MongoCollection<Document> schemaCollection) {
        return MongoIndex.parse((ListIndexesIterable<Document>)schemaCollection.listIndexes()).stream().anyMatch(index -> index.getKeys().size() == 1 && TABLE_NAME_KEY.equals(index.getKeys().get(0).getName()));
    }

    private Set<String> getTableMetadataNames(String schemaName) throws TableNotFoundException {
        MongoDatabase db = this.client.getDatabase(schemaName);
        MongoCursor cursor = db.getCollection(this.schemaCollection).find().projection((Bson)new Document(TABLE_NAME_KEY, (Object)true)).iterator();
        HashSet<String> names = new HashSet<String>();
        while (cursor.hasNext()) {
            names.add(((Document)cursor.next()).getString((Object)TABLE_NAME_KEY));
        }
        return names;
    }

    private void createTableMetadata(SchemaTableName schemaTableName, List<MongoColumnHandle> columns) throws TableNotFoundException {
        String schemaName = schemaTableName.getSchemaName();
        String tableName = schemaTableName.getTableName();
        MongoDatabase db = this.client.getDatabase(schemaName);
        Document metadata = new Document(TABLE_NAME_KEY, (Object)tableName);
        ArrayList<Document> fields = new ArrayList<Document>();
        if (!columns.stream().anyMatch(c -> c.getName().equals("_id"))) {
            fields.add(new MongoColumnHandle("_id", (Type)ObjectIdType.OBJECT_ID, true).getDocument());
        }
        fields.addAll(columns.stream().map(MongoColumnHandle::getDocument).collect(Collectors.toList()));
        metadata.append(FIELDS_KEY, fields);
        MongoCollection schema = db.getCollection(this.schemaCollection);
        if (!this.indexExists((MongoCollection<Document>)schema)) {
            schema.createIndex((Bson)new Document(TABLE_NAME_KEY, (Object)1), new IndexOptions().unique(true));
        }
        schema.insertOne((Object)metadata);
    }

    private boolean deleteTableMetadata(SchemaTableName schemaTableName) {
        String schemaName = this.toRemoteSchemaName(schemaTableName.getSchemaName());
        String tableName = this.toRemoteTableName(schemaName, schemaTableName.getTableName());
        MongoDatabase db = this.client.getDatabase(schemaName);
        if (!this.collectionExists(db, tableName) && ((Document)db.getCollection(this.schemaCollection).find((Bson)new Document(TABLE_NAME_KEY, (Object)tableName)).first()).isEmpty()) {
            return false;
        }
        DeleteResult result = db.getCollection(this.schemaCollection).deleteOne((Bson)new Document(TABLE_NAME_KEY, (Object)tableName));
        return result.getDeletedCount() == 1L;
    }

    private List<Document> guessTableFields(String schemaName, String tableName) {
        MongoDatabase db = this.client.getDatabase(schemaName);
        Document doc = (Document)db.getCollection(tableName).find().first();
        if (doc == null) {
            return ImmutableList.of();
        }
        ImmutableList.Builder builder = ImmutableList.builder();
        for (String key : doc.keySet()) {
            Object value = doc.get((Object)key);
            Optional<TypeSignature> fieldType = this.guessFieldType(value);
            if (fieldType.isPresent()) {
                Document metadata = new Document();
                metadata.append(FIELDS_NAME_KEY, (Object)key);
                metadata.append(FIELDS_TYPE_KEY, (Object)fieldType.get().toString());
                metadata.append(FIELDS_HIDDEN_KEY, (Object)(key.equals("_id") && fieldType.get().equals((Object)ObjectIdType.OBJECT_ID.getTypeSignature()) ? 1 : 0));
                builder.add((Object)metadata);
                continue;
            }
            log.debug("Unable to guess field type from %s : %s", new Object[]{value == null ? "null" : value.getClass().getName(), value});
        }
        return builder.build();
    }

    private Optional<TypeSignature> guessFieldType(Object value) {
        if (value == null) {
            return Optional.empty();
        }
        TypeSignature typeSignature = null;
        if (value instanceof String) {
            typeSignature = VarcharType.createUnboundedVarcharType().getTypeSignature();
        } else if (value instanceof Integer || value instanceof Long) {
            typeSignature = BigintType.BIGINT.getTypeSignature();
        } else if (value instanceof Boolean) {
            typeSignature = BooleanType.BOOLEAN.getTypeSignature();
        } else if (value instanceof Float || value instanceof Double) {
            typeSignature = DoubleType.DOUBLE.getTypeSignature();
        } else if (value instanceof Date) {
            typeSignature = TimestampType.TIMESTAMP_MILLIS.getTypeSignature();
        } else if (value instanceof ObjectId) {
            typeSignature = ObjectIdType.OBJECT_ID.getTypeSignature();
        } else if (value instanceof List) {
            List subTypes = ((List)value).stream().map(this::guessFieldType).collect(Collectors.toList());
            if (subTypes.isEmpty() || subTypes.stream().anyMatch(Optional::isEmpty)) {
                return Optional.empty();
            }
            Set signatures = subTypes.stream().map(Optional::get).collect(Collectors.toSet());
            typeSignature = signatures.size() == 1 ? new TypeSignature("array", signatures.stream().map(TypeSignatureParameter::typeParameter).collect(Collectors.toList())) : new TypeSignature("row", IntStream.range(0, subTypes.size()).mapToObj(idx -> TypeSignatureParameter.namedTypeParameter((NamedTypeSignature)new NamedTypeSignature(Optional.of(new RowFieldName(String.format("%s%d", this.implicitPrefix, idx + 1))), (TypeSignature)((Optional)subTypes.get(idx)).get()))).collect(Collectors.toList()));
        } else if (value instanceof Document) {
            ArrayList<TypeSignatureParameter> parameters = new ArrayList<TypeSignatureParameter>();
            for (String key : ((Document)value).keySet()) {
                Optional<TypeSignature> fieldType = this.guessFieldType(((Document)value).get((Object)key));
                if (!fieldType.isPresent()) continue;
                parameters.add(TypeSignatureParameter.namedTypeParameter((NamedTypeSignature)new NamedTypeSignature(Optional.of(new RowFieldName(key)), fieldType.get())));
            }
            if (!parameters.isEmpty()) {
                typeSignature = new TypeSignature("row", parameters);
            }
        }
        return Optional.ofNullable(typeSignature);
    }

    private String toRemoteSchemaName(String schemaName) {
        Verify.verify((boolean)schemaName.equals(schemaName.toLowerCase(Locale.ENGLISH)), (String)"schemaName not in lower-case: %s", (Object)schemaName);
        if (!this.caseInsensitiveNameMatching) {
            return schemaName;
        }
        for (String remoteSchemaName : this.client.listDatabaseNames()) {
            if (!schemaName.equals(remoteSchemaName.toLowerCase(Locale.ENGLISH))) continue;
            return remoteSchemaName;
        }
        return schemaName;
    }

    private String toRemoteTableName(String schemaName, String tableName) {
        Verify.verify((boolean)tableName.equals(tableName.toLowerCase(Locale.ENGLISH)), (String)"tableName not in lower-case: %s", (Object)tableName);
        if (!this.caseInsensitiveNameMatching) {
            return tableName;
        }
        for (String remoteTableName : this.client.getDatabase(schemaName).listCollectionNames()) {
            if (!tableName.equals(remoteTableName.toLowerCase(Locale.ENGLISH))) continue;
            return remoteTableName;
        }
        return tableName;
    }

    private boolean isView(String schemaName, String tableName) {
        Document listCollectionsCommand = new Document((Map)new ImmutableMap.Builder().put((Object)"listCollections", (Object)1.0).put((Object)"filter", (Object)MongoSession.documentOf(FIELDS_NAME_KEY, tableName)).put((Object)"nameOnly", (Object)true).build());
        Document cursor = (Document)this.client.getDatabase(schemaName).runCommand((Bson)listCollectionsCommand).get((Object)"cursor", Document.class);
        List firstBatch = (List)cursor.get((Object)"firstBatch", List.class);
        if (firstBatch.isEmpty()) {
            return false;
        }
        String type = ((Document)firstBatch.get(0)).getString((Object)FIELDS_TYPE_KEY);
        return "view".equals(type);
    }
}

