/*
 * Decompiled with CFR 0.152.
 */
package io.trino.plugin.mongodb;

import com.google.common.base.Preconditions;
import com.google.common.collect.ImmutableList;
import com.google.common.collect.ImmutableMap;
import io.airlift.log.Logger;
import io.airlift.slice.Slice;
import io.trino.plugin.mongodb.MongoColumnHandle;
import io.trino.plugin.mongodb.MongoIndex;
import io.trino.plugin.mongodb.MongoInsertTableHandle;
import io.trino.plugin.mongodb.MongoOutputTableHandle;
import io.trino.plugin.mongodb.MongoSession;
import io.trino.plugin.mongodb.MongoTable;
import io.trino.plugin.mongodb.MongoTableHandle;
import io.trino.spi.connector.ColumnHandle;
import io.trino.spi.connector.ColumnMetadata;
import io.trino.spi.connector.ConnectorInsertTableHandle;
import io.trino.spi.connector.ConnectorMetadata;
import io.trino.spi.connector.ConnectorNewTableLayout;
import io.trino.spi.connector.ConnectorOutputMetadata;
import io.trino.spi.connector.ConnectorOutputTableHandle;
import io.trino.spi.connector.ConnectorSession;
import io.trino.spi.connector.ConnectorTableHandle;
import io.trino.spi.connector.ConnectorTableMetadata;
import io.trino.spi.connector.ConnectorTableProperties;
import io.trino.spi.connector.Constraint;
import io.trino.spi.connector.ConstraintApplicationResult;
import io.trino.spi.connector.LimitApplicationResult;
import io.trino.spi.connector.NotFoundException;
import io.trino.spi.connector.SchemaTableName;
import io.trino.spi.connector.SchemaTablePrefix;
import io.trino.spi.connector.SortingProperty;
import io.trino.spi.connector.TableNotFoundException;
import io.trino.spi.predicate.TupleDomain;
import io.trino.spi.statistics.ComputedStatistics;
import java.util.Collection;
import java.util.List;
import java.util.Locale;
import java.util.Map;
import java.util.Objects;
import java.util.Optional;
import java.util.OptionalInt;
import java.util.concurrent.atomic.AtomicReference;
import java.util.stream.Collectors;

public class MongoMetadata
implements ConnectorMetadata {
    private static final Logger log = Logger.get(MongoMetadata.class);
    private final MongoSession mongoSession;
    private final AtomicReference<Runnable> rollbackAction = new AtomicReference();

    public MongoMetadata(MongoSession mongoSession) {
        this.mongoSession = Objects.requireNonNull(mongoSession, "mongoSession is null");
    }

    public List<String> listSchemaNames(ConnectorSession session) {
        return this.mongoSession.getAllSchemas();
    }

    public MongoTableHandle getTableHandle(ConnectorSession session, SchemaTableName tableName) {
        Objects.requireNonNull(tableName, "tableName is null");
        try {
            return this.mongoSession.getTable(tableName).getTableHandle();
        }
        catch (TableNotFoundException e) {
            log.debug((Throwable)e, "Table(%s) not found", new Object[]{tableName});
            return null;
        }
    }

    public ConnectorTableMetadata getTableMetadata(ConnectorSession session, ConnectorTableHandle tableHandle) {
        Objects.requireNonNull(tableHandle, "tableHandle is null");
        SchemaTableName tableName = MongoMetadata.getTableName(tableHandle);
        return this.getTableMetadata(session, tableName);
    }

    public List<SchemaTableName> listTables(ConnectorSession session, Optional<String> optionalSchemaName) {
        List schemaNames = (List)optionalSchemaName.map(ImmutableList::of).orElseGet(() -> (ImmutableList)this.listSchemaNames(session));
        ImmutableList.Builder tableNames = ImmutableList.builder();
        for (String schemaName : schemaNames) {
            for (String tableName : this.mongoSession.getAllTables(schemaName)) {
                tableNames.add((Object)new SchemaTableName(schemaName, tableName.toLowerCase(Locale.ENGLISH)));
            }
        }
        return tableNames.build();
    }

    public Map<String, ColumnHandle> getColumnHandles(ConnectorSession session, ConnectorTableHandle tableHandle) {
        MongoTableHandle table = (MongoTableHandle)tableHandle;
        List<MongoColumnHandle> columns = this.mongoSession.getTable(table.getSchemaTableName()).getColumns();
        ImmutableMap.Builder columnHandles = ImmutableMap.builder();
        for (MongoColumnHandle columnHandle : columns) {
            columnHandles.put((Object)columnHandle.getName(), (Object)columnHandle);
        }
        return columnHandles.build();
    }

    public Map<SchemaTableName, List<ColumnMetadata>> listTableColumns(ConnectorSession session, SchemaTablePrefix prefix) {
        Objects.requireNonNull(prefix, "prefix is null");
        ImmutableMap.Builder columns = ImmutableMap.builder();
        for (SchemaTableName tableName : this.listTables(session, prefix)) {
            try {
                columns.put((Object)tableName, (Object)this.getTableMetadata(session, tableName).getColumns());
            }
            catch (NotFoundException notFoundException) {}
        }
        return columns.build();
    }

    private List<SchemaTableName> listTables(ConnectorSession session, SchemaTablePrefix prefix) {
        if (prefix.getTable().isEmpty()) {
            return this.listTables(session, prefix.getSchema());
        }
        return ImmutableList.of((Object)prefix.toSchemaTableName());
    }

    public ColumnMetadata getColumnMetadata(ConnectorSession session, ConnectorTableHandle tableHandle, ColumnHandle columnHandle) {
        return ((MongoColumnHandle)columnHandle).toColumnMetadata();
    }

    public void createTable(ConnectorSession session, ConnectorTableMetadata tableMetadata, boolean ignoreExisting) {
        this.mongoSession.createTable(tableMetadata.getTable(), MongoMetadata.buildColumnHandles(tableMetadata));
    }

    public void dropTable(ConnectorSession session, ConnectorTableHandle tableHandle) {
        MongoTableHandle table = (MongoTableHandle)tableHandle;
        this.mongoSession.dropTable(table.getSchemaTableName());
    }

    public void addColumn(ConnectorSession session, ConnectorTableHandle tableHandle, ColumnMetadata column) {
        this.mongoSession.addColumn(((MongoTableHandle)tableHandle).getSchemaTableName(), column);
    }

    public void dropColumn(ConnectorSession session, ConnectorTableHandle tableHandle, ColumnHandle column) {
        this.mongoSession.dropColumn(((MongoTableHandle)tableHandle).getSchemaTableName(), ((MongoColumnHandle)column).getName());
    }

    public ConnectorOutputTableHandle beginCreateTable(ConnectorSession session, ConnectorTableMetadata tableMetadata, Optional<ConnectorNewTableLayout> layout) {
        List<MongoColumnHandle> columns = MongoMetadata.buildColumnHandles(tableMetadata);
        this.mongoSession.createTable(tableMetadata.getTable(), columns);
        this.setRollback(() -> this.mongoSession.dropTable(tableMetadata.getTable()));
        return new MongoOutputTableHandle(tableMetadata.getTable(), columns.stream().filter(c -> !c.isHidden()).collect(Collectors.toList()));
    }

    public Optional<ConnectorOutputMetadata> finishCreateTable(ConnectorSession session, ConnectorOutputTableHandle tableHandle, Collection<Slice> fragments, Collection<ComputedStatistics> computedStatistics) {
        this.clearRollback();
        return Optional.empty();
    }

    public ConnectorInsertTableHandle beginInsert(ConnectorSession session, ConnectorTableHandle tableHandle) {
        MongoTableHandle table = (MongoTableHandle)tableHandle;
        List<MongoColumnHandle> columns = this.mongoSession.getTable(table.getSchemaTableName()).getColumns();
        return new MongoInsertTableHandle(table.getSchemaTableName(), (List)columns.stream().filter(column -> !column.isHidden()).peek(column -> MongoMetadata.validateColumnNameForInsert(column.getName())).collect(ImmutableList.toImmutableList()));
    }

    public Optional<ConnectorOutputMetadata> finishInsert(ConnectorSession session, ConnectorInsertTableHandle insertHandle, Collection<Slice> fragments, Collection<ComputedStatistics> computedStatistics) {
        return Optional.empty();
    }

    public boolean usesLegacyTableLayouts() {
        return false;
    }

    public ConnectorTableProperties getTableProperties(ConnectorSession session, ConnectorTableHandle table) {
        MongoTableHandle tableHandle = (MongoTableHandle)table;
        Optional partitioningColumns = Optional.empty();
        ImmutableList.Builder localProperties = ImmutableList.builder();
        MongoTable tableInfo = this.mongoSession.getTable(tableHandle.getSchemaTableName());
        Map<String, ColumnHandle> columns = this.getColumnHandles(session, tableHandle);
        for (MongoIndex index : tableInfo.getIndexes()) {
            for (MongoIndex.MongodbIndexKey key : index.getKeys()) {
                if (key.getSortOrder().isEmpty() || columns.get(key.getName()) == null) continue;
                localProperties.add((Object)new SortingProperty((Object)columns.get(key.getName()), key.getSortOrder().get()));
            }
        }
        return new ConnectorTableProperties(TupleDomain.all(), Optional.empty(), partitioningColumns, Optional.empty(), (List)localProperties.build());
    }

    public Optional<LimitApplicationResult<ConnectorTableHandle>> applyLimit(ConnectorSession session, ConnectorTableHandle table, long limit) {
        MongoTableHandle handle = (MongoTableHandle)table;
        if (limit == 0L) {
            return Optional.empty();
        }
        if (limit > Integer.MAX_VALUE) {
            return Optional.empty();
        }
        if (handle.getLimit().isPresent() && (long)handle.getLimit().getAsInt() <= limit) {
            return Optional.empty();
        }
        return Optional.of(new LimitApplicationResult((Object)new MongoTableHandle(handle.getSchemaTableName(), handle.getConstraint(), OptionalInt.of(Math.toIntExact(limit))), true, false));
    }

    public Optional<ConstraintApplicationResult<ConnectorTableHandle>> applyFilter(ConnectorSession session, ConnectorTableHandle table, Constraint constraint) {
        TupleDomain newDomain;
        MongoTableHandle handle = (MongoTableHandle)table;
        TupleDomain<ColumnHandle> oldDomain = handle.getConstraint();
        if (oldDomain.equals((Object)(newDomain = oldDomain.intersect(constraint.getSummary())))) {
            return Optional.empty();
        }
        handle = new MongoTableHandle(handle.getSchemaTableName(), (TupleDomain<ColumnHandle>)newDomain, handle.getLimit());
        return Optional.of(new ConstraintApplicationResult((Object)handle, constraint.getSummary(), false));
    }

    private void setRollback(Runnable action) {
        Preconditions.checkState((boolean)this.rollbackAction.compareAndSet(null, action), (Object)"rollback action is already set");
    }

    private void clearRollback() {
        this.rollbackAction.set(null);
    }

    public void rollback() {
        Optional.ofNullable(this.rollbackAction.getAndSet(null)).ifPresent(Runnable::run);
    }

    private static SchemaTableName getTableName(ConnectorTableHandle tableHandle) {
        return ((MongoTableHandle)tableHandle).getSchemaTableName();
    }

    private ConnectorTableMetadata getTableMetadata(ConnectorSession session, SchemaTableName tableName) {
        MongoTableHandle tableHandle = this.mongoSession.getTable(tableName).getTableHandle();
        ImmutableList columns = ImmutableList.copyOf((Collection)this.getColumnHandles(session, tableHandle).values().stream().map(MongoColumnHandle.class::cast).map(MongoColumnHandle::toColumnMetadata).collect(Collectors.toList()));
        return new ConnectorTableMetadata(tableName, (List)columns);
    }

    private static List<MongoColumnHandle> buildColumnHandles(ConnectorTableMetadata tableMetadata) {
        return tableMetadata.getColumns().stream().map(m -> new MongoColumnHandle(m.getName(), m.getType(), m.isHidden())).collect(Collectors.toList());
    }

    private static void validateColumnNameForInsert(String columnName) {
        if (columnName.contains("$") || columnName.contains(".")) {
            throw new IllegalArgumentException("Column name must not contain '$' or '.' for INSERT: " + columnName);
        }
    }
}

