/*
 * Decompiled with CFR 0.152.
 */
package io.trino.plugin.mongodb;

import com.google.common.annotations.VisibleForTesting;
import com.google.common.base.Preconditions;
import com.google.common.base.Throwables;
import com.google.common.base.Verify;
import com.google.common.cache.Cache;
import com.google.common.collect.ImmutableList;
import com.google.common.collect.ImmutableMap;
import com.google.common.collect.ImmutableSet;
import com.google.common.collect.Ordering;
import com.google.common.collect.Streams;
import com.google.common.primitives.Primitives;
import com.google.common.primitives.Shorts;
import com.google.common.primitives.SignedBytes;
import com.google.common.util.concurrent.UncheckedExecutionException;
import com.mongodb.DBRef;
import com.mongodb.MongoNamespace;
import com.mongodb.client.FindIterable;
import com.mongodb.client.ListIndexesIterable;
import com.mongodb.client.MongoClient;
import com.mongodb.client.MongoCollection;
import com.mongodb.client.MongoCursor;
import com.mongodb.client.MongoDatabase;
import com.mongodb.client.MongoIterable;
import com.mongodb.client.model.Collation;
import com.mongodb.client.model.Filters;
import com.mongodb.client.model.IndexOptions;
import com.mongodb.client.model.Updates;
import com.mongodb.client.result.DeleteResult;
import io.airlift.log.Logger;
import io.airlift.slice.Slice;
import io.trino.cache.EvictableCacheBuilder;
import io.trino.plugin.mongodb.MongoClientConfig;
import io.trino.plugin.mongodb.MongoColumnHandle;
import io.trino.plugin.mongodb.MongoIndex;
import io.trino.plugin.mongodb.MongoTable;
import io.trino.plugin.mongodb.MongoTableHandle;
import io.trino.plugin.mongodb.ObjectIdType;
import io.trino.plugin.mongodb.RemoteTableName;
import io.trino.plugin.mongodb.ptf.Query;
import io.trino.spi.ErrorCodeSupplier;
import io.trino.spi.HostAddress;
import io.trino.spi.StandardErrorCode;
import io.trino.spi.TrinoException;
import io.trino.spi.connector.ColumnHandle;
import io.trino.spi.connector.ColumnMetadata;
import io.trino.spi.connector.SchemaNotFoundException;
import io.trino.spi.connector.SchemaTableName;
import io.trino.spi.connector.TableNotFoundException;
import io.trino.spi.predicate.Domain;
import io.trino.spi.predicate.Range;
import io.trino.spi.predicate.TupleDomain;
import io.trino.spi.type.BigintType;
import io.trino.spi.type.BooleanType;
import io.trino.spi.type.CharType;
import io.trino.spi.type.Chars;
import io.trino.spi.type.DateTimeEncoding;
import io.trino.spi.type.DateType;
import io.trino.spi.type.DecimalType;
import io.trino.spi.type.Decimals;
import io.trino.spi.type.DoubleType;
import io.trino.spi.type.Int128;
import io.trino.spi.type.IntegerType;
import io.trino.spi.type.NamedTypeSignature;
import io.trino.spi.type.RowFieldName;
import io.trino.spi.type.SmallintType;
import io.trino.spi.type.TimeType;
import io.trino.spi.type.TimestampType;
import io.trino.spi.type.TimestampWithTimeZoneType;
import io.trino.spi.type.Timestamps;
import io.trino.spi.type.TinyintType;
import io.trino.spi.type.Type;
import io.trino.spi.type.TypeManager;
import io.trino.spi.type.TypeSignature;
import io.trino.spi.type.TypeSignatureParameter;
import io.trino.spi.type.VarbinaryType;
import io.trino.spi.type.VarcharType;
import java.math.BigDecimal;
import java.time.Instant;
import java.time.LocalDate;
import java.time.LocalDateTime;
import java.time.LocalTime;
import java.time.ZoneOffset;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Comparator;
import java.util.Date;
import java.util.Iterator;
import java.util.List;
import java.util.Locale;
import java.util.Map;
import java.util.Objects;
import java.util.Optional;
import java.util.Set;
import java.util.concurrent.ExecutionException;
import java.util.concurrent.TimeUnit;
import java.util.stream.Collectors;
import java.util.stream.IntStream;
import org.bson.Document;
import org.bson.conversions.Bson;
import org.bson.types.Binary;
import org.bson.types.Decimal128;
import org.bson.types.ObjectId;

public class MongoSession {
    private static final Logger log = Logger.get(MongoSession.class);
    private static final Set<String> SYSTEM_DATABASES = Set.of("admin", "local", "config");
    private static final List<String> SYSTEM_TABLES = Arrays.asList("system.indexes", "system.users", "system.version", "system.views");
    private static final String TABLE_NAME_KEY = "table";
    private static final String COMMENT_KEY = "comment";
    private static final String FIELDS_KEY = "fields";
    private static final String FIELDS_NAME_KEY = "name";
    private static final String FIELDS_TYPE_KEY = "type";
    private static final String FIELDS_HIDDEN_KEY = "hidden";
    private static final String AND_OP = "$and";
    private static final String OR_OP = "$or";
    private static final String EQ_OP = "$eq";
    private static final String NOT_EQ_OP = "$ne";
    private static final String GTE_OP = "$gte";
    private static final String GT_OP = "$gt";
    private static final String LT_OP = "$lt";
    private static final String LTE_OP = "$lte";
    private static final String IN_OP = "$in";
    public static final String DATABASE_NAME = "databaseName";
    public static final String COLLECTION_NAME = "collectionName";
    public static final String ID = "id";
    private static final Collation SIMPLE_COLLATION = Collation.builder().locale("simple").build();
    private static final Map<String, Object> AUTHORIZED_LIST_COLLECTIONS_COMMAND = ImmutableMap.builder().put((Object)"listCollections", (Object)1.0).put((Object)"nameOnly", (Object)true).put((Object)"authorizedCollections", (Object)true).buildOrThrow();
    private static final Ordering<MongoColumnHandle> COLUMN_HANDLE_ORDERING = Ordering.from(Comparator.comparingInt(columnHandle -> columnHandle.getDereferenceNames().size()));
    private final TypeManager typeManager;
    private final MongoClient client;
    private final String schemaCollection;
    private final boolean caseInsensitiveNameMatching;
    private final int cursorBatchSize;
    private final Cache<SchemaTableName, MongoTable> tableCache;
    private final String implicitPrefix;

    public MongoSession(TypeManager typeManager, MongoClient client, MongoClientConfig config) {
        this.typeManager = Objects.requireNonNull(typeManager, "typeManager is null");
        this.client = Objects.requireNonNull(client, "client is null");
        this.schemaCollection = Objects.requireNonNull(config.getSchemaCollection(), "config.getSchemaCollection() is null");
        this.caseInsensitiveNameMatching = config.isCaseInsensitiveNameMatching();
        this.cursorBatchSize = config.getCursorBatchSize();
        this.implicitPrefix = Objects.requireNonNull(config.getImplicitRowFieldPrefix(), "config.getImplicitRowFieldPrefix() is null");
        this.tableCache = EvictableCacheBuilder.newBuilder().expireAfterWrite(1L, TimeUnit.MINUTES).build();
    }

    public void shutdown() {
        this.client.close();
    }

    public List<HostAddress> getAddresses() {
        return (List)this.client.getClusterDescription().getServerDescriptions().stream().map(description -> HostAddress.fromParts((String)description.getAddress().getHost(), (int)description.getAddress().getPort())).collect(ImmutableList.toImmutableList());
    }

    public List<String> getAllSchemas() {
        return (List)Streams.stream(this.listDatabaseNames()).filter(schema -> !SYSTEM_DATABASES.contains(schema)).map(schema -> schema.toLowerCase(Locale.ENGLISH)).collect(ImmutableList.toImmutableList());
    }

    public void createSchema(String schemaName) {
        this.client.getDatabase(schemaName).createCollection(this.schemaCollection);
    }

    public void dropSchema(String schemaName, boolean cascade) {
        MongoDatabase database = this.client.getDatabase(this.toRemoteSchemaName(schemaName));
        if (!cascade) {
            try (MongoCursor collections = database.listCollectionNames().cursor();){
                while (collections.hasNext()) {
                    if (((String)collections.next()).equals(this.schemaCollection)) continue;
                    throw new TrinoException((ErrorCodeSupplier)StandardErrorCode.SCHEMA_NOT_EMPTY, "Cannot drop non-empty schema '%s'".formatted(schemaName));
                }
            }
        }
        database.drop();
    }

    public Set<String> getAllTables(String schema) throws SchemaNotFoundException {
        String schemaName = this.toRemoteSchemaName(schema);
        ImmutableSet.Builder builder = ImmutableSet.builder();
        builder.addAll((Iterable)ImmutableList.copyOf(this.listCollectionNames(schemaName)).stream().filter(name -> !name.equals(this.schemaCollection)).filter(name -> !SYSTEM_TABLES.contains(name)).collect(Collectors.toSet()));
        builder.addAll(this.getTableMetadataNames(schema));
        return builder.build();
    }

    public MongoTable getTable(SchemaTableName tableName) throws TableNotFoundException {
        try {
            return (MongoTable)this.tableCache.get((Object)tableName, () -> this.loadTableSchema(tableName));
        }
        catch (UncheckedExecutionException | ExecutionException e) {
            Throwables.throwIfInstanceOf((Throwable)e.getCause(), TrinoException.class);
            throw new RuntimeException(e);
        }
    }

    public void createTable(RemoteTableName name, List<MongoColumnHandle> columns, Optional<String> comment) {
        if (this.getAllSchemas().stream().noneMatch(schemaName -> schemaName.equalsIgnoreCase(name.getDatabaseName()))) {
            throw new SchemaNotFoundException(name.getDatabaseName());
        }
        this.createTableMetadata(name, columns, comment);
        this.client.getDatabase(name.getDatabaseName()).createCollection(name.getCollectionName());
    }

    public void dropTable(RemoteTableName remoteTableName) {
        this.deleteTableMetadata(remoteTableName);
        this.getCollection(remoteTableName).drop();
        this.tableCache.invalidate((Object)new SchemaTableName(remoteTableName.getDatabaseName(), remoteTableName.getCollectionName()));
    }

    public void setTableComment(MongoTableHandle table, Optional<String> comment) {
        String remoteSchemaName = table.getRemoteTableName().getDatabaseName();
        String remoteTableName = table.getRemoteTableName().getCollectionName();
        Document metadata = this.getTableMetadata(remoteSchemaName, remoteTableName);
        metadata.append(COMMENT_KEY, comment.orElse(null));
        this.client.getDatabase(remoteSchemaName).getCollection(this.schemaCollection).findOneAndReplace((Bson)new Document(TABLE_NAME_KEY, (Object)remoteTableName), (Object)metadata);
        this.tableCache.invalidate((Object)table.getSchemaTableName());
    }

    public void setColumnComment(MongoTableHandle table, String columnName, Optional<String> comment) {
        String remoteSchemaName = table.getRemoteTableName().getDatabaseName();
        String remoteTableName = table.getRemoteTableName().getCollectionName();
        Document metadata = this.getTableMetadata(remoteSchemaName, remoteTableName);
        ImmutableList.Builder columns = ImmutableList.builder();
        for (Document column : this.getColumnMetadata(metadata)) {
            if (column.getString((Object)FIELDS_NAME_KEY).equals(columnName)) {
                column.append(COMMENT_KEY, comment.orElse(null));
            }
            columns.add((Object)column);
        }
        metadata.append(FIELDS_KEY, (Object)columns.build());
        this.client.getDatabase(remoteSchemaName).getCollection(this.schemaCollection).findOneAndReplace((Bson)new Document(TABLE_NAME_KEY, (Object)remoteTableName), (Object)metadata);
        this.tableCache.invalidate((Object)table.getSchemaTableName());
    }

    public void renameTable(MongoTableHandle table, SchemaTableName newName) {
        String oldSchemaName = table.getRemoteTableName().getDatabaseName();
        String oldTableName = table.getRemoteTableName().getCollectionName();
        String newSchemaName = this.toRemoteSchemaName(newName.getSchemaName());
        MongoCollection oldSchema = this.client.getDatabase(oldSchemaName).getCollection(this.schemaCollection);
        Document tableDefinition = (Document)oldSchema.findOneAndDelete((Bson)new Document(TABLE_NAME_KEY, (Object)oldTableName));
        Objects.requireNonNull(tableDefinition, "Table definition not found in schema collection: " + oldTableName);
        MongoCollection newSchema = this.client.getDatabase(newSchemaName).getCollection(this.schemaCollection);
        tableDefinition.append(TABLE_NAME_KEY, (Object)newName.getTableName());
        newSchema.insertOne((Object)tableDefinition);
        if (this.collectionExists(this.client.getDatabase(oldSchemaName), oldTableName)) {
            this.getCollection(table.getRemoteTableName()).renameCollection(new MongoNamespace(newSchemaName, newName.getTableName()));
        }
        this.tableCache.invalidate((Object)table.getSchemaTableName());
    }

    public void addColumn(MongoTableHandle table, ColumnMetadata columnMetadata) {
        String remoteSchemaName = table.getRemoteTableName().getDatabaseName();
        String remoteTableName = table.getRemoteTableName().getCollectionName();
        Document metadata = this.getTableMetadata(remoteSchemaName, remoteTableName);
        ArrayList<Document> columns = new ArrayList<Document>(this.getColumnMetadata(metadata));
        Document newColumn = new Document();
        newColumn.append(FIELDS_NAME_KEY, (Object)columnMetadata.getName());
        newColumn.append(FIELDS_TYPE_KEY, (Object)columnMetadata.getType().getTypeSignature().toString());
        newColumn.append(COMMENT_KEY, (Object)columnMetadata.getComment());
        newColumn.append(FIELDS_HIDDEN_KEY, (Object)false);
        columns.add(newColumn);
        metadata.append(FIELDS_KEY, columns);
        MongoDatabase db = this.client.getDatabase(remoteSchemaName);
        MongoCollection schema = db.getCollection(this.schemaCollection);
        schema.findOneAndReplace((Bson)new Document(TABLE_NAME_KEY, (Object)remoteTableName), (Object)metadata);
        this.tableCache.invalidate((Object)table.getSchemaTableName());
    }

    public void renameColumn(MongoTableHandle table, String source, String target) {
        String remoteSchemaName = table.getRemoteTableName().getDatabaseName();
        String remoteTableName = table.getRemoteTableName().getCollectionName();
        Document metadata = this.getTableMetadata(remoteSchemaName, remoteTableName);
        List columns = (List)this.getColumnMetadata(metadata).stream().map(document -> {
            if (document.getString((Object)FIELDS_NAME_KEY).equals(source)) {
                document.put(FIELDS_NAME_KEY, (Object)target);
            }
            return document;
        }).collect(ImmutableList.toImmutableList());
        metadata.append(FIELDS_KEY, (Object)columns);
        MongoDatabase database = this.client.getDatabase(remoteSchemaName);
        MongoCollection schema = database.getCollection(this.schemaCollection);
        schema.findOneAndReplace((Bson)new Document(TABLE_NAME_KEY, (Object)remoteTableName), (Object)metadata);
        database.getCollection(remoteTableName).updateMany(Filters.empty(), Updates.rename((String)source, (String)target));
        this.tableCache.invalidate((Object)table.getSchemaTableName());
    }

    public void dropColumn(MongoTableHandle table, String columnName) {
        String remoteSchemaName = table.getRemoteTableName().getDatabaseName();
        String remoteTableName = table.getRemoteTableName().getCollectionName();
        Document metadata = this.getTableMetadata(remoteSchemaName, remoteTableName);
        List columns = (List)this.getColumnMetadata(metadata).stream().filter(document -> !document.getString((Object)FIELDS_NAME_KEY).equals(columnName)).collect(ImmutableList.toImmutableList());
        metadata.append(FIELDS_KEY, (Object)columns);
        MongoDatabase database = this.client.getDatabase(remoteSchemaName);
        MongoCollection schema = database.getCollection(this.schemaCollection);
        schema.findOneAndReplace((Bson)new Document(TABLE_NAME_KEY, (Object)remoteTableName), (Object)metadata);
        database.getCollection(remoteTableName).updateMany(Filters.empty(), Updates.unset((String)columnName));
        this.tableCache.invalidate((Object)table.getSchemaTableName());
    }

    public void setColumnType(MongoTableHandle table, String columnName, Type type) {
        String remoteSchemaName = table.getRemoteTableName().getDatabaseName();
        String remoteTableName = table.getRemoteTableName().getCollectionName();
        Document metadata = this.getTableMetadata(remoteSchemaName, remoteTableName);
        List columns = (List)this.getColumnMetadata(metadata).stream().map(document -> {
            if (document.getString((Object)FIELDS_NAME_KEY).equals(columnName)) {
                document.put(FIELDS_TYPE_KEY, (Object)type.getTypeSignature().toString());
                return document;
            }
            return document;
        }).collect(ImmutableList.toImmutableList());
        metadata.replace((Object)FIELDS_KEY, (Object)columns);
        this.client.getDatabase(remoteSchemaName).getCollection(this.schemaCollection).findOneAndReplace((Bson)new Document(TABLE_NAME_KEY, (Object)remoteTableName), (Object)metadata);
        this.tableCache.invalidate((Object)table.getSchemaTableName());
    }

    private MongoTable loadTableSchema(SchemaTableName schemaTableName) throws TableNotFoundException {
        RemoteTableName remoteSchemaTableName = this.toRemoteSchemaTableName(schemaTableName);
        String remoteSchemaName = remoteSchemaTableName.getDatabaseName();
        String remoteTableName = remoteSchemaTableName.getCollectionName();
        Document tableMeta = this.getTableMetadata(remoteSchemaName, remoteTableName);
        ImmutableList.Builder columnHandles = ImmutableList.builder();
        for (Document columnMetadata : this.getColumnMetadata(tableMeta)) {
            MongoColumnHandle columnHandle = this.buildColumnHandle(columnMetadata);
            columnHandles.add((Object)columnHandle);
        }
        MongoTableHandle tableHandle = new MongoTableHandle(schemaTableName, remoteSchemaTableName, Optional.empty());
        return new MongoTable(tableHandle, (List<MongoColumnHandle>)columnHandles.build(), this.getIndexes(remoteSchemaName, remoteTableName), MongoSession.getComment(tableMeta));
    }

    private MongoColumnHandle buildColumnHandle(Document columnMeta) {
        String name = columnMeta.getString((Object)FIELDS_NAME_KEY);
        String typeString = columnMeta.getString((Object)FIELDS_TYPE_KEY);
        boolean hidden = columnMeta.getBoolean((Object)FIELDS_HIDDEN_KEY, false);
        String comment = columnMeta.getString((Object)COMMENT_KEY);
        Type type = this.typeManager.fromSqlType(typeString);
        return new MongoColumnHandle(name, (List<String>)ImmutableList.of(), type, hidden, false, Optional.ofNullable(comment));
    }

    private List<Document> getColumnMetadata(Document doc) {
        if (!doc.containsKey((Object)FIELDS_KEY)) {
            return ImmutableList.of();
        }
        return (List)doc.get((Object)FIELDS_KEY);
    }

    private static Optional<String> getComment(Document doc) {
        return Optional.ofNullable(doc.getString((Object)COMMENT_KEY));
    }

    public MongoCollection<Document> getCollection(RemoteTableName remoteTableName) {
        return this.client.getDatabase(remoteTableName.getDatabaseName()).getCollection(remoteTableName.getCollectionName());
    }

    public List<MongoIndex> getIndexes(String schemaName, String tableName) {
        if (this.isView(schemaName, tableName)) {
            return ImmutableList.of();
        }
        MongoCollection collection = this.client.getDatabase(schemaName).getCollection(tableName);
        return MongoIndex.parse((ListIndexesIterable<Document>)collection.listIndexes());
    }

    public long deleteDocuments(RemoteTableName remoteTableName, TupleDomain<ColumnHandle> constraint) {
        Document filter = MongoSession.buildQuery(constraint);
        log.debug("Delete documents: collection: %s, filter: %s", new Object[]{remoteTableName, filter});
        DeleteResult result = this.getCollection(remoteTableName).deleteMany((Bson)filter);
        return result.getDeletedCount();
    }

    public MongoCursor<Document> execute(MongoTableHandle tableHandle, List<MongoColumnHandle> columns) {
        Set<MongoColumnHandle> projectedColumns = tableHandle.getProjectedColumns();
        Preconditions.checkArgument((projectedColumns.isEmpty() || projectedColumns.containsAll(columns) ? 1 : 0) != 0, (Object)"projectedColumns must be empty or equal to columns");
        Document projection = MongoSession.buildProjection(columns);
        MongoCollection<Document> collection = this.getCollection(tableHandle.getRemoteTableName());
        Document filter = MongoSession.buildFilter(tableHandle);
        FindIterable iterable = collection.find((Bson)filter).projection((Bson)projection).collation(SIMPLE_COLLATION);
        tableHandle.getLimit().ifPresent(arg_0 -> ((FindIterable)iterable).limit(arg_0));
        log.debug("Find documents: collection: %s, filter: %s, projection: %s", new Object[]{tableHandle.getSchemaTableName(), filter, projection});
        if (this.cursorBatchSize != 0) {
            iterable.batchSize(this.cursorBatchSize);
        }
        return iterable.iterator();
    }

    @VisibleForTesting
    static Document buildProjection(List<MongoColumnHandle> columns) {
        Document output = new Document();
        output.append("_id", (Object)0);
        for (MongoColumnHandle column : MongoSession.projectSufficientColumns(columns)) {
            output.append(column.getQualifiedName(), (Object)1);
        }
        return output;
    }

    public static List<MongoColumnHandle> projectSufficientColumns(List<MongoColumnHandle> columnHandles) {
        List sortedColumnHandles = COLUMN_HANDLE_ORDERING.sortedCopy(columnHandles);
        ArrayList<MongoColumnHandle> sufficientColumns = new ArrayList<MongoColumnHandle>();
        for (MongoColumnHandle column : sortedColumnHandles) {
            if (MongoSession.parentColumnExists(sufficientColumns, column)) continue;
            sufficientColumns.add(column);
        }
        return sufficientColumns;
    }

    private static boolean parentColumnExists(List<MongoColumnHandle> existingColumns, MongoColumnHandle column) {
        for (MongoColumnHandle existingColumn : existingColumns) {
            List<String> existingColumnDereferenceNames = existingColumn.getDereferenceNames();
            Verify.verify((column.getDereferenceNames().size() >= existingColumnDereferenceNames.size() ? 1 : 0) != 0, (String)"Selected column's dereference size must be greater than or equal to the existing column's dereference size", (Object[])new Object[0]);
            if (!existingColumn.getBaseName().equals(column.getBaseName()) || !column.getDereferenceNames().subList(0, existingColumnDereferenceNames.size()).equals(existingColumnDereferenceNames)) continue;
            return true;
        }
        return false;
    }

    static Document buildFilter(MongoTableHandle table) {
        ImmutableList.Builder filter = ImmutableList.builder();
        table.getFilter().ifPresent(json -> filter.add((Object)Query.parseFilter(json)));
        filter.add((Object)MongoSession.buildQuery(table.getConstraint()));
        return MongoSession.andPredicate((List<Document>)filter.build());
    }

    @VisibleForTesting
    static Document buildQuery(TupleDomain<ColumnHandle> tupleDomain) {
        Document query = new Document();
        if (tupleDomain.getDomains().isPresent()) {
            for (Map.Entry entry : ((Map)tupleDomain.getDomains().get()).entrySet()) {
                MongoColumnHandle column = (MongoColumnHandle)entry.getKey();
                Optional<Document> predicate = MongoSession.buildPredicate(column, (Domain)entry.getValue());
                predicate.ifPresent(arg_0 -> ((Document)query).putAll(arg_0));
            }
        }
        return query;
    }

    private static Optional<Document> buildPredicate(MongoColumnHandle column, Domain domain) {
        String name = column.getQualifiedName();
        Type type = column.getType();
        if (domain.getValues().isNone() && domain.isNullAllowed()) {
            return Optional.of(MongoSession.documentOf(name, MongoSession.isNullPredicate()));
        }
        if (domain.getValues().isAll() && !domain.isNullAllowed()) {
            return Optional.of(MongoSession.documentOf(name, MongoSession.isNotNullPredicate()));
        }
        ArrayList<Object> singleValues = new ArrayList<Object>();
        ArrayList<Document> disjuncts = new ArrayList<Document>();
        for (Range range : domain.getValues().getRanges().getOrderedRanges()) {
            Optional<Object> translated;
            if (range.isSingleValue()) {
                Optional<Object> translated2 = MongoSession.translateValue(range.getSingleValue(), type);
                if (translated2.isEmpty()) {
                    return Optional.empty();
                }
                singleValues.add(translated2.get());
                continue;
            }
            Document rangeConjuncts = new Document();
            if (!range.isLowUnbounded()) {
                translated = MongoSession.translateValue(range.getLowBoundedValue(), type);
                if (translated.isEmpty()) {
                    return Optional.empty();
                }
                rangeConjuncts.put(range.isLowInclusive() ? GTE_OP : GT_OP, translated.get());
            }
            if (!range.isHighUnbounded()) {
                translated = MongoSession.translateValue(range.getHighBoundedValue(), type);
                if (translated.isEmpty()) {
                    return Optional.empty();
                }
                rangeConjuncts.put(range.isHighInclusive() ? LTE_OP : LT_OP, translated.get());
            }
            Verify.verify((!rangeConjuncts.isEmpty() ? 1 : 0) != 0);
            disjuncts.add(rangeConjuncts);
        }
        if (singleValues.size() == 1) {
            disjuncts.add(MongoSession.documentOf(EQ_OP, singleValues.get(0)));
        } else if (singleValues.size() > 1) {
            disjuncts.add(MongoSession.documentOf(IN_OP, singleValues));
        }
        if (domain.isNullAllowed()) {
            disjuncts.add(MongoSession.isNullPredicate());
        }
        return Optional.of(MongoSession.orPredicate((List)disjuncts.stream().map(disjunct -> new Document(name, disjunct)).collect(ImmutableList.toImmutableList())));
    }

    private static Optional<Object> translateValue(Object trinoNativeValue, Type type) {
        Objects.requireNonNull(trinoNativeValue, "trinoNativeValue is null");
        Objects.requireNonNull(type, "type is null");
        Preconditions.checkArgument((boolean)Primitives.wrap((Class)type.getJavaType()).isInstance(trinoNativeValue), (String)"%s (%s) is not a valid representation for %s", (Object)trinoNativeValue, trinoNativeValue.getClass(), (Object)type);
        if (type == BooleanType.BOOLEAN) {
            return Optional.of(trinoNativeValue);
        }
        if (type == TinyintType.TINYINT) {
            return Optional.of(Long.valueOf(SignedBytes.checkedCast((long)((Long)trinoNativeValue))));
        }
        if (type == SmallintType.SMALLINT) {
            return Optional.of(Long.valueOf(Shorts.checkedCast((long)((Long)trinoNativeValue))));
        }
        if (type == IntegerType.INTEGER) {
            return Optional.of(Long.valueOf(Math.toIntExact((Long)trinoNativeValue)));
        }
        if (type == BigintType.BIGINT) {
            return Optional.of(trinoNativeValue);
        }
        if (type instanceof DecimalType) {
            DecimalType decimalType = (DecimalType)type;
            if (decimalType.isShort()) {
                return Optional.of(Decimal128.parse((String)Decimals.toString((long)((Long)trinoNativeValue), (int)decimalType.getScale())));
            }
            return Optional.of(Decimal128.parse((String)Decimals.toString((Int128)((Int128)trinoNativeValue), (int)decimalType.getScale())));
        }
        if (type instanceof ObjectIdType) {
            return Optional.of(new ObjectId(((Slice)trinoNativeValue).getBytes()));
        }
        if (type instanceof CharType) {
            CharType charType = (CharType)type;
            Slice slice = Chars.padSpaces((Slice)((Slice)trinoNativeValue), (CharType)charType);
            return Optional.of(slice.toStringUtf8());
        }
        if (type instanceof VarcharType) {
            return Optional.of(((Slice)trinoNativeValue).toStringUtf8());
        }
        if (type == DateType.DATE) {
            long days = (Long)trinoNativeValue;
            return Optional.of(LocalDate.ofEpochDay(days));
        }
        if (type == TimeType.TIME_MILLIS) {
            long picos = (Long)trinoNativeValue;
            return Optional.of(LocalTime.ofNanoOfDay(Timestamps.roundDiv((long)picos, (long)1000L)));
        }
        if (type == TimestampType.TIMESTAMP_MILLIS) {
            long epochMicros = (Long)trinoNativeValue;
            long epochSecond = Math.floorDiv(epochMicros, 1000000);
            int nanoFraction = Math.floorMod(epochMicros, 1000000) * 1000;
            Instant instant = Instant.ofEpochSecond(epochSecond, nanoFraction);
            return Optional.of(LocalDateTime.ofInstant(instant, ZoneOffset.UTC));
        }
        if (type == TimestampWithTimeZoneType.TIMESTAMP_TZ_MILLIS) {
            long millisUtc = DateTimeEncoding.unpackMillisUtc((long)((Long)trinoNativeValue));
            Instant instant = Instant.ofEpochMilli(millisUtc);
            return Optional.of(LocalDateTime.ofInstant(instant, ZoneOffset.UTC));
        }
        return Optional.empty();
    }

    private static Document documentOf(String key, Object value) {
        return new Document(key, value);
    }

    private static Document orPredicate(List<Document> values) {
        Preconditions.checkState((!values.isEmpty() ? 1 : 0) != 0);
        if (values.size() == 1) {
            return values.get(0);
        }
        return new Document(OR_OP, values);
    }

    private static Document andPredicate(List<Document> values) {
        Preconditions.checkState((!values.isEmpty() ? 1 : 0) != 0);
        if (values.size() == 1) {
            return values.get(0);
        }
        return new Document(AND_OP, values);
    }

    private static Document isNullPredicate() {
        return MongoSession.documentOf(EQ_OP, null);
    }

    private static Document isNotNullPredicate() {
        return MongoSession.documentOf(NOT_EQ_OP, null);
    }

    private Document getTableMetadata(String schemaName, String tableName) throws TableNotFoundException {
        MongoDatabase db = this.client.getDatabase(schemaName);
        MongoCollection schema = db.getCollection(this.schemaCollection);
        Document doc = (Document)schema.find((Bson)new Document(TABLE_NAME_KEY, (Object)tableName)).first();
        if (doc == null) {
            if (!this.collectionExists(db, tableName)) {
                throw new TableNotFoundException(new SchemaTableName(schemaName, tableName), String.format("Table '%s.%s' not found", schemaName, tableName), null);
            }
            Document metadata = new Document(TABLE_NAME_KEY, (Object)tableName);
            metadata.append(FIELDS_KEY, this.guessTableFields(schemaName, tableName));
            if (!this.indexExists((MongoCollection<Document>)schema)) {
                schema.createIndex((Bson)new Document(TABLE_NAME_KEY, (Object)1), new IndexOptions().unique(true));
            }
            schema.insertOne((Object)metadata);
            return metadata;
        }
        return doc;
    }

    public boolean collectionExists(MongoDatabase db, String collectionName) {
        for (String name : this.listCollectionNames(db.getName())) {
            if (!name.equalsIgnoreCase(collectionName)) continue;
            return true;
        }
        return false;
    }

    private boolean indexExists(MongoCollection<Document> schemaCollection) {
        return MongoIndex.parse((ListIndexesIterable<Document>)schemaCollection.listIndexes()).stream().anyMatch(index -> index.getKeys().size() == 1 && TABLE_NAME_KEY.equals(index.getKeys().get(0).getName()));
    }

    private Set<String> getTableMetadataNames(String schemaName) {
        try (MongoCursor cursor = this.client.getDatabase(schemaName).getCollection(this.schemaCollection).find().projection((Bson)new Document(TABLE_NAME_KEY, (Object)true)).iterator();){
            Set set = (Set)Streams.stream((Iterator)cursor).map(document -> document.getString((Object)TABLE_NAME_KEY)).collect(ImmutableSet.toImmutableSet());
            return set;
        }
    }

    private void createTableMetadata(RemoteTableName remoteSchemaTableName, List<MongoColumnHandle> columns, Optional<String> tableComment) {
        String remoteSchemaName = remoteSchemaTableName.getDatabaseName();
        String remoteTableName = remoteSchemaTableName.getCollectionName();
        MongoDatabase db = this.client.getDatabase(remoteSchemaName);
        Document metadata = new Document(TABLE_NAME_KEY, (Object)remoteTableName);
        ArrayList<Document> fields = new ArrayList<Document>();
        if (!columns.stream().anyMatch(c -> c.getBaseName().equals("_id"))) {
            fields.add(new MongoColumnHandle("_id", (List<String>)ImmutableList.of(), (Type)ObjectIdType.OBJECT_ID, true, false, Optional.empty()).getDocument());
        }
        fields.addAll(columns.stream().map(MongoColumnHandle::getDocument).collect(Collectors.toList()));
        metadata.append(FIELDS_KEY, fields);
        tableComment.ifPresent(comment -> metadata.append(COMMENT_KEY, comment));
        MongoCollection schema = db.getCollection(this.schemaCollection);
        if (!this.indexExists((MongoCollection<Document>)schema)) {
            schema.createIndex((Bson)new Document(TABLE_NAME_KEY, (Object)1), new IndexOptions().unique(true));
        }
        schema.insertOne((Object)metadata);
    }

    private boolean deleteTableMetadata(RemoteTableName remoteTableName) {
        MongoDatabase db = this.client.getDatabase(remoteTableName.getDatabaseName());
        if (!this.collectionExists(db, remoteTableName.getCollectionName()) && ((Document)db.getCollection(this.schemaCollection).find((Bson)new Document(TABLE_NAME_KEY, (Object)remoteTableName.getCollectionName())).first()).isEmpty()) {
            return false;
        }
        DeleteResult result = db.getCollection(this.schemaCollection).deleteOne((Bson)new Document(TABLE_NAME_KEY, (Object)remoteTableName.getCollectionName()));
        return result.getDeletedCount() == 1L;
    }

    private List<Document> guessTableFields(String schemaName, String tableName) {
        MongoDatabase db = this.client.getDatabase(schemaName);
        Document doc = (Document)db.getCollection(tableName).find().first();
        if (doc == null) {
            return ImmutableList.of();
        }
        ImmutableList.Builder builder = ImmutableList.builder();
        for (String key : doc.keySet()) {
            Object value = doc.get((Object)key);
            Optional<TypeSignature> fieldType = this.guessFieldType(value);
            if (fieldType.isPresent()) {
                Document metadata = new Document();
                metadata.append(FIELDS_NAME_KEY, (Object)key);
                metadata.append(FIELDS_TYPE_KEY, (Object)fieldType.get().toString());
                metadata.append(FIELDS_HIDDEN_KEY, (Object)(key.equals("_id") && fieldType.get().equals((Object)ObjectIdType.OBJECT_ID.getTypeSignature()) ? 1 : 0));
                builder.add((Object)metadata);
                continue;
            }
            log.debug("Unable to guess field type from %s : %s", new Object[]{value == null ? "null" : value.getClass().getName(), value});
        }
        return builder.build();
    }

    private Optional<TypeSignature> guessFieldType(Object value) {
        if (value == null) {
            return Optional.empty();
        }
        TypeSignature typeSignature = null;
        if (value instanceof String) {
            typeSignature = VarcharType.createUnboundedVarcharType().getTypeSignature();
        }
        if (value instanceof Binary) {
            typeSignature = VarbinaryType.VARBINARY.getTypeSignature();
        } else if (value instanceof Integer || value instanceof Long) {
            typeSignature = BigintType.BIGINT.getTypeSignature();
        } else if (value instanceof Boolean) {
            typeSignature = BooleanType.BOOLEAN.getTypeSignature();
        } else if (value instanceof Float || value instanceof Double) {
            typeSignature = DoubleType.DOUBLE.getTypeSignature();
        } else if (value instanceof Decimal128) {
            BigDecimal decimal;
            Decimal128 decimal128 = (Decimal128)value;
            try {
                decimal = decimal128.bigDecimalValue();
            }
            catch (ArithmeticException e) {
                return Optional.empty();
            }
            typeSignature = DecimalType.createDecimalType((int)decimal.precision(), (int)decimal.scale()).getTypeSignature();
        } else if (value instanceof Date) {
            typeSignature = TimestampType.TIMESTAMP_MILLIS.getTypeSignature();
        } else if (value instanceof ObjectId) {
            typeSignature = ObjectIdType.OBJECT_ID.getTypeSignature();
        } else if (value instanceof List) {
            List subTypes = ((List)value).stream().map(this::guessFieldType).collect(Collectors.toList());
            if (subTypes.isEmpty() || subTypes.stream().anyMatch(Optional::isEmpty)) {
                return Optional.empty();
            }
            Set signatures = subTypes.stream().map(Optional::get).collect(Collectors.toSet());
            typeSignature = signatures.size() == 1 ? new TypeSignature("array", signatures.stream().map(TypeSignatureParameter::typeParameter).collect(Collectors.toList())) : new TypeSignature("row", IntStream.range(0, subTypes.size()).mapToObj(idx -> TypeSignatureParameter.namedTypeParameter((NamedTypeSignature)new NamedTypeSignature(Optional.of(new RowFieldName(String.format("%s%d", this.implicitPrefix, idx + 1))), (TypeSignature)((Optional)subTypes.get(idx)).get()))).collect(Collectors.toList()));
        } else if (value instanceof Document) {
            ArrayList<TypeSignatureParameter> parameters = new ArrayList<TypeSignatureParameter>();
            for (String key : ((Document)value).keySet()) {
                Optional<TypeSignature> fieldType = this.guessFieldType(((Document)value).get((Object)key));
                if (!fieldType.isPresent()) continue;
                parameters.add(TypeSignatureParameter.namedTypeParameter((NamedTypeSignature)new NamedTypeSignature(Optional.of(new RowFieldName(key)), fieldType.get())));
            }
            if (!parameters.isEmpty()) {
                typeSignature = new TypeSignature("row", parameters);
            }
        } else if (value instanceof DBRef) {
            ArrayList<TypeSignatureParameter> parameters = new ArrayList<TypeSignatureParameter>();
            TypeSignature idFieldType = this.guessFieldType(((DBRef)value).getId()).orElseThrow(() -> new UnsupportedOperationException("Unable to guess $id field type of DBRef from: " + ((DBRef)value).getId()));
            parameters.add(TypeSignatureParameter.namedTypeParameter((NamedTypeSignature)new NamedTypeSignature(Optional.of(new RowFieldName(DATABASE_NAME)), VarcharType.VARCHAR.getTypeSignature())));
            parameters.add(TypeSignatureParameter.namedTypeParameter((NamedTypeSignature)new NamedTypeSignature(Optional.of(new RowFieldName(COLLECTION_NAME)), VarcharType.VARCHAR.getTypeSignature())));
            parameters.add(TypeSignatureParameter.namedTypeParameter((NamedTypeSignature)new NamedTypeSignature(Optional.of(new RowFieldName(ID)), idFieldType)));
            typeSignature = new TypeSignature("row", parameters);
        }
        return Optional.ofNullable(typeSignature);
    }

    public RemoteTableName toRemoteSchemaTableName(SchemaTableName schemaTableName) {
        String remoteSchemaName = this.toRemoteSchemaName(schemaTableName.getSchemaName());
        String remoteTableName = this.toRemoteTableName(remoteSchemaName, schemaTableName.getTableName());
        return new RemoteTableName(remoteSchemaName, remoteTableName);
    }

    private String toRemoteSchemaName(String schemaName) {
        Verify.verify((boolean)schemaName.equals(schemaName.toLowerCase(Locale.ENGLISH)), (String)"schemaName not in lower-case: %s", (Object)schemaName);
        if (!this.caseInsensitiveNameMatching) {
            return schemaName;
        }
        if (SYSTEM_DATABASES.contains(schemaName)) {
            return schemaName;
        }
        for (String remoteSchemaName : this.listDatabaseNames()) {
            if (!schemaName.equals(remoteSchemaName.toLowerCase(Locale.ENGLISH))) continue;
            return remoteSchemaName;
        }
        return schemaName;
    }

    private MongoIterable<String> listDatabaseNames() {
        return this.client.listDatabases().nameOnly(Boolean.valueOf(true)).authorizedDatabasesOnly(Boolean.valueOf(true)).map(result -> result.getString((Object)FIELDS_NAME_KEY));
    }

    private String toRemoteTableName(String schemaName, String tableName) {
        Verify.verify((boolean)tableName.equals(tableName.toLowerCase(Locale.ENGLISH)), (String)"tableName not in lower-case: %s", (Object)tableName);
        if (!this.caseInsensitiveNameMatching) {
            return tableName;
        }
        for (String remoteTableName : this.listCollectionNames(schemaName)) {
            if (!tableName.equals(remoteTableName.toLowerCase(Locale.ENGLISH))) continue;
            return remoteTableName;
        }
        return tableName;
    }

    private List<String> listCollectionNames(String databaseName) {
        MongoDatabase database = this.client.getDatabase(databaseName);
        Document cursor = (Document)database.runCommand((Bson)new Document(AUTHORIZED_LIST_COLLECTIONS_COMMAND)).get((Object)"cursor", Document.class);
        List firstBatch = (List)cursor.get((Object)"firstBatch", List.class);
        return (List)firstBatch.stream().map(document -> document.getString((Object)FIELDS_NAME_KEY)).collect(ImmutableList.toImmutableList());
    }

    private boolean isView(String schemaName, String tableName) {
        Document listCollectionsCommand = new Document((Map)ImmutableMap.builder().put((Object)"listCollections", (Object)1.0).put((Object)"filter", (Object)MongoSession.documentOf(FIELDS_NAME_KEY, tableName)).put((Object)"nameOnly", (Object)true).put((Object)"authorizedCollections", (Object)true).buildOrThrow());
        Document cursor = (Document)this.client.getDatabase(schemaName).runCommand((Bson)listCollectionsCommand).get((Object)"cursor", Document.class);
        List firstBatch = (List)cursor.get((Object)"firstBatch", List.class);
        if (firstBatch.isEmpty()) {
            return false;
        }
        String type = ((Document)firstBatch.get(0)).getString((Object)FIELDS_TYPE_KEY);
        return "view".equals(type);
    }
}

